/******************************************************************
 * @file   rs485_bus.c
 * @author Richard Luo
 * @date   2008-11-07
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include <linux/clk.h>
#include "akdebug.h"
#include "rs485_bus.h"
#include "fntnet.h"
#include "token.h"

static int g_baud = 0;
module_param(g_baud, int, 0);

static int g_cd = 0;
module_param(g_cd, int, 0);

static int g_div = 1;
module_param(g_div, int, 0);

static int g_endtx_times = 0;
static int g_txbufe_times = 0;
static int g_tx_chars = 0;
static int g_rxbrk_times = 0;
static int g_rx_pare_time = 0;
static int g_misc_interrupt = 0;
static int g_timeout = 0;
static int g_rx_chars = 0;
static int g_rx_frames = 0;
static int g_illegal_rx_frames = 0;
static int g_illegal_head = 0;
static int g_illegal_checksum = 0;
static int g_tx_deq_failed_times = 0;
static int g_tx_frames = 0;
static int g_illegal_rx_count = 0;
static int g_rx_count_diff = 0;
static int g_endrx = 0;

/* Use device name ttyS, major 4, minor 64-68.  This is the usual serial port
 * name, but it is legally reserved for the 8250 driver. */
#define PDC_RX_BUF(port)	&(port)->pdc_rx[(port)->pdc_rx_idx]
#define PDC_RX_SWITCH(port)	(port)->pdc_rx_idx = !(port)->pdc_rx_idx

/* Flags for multidrop mode */
#define USART_MULTI_DROP_MODE   0x01    /* USART in Multidrop Mode */
#define USART_MULTI_DROP_DATA   0X02    /* USART Receive data in Multidrop mode */

static struct rs485_priv *hw_obj;

static void port_clear_timeout (struct uart_port *port);
static void port_set_timeout (struct uart_port *port, u32 bits, u32 type);
static void rs485_priv_stop_tx ( struct rs485_priv *ppriv );
static void rs485_priv_start_rx ( struct rs485_priv *ppriv );

void test_enable_retto (void)
{
    UART_PUT_IER(&hw_obj->uart, ATMEL_US_TIMEOUT);
}

/* static void rs485_priv_stop_tx ( struct rs485_priv *ppriv ); */
/* static void rs485_priv_stop_rx ( struct rs485_priv *ppriv ); */

static ssize_t rs485_misc_dump (struct rs485_priv *ppriv, char *buf)
{
    char *p = buf;
//    p += sprintf(p, "token_status:%08x, cur_owner:%02x, self_addr:%02x\n",
//                 ppriv->token_status, ppriv->cur_owner, ppriv->self_addr);
//    p += sprintf(p, "g_baud:%d:%08x \n", g_baud, g_baud);
    p += sprintf(p, "==== g_tx_chars:%d bytes \n", g_tx_chars);
    p += sprintf(p, "g_txbufe_times:%d \n", g_txbufe_times);
    p += sprintf(p, "g_endtx_times:%d \n", g_endtx_times);
    p += sprintf(p, "g_rxbrk_times:%d \n", g_rxbrk_times);
    p += sprintf(p, "g_rx_pare_time:%d \n", g_rx_pare_time);
    p += sprintf(p, "g_misc_interrupt:%d \n", g_misc_interrupt);
    p += sprintf(p, "g_timeout:%d \n", g_timeout);
    p += sprintf(p, "==== g_rx_chars:%d \n", g_rx_chars);
    p += sprintf(p, "==== g_rx_frames:%d \n", g_rx_frames);
    p += sprintf(p, "g_illegal_rx_frames:%d \n", g_illegal_rx_frames);
    p += sprintf(p, "g_illegal_head:%d \n", g_illegal_head);
    p += sprintf(p, "g_illegal_checksum:%d \n", g_illegal_checksum);
    p += sprintf(p, "g_tx_deq_failed_times:%d \n", g_tx_deq_failed_times);
    p += sprintf(p, "==== g_tx_frames:%d \n", g_tx_frames);
    p += sprintf(p, "g_illegal_rx_count:%d \n", g_illegal_rx_count);
    p += sprintf(p, "g_rx_count_diff:%d \n", g_rx_count_diff);
    p += sprintf(p, "g_endrx:%d \n", g_endrx);
    return p - buf;
}

/* #define DEVICE_ATTR(_name,_mode,_show,_store) \ */
/* struct device_attribute dev_attr_##_name = __ATTR(_name,_mode,_show,_store) */

#define PRINT_REGS(addr, strname)  do {                         \
        unsigned mode = __raw_readl((port)->membase + addr);    \
        printk("%-20s:%8.8x\n", strname, mode);                 \
    } while (0)

#define FNTNET_SERIAL_DUMP_REGS(addr, strname)  do {            \
        unsigned mode = __raw_readl((port)->membase + addr);    \
        pbuf += sprintf(pbuf, "%-20s:%8.8x\n", strname, mode);  \
    } while (0)

static ssize_t usart_port_dump (struct uart_port *port, char *buf)
{
    char *pbuf = buf;
    if (port) {
        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_MR, "ATMEL_US_MR");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_IMR, "ATMEL_US_IMR");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_CSR, "ATMEL_US_CSR");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_RHR, "ATMEL_US_RHR");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_BRGR, "ATMEL_US_BRGR");
//        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_RTOR, "ATMEL_US_RTOR");
//        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_TTGR, "ATMEL_US_TTGR");
//        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_FIDI, "ATMEL_US_FIDI");
//        FNTNET_SERIAL_DUMP_REGS(ATMEL_US_NER, "ATMEL_US_NER");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_PDC_PTSR, "ATMEL_PDC_PTSR");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_PDC_TCR, "ATMEL_PDC_TCR");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_PDC_RPR, "ATMEL_PDC_RPR");
        FNTNET_SERIAL_DUMP_REGS(ATMEL_PDC_RCR, "ATMEL_PDC_RCR");

    } else
        pbuf += sprintf(pbuf, "port is NULL \n");
    return (pbuf - buf);
}

static ssize_t atmel_clk_dump (struct clk *pck, char *buf)
{
    char *p = buf;
    unsigned long hz = clk_get_rate(pck);
    p += sprintf(buf, "clock:[%ld K]:[%ld M] \n", (hz >> 10), (hz >> 20) );
    return p - buf;
}

static ssize_t rs485_dump_pdc (const char *prefix, struct atmel_dma_buffer *pdc, char *buf)
{
    char *p = buf;
    p += sprintf(p, "%s \n", prefix);
    p += sprintf(p, "buf: %p \n", pdc->buf);
    p += sprintf(p, "dma_addr: %08x \n", pdc->dma_addr);
    p += sprintf(p, "dma_size: %08x \n", pdc->dma_size);
    p += sprintf(p, "ofs: %08x \n", pdc->ofs);
    return p - buf;
}

static struct sk_buff *debug_tx_skb;
static struct sk_buff *debug_rx_skb;
static struct circ_buf *debug_cir_buf;

void dump_debug_cir_buf (void)
{
    int n, m = 0;
    char buf[16];

    while( (n = serial_buf_get(debug_cir_buf, buf, sizeof(buf)) ) > 0 ) {
        show_bytes("", buf, n);
        m += n;
    }
    if (m == 0)
        PDEBUG("empty circ_buf \n");
}


static ssize_t fntnet_dump_debug_skb (char *buf)
{
    char *p = buf;
    p += show_bytes_dump("fntnet_dump_debug_tx_skb:", debug_tx_skb->data, debug_tx_skb->len, p);
    p += show_bytes_dump("fntnet_dump_debug_rx_skb:", debug_rx_skb->data, debug_rx_skb->len, p);
    return p - buf;
}


static ssize_t rs485_priv_dump2buf (struct rs485_priv *ppriv, char *buf, size_t bufsz)
{
    char *p = buf;
    p += usart_port_dump(&ppriv->uart, p);
    p += atmel_clk_dump(ppriv->clk, p);
    p += rs485_misc_dump(ppriv, p);

    /* p += rs485_dump_pdc("pdc of rx[0]:", &ppriv->pdc_rx[0], p); */
    /* p += rs485_dump_pdc("pdc of rx[1]:", &ppriv->pdc_rx[1], p); */

    p += fntnet_dump_debug_skb(p);

    return (p - buf);
}


static ssize_t fntnet_dump_rs485_state (struct device *dev, struct device_attribute *attr, char *buf)
{
    return rs485_priv_dump2buf(hw_obj, buf, 4096);
}


DEVICE_ATTR(fntnet_debug_usart, S_IRUGO | S_IWUSR, fntnet_dump_rs485_state, NULL);

static int fntnet_create_usart_debug_file (struct platform_device *pdev)
{
    BUG_ON(!pdev);

    if (!debug_tx_skb)
        debug_tx_skb = dev_alloc_skb(FNET_MTU);

    if (!debug_rx_skb)
        debug_rx_skb = dev_alloc_skb(FNET_MTU);

    if (!debug_cir_buf)
        debug_cir_buf = serial_buf_alloc();

	return device_create_file(&pdev->dev, &dev_attr_fntnet_debug_usart);
}

static inline unsigned char *skb_goahead(struct sk_buff *skb, unsigned int len)
{
    unsigned char *tmp = skb_tail_pointer(skb);
	skb->tail += len;
	skb->len -= len;
	return tmp;
}


/** 
 * 
 * 
 * @param ph 
 * 
 * @return -1 on error, 0 on success
 */
static int validate_checksum (struct token_head *ph)
{
    u16 check = ip_fast_csum((unsigned char *)ph, sizeof(*ph) );
//    return (check == 0xFFFF) ? 0 : -1;
    return 0;
}

/** 
 * 
 * 
 * @param start 
 * @param size 
 * 
 * @return index -- start[index] is the start point of frame, else -1 on error.
 */
static int find_frame_head (const char *start)
{
    int ret = -1;
    const char *pm = start;
    int max_idx = 10;
    for (pm = memchr(pm, 'k', max_idx);
         pm;)  {
        if (pm[1] == 'y') {
            ret = (pm - start);
            break;
        }
        ++pm;
        max_idx -= (pm - start);
        if (max_idx <= 0) break;
        pm = memchr(pm, 'k', max_idx);        
    }

    return ret;
}


static void rs485_pdc_frame_end (struct rs485_priv *ppriv)
{
    extern struct net_device *fntnet_devs;
    
    struct uart_port *port = &ppriv->uart;
#ifdef USE_SINGLE_RX_DMA_BUF
	struct atmel_dma_buffer *pdc = PDC_RX_BUF(ppriv);
#else
	struct atmel_dma_buffer *pdc = &ppriv->pdc_rx[0];
#endif    
	unsigned short count = UART_GET_RPR(port) - pdc->dma_addr;	/* current DMA adress */
    struct sk_buff *skb;
    struct token_head *phead;

    BUG_ON(!port->dev);
    BUG_ON(!pdc);
    BUG_ON(!pdc->dma_addr);

    dma_sync_single_for_cpu(port->dev, pdc->dma_addr, pdc->dma_size, DMA_FROM_DEVICE);

    if ( unlikely(count+2 > FNET_MTU) ||
         unlikely(count < sizeof(struct token_head)) ) {
        PDEBUG("!!!! count:%d \n", count);
        ++g_illegal_rx_count;
        if (count > 0)
            show_bytes("ill count", pdc->buf, count < 30 ? count : 30);

        goto set_nex_dma;
    }

#ifdef TEST_TRANSFER_ERRORS_PERC
    {
        int s = find_frame_head(pdc->buf);
        if (-1 == s) {
            g_illegal_head++;
            g_illegal_rx_frames++;
            show_bytes("ill head", pdc->buf, count < 30 ? count : 30);
            goto set_nex_dma;
        }
        phead = (struct token_head*) &pdc->buf[s];
        if (-1 == validate_checksum(phead) ) {
            g_illegal_checksum++;
            g_illegal_rx_frames++;
            goto set_nex_dma;
        }
        count -= s;
        if (count < sizeof(*phead)) {
            PDEBUG("error count value:%d \n", count);
            goto set_nex_dma;                
        }

        g_rx_count_diff += (count - sizeof(*phead));
        count = sizeof(*phead);
    }
#endif

    skb = dev_alloc_skb(count + 2);
    if (skb != NULL) {
        skb->dev = fntnet_devs;
        memcpy(skb_put(skb, count), phead, count);

#ifdef DEBUG
        {
            static int g_ri = 0;
            if (!g_ri && count > 10) {    /* only copy the first packet */
                memcpy(skb_put(debug_rx_skb, count), skb->data, count);
                g_ri = 1;
            }

            serial_buf_put(debug_cir_buf, skb->data, count);
        }
#endif            
        skb->protocol = eth_type_trans(skb, skb->dev);
        fntnet_devs->last_rx = jiffies;
        netif_rx(skb);
        g_rx_chars += count;
        ++g_rx_frames;
    }
set_nex_dma:

    UART_PUT_CR(port, ATMEL_US_RSTSTA);	/* clear error */

#ifdef USE_SINGLE_RX_DMA_BUF
    UART_PUT_RPR(port, pdc->dma_addr);
	UART_PUT_RCR(port, pdc->dma_size);
#else    
	UART_PUT_RNPR(port, pdc->dma_addr);
	UART_PUT_RNCR(port, pdc->dma_size);
	/* Switch to next buffer */
	PDC_RX_SWITCH(ppriv);		/* next PDC buffer */
#endif
    
//    UART_PUT_PTCR(port, ATMEL_PDC_RXTEN);
}

static void rs485_pdc_endtx (struct rs485_priv *ppriv)
{
	struct uart_port *port = &ppriv->uart;
	struct atmel_dma_buffer *pdc = &ppriv->pdc_tx;
	struct sk_buff *skb;

    if (ppriv->data_sent) {  /* if not the first time, need to send a break */

#ifdef USE_BRK_TO_MAKE_FRAME
        /* send a fast break, may cause bug ?? -- need testing */
		UART_PUT_CR(port, ATMEL_US_STTBRK);	/* start break */
        ppriv->break_sent = 1;
        port_set_timeout(port, 16, ATMEL_US_RETTO);
        ppriv->data_sent = 0;
#elif defined USE_MULTIDROP_TO_MAKE_FRAME
        pdc->buf[0] = 0xEC;
        pdc->buf[1] = 0xa1;
        pdc->buf[2] = 0xb2;
        pdc->buf[3] = 0xc3;
        pdc->buf[4] = 0xd4;
        pdc->dma_size = 100;
        pdc->dma_addr = dma_map_single(port->dev, pdc->buf, pdc->dma_size, DMA_TO_DEVICE);
        UART_PUT_CR(port, ATMEL_US_SENDA);
        UART_PUT_TPR(port, pdc->dma_addr);
        UART_PUT_TCR(port, pdc->dma_size);
        UART_PUT_PTCR(port, ATMEL_PDC_TXTEN); /* re-enable PDC transmit */
        ppriv->data_sent = 0;
        g_tx_frames++;
        return;
#elif defined USE_TIMEOUT_TO_MAKE_FRAME        
        rs485_priv_stop_tx(ppriv);
        port_set_timeout(port, 80, ATMEL_US_RETTO);
        ppriv->tx_timeout = 1;
        PDEBUG("stop tx, start to \n");
        return;
#endif
    }

    skb = rs485_dequeue(ppriv);
    if (!skb) {
        rs485_priv_stop_tx(ppriv);
        g_tx_deq_failed_times++;
        return;
    }

#ifdef DEBUG
    {
        static int g_ti = 0;
        if (!g_ti) {    /* only copy the first packet */
            memcpy(skb_put(debug_tx_skb, skb->len), skb->data, skb->len);
            g_ti = 1;
        }
    }
#endif            

    pdc->buf = skb->data;
    pdc->dma_addr = dma_map_single(port->dev, skb->data, skb->len, DMA_TO_DEVICE);
    pdc->dma_size = skb->len;
    pdc->ofs = 0;
    ppriv->data_sent = 1;     /* clear the fresh state */

    g_tx_chars += skb->len;
    UART_PUT_TPR(port, pdc->dma_addr);
    UART_PUT_TCR(port, pdc->dma_size);
    UART_PUT_PTCR(port, ATMEL_PDC_TXTEN); /* re-enable PDC transmit */
}

/*
 * Interrupt handler
 */
static irqreturn_t rs485_bus_interrupt (int irq, void *dev_id)
{
	struct rs485_priv *ppriv = (struct rs485_priv*) dev_id;
	struct uart_port *port = &ppriv->uart;
	u32 status, pending;

	status = UART_GET_CSR(port);
	pending = status & UART_GET_IMR(port);

	while (pending) {

#ifdef USE_MULTIDROP_TO_MAKE_FRAME
        if(pending & ATMEL_US_PARE) {
            g_rx_pare_time++;
            rs485_pdc_frame_end(ppriv);
        }
#endif        

        if (pending & ATMEL_US_ENDTX) {
            ++g_endtx_times;
            rs485_pdc_endtx(ppriv);
        }
        else if (pending & ATMEL_US_TXBUFE) {
            ++g_txbufe_times;
            rs485_pdc_endtx(ppriv);
        }

        if (pending & ATMEL_US_RXBRK) {
            ++g_rxbrk_times;        

#ifdef USE_BRK_TO_MAKE_FRAME
            rs485_pdc_frame_end(ppriv);
#endif

        }

        if (pending & ATMEL_US_TIMEOUT) {
        PDEBUG("get timeout \n");
#ifdef USE_BRK_TO_MAKE_FRAME
            if (ppriv->break_sent) {
                UART_PUT_CR(port, ATMEL_US_STPBRK);	/* stop break */
                ppriv->break_sent = 0;
                ++g_tx_frames;                      /* g_tx_frames frames have been sent ... */
            }
            port_clear_timeout(port);
#elif defined USE_TIMEOUT_TO_MAKE_FRAME
            if (ppriv->data_sent && ppriv->tx_timeout) {
//                UART_PUT_CR(port, ATMEL_US_RSTSTA);
//                UART_PUT_IDR(port, ATMEL_US_TIMEOUT);
                port_set_timeout(port, 50, ATMEL_US_STTTO); /* reset timeout, receiver timeout can not be disabled, so ... */
//                port_clear_timeout(port);
                ppriv->data_sent = ppriv->tx_timeout = 0;
                ++g_tx_frames;                      /* g_tx_frames frames have been sent ... */
                rs485_priv_start_tx(ppriv);
//                rs485_pdc_endtx(ppriv);     /* continue to send */
            } else if (ppriv->rx_timeout) { /* rx frame end */
//                PDEBUG("get rx timeout \n");                
//                rs485_pdc_frame_end (ppriv);
                port_set_timeout(port, 50, ATMEL_US_STTTO); /* reset rx timeout */
//                ppriv->rx_timeout = 0;                
            } else {
                PDEBUG("is there any bug? \n");
                BUG_ON(1);
            }
#endif
            ++g_timeout;
        }

        if (pending & ATMEL_US_ENDRX) {
            ++g_endrx;
        }

		status = UART_GET_CSR(port);
		pending = status & UART_GET_IMR(port);
	}
    
	return IRQ_HANDLED;
}


static inline void rs485_priv_enable_clock (struct rs485_priv *ppriv, struct platform_device *pdev)
{
	struct uart_port *port = &ppriv->uart;
    ppriv->clk = clk_get(&pdev->dev, "usart");
    clk_enable(ppriv->clk);
    port->uartclk = clk_get_rate(ppriv->clk);
}

static inline int rs485_priv_request_irq (	struct rs485_priv *ppriv )
{
    struct uart_port *port = &ppriv->uart;
	/*
	 * Allocate the IRQ
	 */
	ERR_RET_VAL (request_irq(port->irq, rs485_bus_interrupt, IRQF_SHARED, "rs485_bus", ppriv),
                 "rs485_bus request irq failed");
    return 0;
}

static int rs485_priv_enable_dma (struct rs485_priv *ppriv)
{
    int i, ret = 0;
    struct atmel_dma_buffer *pdc;
    struct uart_port *port = &ppriv->uart;

#ifdef USE_SINGLE_RX_DMA_BUF
    const int NUM_OF_RX_DMA_BUF = 1;
#else
    const int NUM_OF_RX_DMA_BUF = 2;
#endif
    
    for (i = 0; i < NUM_OF_RX_DMA_BUF; i++) {
        pdc = &ppriv->pdc_rx[i];
        pdc->buf = kmalloc(FNET_MTU, GFP_KERNEL);
        if (pdc->buf == NULL) {
            ret = -ENOMEM;
            if (0 == i)
                goto free_fnet_irq;
            else
                goto free_dma_rx_0;
        }
        pdc->dma_addr = dma_map_single(port->dev, pdc->buf, FNET_MTU, DMA_FROM_DEVICE);
        pdc->dma_size = FNET_MTU;
        pdc->ofs = 0;

        if (0 == i) {
            UART_PUT_RPR(port, ppriv->pdc_rx[0].dma_addr);
            UART_PUT_RCR(port, FNET_MTU);
            PDEBUG("ATMEL_PDC_RCR: %d:%08x \n", UART_GET_RCR(port), UART_GET_RCR(port) );
        }
        if (1 == i) {
            UART_PUT_RNPR(port, ppriv->pdc_rx[1].dma_addr);
            UART_PUT_RNCR(port, FNET_MTU);
        }
        
    }
    
    pdc = &ppriv->pdc_tx;
    pdc->buf = kmalloc(FNET_MTU, GFP_KERNEL);
    if (NULL == pdc->buf) {
        ret = -ENOMEM;
        goto free_dma_rx_1;
    }

    pdc->dma_addr = dma_map_single(port->dev, pdc->buf, FNET_MTU, DMA_TO_DEVICE);
    pdc->dma_size = FNET_MTU;
    pdc->ofs = 0;

    ppriv->pdc_rx_idx = 0;

    return 0;
    
free_dma_rx_1:
    dma_unmap_single(port->dev, ppriv->pdc_rx[1].dma_addr, FNET_MTU, DMA_FROM_DEVICE);
    kfree(ppriv->pdc_rx[1].buf);
free_dma_rx_0:
    dma_unmap_single(port->dev, ppriv->pdc_rx[0].dma_addr, FNET_MTU, DMA_FROM_DEVICE);
    kfree(ppriv->pdc_rx[0].buf);
free_fnet_irq:
    free_irq(port->irq, port);

    return ret;
}



/** 
 * 
 * 
 * @param port 
 * @param bits in relative with baud rate
 */
static void port_clear_timeout (struct uart_port *port)
{
//	UART_PUT_CR(port, ATMEL_US_RSTRX);
//    UART_PUT_CR(port, ATMEL_US_RSTSTA);
    UART_PUT_IDR(port, ATMEL_US_TIMEOUT);
}


/** 
 * 
 * 
 * @param port 
 * @param bits in relative with baud rate
 */
static void port_set_timeout (struct uart_port *port, u32 bits, u32 type)
{
    UART_PUT_RTOR(port, bits);		/* set UART timeout */
    UART_PUT_CR(port, type);
    UART_PUT_IER(port, ATMEL_US_TIMEOUT);
}

static void port_set_baud (struct uart_port *port)
{
    u32 mode = ATMEL_US_CHRL;

//    if (!g_cd) g_cd = 100;       /* 1M */
//    if (!g_cd) g_cd = 50;       /* 2M bps ok */
//    if (!g_cd) g_cd = 25;       /* 4M bps failed */
    if (!g_cd) g_cd = 12;       /* 8M bps failed */
//    if (!g_cd) g_cd = 33;       /* 3M bps */
    
//    g_cd = (1 << 10)/2;        /* 18k */
//    g_cd = (1 << 7);        /* 73k */
//    g_cd = (1 << 10)/10;        /*  94K failed */
//    g_cd = (2 << 10);        /* defaut 4.7K */

    PDEBUG("g_cd:%08x \n", g_cd&0xFFFF);
    UART_PUT_BRGR(port, (g_cd&0xFFFF) );

    mode |= (1<<8);             /* sync */
    mode |= 1;                  /* rs485 mode */
    UART_PUT_MR(port, mode);

#ifdef USE_MULTIDROP_TO_MAKE_FRAME
    mode = UART_GET_MR(port) & ~ATMEL_US_PAR;
    mode |= ATMEL_US_PAR_MULTI_DROP;
    UART_PUT_MR(port, mode);
    UART_PUT_IER(port, ATMEL_US_PARE);
#endif

}



static inline int rs485_priv_hw_init (struct rs485_priv *ppriv, struct atmel_uart_data *data)
{
	struct uart_port *port = &ppriv->uart;
    
	ppriv->use_dma_rx = data->use_dma_rx;
	ppriv->use_dma_tx = data->use_dma_tx;
    ppriv->data_sent = 0;
    
	/*
	 * Initialize DMA (if necessary)
	 */
	ERR_INFO_RETURN ( (!ppriv->use_dma_rx || !ppriv->use_dma_tx),
                      "error: use_dma_rx is false");

    port->fifosize = FNET_MTU;

    ERR_RET_VAL (rs485_priv_enable_dma(ppriv),
                 "error for rs485_priv_enable_dma");

    port_set_baud(port);

	/*
	 * Finally, enable the serial port
	 */
	UART_PUT_CR(port, ATMEL_US_RSTSTA | ATMEL_US_RSTTX | ATMEL_US_RSTRX);
    rs485_priv_start_rx(ppriv);
    
    return 0;
}


static inline void rs485_priv_hw_destroy (struct rs485_priv *ppriv)
{
    struct uart_port *port = &ppriv->uart;
	struct platform_device *pdev = to_platform_device(port->dev);

	int size = pdev->resource[0].end - pdev->resource[0].start + 1;
	release_mem_region(port->mapbase, size);
	if (port->flags & UPF_IOREMAP) {
		iounmap(port->membase);
		port->membase = NULL;
	}

    dma_unmap_single(port->dev, ppriv->pdc_rx[1].dma_addr, FNET_MTU, DMA_FROM_DEVICE);
    kfree(ppriv->pdc_rx[1].buf);

    dma_unmap_single(port->dev, ppriv->pdc_rx[0].dma_addr, FNET_MTU, DMA_FROM_DEVICE);
    kfree(ppriv->pdc_rx[0].buf);

    dma_unmap_single(port->dev, ppriv->pdc_tx.dma_addr, FNET_MTU, DMA_TO_DEVICE);
    kfree(ppriv->pdc_tx.buf);

    free_irq(port->irq, port);
}

/** 
 * Configure the port from the platform device resource info.
 * 
 * @atmel_port 
 * @pdev 
 */
static int rs485_priv_startup (struct rs485_priv *ppriv, struct platform_device *pdev)
{
    size_t size;
	struct uart_port *port = &ppriv->uart;
	struct atmel_uart_data *data = pdev->dev.platform_data;

	port->iotype        = UPIO_MEM;
	port->flags         = UPF_BOOT_AUTOCONF;
	port->fifosize      = 1;
	port->line          = pdev->id;
	port->dev           = &pdev->dev;
	port->mapbase       = pdev->resource[0].start;
	port->irq           = pdev->resource[1].start;
//	port->ops           = &atmel_pops;

	if (data->regs)
		/* Already mapped by setup code */
		port->membase = data->regs;
	else {
		port->flags	|= UPF_IOREMAP;
		port->membase	= NULL;
	}

    size = pdev->resource[0].end - pdev->resource[0].start + 1;

	if (!request_mem_region(port->mapbase, size, "fntnet_port"))
		return -EBUSY;

	if (port->flags & UPF_IOREMAP) {
		port->membase = ioremap(port->mapbase, size);
		if (port->membase == NULL) {
			release_mem_region(port->mapbase, size);
			return -ENOMEM;
		}
	}


	/*
	 * Ensure that no interrupts are enabled otherwise when
	 * request_irq() is called we could get stuck trying to
	 * handle an unexpected interrupt
	 */
	UART_PUT_IDR(port, -1);

    rs485_priv_enable_clock(ppriv, pdev);

    ERR_RET_VAL(rs485_priv_request_irq(ppriv),
                "error for rs485_priv_request_irq");
    ERR_RET_VAL (rs485_priv_hw_init(ppriv, data),
                 "error for rs485_priv_hw_init");
    
    PDEBUG("================here UART_GET_TCR:%08x \n", UART_GET_TCR(port));

	return 0;
}


static void rs485_priv_shutdown ( struct rs485_priv *ppriv )
{
    struct uart_port *port = &ppriv->uart;

	/* /\* */
	/*  * Ensure everything is stopped. */
	/*  *\/ */
	/* rs485_priv_stop_rx(ppriv); */
	/* rs485_priv_stop_tx(ppriv); */

	/*
	 * Disable all interrupts, port and break condition.
	 */
	UART_PUT_CR(port, ATMEL_US_RSTSTA);
	UART_PUT_IDR(port, -1);
    rs485_priv_hw_destroy (ppriv);
}




/*
 * Start transmitting.
 */
void rs485_priv_start_tx ( struct rs485_priv *ppriv )
{
    struct uart_port *port = &ppriv->uart;

    if (UART_GET_PTSR(port) & ATMEL_PDC_TXTEN) {
        /* The transmitter is already running.  Yes, we
           really need this.*/
        PDEBUG("transmitter is already running \n");
        return;
    }

	UART_PUT_CR(port, ATMEL_US_TXEN);
    UART_PUT_IER(port, ATMEL_US_ENDTX | ATMEL_US_TXBUFE);
    PRINT_REGS(ATMEL_US_IMR, "ATMEL_US_IMR");
}


/*
 * Stop transmitting.
 */
static void rs485_priv_stop_tx ( struct rs485_priv *ppriv )
{
    struct uart_port *port = &ppriv->uart;
    UART_PUT_PTCR(port, ATMEL_PDC_TXTDIS);		/* disable PDC transmit */
    UART_PUT_IDR(port, ATMEL_US_ENDTX | ATMEL_US_TXBUFE);
	UART_PUT_CR(port, ATMEL_US_TXDIS);
}

static void rs485_priv_start_rx ( struct rs485_priv *ppriv )
{
    struct uart_port *port = &ppriv->uart;

	/* UART_PUT_CR(port, ATMEL_US_RXEN | ATMEL_US_TXDIS); */
    /* UART_PUT_IER(port, ATMEL_US_RXBRK | ATMEL_US_ENDRX); */
    /* UART_PUT_PTCR(port, ATMEL_PDC_RXTEN);		/\* enable PDC controller *\/ */

	/* UART_PUT_CR(port, ATMEL_US_RSTSTA | ATMEL_US_RSTRX); */
	/* UART_PUT_CR(port, ATMEL_US_TXEN | ATMEL_US_RXEN);		/\* enable xmit & rcvr *\/ */
    /* UART_PUT_RTOR(port, PDC_RX_TIMEOUT);		/\* set UART timeout *\/ */
    /* UART_PUT_CR(port, ATMEL_US_STTTO); */
    /* UART_PUT_IER(port, ATMEL_US_ENDRX | ATMEL_US_TIMEOUT | ATMEL_US_RXBRK); */
    /* UART_PUT_PTCR(port, ATMEL_PDC_RXTEN);		/\* enable PDC controller *\/ */

	UART_PUT_CR(port, ATMEL_US_RSTSTA | ATMEL_US_RSTRX);
	UART_PUT_CR(port, ATMEL_US_TXEN | ATMEL_US_RXEN);		/* enable xmit & rcvr */
    UART_PUT_IER(port, ATMEL_US_ENDRX);

#ifdef USE_BRK_TO_MAKE_FRAME
    UART_PUT_IER(port, ATMEL_US_RXBRK);
#elif defined USE_TIMEOUT_TO_MAKE_FRAME
    port_set_timeout(port, 50, ATMEL_US_STTTO);
    ppriv->rx_timeout = 1;
#endif    
    
    UART_PUT_PTCR(port, ATMEL_PDC_RXTEN);		/* enable PDC controller */
}


#if 0


/*
 * Stop receiving - port is in process of being closed.
 */
static void rs485_priv_stop_rx ( struct rs485_priv *ppriv )
{
    struct uart_port *port = &ppriv->uart;
    UART_PUT_PTCR(port, ATMEL_PDC_RXTDIS);		/* disable PDC receive */
    UART_PUT_IDR(port, ATMEL_US_ENDRX | ATMEL_US_TIMEOUT);
}

/*
 * Control the transmission of a break signal
 */
static void rs485_priv_break_ctl ( struct rs485_priv *ppriv, int break_state)
{
    struct uart_port *port = &ppriv->uart;    
	if (break_state != 0)
		UART_PUT_CR(port, ATMEL_US_STTBRK);	/* start break */
	else
		UART_PUT_CR(port, ATMEL_US_STPBRK);	/* stop break */
}

#endif


static int usart_remove (struct platform_device *pdev)
{
	device_remove_file(&pdev->dev, &dev_attr_fntnet_debug_usart);
    BUG_ON(!hw_obj);
	clk_disable(hw_obj->clk);
	clk_put(hw_obj->clk);
    rs485_priv_shutdown (hw_obj);

    if (debug_tx_skb) {
        kfree_skb(debug_tx_skb);
        debug_tx_skb = 0;
    }
    
    if (debug_rx_skb) {
        kfree_skb(debug_rx_skb);
        debug_rx_skb = 0;
    }

    if (debug_cir_buf) {
        serial_buf_free(debug_cir_buf);
        debug_cir_buf = 0;
    }
    
	return 0;
}

static int usart_probe (struct platform_device *pdev)
{

    printk (KERN_INFO "pdev->id: %i \n", pdev->id);
	ERR_RET_VAL (rs485_priv_startup(hw_obj, pdev),
                 "error for rs485_priv_startup");

    PDEBUG("================here \n");
    BUG_ON(fntnet_create_usart_debug_file(pdev));
    PDEBUG("================here \n");
    
	return 0;
}

static inline void init_usart_driver (struct platform_driver *drv)
{
    drv->probe = usart_probe;
    drv->remove = usart_remove;
    drv->driver.name = "fntnet_port";
    drv->driver.owner = THIS_MODULE;
}


int rs485_bus_ctor (void *p)
{
    struct fntnet_priv * priv = (struct fntnet_priv *)p;
    hw_obj = &priv->rs485;
    init_usart_driver(&hw_obj->usart_drv);
	ERR_RET_VAL (platform_driver_register( &hw_obj->usart_drv ),
                 "platform_driver_register(&priv->usart_drv) failed");

    rs485_init_queue(hw_obj);
    return 0;
}

void rs485_bus_dtor (void *p)
{
    struct fntnet_priv * priv = (struct fntnet_priv *)p;
	platform_driver_unregister( &priv->rs485.usart_drv );
    hw_obj = NULL;
}

