/******************************************************************
 * @file   psoc_event.c
 * @author Richard Luo
 * @date   2008-08-05
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/irq.h>
#include <linux/interrupt.h>


#include <linux/kernel.h>	/* printk() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/cdev.h>
#include <linux/poll.h>

#include <asm/arch/gpio.h>
#include <asm/arch/at91_pio.h>
#include <linux/proc_fs.h>

#include <asm/system.h>		/* cli(), *_flags */
#include <asm/uaccess.h>	/* copy_*_user */
#include "akdebug.h"
#include "dbgfp.h"

MODULE_LICENSE("Dual BSD/GPL");


static int psoc_event_major;
module_param(psoc_event_major, int, 0);

static atomic_t event_counter = ATOMIC_INIT(0);

typedef struct {
    int pin_;
    int state_; 
    struct timeval tm_;
    const char *name_;
    struct cdev cdev_;
	struct semaphore sem_;
    wait_queue_head_t wait_;
} psoc_event_t;

enum {
    PSOC_STATE_INIT,
    PSOC_STATE_WAIT_SECOND_EDGE,
};


psoc_event_t g_psoc_event = {
    .pin_ = AT91_PIN_PB20,
    .name_ = "psoc_event",
};


#if defined DEBUG

dfp_t *psoc_dfp_inst (void)
{
    static dfp_t *s_dfp = 0;

    /* FIXME: may need double checked lock here. */
    if (s_dfp) return s_dfp;

    while (!s_dfp)
        s_dfp = dfp_alloc();

    return s_dfp;
}

/** 
 * @note can only be called when module removed!!!!
 * 
 */
void psoc_dfp_free (void)
{
    dfp_free(psoc_dfp_inst());
}



int psoc_event_proc (char *buf, char **start, off_t offset,
                     int count, int *eof, void *data)
{
    int n;
//    psoc_interval_accounter_dump();
    n = dfp_mem_dump(psoc_dfp_inst(), buf, count);
	*eof = 1;
	return n;
}

#define dbgpk_buf(format,args...)                       \
    do {                                                \
        dfp_printf(psoc_dfp_inst(),format, ## args);    \
    } while (0)

#else

#define dbgpk_buf(format,args...) do { } while (0)



#endif


static int psoc_open (struct inode *inode, struct file *filp)
{
	psoc_event_t *dev; /* device information */
	dev = container_of(inode->i_cdev, psoc_event_t, cdev_);
	filp->private_data = dev; /* for other methods */
	return 0;
}

static unsigned int psoc_poll (struct file *filp, poll_table *wait)
{
    psoc_event_t *dev = (psoc_event_t*) filp->private_data;
	unsigned int mask = 0;

#if defined DEBUG
    static int count = 0;
    count++;
    dbgpk_buf("calling times: %d, event_counter:%d \n", count, atomic_read(&event_counter) );
#endif
    
	down(&dev->sem_);
	poll_wait(filp, &dev->wait_, wait);
    if (atomic_read(&event_counter) > 0 ) {
        mask |= POLLIN | POLLRDNORM;	/* readable */
        atomic_dec(&event_counter);     /* delete one event */
    }
	up(&dev->sem_);

	return mask;
}

static ssize_t psoc_read (struct file *filp, char __user *buf, size_t count,
                          loff_t *f_pos)
{
    psoc_event_t *dev = (psoc_event_t*) filp->private_data;

    dev = filp->private_data;

	if (down_interruptible (&dev->sem_))
		return -ERESTARTSYS;

    while ( atomic_read(&event_counter) == 0 ) {
		up(&dev->sem_); /* release the lock */
		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;
		PDEBUG("\"%s\" psoc_read: going to sleep\n", current->comm);
		if (wait_event_interruptible(dev->wait_, (atomic_read(&event_counter) > 0)) )
			return -ERESTARTSYS; /* signal: tell the fs layer to handle it */

		/* otherwise loop, but first reacquire the lock */
		if (down_interruptible(&dev->sem_)) /*  */
			return -ERESTARTSYS;
    }

    const int dc = atomic_read(&event_counter);
    if (dc <= 0 || dc > 10) {
        printk(KERN_ERR "ERR: invalid event_counter:%d", dc);
        atomic_set(&event_counter, 1);
    }

    atomic_dec(&event_counter);
    up(&dev->sem_); /* release the lock */
    return 0;
}



struct file_operations psoc_event_ops = {
	.owner = THIS_MODULE,
    .poll = psoc_poll,
    .open = psoc_open,
    .read = psoc_read,
};

static irqreturn_t psoc_event_isr (int irq, void *dev)
{
    static int dbg_int_times = 0;
    psoc_event_t *pdev = (psoc_event_t*) dev;
    atomic_inc(&event_counter);
    wake_up_interruptible(&pdev->wait_);

    dbgpk_buf("wake up finished, event_counter: %d, times:%d \n",
              atomic_read(&event_counter), ++dbg_int_times);
    
	return IRQ_HANDLED;
}

static void psoc_event_destroy (psoc_event_t *pdev)
{
    cdev_del(&pdev->cdev_);    
    free_irq(gpio_to_irq(pdev->pin_), pdev);
}

static int psoc_event_setup (psoc_event_t *pdev)
{
	int err, devno, irq;

    devno = MKDEV(psoc_event_major, 0);

	init_MUTEX(&(pdev->sem_));
    init_waitqueue_head(&pdev->wait_);
	cdev_init(&pdev->cdev_, &psoc_event_ops);
	pdev->cdev_.owner = THIS_MODULE;
	err = cdev_add (&pdev->cdev_, devno, 1);

	/* Fail gracefully if need be */
	if (err) {
		printk(KERN_NOTICE "Error %d setup psoc event module \n", err);
        return -1;
    }

    at91_set_gpio_input(pdev->pin_, 1);
    irq = gpio_to_irq(pdev->pin_);

    err = request_irq(irq, psoc_event_isr, 0, pdev->name_, pdev);

    if (-1 == err) {
        PDEBUG("request_irq failed, irq: [%d] \n", irq);
        cdev_del(&pdev->cdev_);
        return -1;
    }

#ifdef DEBUG
	create_proc_read_entry("psoc_event", 0, NULL, psoc_event_proc, NULL);
#endif
    
    PDEBUG("psoc_event_setup ok \n");
    return 0;
}


int psoc_event_init_module(void)
{
	int result;
	dev_t dev = MKDEV(psoc_event_major, 0);

    /* at91_dump_pio_status(); */

	/*
	 * Register your major, and accept a dynamic number.
	 */
	if (psoc_event_major)
		result = register_chrdev_region(dev, 1, g_psoc_event.name_);
	else {
		result = alloc_chrdev_region(&dev, 0, 1, g_psoc_event.name_);
		psoc_event_major = MAJOR(dev);
	}

    return psoc_event_setup(&g_psoc_event);
}

void psoc_event_cleanup_module(void)
{
    
    psoc_event_destroy(&g_psoc_event);
	unregister_chrdev_region(MKDEV(psoc_event_major, 0), 1);

#ifdef DEBUG
    remove_proc_entry("psoc_event", NULL);
    psoc_dfp_free();
#endif

    printk("bye psoc event \n");
}

module_init(psoc_event_init_module);
module_exit(psoc_event_cleanup_module);




