/******************************************************************
 * @file   psoc_fw.c
 * @author Richard Luo
 * @date   2008-09-22
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include "psoc_fw.h"

typedef struct {
    const struct firmware *pfw_;
    size_t rdptr_;
} fw_itor_t;

static fw_itor_t record_itor;

/** 
 * @brief if @a p is NULL, stand for reset
 * 
 * @param p from user space
 */
void psoc_fw_set_firmware (const struct firmware *p)
{
    if (p) record_itor.pfw_ = p;
    record_itor.rdptr_ = 0;
}


/** 
 * 
 * 
 * 
 * @return 0 -- reach the end
 *         1 -- can be continue
 *        -1 -- illegal @c firmware
 */
static inline int valid_fw_itor (void)
{
    if (record_itor.rdptr_ < record_itor.pfw_->size)
        return 1;
    if (record_itor.rdptr_ == record_itor.pfw_->size)
        return 0;
    return -1;
}

static inline uint8_t *__fw_itor_rdata_fast (void)
{
    return &record_itor.pfw_->data[record_itor.rdptr_];
}

static inline uint8_t *fw_itor_rdata (void)
{
    if (1 == valid_fw_itor())
        return __fw_itor_rdata_fast();
    return NULL;
}

static inline int fw_itor_inc (size_t sz)
{
    record_itor.rdptr_ += sz;
    return valid_fw_itor();
}

/** 
 * 
 * 
 * @param prec 
 * 
 * @return      0 -- reach the end, return the last valid record,
 *              1 -- can be continued,
 *             -1 -- error
 */
int psoc_record_next (psoc_fw_record_t **prec)
{
    psoc_fw_record_t *ret = (psoc_fw_record_t*) fw_itor_rdata();

    if (!ret) return -1;       /* unknow error, caused by illegal input */

    *prec = ret;               /* *prec can only be changed with valid value */

    return fw_itor_inc(sizeof(psoc_fw_record_t) + (*prec)->num_ + 1); /* '1' for checksum */
}


uint8_t twos_complete_checksum (const void *buf, size_t size)
{
    size_t i;
    uint8_t *pb = (uint8_t*) buf;
    uint8_t checksum = 0;

    for (i = 0; i < size; ++i)
        checksum += pb[i];

    checksum = ~checksum;
    ++checksum; 

    return checksum;
}

uint8_t psoc_get_record_checksum ( const psoc_fw_record_t *prec )
{
    size_t size = sizeof(*prec) + prec->num_;
    return twos_complete_checksum(prec, size);
}

/** 
 * 
 * 
 * @param prec 
 * 
 * @return 0 on checksum ok, -1 on error
 */
int psoc_validate_checksum ( const psoc_fw_record_t *prec )
{
    return prec->data_[prec->num_] == psoc_get_record_checksum(prec) ? 0 : -1;
}



int psoc_validate_one_fw_record ( const psoc_fw_record_t *prec )
{
    if (prec->type_ == FWV_DATA)
        return psoc_validate_checksum(prec);
    else {
        psoc_fw_record_t *pnext = 0;
        if ( psoc_record_next(&pnext) )
            return -1;
        return psoc_validate_checksum(pnext);        
    }
}

/** 
 * 
 * 
 * @param fwm 
 * 
 * @return 0 on success, else -1
 */
int psoc_validate_firmware ( const struct firmware *fwm )
{
    int ret;
    psoc_fw_record_t *precord = 0;

    psoc_fw_set_firmware (fwm);

    for (ret = psoc_record_next(&precord); ret >= 0; ret = psoc_record_next(&precord) ) {
        if (psoc_validate_checksum(precord))
            return -1;
        if (0 == ret) {          /* reach the end */
            if (precord->type_ != 1) /* end of file record */
                return -1;
        }
    }

    psoc_fw_set_firmware (fwm); /* reset for reuse */
    
    return 0;
}
