/******************************************************************
 * @file   gpio_reader.c
 * @author Richard Luo
 * @date   2008-05-19
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include <linux/irq.h>
#include <linux/interrupt.h>
#include <asm/arch/gpio.h>
#include <asm/arch/at91_pio.h>
#include <asm/io.h>

#include "reader_buf.h"
#include "gpio_reader.h"

#include "dbgpk.h"
#include "akdebug.h"

typedef struct {
    unsigned long too_small_;
    unsigned long too_big_;
    unsigned long normal_;
} interval_t;

struct reader_interval_accounter {
    suseconds_t small_t_;
    interval_t small_ii_;

    suseconds_t big_t_;
    interval_t big_ii_;

    unsigned long right_read_times_;
    unsigned long wrong_read_times_;
    unsigned long unexpeted_interrupts_;
    
};

struct reader_interval_accounter debug_reader_accounter;

static void debug_show_interval_info ( struct reader_interval_accounter *pi)
{
    const int total_times = pi->right_read_times_ + pi->wrong_read_times_;
    if ( total_times > 0) 
        printk (KERN_DEBUG
                "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n"

                "Total reading times: [%ld] \n"
                "right read times: [%ld] wrong_read_times_: [%ld] unexpeted_interrupts_: [%ld]\n"
                "reading right percent: [%ld%%] \n"

                "small interval@ too small times: [%ld] \n"
                "small interval@ too big times: [%ld] \n"
                "small interval@ normal times: [%ld] \n"

                "big interval@ too small times: [%ld] \n"
                "big interval@ too big times: [%ld] \n"
                "big interval@ normal times: [%ld] \n"

                "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n",

                (pi->right_read_times_ + pi->wrong_read_times_), /* total reading times */
                pi->right_read_times_, pi->wrong_read_times_, pi->unexpeted_interrupts_,
                (pi->right_read_times_*100)/(pi->right_read_times_ + pi->wrong_read_times_),
           
                pi->small_ii_.too_small_,
                pi->small_ii_.too_big_,
                pi->small_ii_.normal_,

                pi->big_ii_.too_small_,
                pi->big_ii_.too_big_,
                pi->big_ii_.normal_
            );
}

/** 
 * 
 * @param rd reader
 * 
 * @return 0 ok, the is valid data in the reader's buffer, user
 * process should wakeup. els -1;
 */
static int finished_one_card_reading (gpio_reader_t *rd)
{
    uint16_t state = rd->state_;
    rd->state_ = RD_STAT_INIT;  /* recover for next reading. */
    WG_EXCEPTION_CLEAR(rd);

    if (RD_STAT_FINISHED_SECOND_EDGE != state) {
        reader_buf_init(rd->rdbuf_); /**< clear invalid data. */
        dbgpk("================ wrong_read_times, state=[%d] \n", (int)state);
        debug_reader_accounter.wrong_read_times_++;
        return -1;
    }

    /* move the wr pointer to the next position */
    reader_buf_wr_inc(rd->rdbuf_);
    debug_reader_accounter.right_read_times_++;
    return 0;
}

/** 
 * @brief one card reading finished can we get here.
 * 
 * @param r the gpio reader address
 */
static void reader_timeout(unsigned long r)
{
    gpio_reader_t *rd = (gpio_reader_t*) r;

    extern wait_queue_head_t * reader_wait_queue (void);

    if (0 == finished_one_card_reading(rd) )
        wake_up_interruptible(reader_wait_queue());
}

/** 
 * @brief to do three jobs:
 *      1. enable the gpio input/output status
 *      2. allocate buffer for saving card data.
 *      3. setup a timer.
 * 
 * @param rd the reader
 * 
 * @return -1 on error, 0 on success.
 */
int gpio_reader_init (gpio_reader_t *rd)
{
    if (!rd) return -1;

    at91_set_gpio_input(rd->pin_data0_, 1);
    at91_set_gpio_input(rd->pin_data1_, 1);
    at91_set_gpio_output(rd->pin_led_, 0);

    rd->rdbuf_ = kmalloc(sizeof(reader_buf_t), GFP_KERNEL );

    if (rd->rdbuf_ == NULL)
        return -1;

    reader_buf_init(rd->rdbuf_);

    init_timer(&rd->timer_);
	rd->timer_.function = reader_timeout;
	rd->timer_.data = (unsigned long) rd;

    return 0;
}

void gpio_reader_destroy(gpio_reader_t *rd)
{
    gpio_reader_release_irq(rd);
    if (rd->rdbuf_)
        kfree(rd->rdbuf_);
    rd->rdbuf_ = NULL;
}

/** 
 * 
 * @param rd one reader
 * @param h irq handler
 * 
 * @return 0 on success, -1 on error
 */
int gpio_reader_request_irq (gpio_reader_t *rd, irq_handler_t h)
{
    int irq_d0, irq_d1, ret;
    const char *name0;
    const char *name1;
    if (!rd) return -1;

    irq_d0 = gpio_to_irq(rd->pin_data0_);
    irq_d1 = gpio_to_irq(rd->pin_data1_);

    name0 = rd->irq_name0_;
    name1 = rd->irq_name1_;

    set_irq_type(irq_d0, IRQ_TYPE_EDGE_BOTH);
    set_irq_type(irq_d1, IRQ_TYPE_EDGE_BOTH);

    ret = request_irq(irq_d0, h, IRQF_DISABLED, name0, rd);
    
    if (ret) {
        printk(KERN_INFO "can't get assigned irq %i, err:[%d] \n", irq_d0, ret);
        return -1;
    }

    ret = request_irq(irq_d1, h, IRQF_DISABLED, name1, rd);
    
    if (ret) {
        printk(KERN_INFO "can't get assigned irq %i, err:[%d] \n", irq_d1, ret);
        free_irq(irq_d0, NULL);
        return -1;
    }

    return 0;
}

/** 
 * @brief must be sure that the irq has been successfully requested
 * before otherwise will crash.
 * 
 */
void gpio_reader_release_irq (gpio_reader_t *rd)
{
    int irq_d0, irq_d1;

    irq_d0 = gpio_to_irq(rd->pin_data0_);
    irq_d1 = gpio_to_irq(rd->pin_data1_);
    free_irq(irq_d0, rd);
    free_irq(irq_d1, rd);
}

/** 
 * @brief for extension when need more readers
 * 
 * @param group pointer to @a n readers.
 * 
 * @return 0 on success, -1 on error.
 */
static int gpio_reader_init_n (gpio_reader_t *group, int n)
{
    int i = 0;

    for (; i < n ; ++i )
        if ( -1 == gpio_reader_init(&group[i]) )
            return -1;
    
    return 0;
}


#define my_abs(x) ({                            \
            int __x = (x);                      \
            (__x < 0) ? -__x : __x;             \
        })


#define dbgpk_tm(format,args...) do {           \
        printk("[%5lu.%06lu]@%d " format,       \
               (unsigned long)tm.tv_sec,        \
               (unsigned long)tm.tv_usec,       \
               __LINE__, ## args);              \
    }while(0)


static inline uint16_t valid_big_interval_time(struct timeval *told, struct timeval *tnew)
{
    const suseconds_t MINIMAL_BIG_INTERVAL = 100, MAXMUM_BIG_INTERVAL = 20000;
    const suseconds_t usec_diff = timval_diff(told, tnew);

#ifdef DEBUG
    BUG_ON(usec_diff < 0);
#else
    if (usec_diff < 0) {
        dbgpk("big time diff error, usec_diff=%ld \n", usec_diff);
        return 0;
    }
#endif    

    debug_reader_accounter.big_t_ += usec_diff;

    if (MAXMUM_BIG_INTERVAL < usec_diff) {
        debug_reader_accounter.big_ii_.too_big_++;
        dbgpk("big too big, usec_diff:[%ld] \n", usec_diff);
        return (uint16_t) RD_STAT_BIG_INTERVAL_TOO_LARGE;
    }
    
    if (MINIMAL_BIG_INTERVAL > usec_diff) {
        debug_reader_accounter.big_ii_.too_small_++;
        dbgpk("big too small, usec_diff:[%ld] \n", usec_diff);
        return (uint16_t) RD_STAT_BIG_INTERVAL_TOO_SMALL;
    }
    debug_reader_accounter.big_ii_.normal_++;
    return 0;
}

/** 
 * 
 * 
 * @param t0 
 * @param t1 
 * 
 * @return 0 on ok, else the invalid reader state returnd and should
 * be set to the corresponding reader.
 */
static inline uint16_t valid_small_interval_time(struct timeval *told, struct timeval *tnew)
{
    const suseconds_t MINIMAL_SMALL_INTERVAL = 2, MAXMUM_SMALL_INTERVAL = 500;
    const suseconds_t usec_diff = timval_diff(told, tnew);

#ifdef DEBUG
    BUG_ON(usec_diff < 0);
#else
    if (usec_diff < 0) {
        dbgpk("small time diff error, usec_diff=%ld \n", usec_diff);
        return 0;
    }
#endif    

    debug_reader_accounter.small_t_ += usec_diff;

    if (MAXMUM_SMALL_INTERVAL < usec_diff) {
        dbgpk ("small too big, %ld \n", usec_diff);
        debug_reader_accounter.small_ii_.too_big_++;
        return (uint16_t) RD_STAT_SMALL_INTERVAL_TOO_LARGE;
    }
    
    if (MINIMAL_SMALL_INTERVAL > usec_diff) {
        dbgpk ("small too small, %ld \n", usec_diff);
        debug_reader_accounter.small_ii_.too_small_++;
        return (uint16_t) RD_STAT_SMALL_INTERVAL_TOO_SMALL;
    }
    debug_reader_accounter.small_ii_.normal_++;
    return 0;
}


void reader_interval_accounter_dump(void)
{
    debug_show_interval_info(&debug_reader_accounter);
}

/** 
 * 1. check whether it's a *valid* interrupt
 *      'valid' is defined as:
 *              a. not overtime, whithin a definded period of time.
 *              b. the data line is correct, corresponding to the previous one.
 * 
 * 2. according to reader's state make a decision whether to save the
 * bit value, and record current time to the reader.
 *
 * @param rd reader
 * @param val 0: interrupt at data0  1: ... data1
 * @param gpio_v: the current gpio's level of data0 or data1 specified by @param val
 * @return -1 on some error occured, 0 on success
 */
static inline int record_reader_state(gpio_reader_t *rd, uint8_t val, uint8_t gpio_v, int irq)
{
    struct timeval tm;

    if (rd->state_ & RD_STAT_BAD) {
        return -1;
    }
    do_gettimeofday(&tm);

    switch (rd->state_) {
    case RD_STAT_INIT: 
        if (gpio_v == 1) {      /* it's normal case */
            rd->state_ = RD_STAT_FINISHED_FIRST_EDGE;
            WG_EXCEPTION_CLEAR(rd); 
        }
        else {                  /* the first positive-edge was lost */
            dbgpk_tm("init lost (+) at D%s \n", val ? "1" : "0");
            rd->state_ = RD_STAT_FINISHED_SECOND_EDGE;
            card_buf_append_bit(reader_buf_wr_card(rd->rdbuf_), val);
            WG_EXCEPTION_SET(rd, val, WG_TPW_BOTH_DELAY);
        }

        rd->prev_bit_val_ = val;
        rd->last_read_time_ = tm; /* record this time */
        break;
        
    case RD_STAT_FINISHED_FIRST_EDGE:
        if (gpio_v == 0) {
            if (val == rd->prev_bit_val_) {
                /* normal case */
                rd->state_ = valid_small_interval_time(&rd->last_read_time_, &tm);
                if ( 0 == rd->state_) {
                    card_buf_append_bit(reader_buf_wr_card(rd->rdbuf_), val);
                    rd->state_ = RD_STAT_FINISHED_SECOND_EDGE;
                    WG_EXCEPTION_CLEAR(rd);
                } else {
                    dbgpk_tm("error with valid_small_interval_time \n");
                }
            }
            else {
                dbgpk_tm("error: seems like impossible case!! \n");
                rd->state_ = RD_STAT_BAD_LOGIC;

                /* dbgpk_tm("at least lost two edge: (-) of D%s and (+) of D%s \n", */
                /*          val ? "0" : "1", val ? "1" : "0"); */
                /* rd->state_ = valid_big_interval_time(&rd->last_read_time_, &tm); */
                /* if ( 0 == rd->state_) { */
                /*     uint8_t prev_lost_v = val ? 0 : 1; */
                /*     dbgpk_tm("fix lost two edge \n"); */
                /*     card_buf_append_bit(reader_buf_wr_card(rd->rdbuf_), prev_lost_v); */
                /*     card_buf_append_bit(reader_buf_wr_card(rd->rdbuf_), val); */
                /*     rd->state_ = RD_STAT_FINISHED_SECOND_EDGE; */
                /*     rd->prev_bit_val_ = val; */
                /* } */

            }
        }
        else {                  /* this is a (+) edge*/
            if (val == rd->prev_bit_val_) {
                dbgpk_tm("may lost big (-) at D%s \n", val ? "1" : "0");
                rd->state_ = valid_big_interval_time(&rd->last_read_time_, &tm);
                if ( 0 == rd->state_ && 0 == rd->last_fix_val_) {
                    const uint8_t prev_lost_v = val;
                    dbgpk_tm("fix lost big (-) \n");
                    card_buf_append_bit(reader_buf_wr_card(rd->rdbuf_), prev_lost_v);
                    rd->state_ = RD_STAT_FINISHED_FIRST_EDGE;
                    WG_EXCEPTION_SET(rd, val, WG_LOST_NEGATIVE);
                }
                else {
                    dbgpk_tm("Error: lost too many?? \n");
                }
            }
            else {
                dbgpk_tm("FIXME: at least lost one (+)--(-) at D%s \n", val ? "1" : "0");
                rd->state_ = RD_STAT_BAD_LOGIC;
            }
        }
        rd->last_read_time_ = tm;
        break;

    case RD_STAT_FINISHED_SECOND_EDGE:
        rd->state_ = valid_big_interval_time(&rd->last_read_time_, &tm);
        switch(rd->state_) {
        case 0:                 /* interval is ok */
            if (gpio_v == 1) {  /* normal case */
                rd->state_ = RD_STAT_FINISHED_FIRST_EDGE;
                WG_EXCEPTION_CLEAR(rd);
            }
            else {              
                if (rd->last_fix_val_)
                    dbgpk_tm("Note: already have an EXP! \n");
                /* the same case as "lost first positive-edge" */
                dbgpk_tm("== fix the delayed (+) \n");
                card_buf_append_bit(reader_buf_wr_card(rd->rdbuf_), val);
                // we can not know currently this one should be a (+)
                // or (-), if (+) then BOTH_DELAY, or have LOST one of
                // them, we append bit here in case of the LOST one.
                WG_EXCEPTION_SET(rd, val, WG_TPW_BOTH_DELAY); 
                rd->state_ = RD_STAT_FINISHED_SECOND_EDGE;
            }
            break;

        case RD_STAT_BIG_INTERVAL_TOO_SMALL:
            dbgpk_tm("Note: small Ints too close: gpio_v:%d \n", gpio_v);
            if (gpio_v == 0 && rd->fix_reason_ == WG_TPW_BOTH_DELAY) { /* already fixed?? */
                /* two delayed interrupts between a very small window, actually it's Tpw */
                rd->state_ = RD_STAT_FINISHED_SECOND_EDGE;
                WG_EXCEPTION_CLEAR(rd); /* good guess! let's return to normal */
            }
            else {
                dbgpk_tm("Can not understand this case!! \n");
                rd->state_ = RD_STAT_BAD_LOGIC;
            }
            break;

        default:
            dbgpk_tm("Error: state:%d \n", rd->state_);
            break;
        };

        rd->last_read_time_ = tm;
        rd->prev_bit_val_ = val;
        break;
    default:
        dbgpk_tm("error: unknow rd->state:[%x] \n", rd->state_);
        rd->state_ = RD_STAT_BAD_LOGIC;
        return -1;
    };

    return 0;
}

/** 
 * 
 * 
 * @param dev_id point to the corresponding reader who raised the interrupt
 * 
 */
static irqreturn_t read_isr(int irq, void *dev_id)
{
    uint8_t val;
    gpio_reader_t *rd;
    int irq_d0, irq_d1, gpio_v;

#ifdef DEBUG_TIME
    suseconds_t diff;
    struct timeval tm_in, tm_out;
    do_gettimeofday(&tm_in);
#endif    



    rd  = (gpio_reader_t*) dev_id;

    irq_d0 = gpio_to_irq(rd->pin_data0_);
    irq_d1 = gpio_to_irq(rd->pin_data1_);

    if (irq == irq_d0) {
        val = 0;
        gpio_v = gpio_get_value(rd->pin_data0_);
    }
    else if (irq == irq_d1) {
        val = 1;
        gpio_v = gpio_get_value(rd->pin_data1_);
    }
    else {
        val = -1;
        return IRQ_NONE;
    }

    record_reader_state(rd, val, gpio_v ? 1 : 0, irq);

    if ( timer_pending(&rd->timer_) )
        mod_timer(&rd->timer_, jiffies + MIN_ONE_CARD_INTERVAL);
    else {
        rd->timer_.expires = jiffies + MIN_ONE_CARD_INTERVAL;
        add_timer(&rd->timer_);
    }


#ifdef DEBUG_TIME
    do_gettimeofday(&tm_out);
    diff = timval_diff(&tm_in, &tm_out);
    printk("isr diff:%ld \n", diff);
#endif    

	return IRQ_HANDLED;
}


/** 
 * 
 * 
 * @param group 
 * @param n 
 * 
 * @return 0 on success, -1 on error.
 */
int gpio_reader_request_irq_n (gpio_reader_t *group, int n)
{
    int i = 0;

    for (; i < n ; ++i )
        if ( -1 == gpio_reader_request_irq( &group[i], read_isr) )
            break;

    if (i != n) { /* faild with the reader:
                   * group[i], but 0...i-1 are ok
                   * and must release them before return -1 */
        int k = 0;
        for (; k < i; ++k)
            gpio_reader_release_irq(&group[k]);
        return -1;
    }
    
    return 0;
}

/** 
 * @brief commit or rollback meaning
 * 
 * @param group to all readers.
 * @param n num of readers
 * 
 * @return 0 if all readers setup are ok.
 */
int gpio_reader_setup_n (gpio_reader_t *group, int n)
{
    if (0 == gpio_reader_init_n(group, n) && 0 == gpio_reader_request_irq_n(group, n) ) {
        return 0;
    }
    return -1;
}

/** 
 * @brief assumed there is a successfull @p gpio_reader_setup_n
 * 
 */
void gpio_reader_destroy_n (gpio_reader_t *group, int n)
{
    int i = 0;

    for ( ; i < n; ++i)
        gpio_reader_destroy(&group[i]);
}



