/******************************************************************
 * @file   gpio_reader.h
 * @author Richard Luo
 * @date   2008-05-19
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#ifndef _GPIO_READER_H
#define _GPIO_READER_H 1

#include <linux/timer.h>
#include <linux/types.h>

// struct timeval {
// 	time_t		tv_sec;		/* seconds */
// 	suseconds_t	tv_usec;	/* microseconds */
// };


enum {

    RD_STAT_INIT = 0,
    RD_STAT_FINISHED_FIRST_EDGE = 1,
    RD_STAT_FINISHED_SECOND_EDGE = 2,
    RD_STAT_BAD = 1 << 14,
    RD_STAT_SMALL_INTERVAL_TOO_SMALL = (RD_STAT_BAD | 3),
    RD_STAT_SMALL_INTERVAL_TOO_LARGE = (RD_STAT_BAD | 4),
    RD_STAT_BIG_INTERVAL_TOO_SMALL = (RD_STAT_BAD | 5),
    RD_STAT_BIG_INTERVAL_TOO_LARGE = (RD_STAT_BAD | 6),


    RD_STAT_UNEXPECTED_INTERRUPT = (RD_STAT_BAD | 8),
    RD_STAT_BAD_LOGIC = (RD_STAT_BAD | 9),
    
};

enum WG_EXCEPTION_REASON {
    WG_OK = 0,                /* no exception happened */
    WG_TPW_BOTH_DELAY = 0x0001,      /* sometimes CPU is too busy but not caused lost of INTs, just delay */
    WG_LOST_POSITIVE = 0x0002,  /* lost (+) */
    WG_LOST_NEGATIVE = 0x0004,  /* lost (-) */
    WG_TPW_HALF_DELAY = 0x0001, /* no lost, just delayed the (-), seems impossible */
};

typedef struct {
    int pin_data0_;
    int pin_data1_;
    int pin_led_;
    int pin_buzz_;
    const char *irq_name0_;
    const char *irq_name1_;
    irq_handler_t handler_;
    reader_buf_t *rdbuf_;
    struct timer_list timer_;

    struct timeval last_read_time_;
    uint16_t prev_bit_val_;
    uint16_t state_;

    uint16_t last_fix_val_;     /* no pending: 0, pending D0: '0', pending D1: '1' */
    uint16_t fix_reason_;       /* see @a WG_EXCEPTION_REASON */

} gpio_reader_t;


/* used for state machine to return normal case */
#define WG_EXCEPTION_CLEAR(rd)                  \
    do {                                        \
        if (rd->last_fix_val_) {                \
            rd->last_fix_val_ = 0;              \
            rd->fix_reason_ = 0;                \
        }                                       \
    } while(0)

#define WG_EXCEPTION_SET(rd, val, reason)       \
    do {                                        \
        rd->last_fix_val_ = val ? '1' : '0';    \
        rd->fix_reason_ = reason;               \
    } while(0)



int gpio_reader_init (gpio_reader_t *rd);
void gpio_reader_destroy(gpio_reader_t *rd);
int gpio_reader_request_irq (gpio_reader_t *rd, irq_handler_t h);
void gpio_reader_release_irq (gpio_reader_t *rd);

int gpio_reader_setup_n (gpio_reader_t *group, int n);
void gpio_reader_destroy_n (gpio_reader_t *group, int n);


static inline suseconds_t timval_diff(struct timeval *told, struct timeval *tnew)
{
    const time_t sec_diff = tnew->tv_sec - told->tv_sec;
    const suseconds_t usec_diff = tnew->tv_usec - told->tv_usec;
    return (sec_diff * 1000000 + usec_diff);
}

#endif /* _GPIO_READER_H */



