/*
 * RTC subsystem, initialize system time on startup
 *
 * Copyright (C) 2005 Tower Technologies
 * Author: Alessandro Zummo <a.zummo@towertech.it>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
*/

#include <linux/rtc.h>

/* IMPORTANT: the RTC only stores whole seconds. It is arbitrary
 * whether it stores the most close value or the value with partial
 * seconds truncated. However, it is important that we use it to store
 * the truncated value. This is because otherwise it is necessary,
 * in an rtc sync function, to read both xtime.tv_sec and
 * xtime.tv_nsec. On some processors (i.e. ARM), an atomic read
 * of >32bits is not possible. So storing the most close value would
 * slow down the sync API. So here we have the truncated value and
 * the best guess is to add 0.5s.
 */

/** 
 * @brief To work around the M41T11's unsteable state, the problem
 * seems in relationship with the super electric capacitor...
 * 
 * @param tm set to 2000-01-01, Sat day
 * 
 */
static void __init set_rtc_to_2000_01_01(struct rtc_device *rtc)
{
    struct rtc_time tm = {
        .tm_sec = 0,
        .tm_min = 0,
        .tm_hour = 0,
        .tm_mday = 6,
        .tm_mon = 6,
        .tm_year = 100,
        .tm_wday = 6,
        .tm_yday = 0,
        .tm_isdst = 0
    };
    rtc_set_time(rtc, &tm);
}

#include <asm/string.h>

/*
 * trying to fix 1307's +100 year
 */
static int __init my_rtc_valid_tm(struct rtc_time *tm)
{
	if ( ((unsigned)tm->tm_mon) >= 12
         || tm->tm_mday < 1
         || ((unsigned)tm->tm_hour) >= 24
         || ((unsigned)tm->tm_min) >= 60
         || ((unsigned)tm->tm_sec) >= 60)
		return EINVAL;

    if (6 <= tm->tm_year && tm->tm_year <= 30) {
		printk("seems like this is a valid ds1307 year:%d \n", tm->tm_year);
        return ERANGE;
    }

	return 0;
}

static int __init trying_fix_m41t11(void)
{
	int err;
    int fixme_i = 0;
	struct rtc_time tm;
	struct rtc_device *rtc = rtc_class_open(CONFIG_RTC_HCTOSYS_DEVICE);

	if (rtc == NULL) {
		printk("%s: unable to open rtc device (%s)\n",
               __FILE__, CONFIG_RTC_HCTOSYS_DEVICE);
		return -ENODEV;
	}

    for (; fixme_i < 2; fixme_i++) {
        memset(&tm, 0, sizeof(tm));
        err = rtc_read_time(rtc, &tm);
        if (err == 0) {
            err = my_rtc_valid_tm(&tm);
            if (err) {
                if (EINVAL == err) {
                    dev_err(rtc->dev.parent,
                            "my_rtc_valid_tm error: try to fix by calling set_rtc_to_2000_01_01... \n");
                    set_rtc_to_2000_01_01(rtc);
                } else {
                    dev_err(rtc->dev.parent,
                            "my_rtc_valid_tm error: it's a 1307 year, +100 should be ok... \n");
                    tm.tm_year += 100;
                    rtc_set_time(rtc, &tm);
                }
                continue;
            }
            else {
                if (fixme_i > 0) {
                    dev_err(rtc->dev.parent,
                            "Fix M41T11 to 1970 ok \n");
                } else {
                    dev_err(rtc->dev.parent,
                            "Original M41T11 is ok, no fix \n");
                }
                break;
            }
        }
        else {
            dev_err(rtc->dev.parent,
                    "rtc_read_time err: %d \n", err);
        }
    }
    rtc_class_close(rtc);
    return err;
}


static int __init rtc_hctosys(void)
{
    int err;
    struct rtc_time tm;
    struct rtc_device *rtc;
    
    err = trying_fix_m41t11();
    if (err) {
        printk("failed trying_fix_m41t11 \n");
        return err;
    }

    rtc = rtc_class_open(CONFIG_RTC_HCTOSYS_DEVICE);

    if (rtc == NULL) {
        printk("%s: unable to open rtc device (%s)\n",
               __FILE__, CONFIG_RTC_HCTOSYS_DEVICE);
        return -ENODEV;
    }


    err = rtc_read_time(rtc, &tm);
    if (err == 0) {
        err = rtc_valid_tm(&tm);
        if (err == 0) {
            struct timespec tv;

            tv.tv_nsec = NSEC_PER_SEC >> 1;

            rtc_tm_to_time(&tm, &tv.tv_sec);

            do_settimeofday(&tv);

            dev_info(rtc->dev.parent,
                     "setting the system clock to "
                     "%d-%02d-%02d %02d:%02d:%02d (%u)\n",
                     tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday,
                     tm.tm_hour, tm.tm_min, tm.tm_sec,
                     (unsigned int) tv.tv_sec);
        }
        else
            dev_err(rtc->dev.parent,
                    "hctosys: invalid date/time\n");
    }
    else
        dev_err(rtc->dev.parent,
                "hctosys: unable to read the hardware clock\n");

    rtc_class_close(rtc);

    return 0;
}

late_initcall(rtc_hctosys);
