/*
 * (C) Copyright 2003
 * Author : Hamid Ikdoumi (Atmel)
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <at91_net.h>
#include <net.h>
#include <ksz8041nl.h>

#ifdef CONFIG_DRIVER_ETHER

#if (CONFIG_COMMANDS & CFG_CMD_NET)

/*
 * Name:
 *	dm9161a_IsPhyConnected
 * Description:
 *	Reads the 2 PHY ID registers
 * Arguments:
 *	p_mac - pointer to AT91S_EMAC struct
 * Return value:
 *	TRUE - if id read successfully
 *	FALSE- if error
 */
static unsigned int ksz8041nl_IsPhyConnected (AT91PS_EMAC p_mac)
{
	unsigned short Id1, Id2;
    
	at91_EmacEnableMDIO (p_mac);
	at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_PHYID1, &Id1);
	at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_PHYID2, &Id2);
	at91_EmacDisableMDIO (p_mac);
    
	if ((Id1 == (KSZ8041NL_PHYID1_OUI >> 6)) &&
	    ((Id2 >> 10) == (KSZ8041NL_PHYID1_OUI & KSZ8041NL_LSB_MASK))) {
	  printf("MICREL KSZ8041NL PHY Detected\n\r");
	  return TRUE;
	}
	
	return FALSE;
}

/*
 * Name:
 *	dm9161a_GetLinkSpeed
 * Description:
 *	Link parallel detection status of MAC is checked and set in the
 *	MAC configuration registers
 * Arguments:
 *	p_mac - pointer to MAC
 * Return value:
 *	TRUE - if link status set succesfully
 *	FALSE - if link status not set
 */
static uchar ksz8041nl_GetLinkSpeed (AT91PS_EMAC p_mac)
{
	unsigned short stat1, stat2;

	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMSR, &stat1))
		return FALSE;

/*     printf("stat1 = %x\n", stat1); */
    
	if (!(stat1 & KSZ8041NL_LINK_STATUS))	/* link status up? */
		return FALSE;

	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_PHY_CTRL_2, &stat2))
		return FALSE;

/*     printf("stat2 = %x\n", stat2); */

	if ((stat1 & KSZ8041NL_100BASE_TX_FD) && (stat2 & KSZ8041NL_100FDX)) {
		/*set Emac for 100BaseTX and Full Duplex  */
		p_mac->EMAC_NCFGR |= AT91C_EMAC_SPD | AT91C_EMAC_FD;
		return TRUE;
	}

	if ((stat1 & KSZ8041NL_10BASE_T_FD) && (stat2 & KSZ8041NL_10FDX)) {
		/*set MII for 10BaseT and Full Duplex  */
		p_mac->EMAC_NCFGR = (p_mac->EMAC_NCFGR &
				~(AT91C_EMAC_SPD | AT91C_EMAC_FD))
				| AT91C_EMAC_FD;
		return TRUE;
	}

	if ((stat1 & KSZ8041NL_100BASE_TX_HD) && (stat2 & KSZ8041NL_100HDX)) {
		/*set MII for 100BaseTX and Half Duplex  */
		p_mac->EMAC_NCFGR = (p_mac->EMAC_NCFGR &
				~(AT91C_EMAC_SPD | AT91C_EMAC_FD))
				| AT91C_EMAC_SPD;
		return TRUE;
	}

	if ((stat1 & KSZ8041NL_10BASE_T_HD) && (stat2 & KSZ8041NL_10HDX)) {
		/*set MII for 10BaseT and Half Duplex  */
		p_mac->EMAC_NCFGR &= ~(AT91C_EMAC_SPD | AT91C_EMAC_FD);
		return TRUE;
	}

	return FALSE;
}


/*
 * Name:
 *	dm9161a_ResetPhy
 * Description:
 *	Resets PHY
 * Arguments:
 *	p_mac - pointer to struct AT91S_EMAC
 * Return value:
 *	TRUE - if link status set succesfully
 *	FALSE - if link status not set
 */
static uchar ksz8041nl_ResetPhy(AT91PS_EMAC p_mac)
{
    unsigned short bmcr;
	int timeout = 10;
	uchar ret = TRUE;
	unsigned short Id1, Id2;

	bmcr = KSZ8041NL_RESET;
	at91_EmacEnableMDIO (p_mac);

/* 	at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_PHYID1, &Id1); */
/* 	at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_PHYID2, &Id2); */

/*     printf("Id1: %x, Id2: %x\n", Id1, Id2); */
    
	at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &bmcr);

	do {
	  at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &bmcr);
	  at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &bmcr);
	  timeout--;
	} while ((bmcr & KSZ8041NL_RESET) && timeout);

	at91_EmacDisableMDIO (p_mac);

	if (!timeout)
	  ret = FALSE;

	return ret;
}


/*
 * Name:
 *	dm9161a_InitPhy
 * Description:
 *	MAC starts checking its link by using parallel detection and
 *	Autonegotiation and the same is set in the MAC configuration registers
 * Arguments:
 *	p_mac - pointer to struct AT91S_EMAC
 * Return value:
 *	TRUE - if link status set succesfully
 *	FALSE - if link status not set
 */
static uchar ksz8041nl_InitPhy (AT91PS_EMAC p_mac)
{
	unsigned int tick = 0;
	unsigned short IntValue;
	uchar ret = TRUE;

	at91_EmacEnableMDIO (p_mac);

	do {
		/* Link status is latched, so read twice to get current value */
		at91_EmacReadPhy(p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMSR, &IntValue);
		at91_EmacReadPhy(p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMSR, &IntValue);
	
		tick++;
	} while (!(IntValue & KSZ8041NL_LINK_STATUS) && (tick < AT91C_ETH_TIMEOUT));

	if (!(IntValue & KSZ8041NL_LINK_STATUS)){
		at91_EmacDisableMDIO (p_mac);
		printf ("No link\n\r");
		ret = FALSE;
		return ret;
	}

	if (!ksz8041nl_GetLinkSpeed (p_mac)) {
		/* Try another time */
		ret = ksz8041nl_GetLinkSpeed (p_mac);
	}

	/* Disable PHY Interrupts */
	at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_MDINTR, &IntValue);
	/* clear FDX, SPD, Link, INTR masks */
	IntValue &= ~(KSZ8041NL_LINK_DOWN_MASK | KSZ8041NL_LINK_UP_MASK);
	at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_MDINTR, &IntValue);
	at91_EmacDisableMDIO (p_mac);

	return (ret);
}


/*
 * Name:
 *	ksz8041nl_AutoNegotiate
 * Description:
 *	MAC Autonegotiates with the partner status of same is set in the
 *	MAC configuration registers
 * Arguments:
 *	dev - pointer to struct net_device
 * Return value:
 *	TRUE - if link status set successfully
 *	FALSE - if link status not set
 */
static uchar ksz8041nl_AutoNegotiate (AT91PS_EMAC p_mac, int *status)
{
	unsigned short value;
	unsigned short PhyAnar;
	unsigned short PhyAnalpar;

	/* Set dm9161 control register */
	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &value))
		return FALSE;
	value &= ~KSZ8041NL_AUTONEG;	/* remove autonegotiation enable */
	value |= KSZ8041NL_ISOLATE;	/* Electrically isolate PHY */
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &value))
		return FALSE;

	/* Set the Auto_negotiation Advertisement Register */
	/* MII advertising for Next page, 100BaseTxFD and HD, 10BaseTFD and HD, IEEE 802.3 */
	PhyAnar = KSZ8041NL_NP | KSZ8041NL_TX_FDX | KSZ8041NL_TX_HDX |
		  KSZ8041NL_10_FDX | KSZ8041NL_10_HDX | KSZ8041NL_AN_IEEE_802_3;
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_ANAR, &PhyAnar))
		return FALSE;

	/* Read the Control Register     */
	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &value))
		return FALSE;

	value |= KSZ8041NL_SPEED_SELECT | KSZ8041NL_AUTONEG | KSZ8041NL_DUPLEX_MODE;
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &value))
		return FALSE;
	/* Restart Auto_negotiation  */
	value |= KSZ8041NL_RESTART_AUTONEG;
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMCR, &value))
		return FALSE;

	/*check AutoNegotiate complete */
	udelay (10000);
	at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_BMSR, &value);
	if (!(value & KSZ8041NL_AUTONEG_COMP))
		return FALSE;

	/* Get the AutoNeg Link partner base page */
	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, KSZ8041NL_ANLPAR, &PhyAnalpar))
		return FALSE;

	if ((PhyAnar & KSZ8041NL_TX_FDX) && (PhyAnalpar & KSZ8041NL_TX_FDX)) {
		/*set MII for 100BaseTX and Full Duplex  */
		p_mac->EMAC_NCFGR |= AT91C_EMAC_SPD | AT91C_EMAC_FD;
		return TRUE;
	}

	if ((PhyAnar & KSZ8041NL_10_FDX) && (PhyAnalpar & KSZ8041NL_10_FDX)) {
		/*set MII for 10BaseT and Full Duplex  */
		p_mac->EMAC_NCFGR = (p_mac->EMAC_NCFGR &
				~(AT91C_EMAC_SPD | AT91C_EMAC_FD))
				| AT91C_EMAC_FD;
		return TRUE;
	}
	return FALSE;
}


/*
 * Name:
 *	at91_GetPhyInterface
 * Description:
 *	Initialise the interface functions to the PHY
 * Arguments:
 *	None
 * Return value:
 *	None
 */
void at91_GetPhyInterface(AT91PS_PhyOps p_phyops)
{
	p_phyops->Init           = ksz8041nl_InitPhy;
	p_phyops->IsPhyConnected = ksz8041nl_IsPhyConnected;
	p_phyops->GetLinkSpeed   = ksz8041nl_GetLinkSpeed;
	p_phyops->AutoNegotiate  = ksz8041nl_AutoNegotiate;
	p_phyops->Reset          = ksz8041nl_ResetPhy;
}

#endif	/* CONFIG_COMMANDS & CFG_CMD_NET */

#endif	/* CONFIG_DRIVER_ETHER */

