/******************************************************************
 * @file   mkenv.c
 * @author Richard Luo
 * @date   2009-03-02
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include <stdio.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <asm/ioctls.h>
#include <sys/ioctl.h>
#include <string.h>
#include <zlib.h>

#include <libgen.h>

#include "at91sam9260ek.h"
#include "environment.h"

static void print_usage(const char *prog)
{
    char buf[1024];
    bcopy(prog, buf, sizeof(buf)-1 );

	printf("usage: %s [OPTION]... [FILE]...\n", basename(buf) );
	puts(
        "  -i --input-file             specify the input text file      \n"
        "  -o --outpu-bin-file         specify the output binary file   \n"
        "  -t --text                   to convert bin file to txt file  \n"
        "  -b --bin                    to convert txt file to bin file  \n"
        );
	exit(1);
}


static const char *input_file = 0;
static const char *output_file = 0;
static const char *conv_type = "bin";

static void parse_opts(int argc, char *argv[])
{
	while (1) {
		static const struct option lopts[] = {
			{ "input-file",             0, 0, 'f' },
			{ "output-bin-file",        0, 0, 'o' },
			{ "text",                   0, 0, 't' },
			{ "bin",                    0, 0, 'b' },
			{ NULL, 0, 0, 0 },
		};
		int c;

		c = getopt_long(argc, argv, "i:o:tb", lopts, NULL);

		if (c == -1)
			break;

		switch (c) {

		case 'i':
			input_file = optarg;
			break;

		case 'o':
            output_file = optarg;
			break;

		case 't':
            conv_type = "txt";
			break;

		case 'b':
			break;

		default:
			print_usage(argv[0]);
			break;
		}
	}

    if (!input_file || !output_file)
        print_usage(argv[0]);
    
}


static char *push_back(char *dst, const char *src)
{
    int len = strlen(src) + 1;
    memcpy(dst, src, len);
    return dst + len;
}


static void write2file(const void *buf, int n, const char *path)
{
    FILE *fp = fopen(path, "w+");
    const char *pbuf = buf;

    assert(fp);

    assert(n == fwrite(pbuf, 1, n, fp) );
    fclose(fp);
}

/* unsigned long crc32_m(crc, buf, len) */
/*     unsigned long crc; */
/*     const char *buf; */
/*     unsigned int len; */
/* { */
/*     crc = crc ^ 0xffffffffL; */
/*     while (len >= 8) */
/*     { */
/*         DO8(buf); */
/*         len -= 8; */
/*     } */
/*     if (len) do { */
/*             DO1(buf); */
/*         } while (--len); */
/*     return crc ^ 0xffffffffL; */
/* } */


static void convert_bin2txt (const char *binfile, const char *txtfile)
{
    char *p, *pe;
    FILE *ifp, *ofp;
    env_t eobj;

    bzero(&eobj, sizeof(eobj) );

    assert( (ifp = fopen(binfile, "r")) );
    assert( (ofp = fopen(txtfile, "w+")) );

    assert (sizeof(eobj) == fread(&eobj, 1, sizeof(eobj), ifp));
    assert (eobj.crc == crc32(0, eobj.data, ENV_SIZE));

    printf("crc is ok \n");

    p = (char*) eobj.data;
    pe = (char*) &eobj.data[sizeof(eobj.data)]; 

    while (*p) {
        fprintf(ofp, "%s\n", p);
        printf("%s\n", p);
        p = memchr(p, '\0', pe - p);
        if (p) assert( *p == '\0');
        ++p;
    }
    fclose(ifp);
    fclose(ofp);
}


#include <ctype.h>

static void convert_txt2bin (const char *txtfile, const char *binfile)
{
    char *p;
    FILE *fp;
    env_t eobj;
    char line[256] = {0};

    assert( (fp = fopen(txtfile, "r")) );

    p = (char*) eobj.data;

    while (line == fgets(line, sizeof(line), fp) ) {
        char *pc;
        pc = strchr(line, '\n');
        if (pc)
            *pc = 0;
        
        if ( !line[0] || !isalpha(line[0]) )
            continue;
        printf("before processing [%s] \n", line);
        p = push_back(p, line);
        bzero(line, sizeof(line) );
    }

    eobj.crc = crc32(0, eobj.data, ENV_SIZE);
    write2file(&eobj, sizeof(eobj), binfile);

    fclose(fp);
}


int main(int argc, char *argv[])
{
    parse_opts(argc, argv);

    if (strcmp(conv_type, "bin") == 0)
        convert_txt2bin(input_file, output_file);
    else
        convert_bin2txt(input_file, output_file);
    
    return 0;
}

