/*------------------------------------------------------------------------
  net9eth
  code related to a linux ethernet device driver

  Copyright (C) 2001,2002 by Joachim Franek (joachim.franek@t-online.de)
  for more information see: rs485.de-franek.de
  Redistribution of this file is permitted under the terms of the GNU Public License (GPL)
  ------------------------------------------------------------------------*/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

#include <linux/config.h>
#include <linux/module.h>

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/interrupt.h>

#include <linux/in.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/skbuff.h>

#include "net9eth.h"

#include <linux/in6.h>
#include <asm/checksum.h>

MODULE_AUTHOR("Joachim Franek");

/*------------------------------------------------------------------------
  This are the load-time options: Module Parameters: n9adr, n9baud
  ------------------------------------------------------------------------*/
static int n9adr = 1;						// this sets the own adsress
MODULE_PARM(n9adr, "b");
MODULE_PARM_DESC(n9adr, "The net9 own address (0-255)");

static int n9baud = 38400;			//this sets the baud rate
MODULE_PARM(n9baud, "i");				//not used yet
MODULE_PARM_DESC(n9baud, "The baudrate (1200, 2400, 4800, 9600, 19200, 38400) not used yet!");


/*------------------------------------------------------------------------
  This structure is private to each device. It is used to pass
  packets in and out, so there is place for a packet
  ------------------------------------------------------------------------*/

struct net9eth_priv
{
	struct net_device_stats stats;
	int status;
	int rx_packetlen;
	u8 *rx_packetdata;
	int tx_packetlen;
	u8 *tx_packetdata;
	struct sk_buff *skb;
	spinlock_t lock;
};

extern struct net_device net9eth_devs[];

#include "net9.h"

/*------------------------------------------------------------------------
  Open: called at "ifconfig ethx x.x.x.x up" time
  ------------------------------------------------------------------------*/
int net9eth_open(struct net_device *dev)
{
    MOD_INC_USE_COUNT;
    memcpy(dev->dev_addr, "\0NET9x", ETH_ALEN);	//mac address
    dev->dev_addr[ETH_ALEN - 1] = n9adr;	// own address as last byte
    netif_start_queue(dev);
    return 0;
}

/*------------------------------------------------------------------------
  ifconfig ethx down
  ------------------------------------------------------------------------*/
int net9eth_release(struct net_device *dev)
{
    netif_stop_queue(dev);
    MOD_DEC_USE_COUNT;
    return 0;
}

/*------------------------------------------------------------------------
  Configuration changes (ifconfig xxx)
  ------------------------------------------------------------------------*/
int net9eth_config(struct net_device *dev, struct ifmap *map)
{
    printk("net9eth config\n");
    if(dev->flags & IFF_UP)
        return -EBUSY;
    return 0;
};

/*------------------------------------------------------------------------
  Receive a packet: retrieve, encapsulate and pass over to upper levels
  ------------------------------------------------------------------------*/
void net9eth_rx(struct net_device *dev, int len, unsigned char *buf)
{
    struct sk_buff *skb;
    struct net9eth_priv *priv = (struct net9eth_priv *)dev->priv;

    skb = dev_alloc_skb(len + 2);
    if(!skb)
	{
        printk("net9eth rx: low on mem - packet dropped\n");
        priv->stats.rx_dropped++;
        return;
	};
    skb_reserve(skb, 2);
    net9eth_read_payload_t2(skb_put(skb, len), len);
    skb->dev = dev;
    skb->protocol = eth_type_trans(skb, dev);
    skb->ip_summed = CHECKSUM_UNNECESSARY;
    priv->stats.rx_packets++;
    netif_rx(skb);
    return;
};

/*------------------------------------------------------------------------
  the interrupt function: only for rx
  ------------------------------------------------------------------------*/
void net9eth_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
    struct net9eth_priv *priv;
    struct net_device *dev = net9eth_devs;

    if(!dev) return;
    priv = (struct net9eth_priv *)dev->priv;
    spin_lock(&priv->lock);
    priv->rx_packetlen = (int)net9_frame_length();
    priv->rx_packetdata = NULL;
    net9eth_rx(dev, priv->rx_packetlen, priv->rx_packetdata);
    spin_unlock(&priv->lock);
    return;
};

/*------------------------------------------------------------------------
  transmit a packet
  ------------------------------------------------------------------------*/
void net9eth_hw_tx(char *buf, int len, struct net_device *dev)
{
    struct iphdr *ih;
    struct net9eth_priv *priv;
    u32 *saddr, *daddr;
    u8 net9_dest;

    if(len < (sizeof(struct ethhdr) + sizeof(struct iphdr)))
	{
        printk("net9eth: error tx length? (%i octets)\n", len);
        return;
	};
    ih = (struct iphdr *)(buf + sizeof(struct ethhdr));
    saddr = &ih->saddr;
    daddr = &ih->daddr;
    net9_dest = ((u8 *) daddr)[3];
    net9eth_write(buf, len, net9_dest, frame_type_2);
    priv = (struct net9eth_priv *)dev->priv;
    priv->stats.tx_packets++;
    priv->stats.tx_bytes += len;
    dev_kfree_skb(priv->skb);
};

/*------------------------------------------------------------------------
  transmit a packet
  ------------------------------------------------------------------------*/
int net9eth_tx(struct sk_buff *skb, struct net_device *dev)
{
    int len;
    char *data;

    struct net9eth_priv *priv = (struct net9eth_priv *)dev->priv;
    len = skb->len < ETH_ZLEN ? ETH_ZLEN : skb->len;
    data = skb->data;
    dev->trans_start = jiffies;
    priv->skb = skb;
    net9eth_hw_tx(data, len, dev);
    return 0;
}

/*------------------------------------------------------------------------
  transmit timeout.
  ------------------------------------------------------------------------*/
void net9eth_tx_timeout(struct net_device *dev)
{
    struct net9eth_priv *priv = (struct net9eth_priv *)dev->priv;
    priv->stats.tx_errors++;
    netif_wake_queue(dev);
    return;
};

/*------------------------------------------------------------------------
  Ioctl commands
  ------------------------------------------------------------------------*/
int net9eth_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    printk("net9eth: ioctl %i not valid\n", cmd);
    return 0;
};

/*------------------------------------------------------------------------
  Return statistics to the caller
  ------------------------------------------------------------------------*/
struct net_device_stats *net9eth_stats(struct net_device *dev)
{
    struct net9eth_priv *priv = (struct net9eth_priv *)dev->priv;
    return &priv->stats;
};

/*------------------------------------------------------------------------
  buildt a ethernet header
  ------------------------------------------------------------------------*/
int net9eth_rebuild_header(struct sk_buff *skb)
{
    struct ethhdr *eth = (struct ethhdr *)skb->data;
    struct net_device *dev = skb->dev;
    char *buf;
    struct iphdr *ih;
    u32 *ip_daddr;

    memcpy(eth->h_source, dev->dev_addr, dev->addr_len);
    memcpy(eth->h_dest, dev->dev_addr, dev->addr_len);

    buf = skb->data;
    ih = (struct iphdr *)(buf + sizeof(struct ethhdr));
    ip_daddr = &ih->daddr;
    eth->h_dest[ETH_ALEN - 1] = (((*ip_daddr) & 0xff000000) >> 24);
    return 0;
};

/*------------------------------------------------------------------------*/
int net9eth_header(struct sk_buff *skb, struct net_device *dev, unsigned short type, void *daddr, void *saddr, unsigned int len)
{
    struct ethhdr *eth = (struct ethhdr *)skb_push(skb, ETH_HLEN);
    char *buf;
    struct iphdr *ih;
    u32 *ip_saddr, *ip_daddr;

    eth->h_proto = htons(type);
    len = skb->len < ETH_ZLEN ? ETH_ZLEN : skb->len;
    buf = skb->data;
    ih = (struct iphdr *)(buf + sizeof(struct ethhdr));
    ip_saddr = &ih->saddr;
    ip_daddr = &ih->daddr;
    if(saddr != 0)
	{
        memcpy(eth->h_source, saddr, dev->addr_len);
	}
    else
	{
        memcpy(eth->h_source, dev->dev_addr, dev->addr_len);
	};
    memcpy(eth->h_dest, dev->dev_addr, dev->addr_len);
    eth->h_dest[ETH_ALEN - 1] = (((*ip_daddr) & 0xff000000) >> 24);
    return (dev->hard_header_len);
};

/*------------------------------------------------------------------------
  ------------------------------------------------------------------------*/
int net9eth_change_mtu(struct net_device *dev, int new_mtu)
{
    unsigned long flags;
    spinlock_t *lock = &((struct net9eth_priv *)dev->priv)->lock;

    if((new_mtu < 68) || (new_mtu > 1500)) return -EINVAL;
    spin_lock_irqsave(lock, flags);
    dev->mtu = new_mtu;
    spin_unlock_irqrestore(lock, flags);
    return 0;
}

/*------------------------------------------------------------------------
  code executes at ifconfig time
  ------------------------------------------------------------------------*/
int net9eth_init(struct net_device *dev)
{

    ether_setup(dev);

    dev->open = net9eth_open;
    dev->stop = net9eth_release;
    dev->set_config = net9eth_config;
    dev->hard_start_xmit = net9eth_tx;
    dev->do_ioctl = net9eth_ioctl;
    dev->get_stats = net9eth_stats;
    dev->change_mtu = net9eth_change_mtu;
    dev->rebuild_header = net9eth_rebuild_header;
    dev->hard_header = net9eth_header;
    dev->flags |= IFF_NOARP;
    dev->hard_header_cache = NULL;
    SET_MODULE_OWNER(dev);
    dev->priv = kmalloc(sizeof(struct net9eth_priv), GFP_KERNEL);
    if(dev->priv == NULL)
        return -ENOMEM;
    memset(dev->priv, 0, sizeof(struct net9eth_priv));
    spin_lock_init(&((struct net9eth_priv *)dev->priv)->lock);
    return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  code runs at insmod and rmmod time
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*------------------------------------------------------------------------
  ------------------------------------------------------------------------*/
struct net_device net9eth_devs[1] =
{
    {
    init:net9eth_init,
    }
};

/*------------------------------------------------------------------------*/
int net9eth_init_module(void)
{
    int result, device_present = 0;

    if(net9eth_get_irq() != 0) return -ENODEV;
    strcpy(net9eth_devs[0].name, "eth%d");
    if((result = register_netdev(net9eth_devs)))
	{
        printk("net9eth: error %i registering device \"%s\"\n", result, net9eth_devs[0].name);
	}
    else
	{
        device_present++;
        printk("net9eth: %i registering device \"%s\"\n", result, net9eth_devs[0].name);
        printk("net9eth: own address %i \n", n9adr);
	};
    EXPORT_NO_SYMBOLS;
    return device_present ? 0 : -ENODEV;
}

/*------------------------------------------------------------------------*/
void net9eth_cleanup(void)
{
    printk("net9eth cleanup module\n");
    net9eth_cleanup_irq();
    kfree(net9eth_devs[0].priv);
    unregister_netdev(net9eth_devs);
    return;
}

/*------------------------------------------------------------------------*/
module_init(net9eth_init_module);
module_exit(net9eth_cleanup);

// that's it
