/******************************************************************
 * @file   card_reader.c
 * @author Richard Luo
 * @date   2008-05-22
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>

#include <linux/init.h>
#include <linux/poll.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>

#include <asm/arch/gpio.h>
#include <asm/arch/at91_pio.h>
#include <linux/proc_fs.h>


#include "dbgpk.h"
#include "reader_buf.h"
#include "gpio_reader.h"
#include "card_reader.h"

static card_reader_dev_t *card_reader_device;

static atomic_t card_reader_available = ATOMIC_INIT(2);

static int reader_major;
module_param(reader_major, int, 0);

MODULE_LICENSE("Dual BSD/GPL");

//      .pin_data0_ = AT91_PIN_PC13, /* test psoc */
//      .irq_name0_ = "psoc_test_interrupt",

static gpio_reader_t all_readers[] = {
    {
      .pin_data0_ = AT91_PIN_PB11,
      .pin_data1_ = AT91_PIN_PB13,
      .pin_led_  = AT91_PIN_PB12,
      .pin_buzz_  = AT91_PIN_PC1,
      .irq_name0_ = "reader0_data0",
      .irq_name1_ = "reader0_data1",
      .state_ = RD_STAT_INIT,
      .last_fix_val_ = 0,
      .fix_reason_ = 0
    },
    {
      .pin_data0_ = AT91_PIN_PC0,
      .pin_data1_ = AT91_PIN_PC2,
      .pin_led_  = AT91_PIN_PB14,
      .pin_buzz_  = AT91_PIN_PC3,
      .irq_name0_ = "reader1_data0",
      .irq_name1_ = "reader1_data1",
      .state_ = RD_STAT_INIT,
      .last_fix_val_ = 0,
      .fix_reason_ = 0
    },
    
};

enum {
    NUM_OF_READERS = sizeof(all_readers)/sizeof(gpio_reader_t)
};


int reader_read_procmem(char *buf, char **start, off_t offset,
                        int count, int *eof, void *data)
{
    extern void reader_interval_accounter_dump (void);
    reader_interval_accounter_dump();
	*eof = 1;
	return 0;
}

wait_queue_head_t* reader_wait_queue(void)
{
    return &card_reader_device->wait_;
}


static int reader_release(struct inode *inode, struct file *filp)
{
	atomic_inc(&card_reader_available); /* release the device */
	return 0;
}

static int reader_open(struct inode *inode, struct file *filp)
{

	card_reader_dev_t *dev; /* device information */

	if (0 >  atomic_sub_return(1, &card_reader_available)) {
        printk(KERN_DEBUG "huhu~ i am busy card_reader_available: %d \n", atomic_read(&card_reader_available) );
		atomic_inc(&card_reader_available);
		return -EBUSY; /* already open */
	}

	dev = container_of(inode->i_cdev, card_reader_dev_t, cdev_);
	filp->private_data = dev; /* for other methods */

	return 0;
}


/** 
 * 
 * 
 * @param dev 
 * 
 * @return -1 if no data, else return the reader no.
 */
static int card_data_available(card_reader_dev_t *dev)
{
    int i = 0;

    for ( i=0; i < dev->num_; ++i ) {
        gpio_reader_t *rd = &dev->rds_[i];
        if ( !reader_buf_empty(rd->rdbuf_) )
            return i;
    }

    return -1;
}

static void bit_swap(void *addr, int bitnum, int posi)
{
    int last_idx = bitnum - 1;
    uint8_t v0 = test_bit(posi, addr) ? 1 : 0;
    uint8_t v1 = test_bit(last_idx-posi, addr) ? 1 : 0;

    BUG_ON(posi < 0 || posi >= bitnum);

    if ( v0 == v1) return;

    if (v0)     set_bit(last_idx-posi, addr);
    else        clear_bit(last_idx-posi, addr);

    if (v1)     set_bit(posi, addr);
    else        clear_bit(posi, addr);

}


static void bits_revert(void *addr, int bitsnum)
{
    int i;

    BUG_ON(bitsnum < 0);
    for (i=0; i < bitsnum/2; ++i)
        bit_swap(addr, bitsnum, i);
}

static void byte_revert(uint8_t *b)
{
    bits_revert(b, 8);
}


/** 
 * @brief 
 * 
 * @param drvdata 
 * @param nbits 
 * @param size 
 * 
 * @return -1 on erro, else total bytes.
 */
static int format_card_data(void *drvdata, int nbits, size_t size, int rdno)
{
    int i;
    typedef struct  {
        uint8_t reader_no_;
        uint8_t num_of_bits_;
        uint8_t num_of_bytes_;
        uint8_t num_of_ignore_bits_;

        uint8_t data_[128];
    } card_format_t;

    int TOTAL_SIZE;

    const int HEADER_SIZE = sizeof(card_format_t) - 128;

    card_format_t f;

    f.num_of_bytes_ = (nbits+7)/8;

    TOTAL_SIZE = f.num_of_bytes_ + HEADER_SIZE;

    if (TOTAL_SIZE > size)      // not enough space for format conv.
        return -1;
    
    f.num_of_bits_ = nbits;
    f.reader_no_ = (uint8_t)rdno;
    f.num_of_ignore_bits_ = 8 - f.num_of_bits_%8;

    /* bits_revert(drvdata, nbits); */

    memcpy(f.data_, drvdata, f.num_of_bytes_);

    for (i=0; i < f.num_of_bytes_; ++i)
        byte_revert(&f.data_[i]);

#ifdef DEBUG_READER
    dbgpk("after format: \n");
    mem_byte_bits_dump(f.data_, f.num_of_bits_);
#endif

    memcpy(drvdata, &f, TOTAL_SIZE);
    return TOTAL_SIZE;
    
}


static ssize_t reader_read(struct file *filp, char __user *buf, size_t count,
                           loff_t *f_pos)
{
    int rdno;                   /* reader no. */
    int retval;
    card_buf_t *cb;
    gpio_reader_t *rd;
    struct card_reader_dev *dev;

    dev = filp->private_data;
	if (down_interruptible (&dev->sem_))
		return -ERESTARTSYS;

    while ( -1 == (rdno = card_data_available(dev)) ) {
		up(&dev->sem_); /* release the lock */
		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;
		PDEBUG("\"%s\" reading: going to sleep\n", current->comm);
		if (wait_event_interruptible(dev->wait_, (0 <= card_data_available(dev))) )
			return -ERESTARTSYS; /* signal: tell the fs layer to handle it */

		/* otherwise loop, but first reacquire the lock */
		if (down_interruptible(&dev->sem_)) /*  */
			return -ERESTARTSYS;

    }

    rd = &dev->rds_[rdno];
    cb = reader_buf_get_for_read(rd->rdbuf_);
    BUG_ON(!cb);

#ifdef DEBUG
    if (cb && (cb->bits_ != 26 &&
               cb->bits_ != 34 &&
               cb->bits_ != 32) ) {
        printk(KERN_DEBUG "Error Maybe!!!!!!!!!!!!!!!!  \n");
        printk("before format: let's see the card data: \n");
        mem_byte_bits_dump(cb->buf_, cb->bits_);
    }
#endif    


    retval = format_card_data(cb->buf_, cb->bits_, MAX_CARD_DATA_SIZE, rdno);

	if (count < retval || copy_to_user(buf, cb->buf_, retval)) {
        printk(KERN_DEBUG "count: [%d], retval:[%d] \n", count, retval);
		retval = -EFAULT;
		goto out;
	}

out:    
    if(cb) card_buf_clear(cb);
    up(&dev->sem_); /* release the lock */

    return retval;
}

/** 
 * @brief for trunning on/off LED and RELAY
 * 
 * @param dev pointer to all reader devs
 * @param rdno to which reader 
 * @param cmd to do what?
 */
static inline void reader_output_ctrls(card_reader_dev_t *dev, int rdno, unsigned cmd)
{
    int pin, level;
    gpio_reader_t *rd;

    if (dev->num_ <= rdno || rdno < 0) {
        dbgpk("invalid rdno: [%d] \n", rdno);
        return;
    }

    rd = &dev->rds_[rdno];

	switch(cmd) {

	case READER_IOCT_LED_ON:
        pin = rd->pin_led_;
        level = 1;
		break;

	case READER_IOCT_LED_OFF:
        pin = rd->pin_led_;
        level = 0;
		break;

    case READER_IOCT_BUZZ_ON:
        pin = rd->pin_buzz_;
        level = 1;
		break;

    case READER_IOCT_BUZZ_OFF:
        pin = rd->pin_buzz_;
        level = 0;
		break;
        

    default:
        pin = -1;
        level = -1;
		break;
    }

    if (level >= 0) {
        gpio_direction_output(pin, level);
    }
}

static int reader_ioctl (struct inode *inode, struct file *filp,
                         unsigned int cmd, unsigned long arg)
{

	int err = 0;

	struct card_reader_dev *dev;
    dev = filp->private_data;

	/* don't even decode wrong cmds: better returning  ENOTTY than EFAULT */
	if (_IOC_TYPE(cmd) != READER_IOC_MAGIC) return -ENOTTY;
	if (_IOC_NR(cmd) > READER_IOC_MAXNR) return -ENOTTY;

	/*
	 * the type is a bitmask, and VERIFY_WRITE catches R/W
	 * transfers. Note that the type is user-oriented, while
	 * verify_area is kernel-oriented, so the concept of "read" and
	 * "write" is reversed
	 */
	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd));
	else if (_IOC_DIR(cmd) & _IOC_WRITE)
		err =  !access_ok(VERIFY_READ, (void __user *)arg, _IOC_SIZE(cmd));
	if (err)
		return -EFAULT;

	switch(cmd) {

	case READER_IOCT_RELAY_ON:
	case READER_IOCT_RELAY_OFF:
	case READER_IOCT_LED_ON:
	case READER_IOCT_LED_OFF:
    case READER_IOCT_BUZZ_ON:
    case READER_IOCT_BUZZ_OFF:
        reader_output_ctrls(dev, arg, cmd);
        break;
	default:  /* redundant, as cmd was checked against MAXNR */
		return -ENOTTY;
	}

	return 0;
}



static unsigned int reader_poll (struct file *filp, poll_table *wait)
{
	card_reader_dev_t *dev; /* device information */
	unsigned int mask = 0;

	dev = filp->private_data;

#if 0
	if (down_interruptible (&dev->sem_))
		return -ERESTARTSYS;
#else
	down(&dev->sem_);
#endif    

	poll_wait(filp, &dev->wait_, wait);

    if (card_data_available(dev) >= 0)
        mask |= POLLIN | POLLRDNORM;	/* readable */

	up(&dev->sem_);

	return mask;
}



struct file_operations card_reader_fops = {
	.owner =     THIS_MODULE,
	.read =	     reader_read,
	.open =	     reader_open,
	.ioctl =     reader_ioctl,
	.release =   reader_release,
    .poll = reader_poll,
};

/** 
 * 
 * @return 0 on success.
 */
static int card_reader_dev_setup (card_reader_dev_t *dev)
{
	int err, devno = MKDEV(reader_major, 0);

	memset(dev, 0, sizeof(struct card_reader_dev));
    dev->rds_ = all_readers;
    dev->num_ = NUM_OF_READERS;

	init_MUTEX(&(dev->sem_));

    if ( gpio_reader_setup_n(dev->rds_, dev->num_) ) {
        printk(KERN_DEBUG "Error: gpio_reader_setup_n failed \n");
        return -1;
    }

    init_waitqueue_head(&dev->wait_);

	cdev_init(&dev->cdev_, &card_reader_fops);
	dev->cdev_.owner = THIS_MODULE;
	err = cdev_add (&dev->cdev_, devno, 1);

	/* Fail gracefully if need be */
	if (err) {
		printk(KERN_NOTICE "Error %d adding card reader \n", err);
        return -1;
    }

    return 0;
}


static void card_reader_dev_clean (card_reader_dev_t *dev)
{
    gpio_reader_destroy_n(dev->rds_, dev->num_);
    cdev_del(&dev->cdev_);
    kfree(dev);
	unregister_chrdev_region(MKDEV(reader_major, 0), 1);
}


static int card_reader_init (void)
{
	int result;
	dev_t dev = MKDEV(reader_major, 0);
	
	/*
	 * Register your major, and accept a dynamic number.
	 */
	if (reader_major)
		result = register_chrdev_region(dev, 1, "card_reader");
	else {
		result = alloc_chrdev_region(&dev, 0, 1, "card_reader");
		reader_major = MAJOR(dev);
	}
    
    printk(KERN_DEBUG "reader_major: [%d] \n", reader_major);

	if (result < 0)
		return result;

	card_reader_device = kmalloc( sizeof(struct card_reader_dev), GFP_KERNEL);
	if (!card_reader_device) {
		result = -ENOMEM;
		goto fail_malloc;
	}

    if ( card_reader_dev_setup(card_reader_device) ) {
        result = -1;            /* may add a meaningful val here. */
        goto fail_dev_setup;
    }
    
	create_proc_read_entry("ak_reader", 0, NULL, reader_read_procmem, NULL);

	return 0; /* succeed */

fail_dev_setup:
    kfree(card_reader_device);
    
fail_malloc:
	unregister_chrdev_region(dev, 1);
	return result;
}

static void card_reader_exit (void)
{
	printk(KERN_ALERT "Goodbye, reader \n");
    card_reader_dev_clean(card_reader_device);
    remove_proc_entry("ak_reader", NULL);
}

module_init(card_reader_init);
module_exit(card_reader_exit);
