/******************************************************************
 * @file   ak_DIP.c
 * @author chunfeng wei
 * @date   2008-10-07
 * 
 * @brief  DIP driver to get panel ID and node type
 * 
 ****************************************************************** 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>

#include <linux/init.h>
#include <linux/poll.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>

#include <linux/kernel.h>
#include <asm/arch/gpio.h>
#include <asm/arch/at91_pio.h>
#include <linux/proc_fs.h>
#include <linux/cdev.h>
#include "ak_DIP.h"

static atomic_t DIP_available = ATOMIC_INIT(1);

static int ak_DIP_major;
module_param(ak_DIP_major, int, 0);

MODULE_LICENSE("Dual BSD/GPL");

typedef struct {
    int dip_ID_;
} dip_t;

typedef struct {
    dip_t *dip_;
    int dip_num_;
    struct cdev cdev_;
}DIP_dev_t;

static dip_t g_dips[] = {
    {
        .dip_ID_ = DIP1,
    },
    {
        .dip_ID_ = DIP2,
    },
    {
        .dip_ID_ = DIP3,
    },
    {
        .dip_ID_ = DIP4,
    },    
    {
        .dip_ID_ = DIP5,
    },
    {
        .dip_ID_ = DIP6,
    },
    {
        .dip_ID_ = DIP7,
    },
    {
        .dip_ID_ = DIP8,
    },
};

DIP_dev_t ak_DIP_dev = {
    .dip_ = g_dips,
    .dip_num_ = ARRAY_SIZE(g_dips),
};


int DIP_read_procmem (char *buf, char **start, off_t offset,
                         int count, int *eof, void *data)
{
    int n;
    n = sprintf(buf, "%s", "hello DIP's proc \n");
	*eof = 1;
	return n;
}

static int ak_DIP_release (struct inode *inode, struct file *filp)
{
	atomic_inc(&DIP_available); /* release the device */
	return 0;
}

static unsigned int ak_dip_value (DIP_dev_t *dev)
{
    unsigned int value = 0;
    int i = 0;
    
    for (i=0; i<dev->dip_num_; i++)
        value |= (!(at91_get_gpio_value(dev->dip_[i].dip_ID_))) << i;

    return value;
}

static int ak_DIP_open (struct inode *inode, struct file *filp)
{

	DIP_dev_t *dev; /* device information */

	if (! atomic_dec_and_test (&DIP_available)) {
		atomic_inc(&DIP_available);
		return -EBUSY; /* already open */
	}

	dev = container_of(inode->i_cdev, DIP_dev_t, cdev_);
	filp->private_data = dev; /* for other methods */

	return 0;
}

static int ak_DIP_ioctl (struct inode *inode, struct file *filp,
                         unsigned int cmd, unsigned long arg)
{

	int err = 0;
	DIP_dev_t *dev;
    unsigned int value;
    int retval = -1;

	/* don't even decode wrong cmds: better returning  ENOTTY than EFAULT */
	if (_IOC_TYPE(cmd) != AK_DIP_IOC_MAGIC) return -ENOTTY;
	if (_IOC_NR(cmd) > AK_DIP_IOC_MAXNR) return -ENOTTY;

	/*
	 * the type is a bitmask, and VERIFY_WRITE catches R/W
	 * transfers. Note that the type is user-oriented, while
	 * verify_area is kernel-oriented, so the concept of "read" and
	 * "write" is reversed
	 */
	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd));
	else if (_IOC_DIR(cmd) & _IOC_WRITE)
		err =  !access_ok(VERIFY_READ, (void __user *)arg, _IOC_SIZE(cmd));
	if (err)
		return -EFAULT;

	switch(cmd) {
        
	case AK_DIP_VALUE_GET:
        dev = filp->private_data;
        value = ak_dip_value(dev);
        retval = put_user(value, (unsigned int __user *)arg);
        retval = 0;
        break;
	default:  /* redundant, as cmd was checked against MAXNR */
		return -ENOTTY;
	}

    return retval;
}                            

struct file_operations DIP_fops = {
	.owner =     THIS_MODULE,
	.open =	     ak_DIP_open,
    .ioctl =     ak_DIP_ioctl,
	.release =   ak_DIP_release,
};

/** 
 * @param dev 
 * @param devno 
 * 
 * @return 0 on success
 */
static int DIP_dev_setup (DIP_dev_t *dev, dev_t devno)
{
    int err, i;
    
    for (i = 0; i < dev->dip_num_; ++i)
        at91_set_gpio_input(dev->dip_[i].dip_ID_, 0);

	cdev_init(&dev->cdev_, &DIP_fops);
	dev->cdev_.owner = THIS_MODULE;
	err = cdev_add (&dev->cdev_, devno, 1);

	if (err) {
		printk(KERN_NOTICE "Error %d DIP cdev setup \n", err);
        return -1;
    }

    return 0;
}

static void DIP_dev_destroy (DIP_dev_t *dev)
{
    cdev_del(&dev->cdev_);
	unregister_chrdev_region(MKDEV(ak_DIP_major, 0), 1);
}

static int ak_DIP_init (void)
{
	int result;
	dev_t dev = MKDEV(ak_DIP_major, 0);
	
	/*
	 * Register your major, and accept a dynamic number.
	 */
	if (ak_DIP_major)
		result = register_chrdev_region(dev, 1, "ak_DIP");
	else {
		result = alloc_chrdev_region(&dev, 0, 1, "ak_DIP");
		ak_DIP_major = MAJOR(dev);
	}
    
	if (result < 0)
		return result;

    result = DIP_dev_setup(&ak_DIP_dev, dev);

    if (result)
        unregister_chrdev_region(dev, 1);
    
#ifdef DIP_USE_PROC /* only when available */
	create_proc_read_entry("ak_DIP", 0, NULL, DIP_read_procmem, NULL);
#endif

	return result;
}

static void ak_DIP_exit (void)
{

    DIP_dev_destroy(&ak_DIP_dev);
	printk(KERN_ALERT "Goodbye, DIP \n");
}
 
module_init(ak_DIP_init);
module_exit(ak_DIP_exit);
