/******************************************************************
 * @file   rs485_bus.h
 * @author Richard Luo
 * @date   2008-11-07
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#ifndef _RS485_BUS_H
#define _RS485_BUS_H

#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/list.h>
#include <linux/dma-mapping.h>
#include <linux/serial.h>
#include <asm/arch-at91/board.h>
#include <linux/skbuff.h>

#include "atmel_serial.h"

struct rs485_priv {
	struct uart_port            uart;		/* uart */
	struct clk                  *clk;		/* uart clock */

	short                       use_dma_rx;	/* enable PDC receiver */
	short                       pdc_rx_idx;	/* current PDC RX buffer */
	struct atmel_dma_buffer     pdc_rx[2];	/* PDC receier */

	short                       use_dma_tx;	/* enable PDC transmitter */
	struct atmel_dma_buffer     pdc_tx;		/* PDC transmitter */


	// /* Transmit */
	// struct sk_buff *skb;			/* holds skb until xmit interrupt completes */
	// dma_addr_t skb_physaddr;		/* phys addr from pci_map_single */
	// int skb_length;				/* saved skb length for pci_unmap_single */

    struct sk_buff_head         txq; // socket buffer queue

    uint8_t                     rx_timeout:1; /* start of frame end processing */
    uint8_t                     tx_timeout:1; /* start of frame end processing */
    uint8_t                     data_sent:1;   /* first time to get token to send packet */
    uint8_t                     break_sent:1;
    int                         token_status:28;          /* token state */

    struct platform_driver      usart_drv;
    u16                         cur_owner; /* address for current token owner, 0x8000 for unknown owner or free case */
    u16                         self_addr; 
};

static inline
struct uart_port *rs485_priv_usart (struct rs485_priv *ppriv)
{
    return &ppriv->uart;
}

static inline void
rs485_enqueue (struct rs485_priv *ppriv, struct sk_buff *newskb)
{
    skb_queue_tail(&ppriv->txq, newskb);
}

static inline struct sk_buff*
rs485_dequeue (struct rs485_priv *ppriv)
{
    return skb_dequeue(&ppriv->txq);
}

static inline int rs485_queue_empty (struct rs485_priv *ppriv)
{
    return skb_queue_empty(&ppriv->txq);
}

static inline void
rs485_init_queue (struct rs485_priv *ppriv)
{
	skb_queue_head_init(&ppriv->txq);
}


//#define FNET_MTU		(L1_CACHE_BYTES << 3)
#define FNET_MTU		(2048)
#define PDC_RX_TIMEOUT		(3 * 10)		/* 3 bytes */

enum RS485_TOKEN_STATES {
    TOKEN_INIT                          = 0x0000, /* after system boot up ...  */

    TOKEN_LISTEN_MASK                   = 0x0007, /* bit0, bit1, bit2 */
    TOKEN_LISTEN_STATE                  = 0x0004, 
    TOKEN_LISTEN_SUBSTATE_UNKNOW        = 0x0000,
    TOKEN_LISTEN_SUBSTATE_BUSY          = 0x0001,
    TOKEN_LISTEN_SUBSTATE_FREE          = 0x0002,
    TOKEN_LISTEN_UNKNOW                 = TOKEN_LISTEN_STATE | TOKEN_LISTEN_SUBSTATE_UNKNOW, /* from @a TOKEN_INIT */
    TOKEN_LISTEN_FREE                   = TOKEN_LISTEN_STATE | TOKEN_LISTEN_SUBSTATE_FREE, 
    TOKEN_LISTEN_BUSY                   = TOKEN_LISTEN_STATE | TOKEN_LISTEN_SUBSTATE_BUSY, 

    TOKEN_GRAB_SENT                     = 0x0010,
    TOKEN_GRAB_MAYBE_OTHER              = 0x0020, /* have  */
    TOKEN_OWNER                         = 0x0040,
};

static inline int token_state (int stat)
{
    return (stat & TOKEN_LISTEN_MASK);
}

static inline int token_listen (int stat)
{
    return (stat & TOKEN_LISTEN_STATE) ? 1 : 0;
}

static inline int token_listen_free (int stat)
{
    return token_state(stat) == TOKEN_LISTEN_FREE ? 1 : 0;
}

static inline int token_listen_busy (int stat)
{
    return token_state(stat) == TOKEN_LISTEN_BUSY ? 1 : 0;
}

static inline int token_listen_unknow (int stat)
{
    return token_state(stat) == TOKEN_LISTEN_UNKNOW ? 1 : 0;
}


/** 
 * 
 * 
 * @param n num of @p nano_second
 */
static inline void pdelay (unsigned n)
{
    n /= 25;
    while(n--)
        asm("nop");
}


int rs485_bus_ctor (void * priv);
void rs485_bus_dtor (void * priv);
    
void rs485_priv_start_tx ( struct rs485_priv *ppriv );

#endif /* _RS485_BUS_H */



