/******************************************************************
 * @file   token.c
 * @author Richard Luo
 * @date   2008-11-10
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include <linux/random.h>

#include "token.h"


enum TOKEN_FRAME_TYPE_n_CODE_CONSTS {
    TOKEN_TYPE_TOKEN    = 0x1111,
    TOKEN_TYPE_DATA     = 0x2222,

    TOKEN_CODE_GRAB     = 0x3333,
    TOKEN_CODE_ACK      = 0x4444,
    TOKEN_CODE_RELEASE  = 0x5555,
    
    TOKEN_INVAL_ADDR    = 0x8000,
    TOKEN_FREE_ADDR     = TOKEN_INVAL_ADDR,

};

enum TIMEOUT_BASE_VALUE {
    TOKEN_INIT_TBASE    = 50 * 8,  /* 50 byte */
    TOKEN_INIT_TRANGE   = 25 * 8,  /* 25 byte */

    TOKEN_BUSY_TBASE    = 40 * 8,
    TOKEN_BUSY_TRANGE   = 20 * 8,

    TOKEN_FREE_TBASE    = 30 * 8,
    TOKEN_FREE_TRANGE   = 15 * 8,

    SHORT_TBASE         = 20 * 8,
    SHORT_TRANGE        = 10 * 8,

    MIDDLE_TBASE        = 40 * 8,
    MIDDLE_TRANGE       = 20 * 8,

    LONG_TBASE          = 60 * 8,
    LONG_TRANGE         = 30 * 8,
};

struct token_frame {
    u16 daddr;                  /* dst mac addr of rs485 */
    u16 saddr;                  /* src mac addr of rs485 */
    u16 length;                 /* token frame: fixed with 12. data frame: 12+ */
    u16 type;                   /* token frame, data frame */
    u16 code;                   /* token grab, token ack, token release */
    u16 check;                  /* checksum */
} __attribute((packed));


static inline u16 get_token_owner_addr (const struct token_frame *frame)
{
    if (frame->type == TOKEN_TYPE_DATA)
        return frame->saddr;

    if(frame->type == TOKEN_TYPE_TOKEN &&
       frame->code == TOKEN_CODE_ACK)
        return frame->daddr;

    return TOKEN_INVAL_ADDR;
}


/** 
 * 
 * 
 * @param daddr 
 * @param saddr 
 * @param type 
 * @param oframe output result
 * 
 */
static void make_token_frame (u16 daddr, u16 saddr, u16 type, u16 code, struct token_frame *oframe)
{
    oframe->daddr = htons(daddr);
    oframe->saddr = htons(saddr);
    oframe->type = htons(type);
    oframe->code = htons(code);
    oframe->length = htons(12);
    oframe->check = 0;
	oframe->check = ip_fast_csum((unsigned char *)oframe, oframe->length);
}

#define NEED_TO_SEND()          1
#define START_TX_FRAME()        do { } while(0)


/** 
 * 
 * 
 * @param buf 
 * @param len buf length
 * @param ohead output result, head info, in host byte order.
 * 
 * @return -1 on error
 */
static int parse_token_frame (void *buf, size_t len, struct token_frame *ohead)
{
    u16 length;
    struct token_frame *iframe = (struct token_frame*) buf;

    ERR_INFO_RETURN ((len < 12),
                     "frame length the too small");

    length = ntohs(iframe->length);
    
    ERR_INFO_RETURN ((len < length),
                     "invalid frame: length the too large");

    ERR_RET_VAL (validate_checksum(iframe),
                 "invalid frame: checksum error" );

    ohead->length = length;
    ohead->daddr = ntohs(iframe->daddr);
    ohead->saddr = ntohs(iframe->saddr);
    ohead->type = ntohs(iframe->type);

    return 0;
}


/** 
 * 
 * 
 * @param ppriv rs485 bus object
 * @param status interrupt status 
 * 
 * @return 0 on success and continue, 1 on success and maybe for some other use
 */
typedef int token_handle_func_t (struct rs485_priv *ppriv, unsigned int status);


static u32 random_range (u32 start, u32 range)
{
    return (start + random32()%range);
}

static void set_random_timeout (struct uart_port *port, u32 start, u32 range)
{
    
    u16 timeval = random_range(start, range) & 0xff;

    UART_PUT_RTOR(port, timeval);		/* set UART timeout */
    UART_PUT_CR(port, ATMEL_US_RETTO);
}


static int parse_recv_paket (struct rs485_priv *ppriv, struct token_frame *ohead)
{
    struct atmel_dma_buffer *pdc = PDC_RX_BUF(ppriv);
    size_t count = UART_GET_RPR(rs485_priv_usart(ppriv)) - pdc->dma_addr;	/* current DMA adress */

    BUG_ON(FNET_MTU < count || cout  == 0);

    ERR_RET_VAL ( parse_token_frame(pdc->buf, count, ohead),
                  "error for parse_recv_paket");
    
    return 0;
}



static void token_hw_tx (struct rs485_priv *ppriv, const void *buf, size_t len)
{
	struct atmel_uart_port *atmel_port = (struct atmel_uart_port *) port;

	if (atmel_port->use_dma_tx) {
		if (UART_GET_PTSR(port) & ATMEL_PDC_TXTEN) {
			/* The transmitter is already running.  Yes, we
			   really need this.*/
			return;
        }

		UART_PUT_IER(port, ATMEL_US_ENDTX | ATMEL_US_TXBUFE);
		UART_PUT_PTCR(port, ATMEL_PDC_TXTEN);		/* re-enable PDC transmit */
	}
	else
		UART_PUT_IER(port, ATMEL_US_TXRDY);
}


static inline void send_token_ack (struct rs485_priv *ppriv, struct token_frame *head)
{
    /* send token ack */
    struct token_frame token_ack;
    make_token_frame(head->saddr, ppriv->self_addr,
                     TOKEN_TYPE_TOKEN, TOKEN_CODE_ACK, &token_ack);
    token_hw_tx(&token_ack, sizeof(token_ack)); /* send token ack */
    ppriv->cur_owner = head->saddr;
    ppriv->token_status = TOKEN_LISTEN_BUSY;
}

/** 
 * @brief 
 * 
 * @param ppriv 
 * @param status 
 * 
 * @return 
 */
static int token_handle_rxbrk (struct rs485_priv *ppriv, unsigned int status)
{
    struct uart_port *port = rs485_priv_usart(ppriv);
    struct token_frame head;
    u16 owner_addr;
    
	UART_PUT_CR(port, ATMEL_US_RXBRK);

	BUG_ON( !(status & ATMEL_US_RXBRK) );
	BUG_ON( status & (ATMEL_US_PARE | ATMEL_US_FRAME) ); /* should be processed before get here. */
    port->icount.brk++;

    switch (ppriv->token_status) {
    case TOKEN_INIT:
        BUG_ON(1); break;

    case TOKEN_LISTEN_UNKNOW:
        if ( parse_recv_paket(ppriv, &head) ) {
            /* 收到非法包,总线上有冲突 ?? recorded it here */
            /* no need to change status */
            set_random_timeout(port, LONG_TBASE, LONG_TRANGE); /* 总线可能忙,使用较长的延时 */
            break;
        }

        owner_addr = get_token_owner_addr(&head);
        if (TOKEN_INVAL_ADDR == owner_addr)  {
            if (TOKEN_TYPE_TOKEN == head->type && /* receive token grab to myself */
                TOKEN_CODE_GRAB == head->code &&
                head->daddr == ppriv->self_addr)
                send_token_ack(ppriv, head);
            else if (TOKEN_TYPE_TOKEN == head->type &&
                     TOKEN_CODE_RELEASE == head->code) {
                /* token status change to token free status */
                ppriv->token_status = TOKEN_LISTEN_FREE;
                ppriv->cur_owner = TOKEN_FREE_ADDR; /* 好像是多余的, TOKEN_LISTEN_FREE 已经表明总线free了 */
                set_random_timeout(port, TOKEN_FREE_TBASE, TOKEN_FREE_TRANGE);
            }
            else {
                /* 非法的packet在之前早就处理过了. */
                BUG_ON(1);
            }

        } else {                /* 取得合法的总线owner, 1. pase出data包, 2. parse出A->B的token ack包 */
            ppriv->cur_owner = owner_addr;
            ppriv->token_status = TOKEN_LISTEN_BUSY;
            set_random_timeout(port, TOKEN_BUSY_TBASE, TOKEN_BUSY_TRANGE);
        }
        break;

    case TOKEN_LISTEN_BUSY:
        if ( parse_recv_paket(ppriv, &head) ) {
            /* 收到非法包,总线上有冲突 ?? recorded it here */
            BUG_ON(1); /* 总线不可能忙,状态乱掉!! */
            break;
        }

        if (TOKEN_TYPE_TOKEN == head->type &&
            TOKEN_CODE_RELEASE == head->code) {
            /* token status change to token free status */
            ppriv->token_status = TOKEN_LISTEN_FREE;
            ppriv->cur_owner = TOKEN_FREE_ADDR; /* 好像是多余的, TOKEN_LISTEN_FREE 已经表明总线free了 */
            set_random_timeout(port, TOKEN_FREE_TBASE, TOKEN_FREE_TRANGE);
            break;
        }

        BUG_ON(get_token_owner_addr(&head) != ppriv->owner_addr);
        break;

    case TOKEN_LISTEN_FREE:
        if ( parse_recv_paket(ppriv, &head) ) {
            /* 收到非法包,总线上有冲突, 正常情况, 有两个以上节点在竞争总线, 延时的长度可以动态 */
            set_random_timeout(port, LONG_TBASE, LONG_TRANGE); /* 有竞争,使用较长的延时 */
            break;
        }

        BUG_ON(head->type != TOKEN_TYPE_TOKEN);

        if (head->code == TOKEN_CODE_GRAB) {
            if (frame->daddr == ppriv->self_addr)
                send_token_ack(ppriv, head);
        } else if (head->code == TOKEN_CODE_ACK) {
            ppriv->cur_owner = head->saddr;
            ppriv->token_status = TOKEN_LISTEN_BUSY;
            set_random_timeout(port, TOKEN_BUSY_TBASE, TOKEN_BUSY_TRANGE);
        } else
            BUG_ON(1);

        break;

    case TOKEN_GRAB_SENT:
        if ( parse_recv_paket(ppriv, &head) ) {
            /* 收到非法包,总线上有冲突, 正常情况, 有两个以上节点在竞争总线 */
            set_random_timeout(port, LONG_TBASE, LONG_TRANGE); /* 总线可能忙,使用较长的延时 */
            /* 改变状态,等到timeout继续发grab */
            ppriv->token_status = TOKEN_LISTEN_FREE;
            break;
        }

        if (head->type == TOKEN_TYPE_TOKEN &&
            head->code == TOKEN_CODE_ACK &&
            head->daddr == ppriv->self_addr) {
            /* became token owner */
            ppriv->owner_addr = ppriv->self_addr;
            ppriv->token_status = TOKEN_OWNER;
            /* start hw transfer */
            break;
        }
        /* 发送了grab, 要么就什么都没收到, 要么就收到token ack才对 */
        BUG_ON(1);

    case TOKEN_OWNER:
        /* 有人乱说话 */
        BUG_ON(1);        

    default:
        BUG_ON(1);
    };
    return 0;
}

static int token_handle_timeout (struct rs485_priv *ppriv, unsigned int status)
{

    if (token_listen(ppriv->token_status))
        if (NEED_TO_SEND()) {
            SEND_TOKEN_GRAB();
            ppriv->token_status = TOKEN_GRAB_SENT;
        } else {
            ppriv->token_status = TOKEN_LISTEN_FREE;
            set_random_timeout(port, LONG_TBASE, LONG_TRANGE);
        }

    switch () {
    case TOKEN_LISTEN_UNKNOW:   /* 表明总线空闲 */
    case TOKEN_LISTEN_BUSY:
    case TOKEN_LISTEN_FREE:
        break;

    case TOKEN_GRAB_SENT:
        

    case TOKEN_OWNER:
        /* 有人乱说话 */
        BUG_ON(1);        

    default:
        BUG_ON(1);
    };

}






