/******************************************************************
 * @file   psoc_bitbang.h
 * @author Richard Luo
 * @date   2008-09-17
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#ifndef _PSOC_BITBANG_H
#define _PSOC_BITBANG_H 1

#include <asm/arch/gpio.h>
#include <asm/arch/at91_pio.h>
#include <linux/delay.h>

#include <linux/types.h>
#include "akdebug.h"

#define psoc_clk_bit    AT91_PIN_PB21
#define psoc_data_bit   AT91_PIN_PB22
#define psoc_reset_bit  AT91_PIN_PB10

// #define psoc_data_bit   AT91_PIN_PB10
// #define psoc_reset_bit  AT91_PIN_PB22

#define PSOC_CLK_HZ     250000
#define _1M_HZ          1000000

// unit: us
#define _1CLK_CYCLE     (_1M_HZ/PSOC_CLK_HZ)
#define HALF_CYCLE      (_1CLK_CYCLE/2)

// #define HIGH_EDGE_CTIME     800
// #define LOW_EDGE_CTIME      2000

#define HIGH_EDGE_CTIME     20
#define LOW_EDGE_CTIME      50


enum sp_flag {
    SP_DOUBLE_CYCLE = 1,
    SP_HALF_DELAY_LOW,
    SP_HIGH_Z_BEG,
    SP_HIGH_Z_END,
};

enum vec_type_flags {
    VEC_TYPE_MASK = 0xf000,
    VEC_TYPE_PURE_OUTPUT = 0x1000,       // just for sending
    VEC_TYPE_PURE_INPUT = 0x2000,        // just for sampling
    VEC_TYPE_MISC = 0x4000,              // misc input(sample) and output bits.
    
    SAMP_TYPE_MASK = 0x000f,
    SAMP_ENOUGH_NUM = 0x0001|VEC_TYPE_PURE_INPUT,
    SAMP_HIGH_TO_LOW = 0x0002|VEC_TYPE_PURE_INPUT,

    SAMP_HIGH_TO_LOW_LIMIT = 200000,
};

typedef struct {
    short idx_;
    short flag_;
} special_point_t;

typedef struct {
    u32 data_[32];
    int n_;                      /* number of valid bit */
    int cycle_l, cycle_h;

    special_point_t sp_[64];    /* for send special clock for data bits */

    int vec_type_;             /* for sample data, end condition type */

} psoc_vector_t;

typedef struct {
    psoc_vector_t *grp_[10];
    int n_;
} vector_grp_t;

static inline void make_read_byte_vector_group (vector_grp_t *vgrp, uint8_t address);

static inline vector_grp_t * get_read_byte_vector_group (uint8_t address);

/** 
 * 
 * 
 * @param n num of @p nano_second
 */
static inline void pdelay (unsigned n)
{
    n /= 25;
    while(n--)
        asm("nop");
}

static inline void psoc_setclk (int is_on)
{
	at91_set_gpio_value(psoc_clk_bit, is_on);
}

static inline void psoc_setdata (int is_on)
{
	at91_set_gpio_value(psoc_data_bit, is_on);
}

static inline void psoc_setrst (int is_on)
{
	at91_set_gpio_value(psoc_reset_bit, is_on);
}


static inline void psoc_init_gpio (void)
{
	at91_set_gpio_output(psoc_clk_bit, 0);
	at91_set_gpio_output(psoc_data_bit, 0);
	at91_set_gpio_output(psoc_reset_bit, 0);
}

static inline void psoc_exit_gpio (void)
{
	at91_set_gpio_output(psoc_clk_bit, 0);
	at91_set_gpio_output(psoc_data_bit, 0);
	at91_set_gpio_output(psoc_reset_bit, 0);
    
	at91_set_gpio_input(psoc_clk_bit, 0);
	at91_set_gpio_input(psoc_data_bit, 0);
	at91_set_gpio_input(psoc_reset_bit, 0);
}



static inline int vect_type (const psoc_vector_t *pv)
{
    return (pv->vec_type_ & VEC_TYPE_MASK);
}

static inline int vect_sample_type (const psoc_vector_t *pv)
{
    return (pv->vec_type_);
}

/** 
 * @note send one bit according to the falling edge.
 * 
 * @param val 0 or 1
 * @param cycle_l number of nano-seconds to maintain low level
 * @param cycle_h number of nano-seconds to maintain high level
 */
static inline void send_bit_on_falling (u8 val, int cycle_l, int cycle_h)
{
    psoc_setclk(1);
    psoc_setdata(val);
    pdelay(cycle_h);
    psoc_setclk(0);
    pdelay(cycle_l);
}

static inline void send_bit_on_leading (u8 val, int cycle_l, int cycle_h)
{
    psoc_setclk(0);
    psoc_setdata(val);
    pdelay(cycle_l);
    psoc_setclk(1);
    pdelay(cycle_h);
}


static inline void send_bit (u8 val, int cycle_l, int cycle_h)
{

#ifdef SEND_BIT_ON_LEADING
    send_bit_on_leading(val, cycle_l, cycle_h);
#else
    send_bit_on_falling(val, cycle_l, cycle_h);
#endif

}

static inline void psoc_send_output_vecor (const psoc_vector_t *pv)
{
    int i, j;
    int cl = 22*(pv->cycle_l/38), ch = 22*(pv->cycle_h/38);

    // default value
    if (!cl) cl = LOW_EDGE_CTIME;
    if (!ch) ch = HIGH_EDGE_CTIME;

    if ( PIO_ST_IO_OUTPUT != at91_get_pio_function(psoc_data_bit) )
        at91_set_gpio_output(psoc_data_bit, 1);

    for (i=0,j=0; i < pv->n_; ++i) {
        if (pv->sp_[j].flag_ && pv->sp_[j].idx_ == i) {
            switch (pv->sp_[j].flag_) {
            case SP_DOUBLE_CYCLE: {
                const int cycle = (cl + ch) * 100/86;
                const int ccl = 200;
                send_bit(test_bit(i, (void*)pv->data_), ccl, cycle-ccl);
                break;
            }
            case SP_HALF_DELAY_LOW:
                send_bit(test_bit(i, (void*)pv->data_), 2*cl, ch);
                break;
            };
            ++j;
        } else
            send_bit(test_bit(i, (void*)pv->data_), cl, ch);
    }
}

static inline void conv_vec_2_string_one_byte (const psoc_vector_t *pv, int byte_idx)
{
    int i;
    const int start_bit = byte_idx << 3;
    
    printk("\"");
    for (i=0; i < 8; ++i)
        printk("%d", test_bit(start_bit + i, (void*)pv->data_));
    printk("\"");
}

static inline void psoc_conv_vec_2_string (const char *name, const psoc_vector_t *pv)
{
    int i, nbytes = (pv->n_ + 7)/8;
    printk("%s:\n", name);
    for (i=0; i < nbytes; ++i) {
        conv_vec_2_string_one_byte(pv, i);
        if (i && 0 == (i+1)%8) printk("\n");
    }
    printk("\n");
}


static inline void clock_hi_low_hi (int cl, int ch)
{

    psoc_setclk(1);
    pdelay(ch);

    psoc_setclk(0);
    pdelay(cl);

    psoc_setclk(1);
    pdelay(ch);    

}

static inline void psoc_high_z_beg (int cl, int ch)
{
    clock_hi_low_hi(cl, ch);
}

static inline void psoc_high_z_end (int cl, int ch)
{

#if 1
    psoc_setclk(1);
    pdelay(ch);
    
    psoc_setclk(0);
    pdelay(cl);


#else
    clock_hi_low_hi(cl, ch);
#endif

}


static inline int psoc_get_bit_on_leading (int cycle_l, int cycle_h)
{
    int v;
    psoc_setclk(0);
    pdelay(cycle_l);
    psoc_setclk(1);
    v = at91_get_gpio_value(psoc_data_bit);
    pdelay(cycle_h);
    return v;
}

static inline int psoc_get_bit_on_falling (int cycle_l, int cycle_h)
{
    int v;
    psoc_setclk(1);
    pdelay(cycle_h);
    psoc_setclk(0);             // sample on falling edge.
    v = at91_get_gpio_value(psoc_data_bit);
    pdelay(cycle_l);
    return v;
}

// #define SAMPLE_ON_FALLING

static inline int psoc_get_bit (int cycle_l, int cycle_h)
{

#ifdef SAMPLE_ON_FALLING
    return psoc_get_bit_on_falling(cycle_l, cycle_h);
#else
    return psoc_get_bit_on_leading(cycle_l, cycle_h);
#endif    

}

/** 
 * @brief according to data sheet no clock required for waiting and
 * polling
 * 
 * @param cycle_l 
 * @param cycle_h 
 * 
 * @return 
 */
static inline int psoc_get_bit_no_clk (int cycle_l, int cycle_h)
{
    int v;
    pdelay(cycle_h);
    v = at91_get_gpio_value(psoc_data_bit);
    pdelay(cycle_l);
    return v;
}

static inline void psoc_sample_n (psoc_vector_t *pv, int cl, int ch)
{
    int i, j;
    for (i=0, j=0; i < pv->n_; ++i) {
        if (pv->sp_[j].flag_ && pv->sp_[j].idx_ == i) {
            if (SP_HIGH_Z_BEG == pv->sp_[j].flag_)
                psoc_high_z_beg(cl, ch);
            else if (SP_HIGH_Z_END == pv->sp_[j].flag_)
                psoc_high_z_end(cl, ch);
            else WARN_ON(1);
            ++j;
            continue;
        }
        
        if (psoc_get_bit(cl, ch))
            set_bit(i, (void*)pv->data_);
        else
            clear_bit(i, (void*)pv->data_);
    }
}

static inline int psoc_detect_high_to_low (int cl, int ch)
{
    int i = 0, limit = SAMP_HIGH_TO_LOW_LIMIT;

    for (; i < limit; ++i)
        if (psoc_get_bit(cl, ch))  /* detect high */ 
            break;

    for (; i < limit; ++i)
        if (!psoc_get_bit(cl, ch))  /* detect low */
            break;

    return i;
}


static inline void psoc_sample_vector (psoc_vector_t *pv)
{
    int cl = 22*(pv->cycle_l/38), ch = 22*(pv->cycle_h/38);

    // default value
    if (!cl) cl = LOW_EDGE_CTIME;
    if (!ch) ch = HIGH_EDGE_CTIME;

    if ( PIO_ST_IO_INPUT != at91_get_pio_function(psoc_data_bit) )
        at91_set_gpio_input(psoc_data_bit, 0);

    switch (vect_sample_type(pv)) {

    case SAMP_HIGH_TO_LOW:
        pv->n_ = psoc_detect_high_to_low(cl, ch);
        break;

    case SAMP_ENOUGH_NUM:
        psoc_sample_n(pv, cl, ch);
        break;

    default:
        BUG_ON(1);

    };
}



static inline void psoc_process_vector (psoc_vector_t *pv)
{
    switch ( vect_type(pv) ) {

    case VEC_TYPE_PURE_OUTPUT:
        psoc_send_output_vecor(pv);
//        ak_debug("finished one VEC_TYPE_PURE_OUTPUT \n");
        break;

    case VEC_TYPE_PURE_INPUT:
        psoc_sample_vector(pv);
//        ak_debug("finished one VEC_TYPE_PURE_INPUT \n");
        break;
    };
}               

static inline void psoc_process_vector_grp (vector_grp_t *pgrp)
{
    int i;
    for (i=0; i < pgrp->n_; ++i)
        psoc_process_vector(pgrp->grp_[i]);
}


static inline void psoc_dump_vector (const char *name, const psoc_vector_t *pv)
{
    int i;
    const char *type;

    switch (pv->vec_type_) {
    case SAMP_ENOUGH_NUM:
        type = "SAMP_ENOUGH_NUM";
        break;
    case VEC_TYPE_PURE_INPUT:
        type = "VEC_TYPE_PURE_INPUT";
        break;
    case VEC_TYPE_PURE_OUTPUT:
        type = "VEC_TYPE_PURE_OUTPUT";
        break;
    case SAMP_HIGH_TO_LOW:
        type = "SAMP_HIGH_TO_LOW";
        break;
    default:
        type = "UNKNOWN_VEC_TYPE!!!";
    };

    printk("================================================================\n");
    printk("vector value of %s:\n"
           "\t num of bits:%d\n"
           "\t type:%s \n",
           name, pv->n_, type);

    if (pv->vec_type_ == SAMP_HIGH_TO_LOW) return;

    for (i = 0; i < pv->n_; ++i) {

        if (i && 0 == i%8)
            printk(" ");
        if (i && 0 == i%64)
            printk("\n");
        
        printk("%d", test_bit(i, (void*)pv->data_) ? 1 : 0);
    }
    printk("\n================================================================");
    printk("\n\n\n");
}



static inline void psoc_reset (void)
{
    psoc_setrst(1);
    pdelay(40000);    
    psoc_setrst(0);
    pdelay(49000);
}

static inline int set_bits_by_string (void *data, int start, const char *string)
{
    int i;
    const int end = start + strlen(string);

    for (i = start; i < end; ++i)
        if (string[i-start] == '1')
            set_bit(i, data);
        else
            clear_bit(i, data);

    return end;
}

static inline int set_bits_by_value (void *data, int start, uint8_t value, int len)
{
    int i;
    const int end = start + len;
    const uint8_t f_val = 1 << 7;
    
    for (i = start; i < end; ++i, value <<= 1)
        if (value & f_val)
            set_bit(i, data);
        else
            clear_bit(i, data);

    return end;
}

/** 
 * 
 * 
 * @param pv 
 * @param pos 
 * @param type SP_HIGH_Z_BEG or SP_HIGH_Z_END
 * 
 * @return 
 */
static inline int set_bits_high_z (psoc_vector_t *pv, int pos, int type)
{
    int i = 0;

    for (i=0; pv->sp_[i].flag_ && pv->sp_[i].idx_ < pos; ++i)
        ;
    pv->sp_[i].flag_ = type;
    pv->sp_[i].idx_ = pos;
    return ++pos;               /* point to next position */
}


static inline void make_write_byte_vector (psoc_vector_t *pv, uint8_t address, uint8_t value )
{
    int next;
    void *data = (void*) pv->data_;

    memset(pv, 0, sizeof(*pv) );
    
    pv->n_ = 22;
    pv->vec_type_ = VEC_TYPE_PURE_OUTPUT;

    next = set_bits_by_string(data, 0, "10010");
    next = set_bits_by_value(data, next, (address << 2), 6);
    next = set_bits_by_value(data, next, value, 8);
    next = set_bits_by_string(data, next, "111");

    WARN_ON(next != pv->n_);
}

static inline psoc_vector_t * get_write_byte_vector (uint8_t address, uint8_t value)
{
    static psoc_vector_t write_byte_vector = {
        .data_ = {
            0x00380009,         /* 0b10010aaaaaadddddddd111 */
        },
        .n_ = 22,
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
    };

    void *data = (void*) write_byte_vector.data_;
    int next = set_bits_by_value(data, 5, (address << 2), 6);
    (void) set_bits_by_value(data, next, value, 8);

    return &write_byte_vector;
}


static inline
void make_set_block_num_vector (psoc_vector_t *pv, uint8_t block_no)
{
    int next;
    void *data = (void*) pv->data_;

    memset(pv, 0, sizeof(*pv) );

    pv->n_ = 22;
    pv->vec_type_ = VEC_TYPE_PURE_OUTPUT;

    next = set_bits_by_string(data, 0, "10011111010");
    next = set_bits_by_value(data, next, block_no, 8);
    next = set_bits_by_string(data, next, "111");

    WARN_ON(next != pv->n_);
}

static inline psoc_vector_t * get_block_num_vector (uint8_t block_no)
{

    static psoc_vector_t block_no_vector = {
        .data_ = {
            0x003802f9,
        },
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 22,
    };

    void *data = (void*) block_no_vector.data_;
    (void) set_bits_by_value(data, 11, block_no, 8);

    return &block_no_vector;
}

static inline void send_40_zeros (void)
{
    static psoc_vector_t _40_zeros = { 
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 40,
    };
    psoc_send_output_vecor(&_40_zeros);
}
    
static inline int psoc_wait_and_poll (void)
{
    static psoc_vector_t hi2low_vec = { 
        .vec_type_ = SAMP_HIGH_TO_LOW
    };

    psoc_sample_vector(&hi2low_vec);

    ERR_INFO_RETURN ( (hi2low_vec.n_ == SAMP_HIGH_TO_LOW_LIMIT),
                      "error for detect high to low");
    send_40_zeros();
    return 0;
}

static inline int send_wait_poll_vector (const psoc_vector_t *pv)
{
    psoc_send_output_vecor(pv);
    ERR_INFO_RETURN (psoc_wait_and_poll(),
                     "error for psoc_wait_and_poll");
    return 0;
}



enum PSOC_FW_BLOCK_TYPE {
    BLOCK_TYPE_DATA,
    BLOCK_TYPE_SECURE_HEAD,
    BLOCK_TYPE_SECURE_DATA,
    BLOCK_TYPE_DEV_CHECKSUM_HEAD,
    BLOCK_TYPE_DEV_CHECKSUM_DATA,
};


static inline const char * program_vector_type_info (int type)
{
    switch (type) {
    case BLOCK_TYPE_DATA:
        return "BLOCK_TYPE_DATA";
    case BLOCK_TYPE_SECURE_DATA:
        return "BLOCK_TYPE_SECURE_DATA";
    case BLOCK_TYPE_DEV_CHECKSUM_HEAD:
        return "BLOCK_TYPE_DEV_CHECKSUM_HEAD";
    case BLOCK_TYPE_DEV_CHECKSUM_DATA:
        return "BLOCK_TYPE_DEV_CHECKSUM_DATA";
    case BLOCK_TYPE_SECURE_HEAD: // FIXME: show use pointer to express error
        return "BLOCK_TYPE_SECURE_HEAD";
    default:
        return "INVALID_PROGRAM_VECTOR_TYPE";
    };
}

static inline psoc_vector_t * get_program_vector (int type)
{
    static psoc_vector_t prog_data_block_vec = { 
        .data_ = {
            0xfe7d41f9, 0x8077bed4,
            0xf9e00def, 0x813e7ae0,
            0xef8057bf, 0x0ef9f805,
            0xbe193ef8, 0x01ef812f,
            0xf800fbe8, 0x000e91fe,
        },
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 308,
    };

    static psoc_vector_t prog_secure_data_vec = { 
        .data_ = {
            0xfe7951f9, 0x8077bed4,
            0xf9e00def, 0x813e7ae0,
            0xef8057bf, 0x0ef9f805,
            0xbe193ef8, 0x01ef812f,
            0xf800fbe4, 0x000e91fe,
        },
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 308,
    };

    static psoc_vector_t verify_checksum_vec = { 
        .data_ = {
            0xdef8077b, 0xae0f9e00,
            0x7bf813e7, 0x805ef805,
            0xef80ef9f, 0x12fbe193,
            0xbe00be78, 0x03eff007,
            0x3a47fbe0, 
        },
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 286
    };


    switch (type) {
    case BLOCK_TYPE_DATA:
        return &prog_data_block_vec;
    case BLOCK_TYPE_SECURE_DATA:
        return &prog_secure_data_vec;

    case BLOCK_TYPE_DEV_CHECKSUM_DATA:
        return &verify_checksum_vec;
        
    case BLOCK_TYPE_SECURE_HEAD: // FIXME: show use pointer to express error
    case BLOCK_TYPE_DEV_CHECKSUM_HEAD:
    default:
        return NULL;
    };
    
}

/** 
 * 
 * 
 * @param type 
 * @param data 
 * @param len data length
 * @param block_no block number address
 * @param start_addr address whthin the block
 * 
 * @return 0 on success, -1 on error
 */
static inline
int psoc_write_block (int type, const uint8_t *data, size_t len, uint8_t block_no, uint8_t start_addr)
{
    int i;
    uint8_t end_addr = start_addr + len;
    psoc_vector_t *pvec;

    ak_debug("psoc_write_block, block_no:%d, start_addr:%d, len:%d \n", block_no, start_addr, len);

    if (end_addr > 64) {
        WARN_ON(1);
        return -1;
    }

    /* write one block to SRAM */
    for (i=start_addr; i < end_addr; ++i)
        psoc_send_output_vecor( get_write_byte_vector(i, data[i]) );

    psoc_send_output_vecor( get_block_num_vector(block_no) );

    pvec = get_program_vector(type);
    if (pvec)
        return send_wait_poll_vector(pvec);

    return 0;
}


static inline int write_one_block (const uint8_t *data, size_t len, uint8_t block_no)
{
    return psoc_write_block(BLOCK_TYPE_DATA, data, len, block_no, 0);
}

static inline int psoc_write_data (const uint8_t *data, int len)
{
    int i, nblocks = (len+63) / 64, remain = len % 64;

    for (i=0; i < nblocks; ++i, data += 64)
        if (write_one_block(data, 64, i))
            return -1;
    
    return write_one_block(data, remain, i);
}



static inline void retrieve_checksum (psoc_vector_t *checksum_v1, psoc_vector_t *checksum_v2, uint16_t *value)
{
    int i;

    for (i=8; i >=1; --i, *value <<= 1) 
        *value |= test_bit(i, (void*)checksum_v1->data_);

    for (i=8; i >=1; --i, *value <<= 1) 
        *value |= test_bit(i, (void*)checksum_v2->data_);
}

static inline void retrieve_read_byte (psoc_vector_t *pv, uint8_t *c)
{
    uint8_t i, cc;
    for (i=8, cc=0; i >= 1; --i) 
        if (test_bit(i, (void*)pv->data_) )
            cc |= 1 << (8-i);
    *c = cc;
}



static psoc_vector_t read_byte_v1 = {
    .data_ = {
        0x0d,
    },
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
    .n_ = 11,                /* 0b10110aaaaaa */
};

/* sample vector for read one byte */
static psoc_vector_t read_byte_v2 = {
    .n_ = 10,
    .sp_ = {
        {0, SP_HIGH_Z_BEG},
        {9, SP_HIGH_Z_END},
        {0, 0},
    },
    .vec_type_ = SAMP_ENOUGH_NUM,
};

static psoc_vector_t read_byte_v3 = { /* end output one bit of '1' */
    .data_ = { 0x1 },
    .n_ = 1,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};
    
static vector_grp_t read_byte_vec_grp = {
    .grp_ = { &read_byte_v1, &read_byte_v2, &read_byte_v3 },
    .n_ = 3,
};



static vector_grp_t * make_read_checksum_vector_group (void)
{
    static psoc_vector_t read_checksum_v1 = {
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 11,
    };

    static psoc_vector_t read_checksum_v2 = {
        .n_ = 10,
        .sp_ = {
            {0, SP_HIGH_Z_BEG},
            {9, SP_HIGH_Z_END},
            {0, 0},
        },
        .vec_type_ = SAMP_ENOUGH_NUM,
    };

    static psoc_vector_t read_checksum_v3 = {
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 12,
    };

    static psoc_vector_t read_checksum_v4 = {
        .n_ = 10,
        .sp_ = {
            {0, SP_HIGH_Z_BEG},
            {9, SP_HIGH_Z_END},
            {0, 0},
        },
        .vec_type_ = SAMP_ENOUGH_NUM,
    };
    
    static psoc_vector_t read_checksum_v5 = { /* end output one bit of '1' */
        .data_ = { 0x1 },
        .n_ = 1,
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
    };


    static vector_grp_t read_checksum_vec_group = {
        .grp_ = { &read_checksum_v1, &read_checksum_v2, &read_checksum_v3, &read_checksum_v4, &read_checksum_v5 },
        .n_ = 5,
    };

    psoc_vector_t *v;

    memset(read_checksum_v2.data_, 0, sizeof(read_checksum_v2.data_) );
    memset(read_checksum_v4.data_, 0, sizeof(read_checksum_v4.data_) );

    v = &read_checksum_v1;
    set_bits_by_string((void*)v->data_, 0, "10111111001");

    v = &read_checksum_v3;
    set_bits_by_string((void*)v->data_, 0, "110111111000");

    return &read_checksum_vec_group;
}


static inline void make_read_byte_vector_group (vector_grp_t *vgrp, uint8_t address)
{
    int next;
    psoc_vector_t *v;

    v = vgrp->grp_[0];
    memset(v, 0, sizeof(*v));
    next = set_bits_by_string((void*)v->data_, 0, "10110");
    next = set_bits_by_value((void*)v->data_, next, (address << 2), 6);
    v->vec_type_ = VEC_TYPE_PURE_OUTPUT;
    v->n_ = next;

    v = vgrp->grp_[1];
    memset(v, 0, sizeof(*v));
    (void) set_bits_high_z( (void*)v->data_, 0, SP_HIGH_Z_BEG);
    (void) set_bits_high_z( (void*)v->data_, 9, SP_HIGH_Z_END);
    v->vec_type_ = SAMP_ENOUGH_NUM;
    v->n_ = 10;
    
    v = vgrp->grp_[2];
    memset(v, 0, sizeof(*v));    
    v->vec_type_ = VEC_TYPE_PURE_OUTPUT;
    v->data_[0] = 0xff;
    v->n_ = 1;

}


static inline void read_one_byte (uint8_t *c, uint8_t address)
{
    vector_grp_t *rgrp = get_read_byte_vector_group(address);
    psoc_process_vector_grp(rgrp);
    retrieve_read_byte(rgrp->grp_[1], c);
}

//#define ORG_READ_ONE_BYTE

static inline vector_grp_t * get_read_byte_vector_group (uint8_t address)
{

#ifdef ORG_READ_ONE_BYTE
    static psoc_vector_t read_byte_vecs[3];
    static vector_grp_t read_group = {
        .grp_ = { &read_byte_vecs[0], &read_byte_vecs[1], &read_byte_vecs[2] },
        .n_ = 3,
    };

    make_read_byte_vector_group(&read_group, address);
    return &read_group;
#else

    set_bits_by_value((void*)read_byte_v1.data_, 5, (address << 2), 6);
    pdelay(8500);
//    pdelay(10000);
    return &read_byte_vec_grp;

#endif    


}


/** 
 * 
 * 
 * @param buf 
 * @param size size of @a buf
 * @param block_no specify from which block to read
 * @param start_addr address whithin the block specified by @a block_no
 */
static inline int read_from_block (uint8_t *buf, size_t size, uint8_t block_no, uint8_t start_addr)
{
    size_t i;
    static psoc_vector_t verify_setup_vec = { 
        .data_ = {
            0xdef8077b, 0xae0f9e00,
            0x7bf813e7, 0x805ef805,
            0xef80ef9f, 0x12fbe193,
            0xbf001ef8, 0x1fef800f,
            0x000000e9, 
        },
        .vec_type_ = VEC_TYPE_PURE_OUTPUT,
        .n_ = 264
    };

    uint8_t end_addr = start_addr + size;

    if (end_addr > 64)
        return -1;

    psoc_send_output_vecor( get_block_num_vector(block_no) );

    if (-1 == send_wait_poll_vector(&verify_setup_vec) )
        return -1;

    for (i=0; i < size; ++i)
        read_one_byte(&buf[i], start_addr+i);

//    aways_show_bytes("read_from_block:", buf, size);

    return 0;
}

static inline
int psoc_safe_write_data_block (const uint8_t *data, size_t len, uint8_t block_no, uint8_t start_addr)
{
    uint8_t valid_buf[len];

    ERR_INFO_RETURN (psoc_write_block(BLOCK_TYPE_DATA, data, len, block_no, start_addr),
                     "psoc_write_block failed");

    ERR_INFO_RETURN (read_from_block(valid_buf, sizeof(valid_buf), block_no, start_addr),
                     "read_from_block failed");

    return (0 == memcmp(valid_buf, data, len) ? 0 : -1);
}


#endif /* _PSOC_BITBANG_H */



