/******************************************************************
 * @file   ak_relay.c
 * @author Richard Luo
 * @date   2008-05-22
 * 
 * @brief  relay and led's driver
 * 
 ****************************************************************** 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>

#include <linux/init.h>
#include <linux/poll.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>

#include <linux/kernel.h>
#include <asm/arch/gpio.h>
#include <asm/arch/at91_pio.h>
#include <linux/proc_fs.h>
#include <linux/cdev.h>
#include "ak_relay.h"

static atomic_t relay_available = ATOMIC_INIT(1);

static int ak_relay_major;
module_param(ak_relay_major, int, 0);

MODULE_LICENSE("Dual BSD/GPL");

typedef struct {
    int pin_relay_;
} relay_pins_t;

typedef struct {
    relay_pins_t *relay_;
    int num_;
    struct cdev cdev_;
} relay_dev_t;

static relay_pins_t g_relay_pins[] = {
    {
      .pin_relay_ = AT91_PIN_PB30,
    },
    {
      .pin_relay_ = AT91_PIN_PA8,
    },
    {
      .pin_relay_ = AT91_PIN_PA2,
    },
    {
      .pin_relay_ = AT91_PIN_PA6,
    },
    {
      .pin_relay_ = AT91_PIN_PC1,
    },
    {
      .pin_relay_ = AT91_PIN_PC3,
    },

};

relay_dev_t ak_relay_dev = {
    .relay_ = g_relay_pins,
    .num_ = ARRAY_SIZE(g_relay_pins),
};


int ak_relay_read_procmem (char *buf, char **start, off_t offset,
                         int count, int *eof, void *data)
{
    int n;
    n = sprintf(buf, "%s", "hello relay's proc \n");
	*eof = 1;
	return n;
}

static int ak_relay_release (struct inode *inode, struct file *filp)
{
	atomic_inc(&relay_available); /* release the device */
	return 0;
}

static int ak_relay_open (struct inode *inode, struct file *filp)
{

	relay_dev_t *dev; /* device information */

	/* if (! atomic_dec_and_test (&relay_available)) { */
	/* 	atomic_inc(&relay_available); */
	/* 	return -EBUSY; /\* already open *\/ */
	/* } */

	dev = container_of(inode->i_cdev, relay_dev_t, cdev_);
	filp->private_data = dev; /* for other methods */

	return 0;
}

/** 
 * @brief for trunning on/off LED and RELAY
 * 
 * @param dev pointer to all relay devs
 * @param relayno to which relay 
 * @param cmd to do what?
 */
static inline int relay_output_ctrls (relay_dev_t *dev, int relayno, unsigned cmd)
{
    int pin, level, ret = 0;
    relay_pins_t *pdev;
    
    if (dev->num_ <= relayno || relayno < 0) {
        printk("invalid relayno: [%d] \n", relayno);
        return -EINVAL;
    }

    pdev = &dev->relay_[relayno];

	switch(cmd) {
	case AK_RELAY_ON:
        pin = pdev->pin_relay_;
        level = 1;
		break;

	case AK_RELAY_OFF:
        pin = pdev->pin_relay_;
        level = 0;
		break;

    default:
        pin = -1;
        level = -1;
		break;
    }

    if (level >= 0)
        gpio_direction_output(pin, level);
    else
        ret = -EINVAL;

    return ret;
}

static int ak_relay_ioctl (struct inode *inode, struct file *filp,
                         unsigned int cmd, unsigned long arg)
{

	int err = 0;
	relay_dev_t *dev;

	/* don't even decode wrong cmds: better returning  ENOTTY than EFAULT */
	if (_IOC_TYPE(cmd) != AK_RELAY_IOC_MAGIC) return -ENOTTY;
	if (_IOC_NR(cmd) > AK_RELAY_IOC_MAXNR) return -ENOTTY;

	/*
	 * the type is a bitmask, and VERIFY_WRITE catches R/W
	 * transfers. Note that the type is user-oriented, while
	 * verify_area is kernel-oriented, so the concept of "read" and
	 * "write" is reversed
	 */
	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd));
	else if (_IOC_DIR(cmd) & _IOC_WRITE)
		err =  !access_ok(VERIFY_READ, (void __user *)arg, _IOC_SIZE(cmd));
	if (err)
		return -EFAULT;

	switch(cmd) {

	case AK_RELAY_ON:
	case AK_RELAY_OFF:
        dev = filp->private_data;
        return relay_output_ctrls(dev, arg, cmd);
        break;
	default:  /* redundant, as cmd was checked against MAXNR */
		return -ENOTTY;
	}

	return 0;
}


struct file_operations relay_fops = {
	.owner =     THIS_MODULE,
	.open =	     ak_relay_open,
	.ioctl =     ak_relay_ioctl,
	.release =   ak_relay_release,
};

/** 
 * @param dev 
 * @param devno 
 * 
 * @return 0 on success
 */
static int relay_dev_setup (relay_dev_t *dev, dev_t devno)
{
    int err, i;
    
    for (i = 0; i < dev->num_; ++i)
        at91_set_gpio_output(dev->relay_[i].pin_relay_, 0);

	cdev_init(&dev->cdev_, &relay_fops);
	dev->cdev_.owner = THIS_MODULE;
	err = cdev_add (&dev->cdev_, devno, 1);

	if (err) {
		printk(KERN_NOTICE "Error %d relay cdev setup \n", err);
        return -1;
    }

    return 0;
}

static void relay_dev_destroy (relay_dev_t *dev)
{
    cdev_del(&dev->cdev_);
	unregister_chrdev_region(MKDEV(ak_relay_major, 0), 1);
}

static int ak_relay_init (void)
{
	int result;
	dev_t dev = MKDEV(ak_relay_major, 0);
	
	/*
	 * Register your major, and accept a dynamic number.
	 */
	if (ak_relay_major)
		result = register_chrdev_region(dev, 1, "ak_relay");
	else {
		result = alloc_chrdev_region(&dev, 0, 1, "ak_relay");
		ak_relay_major = MAJOR(dev);
	}
    
	if (result < 0)
		return result;

    result = relay_dev_setup(&ak_relay_dev, dev);

    if (result)
        unregister_chrdev_region(dev, 1);
    
#ifdef RELAY_USE_PROC /* only when available */
	create_proc_read_entry("ak_relay", 0, NULL, relay_read_procmem, NULL);
#endif

	return result;
}

static void ak_relay_exit (void)
{

    relay_dev_destroy(&ak_relay_dev);
	printk(KERN_ALERT "Goodbye, relay \n");
}

module_init(ak_relay_init);
module_exit(ak_relay_exit);
