/* crypto/test/aes_ctstest.c */
/* ====================================================================
 * Copyright (c) 2008 The OpenSSL Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit. (http://www.OpenSSL.org/)"
 *
 * 4. The names "OpenSSL Toolkit" and "OpenSSL Project" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    licensing@OpenSSL.org.
 *
 * 5. Products derived from this software may not be called "OpenSSL"
 *    nor may "OpenSSL" appear in their names without prior written
 *    permission of the OpenSSL Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit (http://www.OpenSSL.org/)"
 *
 * THIS SOFTWARE IS PROVIDED BY THE OpenSSL PROJECT ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE OpenSSL PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */

#include "crypt.h"
#ifdef PERFTEST
# include <sys/time.h>
# define NUM_ITERATIONS  1000000
#endif

/* test vectors from RFC 3962 */
static const unsigned char test_key[16] = "chicken teriyaki";
static const unsigned char test_input[64] =
                "I would like the" " General Gau's C"
                "hicken, please, " "and wonton soup.";
static const unsigned char test_iv[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

static const unsigned char vector_17[17] =
{0xc6,0x35,0x35,0x68,0xf2,0xbf,0x8c,0xb4, 0xd8,0xa5,0x80,0x36,0x2d,0xa7,0xff,0x7f,
 0x97};
static const unsigned char vector_31[31] =
{0xfc,0x00,0x78,0x3e,0x0e,0xfd,0xb2,0xc1, 0xd4,0x45,0xd4,0xc8,0xef,0xf7,0xed,0x22,
 0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5};
static const unsigned char vector_32[32] =
{0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8,
 0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84};
static const unsigned char vector_47[47] =
{0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
 0xb3,0xff,0xfd,0x94,0x0c,0x16,0xa1,0x8c, 0x1b,0x55,0x49,0xd2,0xf8,0x38,0x02,0x9e,
 0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5};
static const unsigned char vector_48[48] =
{0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
 0x9d,0xad,0x8b,0xbb,0x96,0xc4,0xcd,0xc0, 0x3b,0xc1,0x03,0xe1,0xa1,0x94,0xbb,0xd8,
 0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8};
static const unsigned char vector_64[64] =
{0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
 0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8,
 0x48,0x07,0xef,0xe8,0x36,0xee,0x89,0xa5, 0x26,0x73,0x0d,0xbc,0x2f,0x7b,0xc8,0x40,
 0x9d,0xad,0x8b,0xbb,0x96,0xc4,0xcd,0xc0, 0x3b,0xc1,0x03,0xe1,0xa1,0x94,0xbb,0xd8};

static AES_KEY encks, decks;

void test_vector(const unsigned char *vector,int len)
{
    unsigned char cleartext[64];
    unsigned char iv[sizeof(test_iv)];
    unsigned char ciphertext[64];
    int tail;

    printf("vector_%d\n",len); fflush(stdout);

    if ((tail=len%16) == 0) tail = 16;
    tail += 16;

    /* Test CTS block-based encryption */
#ifdef PERFTEST
    int i;
    struct timeval tv1, tv2, res;
    /* Get encryption start time */
    gettimeofday(&tv1, NULL);
    for( i = 0; i < NUM_ITERATIONS; i++ ) {
    /* Still need to do this memcpy() for performance test */
#endif
    memcpy(iv,test_iv,sizeof(test_iv));
    CRYPTO_cts128_encrypt_block(test_input,ciphertext,len,&encks,iv,(block128_f)AES_encrypt);
#ifdef PERFTEST
    }
    /* Get encryption finish time */
    gettimeofday(&tv2, NULL);
    timersub(&tv2,&tv1,&res);
    fprintf(stdout,"Encrypt (block-based): Time for %d calls: %ld.%ld s\n",
            NUM_ITERATIONS, res.tv_sec, res.tv_usec );
#endif
    if (memcmp(ciphertext,vector,len))
        fprintf(stderr,"output_%d mismatch\n",len), exit(1);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        fprintf(stderr,"iv_%d mismatch\n",len), exit(1);

    /* Test CTS block-based decryption */
#ifdef PERFTEST
    /* Get decryption start time */
    gettimeofday(&tv1, NULL);
    for( i = 0; i < NUM_ITERATIONS; i++ ) {
    /* Still need to do this memcpy() for performance test */
#endif
    memcpy(iv,test_iv,sizeof(test_iv));
    CRYPTO_cts128_decrypt_block(ciphertext,cleartext,len,&decks,iv,(block128_f)AES_decrypt);
#ifdef PERFTEST
    }
    /* Get decryption finish time */
    gettimeofday(&tv2, NULL);
    timersub(&tv2,&tv1,&res);
    fprintf(stdout,"Decrypt (block-based): Time for %d calls: %ld.%ld s\n",
            NUM_ITERATIONS, res.tv_sec, res.tv_usec );
#endif
    if (memcmp(cleartext,test_input,len))
        fprintf(stderr,"input_%d mismatch\n",len), exit(2);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        fprintf(stderr,"iv_%d mismatch\n",len), exit(2);

    /* Test CTS streamed encryption */
#ifdef PERFTEST
    /* Get encryption start time */
    gettimeofday(&tv1, NULL);
    for( i = 0; i < NUM_ITERATIONS; i++ ) {
    /* Still need to do this memcpy() for performance test */
#endif
    memcpy(iv,test_iv,sizeof(test_iv));
    CRYPTO_cts128_encrypt(test_input,ciphertext,len,&encks,iv,(cbc128_f)AES_cbc_encrypt);
#ifdef PERFTEST
    }
    /* Get encryption finish time */
    gettimeofday(&tv2, NULL);
    timersub(&tv2,&tv1,&res);
    fprintf(stdout,"Encrypt (streamed): Time for %d calls: %ld.%ld s\n",
            NUM_ITERATIONS, res.tv_sec, res.tv_usec );
#endif
    if (memcmp(ciphertext,vector,len))
        fprintf(stderr,"output_%d mismatch\n",len), exit(3);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        fprintf(stderr,"iv_%d mismatch\n",len), exit(3);

    /* Test CTS streamed decryption */
#ifdef PERFTEST
    /* Get decryption start time */
    gettimeofday(&tv1, NULL);
    for( i = 0; i < NUM_ITERATIONS; i++ ) {
    /* Still need to do this memcpy() for performance test */
#endif
    memcpy(iv,test_iv,sizeof(test_iv));
    CRYPTO_cts128_decrypt(ciphertext,cleartext,len,&decks,iv,(cbc128_f)AES_cbc_encrypt);
#ifdef PERFTEST
    }
    /* Get decryption finish time */
    gettimeofday(&tv2, NULL);
    timersub(&tv2,&tv1,&res);
    fprintf(stdout,"Decrypt (streamed): Time for %d calls: %ld.%ld s\n",
            NUM_ITERATIONS, res.tv_sec, res.tv_usec );
#endif
    if (memcmp(cleartext,test_input,len))
        fprintf(stderr,"input_%d mismatch\n",len), exit(4);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        fprintf(stderr,"iv_%d mismatch\n",len), exit(4);
}

int main(int argc, char **argv)
{
    AES_set_encrypt_key(test_key,128,&encks);
    AES_set_decrypt_key(test_key,128,&decks);

    test_vector(vector_17,sizeof(vector_17));
    test_vector(vector_31,sizeof(vector_31));
    test_vector(vector_32,sizeof(vector_32));
    test_vector(vector_47,sizeof(vector_47));
    test_vector(vector_48,sizeof(vector_48));
    test_vector(vector_64,sizeof(vector_64));
    exit(0);
}
