/* Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
 * All rights reserved.
 *
 * This package is an SSL implementation written
 * by Eric Young (eay@cryptsoft.com).
 * The implementation was written so as to conform with Netscapes SSL.
 *
 * This library is free for commercial and non-commercial use as long as
 * the following conditions are aheared to.  The following conditions
 * apply to all code found in this distribution, be it the RC4, RSA,
 * lhash, DES, etc., code; not just the SSL code.  The SSL documentation
 * included with this distribution is covered by the same copyright terms
 * except that the holder is Tim Hudson (tjh@cryptsoft.com).
 *
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * If this package is used in a product, Eric Young should be given attribution
 * as the author of the parts of the library used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    "This product includes cryptographic software written by
 *     Eric Young (eay@cryptsoft.com)"
 *    The word 'cryptographic' can be left out if the rouines from the library
 *    being used are not cryptographic related :-).
 * 4. If you include any Windows specific code (or a derivative thereof) from
 *    the apps directory (application code) you must include an acknowledgement:
 *    "This product includes software written by Tim Hudson (tjh@cryptsoft.com)"
 *
 * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence
 * [including the GNU Public Licence.]
 */

#ifndef __CRYPT_H
# define __CRYPT_H

/*! \mainpage

   Introduction

   This library enables users to utilise the RSA and AES encryption functions
   in the OpenSSL package but without having to pull in all the associated
   functionality when not required. This allows the library to more portable
   to platforms with less resources.

   This header file is the main API for the reduced functionality library
   although the definitions and function prototypes are extracted from a
   number of the original OpenSSL header files.

   Pseudo-random number generator function prototypes.

   <ul>
   <li><em>\ref ::RAND_seed</em></li>
   <li><em>\ref ::RAND_bytes</em></li>
   <li><em>\ref ::RAND_cleanup</em></li>
   <li><em>\ref ::RAND_add</em></li>
   <li><em>\ref ::RAND_pseudo_bytes</em></li>
   <li><em>\ref ::RAND_status</em></li>
   </ul>

   RSA Encryption function prototypes.

   <ul>
   <li><em>\ref ::RSA_new</em></li>
   <li><em>\ref ::RSA_free</em></li>
   <li><em>\ref ::RSA_check_key</em></li>
   <li><em>\ref ::RSA_size</em></li>
   <li><em>\ref ::RSA_public_encrypt</em></li>
   <li><em>\ref ::RSA_private_decrypt</em></li>
   <li><em>\ref ::RSA_private_encrypt</em></li>
   <li><em>\ref ::RSA_public_decrypt</em></li>
   </ul>

   AES Encryption function prototypes.

   <ul>
   <li><em>\ref ::AES_set_encrypt_key</em></li>
   <li><em>\ref ::AES_set_decrypt_key</em></li>
   <li><em>\ref ::AES_encrypt</em></li>
   <li><em>\ref ::AES_decrypt</em></li>
   <li><em>\ref ::AES_ecb_encrypt</em></li>
   <li><em>\ref ::AES_cbc_encrypt</em></li>
   <li><em>\ref ::AES_cfb1_encrypt</em></li>
   <li><em>\ref ::AES_cfb8_encrypt</em></li>
   <li><em>\ref ::AES_cfb128_encrypt</em></li>
   <li><em>\ref ::AES_ofb128_encrypt</em></li>
   <li><em>\ref ::AES_ctr128_encrypt</em></li>
   <li><em>\ref ::AES_ige_encrypt</em></li>
   <li><em>\ref ::AES_bi_ige_encrypt</em></li>
   <li><em>\ref ::AES_wrap_key</em></li>
   <li><em>\ref ::AES_unwrap_key</em></li>
   </ul>

 */

/*! \file crypt.h
    \brief Main crypto library header file
*/

# if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
# endif

/*!
 * Standard header files. These could be explicitly included from each C
 * source file if required, they are just included here for convenience.
 */
# include <stdio.h>
# include <stdlib.h>
# include <string.h>

/*!
 * Header files required for RSA encryption.
 */
# include <rand.h>
# include <rsa.h>

/*!
 * Header files required for AES encryption.
 */
# include <aes.h>
# include <modes.h>


/*
 * \section 1 Pseudo-random number generator function prototypes.
 *
 * These routines can be used to generate pseudo random numbers and can be
 * used to 'seed' the pseudo random number generator (RNG).  The RNG make no
 * effort to reproduce the same random number stream with each execution.
 */

/*!
 * This routine adds more 'seed' data to the RNG state.
 *
 * RAND_add() mixes the num bytes at buf into the PRNG state. Thus
 * if the data at buf are unpredictable to an adversary, this
 * increases the uncertainty about the state and makes the PRNG output
 * less predictable. Suitable input comes from user interaction (random
 * key presses, mouse movements) and certain hardware events. The
 * entropy argument is (the lower bound of) an estimate of how much
 * randomness is contained in buf, measured in bytes. Details about
 * sources of randomness and how to estimate their entropy can be found
 * in the literature, e.g. RFC 1750.
 *
 * RAND_add() may be called with sensitive data such as user entered
 * passwords. The seed values cannot be recovered from the PRNG output.
 *
 * \param buf
 * Buffer containing the seed
 *
 * \param num
 * Length of seed
 *
 * \param entropy
 * Lower bound of estimated entropy of buffer randomness
 */
extern void
RAND_add (const void *buf, int num, double entropy);

/*!
 * This routine adds more 'seed' data to the RNG state.
 *
 * 'num' bytes are are taken from 'buf' and added to the RNG state.
 *
 * This routine can be called with sensitive data such as user entered
 * passwords.  This sensitive data is in no way recoverable from the RAND
 * library routines or state.  Try to pass as much data from 'random'
 * sources as possible into the RNG via this function.
 *
 * RAND_seed() is equivalent to RAND_add() when num == entropy.
 *
 * \param buf
 * Buffer containing the seed
 *
 * \param num
 * Length of seed
 */
extern void
RAND_seed (const void *buf, int num);

/*!
 * RAND_bytes() puts num cryptographically strong pseudo-random bytes
 * into buf. An error occurs if the PRNG has not been seeded with
 * enough randomness to ensure an unpredictable byte sequence.
 *
 * Ensure that the RAND_seed() function has been called before using
 * this routine.
 *
 * \param buf
 * Buffer to be filled with random data
 *
 * \param num
 * Length of randon data added to the buffer
 *
 * \returns
 * Returns 1 on success, 0 otherwise.
 * Can also return -1 if the function is not supported by the current RAND
 * method (although this is not supported in this version of the library).
 */
extern int
RAND_bytes (unsigned char *buf, int num);

/*!
 *
 * RAND_pseudo_bytes() puts num pseudo-random bytes into buf.
 * Pseudo-random byte sequences generated by RAND_pseudo_bytes() will be
 * unique if they are of sufficient length, but are not necessarily
 * unpredictable. They can be used for non-cryptographic purposes and for
 * certain purposes in cryptographic protocols, but usually not for key
 * generation etc.
 *
 * The contents of buf is mixed into the entropy pool before retrieving
 * the new pseudo-random bytes unless disabled at compile time.
 *
 * Ensure that the RAND_seed() function has been called before using this
 * routine.
 *
 * \param buf
 * Buffer to be filled with random data
 *
 * \param num
 * Length of randon data added to the buffer
 *
 * \returns
 * Returns 1 if the bytes generated are cryptographically strong, 0 otherwise.
 * Can also return -1 if the function is not supported by the current RAND
 * method (although this is not supported in this version of the library).
 */
extern int
RAND_pseudo_bytes (unsigned char *buf, int num);

/*!
 * This routine cleans up after using the RNG by clearing all the RNG
 * state information.
 */
extern void
RAND_cleanup (void);

/*!
 * This routine returns the current state of the RNG. This is used to test
 * if the PRNG has been seeded with enough data.
 *
 * \returns
 * Non-zero if RNG has sufficient entropy, otherwise zero
 */
extern int
RAND_status (void);


/*
 * \section 2 RSA Encryption function prototypes.
 *
 * Note that only RSA_public_encrypt() and RSA_private_decrypt() should be
 * used for encrypting and decrypting data.
 */

/*!
 * RSA_new() allocates and initializes an RSA structure. It is equivalent to
 * calling RSA_new_method(NULL).
 *
 * \returns
 * If the allocation fails, NULL is returned and an error code is set that
 * can be obtained by ERR_get_error() (if compiled in). Otherwise it
 * returns a pointer to the newly allocated structure.
 */
extern RSA
*RSA_new (void);

/*!
 * RSA_free() frees the RSA structure and its components. The key is
 * erased before the memory is returned to the system.
 *
 * \param r
 * RSA structure to be freed.
 */
extern void
RSA_free (RSA *r);

/*!
 * This function validates RSA keys. It checks that p and q are in fact
 * prime, and that n = p*q.
 *
 * It also checks that d*e = 1 mod (p-1*q-1), and that dmp1, dmq1 and iqmp
 * are set correctly or are NULL.
 *
 * As such, this function can not be used with any arbitrary RSA key object,
 * even if it is otherwise fit for regular RSA operation.
 *
 * This function does not work on RSA public keys that have only the modulus
 * and public exponent elements populated. It performs integrity checks on all
 * the RSA key material, so the RSA key structure must contain all the private
 * key data too.
 *
 * \param r
 * RSA key structure to be checked.
 *
 * \returns
 * RSA_check_key() returns 1 if rsa is a valid RSA key, and 0 otherwise.
 * -1 is returned if an error occurs while checking the key.
 *
 * If the key is invalid or an error occurred, the reason code can be
 * obtained by ERR_get_error() (if compiled in).
 */
extern int
RSA_check_key (const RSA *r);

/*!
 * This function returns the RSA modulus size in bytes. It can be used to
 * determine how much memory must be allocated for an RSA encrypted
 * value.
 *
 * \param r
 * RSA key structure to be checked. Must not be NULL
 *
 * \returns
 * Returns RSA modulus size in bytes if rsa is a valid RSA key
 * -1 is returned if an error occurs while checking the key.
 */
extern int
RSA_size (const RSA *r);

/*!
 * RSA_public_encrypt() encrypts the flen bytes at from (usually a
 * session key) using the public key rsa and stores the ciphertext in
 * to. to must point to RSA_size(rsa) bytes of memory.
 *
 * flen must be exactly RSA_size(rsa) for RSA_NO_PADDING and the random
 * number generator must be seeded prior to calling RSA_public_encrypt().
 *
 * \param flen
 * Length of data to be encrypted
 *
 * \param from
 * Buffer containing plaintext to be encrypted
 *
 * \param to
 * Buffer containing ciphertest
 *
 * \param rsa
 * Public key to be used for encryption
 *
 * \param padding
 * Padding type to be used. Only RSA_NO_PADDING is supported in this
 * version of the library
 *
 * \returns
 * Returns the size of the encrypted data (i.e. RSA_size(rsa)).
 * -1 is returned if an error occurs while encrypting the data.
 * If an error occurred, the reason code can be obtained by ERR_get_error()
 * (if compiled in).
 */
extern int
RSA_public_encrypt (int flen, const unsigned char *from,
                    unsigned char *to, RSA * rsa, int padding);

/*!
 * RSA_private_decrypt() decrypts the flen bytes at from using the
 * private key rsa and stores the plaintext in to. to must point
 * to a memory section large enough to hold the decrypted data.
 * padding is the padding mode that was used to encrypt the data.
 *
 * flen must be exactly RSA_size(rsa) for RSA_NO_PADDING and the random
 * number generator must be seeded prior to calling RSA_public_encrypt().
 *
 * \param flen
 * Length of data to be decrypted
 *
 * \param from
 * Buffer containing ciphertext to be decrypted
 *
 * \param to
 * Buffer containing plaintext
 *
 * \param rsa
 * Private key to be used for decryption
 *
 * \param padding
 * Padding type used in ciphertext. Only RSA_NO_PADDING is supported in this
 * version of the library
 *
 * \returns
 * Returns the size of the recovered plaintext.
 * -1 is returned if an error occurs while decrypting the data.
 * If an error occurred, the reason code can be obtained by ERR_get_error()
 * (if compiled in).
 */
extern int
RSA_private_decrypt (int flen, const unsigned char *from,
                     unsigned char *to, RSA * rsa, int padding);

/*!
 * RSA_private_encrypt() signs the flen bytes at from (usually a
 * message digest with an algorithm identifier) using the private key
 * rsa and stores the signature in to. to must point to RSA_size(rsa)
 * bytes of memory.
 *
 * flen must be exactly RSA_size(rsa) for RSA_NO_PADDING and the random
 * number generator must be seeded prior to calling RSA_private_encrypt().
 *
 * \param flen
 * Length of data to be encrypted
 *
 * \param from
 * Buffer containing plaintext to be encrypted
 *
 * \param to
 * Buffer containing ciphertest
 *
 * \param rsa
 * Private key to be used for encryption
 *
 * \param padding
 * Padding type to be used. Only RSA_NO_PADDING is supported in this
 * version of the library
 *
 * \returns
 * Returns the size of the signature data (i.e. RSA_size(rsa)).
 * -1 is returned if an error occurs while encrypting the data.
 * If an error occurred, the reason code can be obtained by ERR_get_error()
 * (if compiled in).
 */
extern int
RSA_private_encrypt (int flen, const unsigned char *from,
                     unsigned char *to, RSA * rsa, int padding);

/*!
 * RSA_public_decrypt() recovers the message digest from the flen
 * padding is the padding mode that was used to sign the data.
 *
 * flen must be exactly RSA_size(rsa) for RSA_NO_PADDING and the random
 * number generator must be seeded prior to calling RSA_public_encrypt().
 *
 * \param flen
 * Length of data to be decrypted
 *
 * \param from
 * Buffer containing ciphertext to be decrypted
 *
 * \param to
 * Buffer containing plaintext
 *
 * \param rsa
 * Public key to be used for decryption
 *
 * \param padding
 * Padding type used in ciphertext. Only RSA_NO_PADDING is supported in this
 * version of the library
 *
 * \returns
 * Returns the size of the recovered plaintext.
 * -1 is returned if an error occurs while decrypting the data.
 * If an error occurred, the reason code can be obtained by ERR_get_error()
 * (if compiled in).
 */
extern int
RSA_public_decrypt (int flen, const unsigned char *from,
                    unsigned char *to, RSA * rsa, int padding);


/*
 *  \section 3 AES Encryption function prototypes.
 */

/*!
 * This function expands the cipher key into the AES encryption key schedule.
 *
 * \param userKey
 * Caller supplied cipher key
 *
 * \param bits
 * Length of supplied key in bits. Only 128, 192 and 256 are allowable values
 *
 * \param key
 * Pointer to AES_KEY structure which is filled in on return
 *
 * \returns
 * Returns 0 on success, and a negaive error code otherwise.
 */
extern int
AES_set_encrypt_key(const unsigned char *userKey,
                    const int bits,
                    AES_KEY *key);

/*!
 * This function expands the cipher key into the AES decryption key schedule.
 *
 * \param userKey
 * Caller supplied cipher key
 *
 * \param bits
 * Length of supplied key in bits. Only 128, 192 and 256 are allowable values
 *
 * \param key
 * Pointer to AES_KEY structure which is filled in on return
 *
 * \returns
 * Returns 0 on success, and a negaive error code otherwise.
 */
extern int
AES_set_decrypt_key(const unsigned char *userKey,
                    const int bits,
                    AES_KEY *key);

/*!
 * This function encrypts the supplied plaintext buffer into the supplied
 * ciphertext buffer using the supplied AES encryption key. The buffers
 * must be a single block size in length (16 bytes) and can overlap.
 *
 * \param in
 * Buffer containing plaintext to be encrypted
 *
 * \param out
 * Buffer containing encrypted ciphertest
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 */
extern void
AES_encrypt(const unsigned char *in,
            unsigned char *out,
            const AES_KEY *key);

/*!
 * This function decrypts the supplied ciphertext buffer into the supplied
 * plaintext buffer using the supplied AES decryption key. The buffers
 * must be a single block size in length (16 bytes) and can overlap.
 *
 * \param in
 * Buffer containing ciphertest to be decrypted
 *
 * \param out
 * Buffer containing decrypted plaintext
 *
 * \param key
 * Pointer to AES_KEY decryption key structure
 */
extern void
AES_decrypt(const unsigned char *in,
            unsigned char *out,
            const AES_KEY *key);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. The buffers
 * must be a single block size in length (16 bytes). This function uses the
 * simplest AES block cipher encryption mode called Electronic Codebook (ECB)
 * mode.
 *
 * In this mode the message is divided into blocks and each block is
 * encrypted or decrypted separately by calling this function.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param enc
 * Can be either AES_ENCRYPT or AES_DECRYPT
 */
extern void
AES_ecb_encrypt(const unsigned char *in,
                unsigned char *out,
                const AES_KEY *key,
                const int enc);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses the AES Cipher-Block Chaining (CBC) mode.
 *
 * In this mode each block of plaintext is XORed with the previous ciphertext
 * block before being encrypted. This way, each ciphertext block is dependent
 * on all plaintext blocks processed up to that point. Also, to make each
 * message unique, an initialization vector must be used in the first block.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (must be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param ivec
 * Input buffer containing initialisation vector. Must be block size long.
 *
 * \param enc
 * Can be either AES_ENCRYPT or AES_DECRYPT
 */
extern void
AES_cbc_encrypt(const unsigned char *in,
                unsigned char *out,
                size_t length,
                const AES_KEY *key,
                unsigned char *ivec,
                const int enc);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses the AES Cipher Feedback (CFB) mode.
 *
 * This mode is a close relative of CBC mode and makes a block cipher into
 * a self-synchronizing stream cipher. Operation is very similar; in
 * particular, CFB decryption is almost identical to CBC encryption performed
 * in reverse. Again an initialization vector must be used in the first block.
 *
 * This mode uses 128-bit Cipher Feedback Mode. See NIST Standard 800-38A
 * for more details.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (does not have to be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param ivec
 * Input buffer containing initialisation vector. Must be block size long.
 *
 * \param num
 * The extra state information to record how much of the 128bit block we
 * have used is contained in *num
 *
 * \param enc
 * Can be either AES_ENCRYPT or AES_DECRYPT
 */
extern void
AES_cfb128_encrypt(const unsigned char *in,
                   unsigned char *out,
                   size_t length,
                   const AES_KEY *key,
                   unsigned char *ivec,
                   int *num,
                   const int enc);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses the AES Cipher Feedback (CFB) mode.
 *
 * This mode is a close relative of CBC mode and makes a block cipher into
 * a self-synchronizing stream cipher. Operation is very similar; in
 * particular, CFB decryption is almost identical to CBC encryption performed
 * in reverse. Again an initialization vector must be used in the first block.
 *
 * This mode uses 1-bit Cipher Feedback Mode instead of 128 bits. See NIST
 * Standard 800-38A for more details.
 *
 * Note: This expects the input to be packed, MS bit first
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (does not have to be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param ivec
 * Input buffer containing initialisation vector. Must be block size long.
 *
 * \param num
 * The extra state information to record how much of the 128bit block we
 * have used is contained in *num
 *
 * \param enc
 * Can be either AES_ENCRYPT or AES_DECRYPT
 */
extern void
AES_cfb1_encrypt(const unsigned char *in,
                 unsigned char *out,
                 size_t length,
                 const AES_KEY *key,
                 unsigned char *ivec,
                 int *num,
                 const int enc);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses the AES Cipher Feedback (CFB) mode.
 *
 * This mode is a close relative of CBC mode and makes a block cipher into
 * a self-synchronizing stream cipher. Operation is very similar; in
 * particular, CFB decryption is almost identical to CBC encryption performed
 * in reverse. Again an initialization vector must be used in the first block.
 *
 * This mode uses 8-bit Cipher Feedback Mode instead of 128 bits. See NIST
 * Standard 800-38A for more details.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (does not have to be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param ivec
 * Input buffer containing initialisation vector. Must be block size long.
 *
 * \param num
 * The extra state information to record how much of the 128bit block we
 * have used is contained in *num
 *
 * \param enc
 * Can be either AES_ENCRYPT or AES_DECRYPT
 */
extern void
AES_cfb8_encrypt(const unsigned char *in,
                 unsigned char *out,
                 size_t length,
                 const AES_KEY *key,
                 unsigned char *ivec,
                 int *num,
                 const int enc);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses the AES Output Feedback (OFB) mode.
 *
 * The output feedback (OFB) mode makes a block cipher into a synchronous
 * stream cipher. It generates keystream blocks, which are then XORed with
 * the plaintext blocks to get the ciphertext. Just as with other stream
 * ciphers, flipping a bit in the ciphertext produces a flipped bit in the
 * plaintext at the same location. This property allows many error correcting
 * codes to function normally even when applied before encryption.
 *
 * Because of the symmetry of the XOR operation, encryption and decryption
 * are exactly the same.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (must be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param ivec
 * Input buffer containing initialisation vector. Must be block size long.
 *
 * \param num
 * The extra state information to record how much of the 128bit block we
 * have used is contained in *num
 */
extern void
AES_ofb128_encrypt(const unsigned char *in,
                   unsigned char *out,
                   size_t length,
                   const AES_KEY *key,
                   unsigned char *ivec,
                   int *num);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses the AES Counter (CTR) mode.
 *
 * Like OFB, counter mode turns a block cipher into a stream cipher. It
 * generates the next keystream block by encrypting successive values of
 * a "counter". The counter can be any function which produces a sequence
 * which is guaranteed not to repeat for a long time, although an actual
 * counter is the simplest way to ensure this.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (must be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param ivec
 * Input buffer containing initialisation vector (also known as a "nonce"
 * in this encryption mode). Must be block size long.
 *
 * \param ecount_buf
 * Input/Output buffer containing counter information. Must be initialised
 * to zeroes before first call to AES_ctr128_encrypt()
 *
 * \param num
 * The extra state information to record how much of the 128bit block we
 * have used is contained in *num
 */
extern void
AES_ctr128_encrypt(const unsigned char *in,
                   unsigned char *out,
                   size_t length,
                   const AES_KEY *key,
                   unsigned char *ivec,
                   unsigned char *ecount_buf,
                   unsigned int *num);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses a variant of the AES Cipher-Block Chaining (CBC) mode
 * known as Infinite Garble Extension (IGE) mode. It has the property
 * that errors are propagated forward indefinitely.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (must be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param ivec
 * Input buffer containing initialisation vector. Must be 2*block size long
 * for this mode.
 *
 * \param enc
 * Can be either AES_ENCRYPT or AES_DECRYPT
 */
extern void
AES_ige_encrypt(const unsigned char *in,
                unsigned char *out,
                size_t length,
                const AES_KEY *key,
                unsigned char *ivec,
                const int enc);

/*!
 * This function encrypts or decrypts the supplied input buffer into the
 * supplied output buffer using the supplied AES encryption key. This
 * function uses a variant of the AES Cipher-Block Chaining (CBC) mode
 * known as Bi-directional Infinite Garble Extension (IGE) mode. In this
 * mode errors are propagated indefinitely in both directions i.e. any
 * change to the ciphertext will cause all of the plaintext to be corrupted.
 *
 * \param in
 * Input buffer containing either plaintext or ciphertext
 *
 * \param out
 * Output buffer containing either plaintext or ciphertext
 *
 * \param length
 * Length of input buffer (must be a multiple of the block size)
 *
 * \param key
 * Pointer to AES_KEY encryption key 1 structure
 *
 * \param key2
 * Pointer to AES_KEY encryption key 2 structure
 *
 * \param ivec
 * Input buffer containing initialisation vector. Must be 4*block size long
 * for this mode.
 *
 * \param enc
 * Can be either AES_ENCRYPT or AES_DECRYPT
 */
extern void
AES_bi_ige_encrypt(const unsigned char *in,
                   unsigned char *out,
                   size_t length,
                   const AES_KEY *key,
                   const AES_KEY *key2,
                   const unsigned char *ivec,
                   const int enc);

/*!
 * This function encrypts the supplied input buffer into the supplied output
 * buffer using the supplied AES encryption key. This function implements
 * RFC3394 compatible AES key wrapping. The buffers must not overlap
 * and the output buffer must be 8 bytes longer than the input buffer.
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param iv
 * Input buffer containing initialisation vector. Must be block size long.
 *
 * \param out
 * Output buffer containing ciphertext
 *
 * \param in
 * Input buffer containing plaintext
 *
 * \param inlen
 * Length of input buffer (must be a multiple of the block size)
 *
 * \return
 * Returns inlen + 8 on success, -1 if an error occurs
 */
extern int
AES_wrap_key(AES_KEY *key,
             const unsigned char *iv,
             unsigned char *out,
             const unsigned char *in,
             unsigned int inlen);

/*!
 * This function decrypts the supplied input buffer into the supplied output
 * buffer using the supplied AES decryption key. This function implements
 * RFC3394 compatible AES key unwrapping. The buffers must not overlap
 * and the input buffer may be 8 bytes longer than the output buffer.
 *
 * \param key
 * Pointer to AES_KEY encryption key structure
 *
 * \param iv
 * Input buffer containing initialisation vector. Must be block size long.
 *
 * \param out
 * Output buffer containing plaintext
 *
 * \param in
 * Input buffer containing ciphertext
 *
 * \param inlen
 * Length of input buffer
 *
 * \return
 * Returns inlen on success, 0 or -1 if an error occurs
 */
extern int
AES_unwrap_key(AES_KEY *key,
               const unsigned char *iv,
               unsigned char *out,
               const unsigned char *in,
               unsigned int inlen);

# if defined(__cplusplus) || defined(c_plusplus)
}
# endif
#endif /* __CRYPT_H */
