/*!
 * \file    rsa_compat.c
 * \brief   Provides a wrapper layer to RSA encryption/decryption to ensure
 *          compatibility with the previous RSA encryption/decryption used
 *          by Honeywell.
 *
 * \par File contents
 * The decrypting side should create an RSA structure containing the
 * private/public key details as normal. RSA_compat_private_decrypt() may
 * then be used to decrypt incoming data.
 *
 * RSA_compat_extract_public_key() may be used to extract the public key in
 * the form of a buffer which may be passed to the encryptor.
 *
 * RSA_compat_public_encrypt() may then be used with the provided public key
 * in buffer form to encrypt data.
 */

#include "bn.h"
#include "rsa_compat.h"

/**
 * Reverse the given buffer.
 *
 * \param buffer
 * The buffer to reverse.
 *
 * \param bufferLen
 * The size of the supplied buffer.
 */
static void _RSA_compat_reverse_buffer(unsigned char * buffer,
                                       unsigned int bufferLen)
{
    unsigned int count;
    unsigned int numSwaps = (bufferLen / 2);

    for (count = 0; count < numSwaps; count++)
    {
        unsigned char tmp = buffer[count];
        buffer[count] = buffer[(bufferLen - 1) - count];
        buffer[(bufferLen - 1) - count] = tmp;
    }
}

/**
 * Allocate a buffer and asign it to the reverse of the given buffer.
 *
 * \param src
 * The buffer to create in reverse.
 *
 * \param srcLen
 * The length of the buffer to reverse.
 *
 * \returns
 * The created buffer, or NULL on error.
 */
static unsigned char *
_RSA_compat_create_reverse_buffer(const unsigned char * src,
                                  unsigned int        srcLen)
{
    unsigned char *dest = OPENSSL_malloc(srcLen);

    if (dest)
    {
        unsigned int count;

        for(count = 0; count < srcLen; count++)
        {
            dest[count] = src[(srcLen - 1) - count ];
        }
    }

    return dest;
}

/**
 * Allocate and return a BIGNUM with the exponent used by the compatible
 * Honeywell code.
 *
 * \returns
 * The allocated BIGNUM.
 */
static BIGNUM * _RSA_compat_get_exponent(void)
{
    BIGNUM *e = BN_new();

    if (e)
    {
        BN_set_word(e, RSA_3);
    }

    return e;
}

#ifndef OPENSSL_PUBLIC_ENCRYPT_ONLY
/**
 * Extract the publick key in a way that is compatible with previous code.
 * This will create a key which can be passed and used to encrypt using
 * RSA_compat_private_decrypt().
 *
 * \param rsa
 * RSA structure containing the key.
 *
 * \apram keyBuffer
 * The buffer to fill with the key.
 *
 * \param bufferLen
 * The size of the provided buffer.
 *
 * \returns
 * The size of the returned key, -1 on error.
 */
int RSA_compat_extract_public_key (RSA              * rsa,
                                   unsigned char    * keyBuffer,
                                   unsigned int     bufferLen )
{
    int ret = -1;

    /*
     * Check the n field (which contains the public key) is present and the
     * buffer is large enough to copy the key into.
     */
    if ((rsa->n) &&
        ((bufferLen * 8) >= BN_num_bits(rsa->n)))
    {
        ret = BN_bn2bin(rsa->n, keyBuffer);

        if (ret >= 0)
        {
            _RSA_compat_reverse_buffer(keyBuffer, ret);
        }
    }

    return ret;
}

/**
 * Decrypt the given data with the given key.
 *
 * \param dataLen
 * The size of the data block to be decrypted. The 'to' buffer must also
 * be at least this size.
 *
 * \param from
 * The data to be decrypted.
 *
 * \param to
 * On success, the decrypted data.
 *
 * \param rsa
 * The key used to decrypt.
 *
 * \returns
 * The size of the recovered plaintext, -1 on error.
 */
int RSA_compat_private_decrypt (int                 dataLen,
                                const unsigned char * from,
                                unsigned char       * to,
                                RSA                 * rsa)
{
    int ret = -1;

    unsigned char * tmpBuffer;

    /*
     * To keep compatibility with the previous honeywell code we need
     * to reverse the ciphertext before encryption, then reverse the
     * recovered plaintext.
     */
    tmpBuffer = _RSA_compat_create_reverse_buffer( from, dataLen );

    if (tmpBuffer)
    {
        ret = RSA_private_decrypt(dataLen,
                                  tmpBuffer,
                                  to,
                                  rsa,
                                  RSA_NO_PADDING );

        _RSA_compat_reverse_buffer(to, dataLen);


        OPENSSL_free(tmpBuffer);
    }

    return ret;
}
#endif /* OPENSSL_PUBLIC_ENCRYPT_ONLY */

/**
 * Encrypt the given data with the given key.
 *
 * \param dataLen
 * The size of the data block to be encrypted. The 'to' buffer must also
 * be at least this size.
 *
 * \param from
 * The data to be encrypted.
 *
 * \param to
 * On success, the encrypted data.
 *
 * \param key
 * The key to use for the encryption.
 *
 * \param keyLen
 * The legnth of the key.
 *
 * \returns
 * The size of the encrypted data, -1 on error.
 */
int RSA_compat_public_encrypt (int                  dataLen,
                               const unsigned char  * from,
                               unsigned char        * to,
                               const unsigned char  * key,
                               unsigned int         keyLen)
{
    int ret = -1;

    unsigned char * tmpBuffer;

    /*
     * To keep compatibility with the previous honeywell code we need
     * to reverse the plaintext and key before encryption, then reverse the
     * created ciphertext.
     */
    tmpBuffer = _RSA_compat_create_reverse_buffer( from, dataLen );

    if (tmpBuffer)
    {
        unsigned char * tmpKey;

        tmpKey = _RSA_compat_create_reverse_buffer( key, keyLen );

        if (tmpKey)
        {
            RSA *rsa;

            rsa = RSA_new();

            if (rsa)
            {
                rsa->e = _RSA_compat_get_exponent();
                rsa->n = BN_bin2bn( tmpKey, keyLen, NULL );

                if (rsa->e && rsa->n)
                {
                    ret = RSA_public_encrypt(dataLen,
                                             tmpBuffer,
                                             to,
                                             rsa,
                                             RSA_NO_PADDING );

                    _RSA_compat_reverse_buffer(to, dataLen);
                }

                RSA_free(rsa);
            }

            OPENSSL_free(tmpKey);
        }

        OPENSSL_free(tmpBuffer);
    }

    return ret;
}
