/* ----------------------------------------------------------------------------
 *         ATMEL Microcontroller Software Support
 * ----------------------------------------------------------------------------
 * Copyright (c) 2008, Atmel Corporation
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Atmel's name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ----------------------------------------------------------------------------
 */

//------------------------------------------------------------------------------
//         Headers
//------------------------------------------------------------------------------

#include <board.h>
#include <pio/pio.h>
#include <dbgu/dbgu.h>

#include <crypt.h>

//------------------------------------------------------------------------------
//         Local variables
//------------------------------------------------------------------------------

/// Pio pins to configure.
static const Pin pins[] = {PINS_DBGU};

//------------------------------------------------------------------------------
//         Global functions
//------------------------------------------------------------------------------


#define SetKey \
  key->n = BN_bin2bn(n, sizeof(n)-1, key->n); \
  key->e = BN_bin2bn(e, sizeof(e)-1, key->e); \
  key->d = BN_bin2bn(d, sizeof(d)-1, key->d); \
  key->p = BN_bin2bn(p, sizeof(p)-1, key->p); \
  key->q = BN_bin2bn(q, sizeof(q)-1, key->q); \
  key->dmp1 = BN_bin2bn(dmp1, sizeof(dmp1)-1, key->dmp1); \
  key->dmq1 = BN_bin2bn(dmq1, sizeof(dmq1)-1, key->dmq1); \
  key->iqmp = BN_bin2bn(iqmp, sizeof(iqmp)-1, key->iqmp); \
  memcpy(c, ctext_ex, sizeof(ctext_ex) - 1); \
  return (sizeof(ctext_ex) - 1);

static int key1(RSA *key, unsigned char *c)
{
    static unsigned char n[] =
          "\x00\xAA\x36\xAB\xCE\x88\xAC\xFD\xFF\x55\x52\x3C\x7F\xC4\x52\x3F"
          "\x90\xEF\xA0\x0D\xF3\x77\x4A\x25\x9F\x2E\x62\xB4\xC5\xD9\x9C\xB5"
          "\xAD\xB3\x00\xA0\x28\x5E\x53\x01\x93\x0E\x0C\x70\xFB\x68\x76\x93"
          "\x9C\xE6\x16\xCE\x62\x4A\x11\xE0\x08\x6D\x34\x1E\xBC\xAC\xA0\xA1"
          "\xF5";

    static unsigned char e[] = "\x11";

    static unsigned char d[] =
          "\x0A\x03\x37\x48\x62\x64\x87\x69\x5F\x5F\x30\xBC\x38\xB9\x8B\x44"
          "\xC2\xCD\x2D\xFF\x43\x40\x98\xCD\x20\xD8\xA1\x38\xD0\x90\xBF\x64"
          "\x79\x7C\x3F\xA7\xA2\xCD\xCB\x3C\xD1\xE0\xBD\xBA\x26\x54\xB4\xF9"
          "\xDF\x8E\x8A\xE5\x9D\x73\x3D\x9F\x33\xB3\x01\x62\x4A\xFD\x1D\x51";

    static unsigned char p[] =
          "\x00\xD8\x40\xB4\x16\x66\xB4\x2E\x92\xEA\x0D\xA3\xB4\x32\x04\xB5"
          "\xCF\xCE\x33\x52\x52\x4D\x04\x16\xA5\xA4\x41\xE7\x00\xAF\x46\x12"
          "\x0D";

    static unsigned char q[] =
          "\x00\xC9\x7F\xB1\xF0\x27\xF4\x53\xF6\x34\x12\x33\xEA\xAA\xD1\xD9"
          "\x35\x3F\x6C\x42\xD0\x88\x66\xB1\xD0\x5A\x0F\x20\x35\x02\x8B\x9D"
          "\x89";

    static unsigned char dmp1[] =
          "\x59\x0B\x95\x72\xA2\xC2\xA9\xC4\x06\x05\x9D\xC2\xAB\x2F\x1D\xAF"
          "\xEB\x7E\x8B\x4F\x10\xA7\x54\x9E\x8E\xED\xF5\xB4\xFC\xE0\x9E\x05";

    static unsigned char dmq1[] =
          "\x00\x8E\x3C\x05\x21\xFE\x15\xE0\xEA\x06\xA3\x6F\xF0\xF1\x0C\x99"
          "\x52\xC3\x5B\x7A\x75\x14\xFD\x32\x38\xB8\x0A\xAD\x52\x98\x62\x8D"
          "\x51";

    static unsigned char iqmp[] =
          "\x36\x3F\xF7\x18\x9D\xA8\xE9\x0B\x1D\x34\x1F\x71\xD0\x9B\x76\xA8"
          "\xA9\x43\xE1\x1D\x10\xB2\x4D\x24\x9F\x2D\xEA\xFE\xF8\x0C\x18\x26";

    static unsigned char ctext_ex[] =
          "\x1b\x8f\x05\xf9\xca\x1a\x79\x52\x6e\x53\xf3\xcc\x51\x4f\xdb\x89"
          "\x2b\xfb\x91\x93\x23\x1e\x78\xb9\x92\xe6\x8d\x50\xa4\x80\xcb\x52"
          "\x33\x89\x5c\x74\x95\x8d\x5d\x02\xab\x8c\x0f\xd0\x40\xeb\x58\x44"
          "\xb0\x05\xc3\x9e\xd8\x27\x4a\x9d\xbf\xa8\x06\x71\x40\x94\x39\xd2";

    SetKey;
}

void test_rsa_vector( void )
{
    int err=0;
    int v;
    RSA key;
    unsigned char ptext[256];
    unsigned char ctext[256];
    static unsigned char ptext_ex[] =
          "\x00\xAA\x36\xAB\xCE\x88\xAC\xFD\xFF\x55\x52\x3C\x7F\xC4\x52\x3F"
          "\x90\xEF\xA0\x0D\xF3\x77\x4A\x25\x9F\x2E\x62\xB4\xC5\xD9\x9C\xB5"
          "\xAD\xB3\x00\xA0\x28\x5E\x53\x01\x93\x0E\x0C\x70\xFB\x68\x76\x93"
          "\x9C\xE6\x16\xCE\x62\x4A\x11\xE0\x08\x6D\x34\x1E\xBC\xAC\xA0\xA1";

    unsigned char ctext_ex[256];
    int plen;
    int clen = 0;
    int num;

    plen = sizeof(ptext_ex) - 1;

    for (v = 0; v < 2; v++)
    {
        /* Set up static RSA key structure */
        memset( &key, 0, sizeof(key));
        key.references=1;
        key.flags=key.meth->flags;
        key.meth = RSA_get_default_method();
        if( key.meth->init != NULL ) {
            key.meth->init(&key);
        }
        clen = key1(&key, ctext_ex);
        key.flags &= ~RSA_FLAG_BLINDING;
        key.flags |= RSA_FLAG_NO_BLINDING;
        if (v == 1) {
            key.flags |= RSA_FLAG_NO_CONSTTIME;
            printf("RSA Test: Constant Time Operations Disabled\n\r");
        } else {
            printf("RSA Test: Constant Time Operations Enabled\n\r");
        }

        /* Encrypt data with public key */
        num = RSA_public_encrypt(plen, ptext_ex, ctext, &key,
                                 RSA_NO_PADDING);
        if (num != clen)
        {
            printf("*** Public encryption failed! ***\n\r");
            err=1;
            continue;
        }
        /* Decrypt data with private key */
        num = RSA_private_decrypt(num, ctext, ptext, &key,
                                  RSA_NO_PADDING);
        if ( num != plen || memcmp(ptext, ptext_ex, num) != 0)
        {
            printf("*** Private decryption failed! ***\n\r");
            err=1;
        }
        else
        {
            printf(" - Public encryption/ private decryption ok\n\r");
        }
    }
}

/* Test vectors from NIST KAT AES */
static const unsigned char cbc_test_key[32] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
                                           0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
static const unsigned char cbc_test_iv[AES_BLOCK_SIZE] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

static const unsigned char CBCGFSbox128_plaintext[AES_BLOCK_SIZE] =
{
    0xf3,0x44,0x81,0xec,0x3c,0xc6,0x27,0xba,
    0xcd,0x5d,0xc3,0xfb,0x08,0xf2,0x73,0xe6
};
static const unsigned char CBCGFSbox128_ciphertext[AES_BLOCK_SIZE] =
{
    0x03,0x36,0x76,0x3e,0x96,0x6d,0x92,0x59,
    0x5a,0x56,0x7c,0xc9,0xce,0x53,0x7f,0x5e
};

static const unsigned char CBCGFSbox192_plaintext[AES_BLOCK_SIZE] =
{
    0x1b,0x07,0x7a,0x6a,0xf4,0xb7,0xf9,0x82,
    0x29,0xde,0x78,0x6d,0x75,0x16,0xb6,0x39
};

static const unsigned char CBCGFSbox192_ciphertext[AES_BLOCK_SIZE] =
{
    0x27,0x5c,0xfc,0x04,0x13,0xd8,0xcc,0xb7,
    0x05,0x13,0xc3,0x85,0x9b,0x1d,0x0f,0x72
};

static const unsigned char CBCGFSbox256_plaintext[AES_BLOCK_SIZE] =
{
    0x01,0x47,0x30,0xf8,0x0a,0xc6,0x25,0xfe,
    0x84,0xf0,0x26,0xc6,0x0b,0xfd,0x54,0x7d
};

static const unsigned char CBCGFSbox256_ciphertext[AES_BLOCK_SIZE] =
{
    0x5c,0x9d,0x84,0x4e,0xd4,0x6f,0x98,0x85,
    0x08,0x5e,0x5d,0x6a,0x4f,0x94,0xc7,0xd7
};

static const unsigned char ECBGFSbox128_plaintext[AES_BLOCK_SIZE] =
{
    0xf3,0x44,0x81,0xec,0x3c,0xc6,0x27,0xba,
    0xcd,0x5d,0xc3,0xfb,0x08,0xf2,0x73,0xe6
};
static const unsigned char ECBGFSbox128_ciphertext[AES_BLOCK_SIZE] =
{
    0x03,0x36,0x76,0x3e,0x96,0x6d,0x92,0x59,
    0x5a,0x56,0x7c,0xc9,0xce,0x53,0x7f,0x5e
};

static const unsigned char ECBGFSbox192_plaintext[AES_BLOCK_SIZE] =
{
    0x1b,0x07,0x7a,0x6a,0xf4,0xb7,0xf9,0x82,
    0x29,0xde,0x78,0x6d,0x75,0x16,0xb6,0x39
};

static const unsigned char ECBGFSbox192_ciphertext[AES_BLOCK_SIZE] =
{
    0x27,0x5c,0xfc,0x04,0x13,0xd8,0xcc,0xb7,
    0x05,0x13,0xc3,0x85,0x9b,0x1d,0x0f,0x72
};

static const unsigned char ECBGFSbox256_plaintext[AES_BLOCK_SIZE] =
{
    0x01,0x47,0x30,0xf8,0x0a,0xc6,0x25,0xfe,
    0x84,0xf0,0x26,0xc6,0x0b,0xfd,0x54,0x7d
};

static const unsigned char ECBGFSbox256_ciphertext[AES_BLOCK_SIZE] =
{
    0x5c,0x9d,0x84,0x4e,0xd4,0x6f,0x98,0x85,
    0x08,0x5e,0x5d,0x6a,0x4f,0x94,0xc7,0xd7
};


/* test vectors from RFC 3962 */
static const unsigned char cts_test_key[16] = "chicken teriyaki";
static const unsigned char cts_test_input[64] =
                "I would like the" " General Gau's C"
                "hicken, please, " "and wonton soup.";
static const unsigned char cts_test_iv[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

static const unsigned char vector_17[17] =
{0xc6,0x35,0x35,0x68,0xf2,0xbf,0x8c,0xb4, 0xd8,0xa5,0x80,0x36,0x2d,0xa7,0xff,0x7f,
 0x97};
static const unsigned char vector_31[31] =
{0xfc,0x00,0x78,0x3e,0x0e,0xfd,0xb2,0xc1, 0xd4,0x45,0xd4,0xc8,0xef,0xf7,0xed,0x22,
 0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5};
static const unsigned char vector_32[32] =
{0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8,
 0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84};
static const unsigned char vector_47[47] =
{0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
 0xb3,0xff,0xfd,0x94,0x0c,0x16,0xa1,0x8c, 0x1b,0x55,0x49,0xd2,0xf8,0x38,0x02,0x9e,
 0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5};
static const unsigned char vector_48[48] =
{0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
 0x9d,0xad,0x8b,0xbb,0x96,0xc4,0xcd,0xc0, 0x3b,0xc1,0x03,0xe1,0xa1,0x94,0xbb,0xd8,
 0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8};
static const unsigned char vector_64[64] =
{0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0, 0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
 0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5, 0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8,
 0x48,0x07,0xef,0xe8,0x36,0xee,0x89,0xa5, 0x26,0x73,0x0d,0xbc,0x2f,0x7b,0xc8,0x40,
 0x9d,0xad,0x8b,0xbb,0x96,0xc4,0xcd,0xc0, 0x3b,0xc1,0x03,0xe1,0xa1,0x94,0xbb,0xd8};


static AES_KEY encks, decks;

void test_cbc_vector( const unsigned char *ptext_in,
                      const unsigned char *ctext_out,
                      int len )
{
    unsigned char cleartext[64];
    unsigned char iv[sizeof(cbc_test_iv)];
    unsigned char ciphertext[64];

    /* Test CBC Encryption */
    memcpy(iv,cbc_test_iv,sizeof(cbc_test_iv));
    AES_cbc_encrypt(ptext_in, ciphertext, len, &encks, iv, 1 );
    if (memcmp(ciphertext, ctext_out, len))
        printf(" *** encryption output mismatch ***\n\r");
    else
        printf(" - encryption output match\n\r");

    /* Test CBC Decryption */
    memcpy(iv,cbc_test_iv,sizeof(cbc_test_iv));
    AES_cbc_encrypt(ctext_out, cleartext, len, &decks, iv, 0);
    if (memcmp(cleartext, ptext_in, len))
        printf(" *** decryption output mismatch ***\n\r");
    else
        printf(" - decryption output match\n\r");
}

void test_cts_vector(const unsigned char *vector,int len)
{
    unsigned char cleartext[64];
    unsigned char iv[sizeof(cts_test_iv)];
    unsigned char ciphertext[64];
    int tail;

    printf("cts_vector_%d\n\r",len);

    if ((tail=len%16) == 0) tail = 16;
    tail += 16;

    /* test block-based encryption */
    memcpy(iv,cts_test_iv,sizeof(cts_test_iv));
    CRYPTO_cts128_encrypt_block(cts_test_input, ciphertext, len,
                                &encks, iv, (block128_f)AES_encrypt);
    if (memcmp(ciphertext,vector,len))
        printf("output_%d mismatch\n\r",len);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        printf("iv_%d mismatch\n\r",len);

    /* test block-based decryption */
    memcpy(iv,cts_test_iv,sizeof(cts_test_iv));
    CRYPTO_cts128_decrypt_block(ciphertext,cleartext,len,&decks,iv,(block128_f)AES_decrypt);
    if (memcmp(cleartext,cts_test_input,len))
        printf("input_%d mismatch\n\r",len);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        printf("iv_%d mismatch\n\r",len);

    /* test streamed encryption */
    memcpy(iv,cts_test_iv,sizeof(cts_test_iv));
    CRYPTO_cts128_encrypt(cts_test_input,ciphertext,len,&encks,iv,(cbc128_f)AES_cbc_encrypt);
    if (memcmp(ciphertext,vector,len))
        printf("output_%d mismatch\n\r",len);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        printf("iv_%d mismatch\n\r",len);

    /* test streamed decryption */
    memcpy(iv,cts_test_iv,sizeof(cts_test_iv));
    CRYPTO_cts128_decrypt(ciphertext,cleartext,len,&decks,iv,(cbc128_f)AES_cbc_encrypt);
    if (memcmp(cleartext,cts_test_input,len))
        printf("input_%d mismatch\n\r",len);
    if (memcmp(iv,vector+len-tail,sizeof(iv)))
        printf("iv_%d mismatch\n\r",len);

    printf("cts_vector_%d done!\n\r",len);
}

//------------------------------------------------------------------------------
/// Default main() function. Initializes the DBGU and writes a string on the
/// DBGU.
//------------------------------------------------------------------------------
int main(void)
{
    PIO_Configure(pins, PIO_LISTSIZE(pins));
    DBGU_Configure(DBGU_STANDARD, 115200, BOARD_MCK);
    printf("\n\r-- RSA/AES Crypto Test --\n\r\n\r");

    /* RSA Encryption/decryption */
    printf("RSA test key length 1024 bits\n\r");
    test_rsa_vector();

    /* Test AES 128 bit CBC Encryption/decryption */
    printf("AES CBC test key length %d bits\n\r", 128);
    AES_set_encrypt_key(cbc_test_key,128,&encks);
    AES_set_decrypt_key(cbc_test_key,128,&decks);
    test_cbc_vector( CBCGFSbox128_plaintext,
                     CBCGFSbox128_ciphertext,
                     sizeof( CBCGFSbox128_plaintext ));

    /* Test AES 192 bit CBC Encryption/decryption */
    printf("AES CBC test key length %d bits\n\r", 192);
    AES_set_encrypt_key(cbc_test_key,192,&encks);
    AES_set_decrypt_key(cbc_test_key,192,&decks);
    test_cbc_vector( CBCGFSbox192_plaintext,
                     CBCGFSbox192_ciphertext,
                     sizeof( CBCGFSbox192_plaintext ));

    /* Test AES 256 bit CBC Encryption/decryption */
    printf("AES CBC test key length %d bits\n\r", 256);
    AES_set_encrypt_key(cbc_test_key,256,&encks);
    AES_set_decrypt_key(cbc_test_key,256,&decks);
    test_cbc_vector( CBCGFSbox256_plaintext,
                     CBCGFSbox256_ciphertext,
                     sizeof( CBCGFSbox256_plaintext ));

    /* Test AES 128 bit CTS Encryption/decryption */
    printf("AES CTS test key length %d bits\n\r", 128);
    AES_set_encrypt_key(cts_test_key,128,&encks);
    AES_set_decrypt_key(cts_test_key,128,&decks);
    test_cts_vector(vector_17,sizeof(vector_17));
    test_cts_vector(vector_31,sizeof(vector_31));
    test_cts_vector(vector_32,sizeof(vector_32));
    test_cts_vector(vector_47,sizeof(vector_47));
    test_cts_vector(vector_48,sizeof(vector_48));
    test_cts_vector(vector_64,sizeof(vector_64));

    printf("\n\r-- RSA/AES Crypto Test Complete --\n\r");

    return 0;
}
