/* crypto/test/aes_cbctest.c */
/* ====================================================================
 * Copyright (c) 2008 The OpenSSL Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit. (http://www.OpenSSL.org/)"
 *
 * 4. The names "OpenSSL Toolkit" and "OpenSSL Project" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    licensing@OpenSSL.org.
 *
 * 5. Products derived from this software may not be called "OpenSSL"
 *    nor may "OpenSSL" appear in their names without prior written
 *    permission of the OpenSSL Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit (http://www.OpenSSL.org/)"
 *
 * THIS SOFTWARE IS PROVIDED BY THE OpenSSL PROJECT ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE OpenSSL PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */

#include <crypt.h>
#ifdef PERFTEST
# include <sys/time.h>
# define NUM_ITERATIONS  1000000
#endif

/* Test vectors from NIST KAT AES */
static const unsigned char test_key[32] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
                                           0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
static const unsigned char test_iv[AES_BLOCK_SIZE] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

static const unsigned char CBCGFSbox128_plaintext[AES_BLOCK_SIZE] =
{
    0xf3,0x44,0x81,0xec,0x3c,0xc6,0x27,0xba,
    0xcd,0x5d,0xc3,0xfb,0x08,0xf2,0x73,0xe6
};
static const unsigned char CBCGFSbox128_ciphertext[AES_BLOCK_SIZE] =
{
    0x03,0x36,0x76,0x3e,0x96,0x6d,0x92,0x59,
    0x5a,0x56,0x7c,0xc9,0xce,0x53,0x7f,0x5e
};

static const unsigned char CBCGFSbox192_plaintext[AES_BLOCK_SIZE] =
{
    0x1b,0x07,0x7a,0x6a,0xf4,0xb7,0xf9,0x82,
    0x29,0xde,0x78,0x6d,0x75,0x16,0xb6,0x39
};

static const unsigned char CBCGFSbox192_ciphertext[AES_BLOCK_SIZE] =
{
    0x27,0x5c,0xfc,0x04,0x13,0xd8,0xcc,0xb7,
    0x05,0x13,0xc3,0x85,0x9b,0x1d,0x0f,0x72
};

static const unsigned char CBCGFSbox256_plaintext[AES_BLOCK_SIZE] =
{
    0x01,0x47,0x30,0xf8,0x0a,0xc6,0x25,0xfe,
    0x84,0xf0,0x26,0xc6,0x0b,0xfd,0x54,0x7d
};

static const unsigned char CBCGFSbox256_ciphertext[AES_BLOCK_SIZE] =
{
    0x5c,0x9d,0x84,0x4e,0xd4,0x6f,0x98,0x85,
    0x08,0x5e,0x5d,0x6a,0x4f,0x94,0xc7,0xd7
};

static AES_KEY encks, decks;

void test_vector( const unsigned char *ptext_in,
                  const unsigned char *ctext_out,
                  int len )
{
    unsigned char cleartext[64];
    unsigned char iv[sizeof(test_iv)];
    unsigned char ciphertext[64];

    /* Test CBC Encryption */
#ifdef PERFTEST
    int i;
    struct timeval tv1, tv2, res;
    /* Get encryption start time */
    gettimeofday(&tv1, NULL);
    for( i = 0; i < NUM_ITERATIONS; i++ ) {
    /* Still need to do this memcpy() for performance test */
#endif
    memcpy(iv,test_iv,sizeof(test_iv));
    AES_cbc_encrypt(ptext_in, ciphertext, len, &encks, iv, 1 );
#ifdef PERFTEST
    }
    /* Get encryption finish time */
    gettimeofday(&tv2, NULL);
    timersub(&tv2,&tv1,&res);
    fprintf(stdout,"Encrypt: Time for %d calls: %ld.%ld s\n",
            NUM_ITERATIONS, res.tv_sec, res.tv_usec );
#endif
    if (memcmp(ciphertext, ctext_out, len))
        fprintf(stderr," *** encryption output mismatch ***\n"), exit(1);
    else
        fprintf(stdout," - encryption output match\n");

    /* Test CBC Decryption */
#ifdef PERFTEST
    /* Get decryption start time */
    gettimeofday(&tv1, NULL);
    for( i = 0; i < NUM_ITERATIONS; i++ ) {
    /* Still need to do this memcpy() for performance test */
#endif
    memcpy(iv,test_iv,sizeof(test_iv));
    AES_cbc_encrypt(ctext_out, cleartext, len, &decks, iv, 0);
#ifdef PERFTEST
    }
    /* Get decryption finish time */
    gettimeofday(&tv2, NULL);
    timersub(&tv2,&tv1,&res);
    fprintf(stdout,"Decrypt: Time for %d calls: %ld.%ld s\n",
            NUM_ITERATIONS, res.tv_sec, res.tv_usec );
#endif
    if (memcmp(cleartext, ptext_in, len))
        fprintf(stderr," *** decryption output mismatch ***\n"), exit(2);
    else
        fprintf(stderr," - decryption output match\n");
}

int main(int argc, char **argv)
{
    /* Test AES 128 bit CBC Encryption/decryption */
    printf("AES CBC test key length %d bits\n", 128);
    AES_set_encrypt_key(test_key,128,&encks);
    AES_set_decrypt_key(test_key,128,&decks);
    test_vector( CBCGFSbox128_plaintext,
                 CBCGFSbox128_ciphertext,
                 sizeof( CBCGFSbox128_plaintext ));

    /* Test AES 192 bit CBC Encryption/decryption */
    printf("AES CBC test key length %d bits\n", 192);
    AES_set_encrypt_key(test_key,192,&encks);
    AES_set_decrypt_key(test_key,192,&decks);
    test_vector( CBCGFSbox192_plaintext,
                 CBCGFSbox192_ciphertext,
                 sizeof( CBCGFSbox192_plaintext ));

    /* Test AES 256 bit CBC Encryption/decryption */
    printf("AES CBC test key length %d bits\n", 256);
    AES_set_encrypt_key(test_key,256,&encks);
    AES_set_decrypt_key(test_key,256,&decks);
    test_vector( CBCGFSbox256_plaintext,
                 CBCGFSbox256_ciphertext,
                 sizeof( CBCGFSbox256_plaintext ));

    exit(0);
}
