/*!
 * \file    rsa_compat_test.c
 * \brief   Provides a test for the honeywell RSA compatibility layer.
 */

#include <crypt.h>
#ifndef OPENSSL_NO_ERR
# include <err.h>
#endif
#include "rsa_compat.h"

/*
 * We are testing 1024 bit encryption so 128 bytes.
 */
#define TEST_KEY_AND_BLOCK_SIZE     128

/*
 * A typical data packet which will be encrypted.
 */
static unsigned char _compatTestData[] =
{
    0xbe, 0x01, 0x55, 0xaa, 0x74, 0x8f, 0xc4, 0xba,
    0x56, 0x93, 0x11, 0x09, 0x02, 0x45, 0xfe, 0xcd,
    0x13, 0x98, 0x00, 0x12, 0x55, 0xaa, 0x74, 0x8f,
    0xc4, 0xba, 0x56, 0x93, 0x11, 0x09, 0x02, 0x45,
    0xfe, 0xcd, 0x13, 0x98, 0xc6, 0x89, 0xef, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

#ifdef OPENSSL_NO_RSA
int main(int argc, char *argv[])
{
    printf("No RSA support\n");
    return(0);
}
#else
# define SetKey \
  key->n = BN_bin2bn(n, sizeof(n)-1, key->n); \
  key->e = BN_bin2bn(e, sizeof(e)-1, key->e); \
  key->d = BN_bin2bn(d, sizeof(d)-1, key->d); \
  key->p = BN_bin2bn(p, sizeof(p)-1, key->p); \
  key->q = BN_bin2bn(q, sizeof(q)-1, key->q); \
  key->dmp1 = BN_bin2bn(dmp1, sizeof(dmp1)-1, key->dmp1); \
  key->dmq1 = BN_bin2bn(dmq1, sizeof(dmq1)-1, key->dmq1); \
  key->iqmp = BN_bin2bn(iqmp, sizeof(iqmp)-1, key->iqmp);

static void key1(RSA *key)
    {
    static unsigned char n[] =
"\x00\xAC\xC5\x1B\xE1\x13\xF8\xF1\xCF\xA2\x17\x40\x62\xBC\x89\xBB"
"\x81\x68\x26\x15\x82\x35\x60\x33\x44\x76\xC8\x36\xC2\x79\x6D\x1D"
"\x29\xE2\x9C\xDF\xEB\x38\xE4\xC5\xC5\x6B\xBD\x7C\x53\xDB\xBE\xBA"
"\x1E\x9C\xD3\x20\xF0\x62\x12\xEC\x52\xB1\xEC\xFC\xC2\x4A\x57\xDE"
"\x2B\x05\x2B\x79\xE5\xF0\xAE\xB6\x67\x12\x45\xB0\x11\xA6\xA3\x37"
"\xBA\x61\x9C\x89\xD3\x4B\xB6\x30\xA0\xA7\x53\xF8\xBF\x1F\xF5\x5D"
"\x29\xF9\x01\x3C\xCC\xB5\x72\x84\xB8\x3C\x04\x8E\xD8\xA8\x53\x30"
"\xC6\x96\xE5\x3E\xC5\xBD\x75\x46\x03\x77\x82\x56\x69\x8A\x2B\xE9"
"\x91";

    static unsigned char e[] = "\x03";

    static unsigned char d[] =
"\x73\x2E\x12\x96\x0D\x50\xA1\x35\x16\xBA\x2A\xEC\x7D\xB1\x27\xAB"
"\x9A\xC4\x0E\x56\xCE\x40\x22\x2D\xA4\x85\x79\xD6\xFB\x9E\x13\x71"
"\x41\xBD\xEA\x9C\xD0\x98\x83\xD8\xF2\x7E\x52\xE2\x92\x7F\x26\xBF"
"\x13\x37\x6B\x4A\xEC\x0C\x9D\x8C\x76\x9D\xFD\xD6\xDC\x3A\x94\x1B"
"\x94\xAE\xC9\xE7\x08\xF6\x7E\x35\xAF\x5F\xCF\x30\xCC\x43\xB4\xA9"
"\x40\x6E\x85\x76\xF0\xB9\xFE\x77\x10\xA8\x96\x44\x5A\x2F\xC0\x0D"
"\x27\xDC\x7B\x48\x58\xB8\x11\x05\x90\xC3\xE2\x0A\xA5\x69\x6D\xDE"
"\xC2\x20\x50\xCF\x84\x72\x86\x4B\xCB\xFB\x19\x75\x1E\xCC\xA0\x4B";

    static unsigned char p[] =
"\x00\xE5\x08\xEC\x5E\x4D\xC8\xA1\x50\x6D\x79\xFB\x52\x43\x39\xC0"
"\x2E\x63\x32\xD6\x33\x1D\xDC\x89\x6F\xB8\x54\x09\xAB\xB5\x2A\xCD"
"\x69\x2D\xB4\xEC\xBD\x02\xAB\xD0\x2D\xD6\x37\x0F\x20\x34\xDC\xA1"
"\x2D\xE0\xAD\x97\xB2\x6D\x9C\xAE\x82\xEF\xD9\x22\x39\x80\x20\x30"
"\x9D";

    static unsigned char q[] =
"\x00\xC1\x1C\x5E\xAD\x15\x74\x57\xC6\x1D\xBB\xFD\xF6\x31\x03\xE8"
"\x8E\x1D\xC3\xEB\x6D\xC4\xC2\xA9\x7E\x56\x03\x0D\xAC\xE3\x82\xEF"
"\xAD\x0F\x81\x97\x23\x2D\xB2\x9B\x02\x0C\xA7\xAC\xA8\x7B\x58\x6A"
"\xCA\x93\x07\x2D\xDC\x09\x2C\xCE\x0E\xD5\xB0\x8E\x00\x5B\xD8\xC8"
"\x85";

    static unsigned char dmp1[] =
"\x00\x98\xB0\x9D\x94\x33\xDB\x16\x35\x9E\x51\x52\x36\xD7\x7B\xD5"
"\x74\x42\x21\xE4\x22\x13\xE8\x5B\x9F\xD0\x38\x06\x72\x78\xC7\x33"
"\x9B\x73\xCD\xF3\x28\xAC\x72\x8A\xC9\x39\x7A\x0A\x15\x78\x93\x16"
"\x1E\x95\xC9\x0F\xCC\x49\x13\x1F\x01\xF5\x3B\x6C\x26\x55\x6A\xCB"
"\x13";

    static unsigned char dmq1[] =
"\x00\x80\xBD\x94\x73\x63\xA2\xE5\x2E\xBE\x7D\x53\xF9\x76\x02\x9B"
"\x09\x69\x2D\x47\x9E\x83\x2C\x70\xFE\xE4\x02\x09\x1D\xED\x01\xF5"
"\x1E\x0A\x56\x64\xC2\x1E\x77\x12\x01\x5D\xC5\x1D\xC5\xA7\x90\x47"
"\x31\xB7\x5A\x1E\x92\xB0\xC8\x89\x5F\x39\x20\x5E\xAA\xE7\xE5\xDB"
"\x03";

    static unsigned char iqmp[] =
"\x00\xAB\x1A\x76\x0A\xA3\xD2\xE2\x83\x99\x6D\x55\xF0\x9B\x91\x87"
"\xA1\x5C\x41\x50\x7B\x34\xF6\xE3\x69\x9D\xEA\xFC\x88\xFC\x9F\xE7"
"\xFF\x8F\x4B\x79\x58\x93\xB7\xE5\xCD\x3F\x69\x36\x16\x0B\x5C\x0C"
"\xA4\xD6\xF6\x4A\xEF\xED\xDA\x3A\x11\x6D\xD8\xC0\xF8\x1E\x11\xC5"
"\x0F";

    SetKey;
    }


/**
 * Perform the RSA compatibility test.
 *  (1) Create a private/public key pair.
 *  (2) Extract the public key in buffer form.
 *  (3) Use the extracted key to encrypt a typical block of data.
 *  (4) Use the original private key to decrypt the data.
 *  (5) Verify the origial data was recovered.
 */
static int RSA_compat_test( RSA *rsa )
{
    int err = 1;

    unsigned char key[TEST_KEY_AND_BLOCK_SIZE];

    if (RSA_compat_extract_public_key(rsa, key, sizeof(key)) < 0)
    {
        printf("RSA_compat_test: Couldn't extract public key\n");
    }
    else
    {
        unsigned char cipher[TEST_KEY_AND_BLOCK_SIZE];

        if (RSA_compat_public_encrypt(sizeof(cipher),
                                      _compatTestData,
                                      cipher,
                                      key,
                                      sizeof( key )) < 0)
        {
            printf("RSA_compat_test: Couldn't encrypt\n");
        }
        else
        {
            unsigned char decipher[TEST_KEY_AND_BLOCK_SIZE];

            if (RSA_compat_private_decrypt(sizeof(cipher),
                                               cipher,
                                               decipher,
                                               rsa ) < 0)
            {
                printf("RSA_compat_test: Couldn't decrypt\n");
            }
            else
            {
                if (memcmp(_compatTestData,
                           decipher,
                           sizeof(_compatTestData)) != 0)
                {
                    printf("RSA_compat_test: Decrypt mismatch\n");
                }
                else
                {
                    printf("RSA_compat_test: Pass\n");
                    err = 0;
                }
            }
        }
    }

    return err;
}

int main(int argc, char *argv[])
{
    int err=0;
    RSA *key;

    CRYPTO_malloc_debug_init();
    CRYPTO_dbg_set_options(V_CRYPTO_MDEBUG_ALL);
    CRYPTO_mem_ctrl(CRYPTO_MEM_CHECK_ON);

    key = RSA_new();

    key1(key);

    RSA_blinding_off(key);

    err = RSA_compat_test( key );

    RSA_free(key);

    CRYPTO_cleanup_all_ex_data();
# ifndef OPENSSL_NO_ERR
    ERR_remove_thread_state(NULL);
# endif

# ifndef OPENSSL_NO_FP_API
    CRYPTO_mem_leaks_fp(stderr);
# endif

# ifdef OPENSSL_SYS_NETWARE
    if (err) printf("ERROR: %d\n", err);
# endif
    return err;
}
#endif
