//==========================================================================
//
//      if_lm3s9xxx.c
//
//	    Ethernet driver for Cortex-m3 LM3S9XXX coldfires
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2011 Free Software Foundation, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    stuartw
// Contributors:
// Date:         2011-08-11
// Description:  hardware driver for LM3S9XXX ethernet devices.
//
//####DESCRIPTIONEND####
//==========================================================================

#include <pkgconf/system.h>
#ifdef CYGPKG_IO_ETH_DRIVERS
#include <pkgconf/io_eth_drivers.h>
#endif
#include <pkgconf/devs_eth_lm3s9xxx.h>

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/hal/hal_arch.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/hal/rom_api.h>
#include <cyg/infra/diag.h>
#include <cyg/hal/hal_if.h>
#include <cyg/hal/drv_api.h>
#include <cyg/io/eth/netdev.h>
#include <cyg/io/eth/eth_drv.h>

#include <string.h>

#ifdef CYGPKG_NET
#include <pkgconf/net.h>
#include <net/if.h>  /* Needed for struct ifnet */
#endif


#include <cyg/hal/var_io.h>
#include <cyg/hal/rom_api.h>

#ifdef ETH_LM3S9XXX_DEBUG
# define ETH_LM3S9XXX_DIAG( x ) diag_printf x
#else
# define ETH_LM3S9XXX_DIAG( x )
#endif /* ETH_LM3S9XXX_DEBUG */

/*
 * Maximum buffer size used in the driver. This defines the maximum packet we
 * can send or receive.
 */
#define ETH_LM3S9XXX_BUFFER_SIZE    1600

/*
 * Control structure for the ethernet driver.
 */
typedef struct lm3s9xxx_eth
{
    /*
     * The base address passed to the ROM functions.
     */
    unsigned long base_address;
    
    /*
     * Handle for the ISD
     */
    cyg_handle_t    interrupt_handle;

    /*
     * Interrupt data required by eCos.
     */
    cyg_interrupt   interrupt_data;

    /*
     * Buffer used when transmitting a packet if the packet is supplied in
     * a list of parts.
     */
    unsigned char   tx_data[ ETH_LM3S9XXX_BUFFER_SIZE ];

    /*
     * Buffer which we copy data into when receiving a packet.
     */
    unsigned char   rx_data[ ETH_LM3S9XXX_BUFFER_SIZE ];

    /*
     * Amount of data in the rx_data buffer.
     */
    cyg_uint16      rx_data_len;

    /*
     * True if we have an outstanding transmission completion which has
     * still to be reported to the upper layers.
     */
    cyg_bool    tx_complete;

    /*
     * True if we have a received packet ready to be handed to the upper
     * layers.
     */
    cyg_bool    rx_packet;

    /*
     * When we are passed a packet to trasmit we are also passed a key.
     * We store it here so that we can pass the key back when we report the
     * transmission as complete.
     */
    unsigned long   tx_key;

    /*
     * True if a transmission is in progress.
     */
    cyg_bool        tx_in_progress;

    /*
     * True if the ethernet functionality has been started.
     */
    cyg_uint8       started;

    /*
     * True if we can accept a new packet for transmission.
     */
    cyg_uint8       tx_can_send;

    /*
     * The mac address fort he interface.
     */
    cyg_uint8       mac[6];

    /*
     * Set to true when the interrupt is deliberately disabled. This stops the
     * DSR re-enabling the interrupt.
     */
    cyg_bool        int_disabled;

} lm3s9xxx_eth;

/*
 * Structure used to define the pins needed to be configured for the Ethernet
 * Interface. We only need to do this for the LEDs.
 */
typedef struct
{
    cyg_uint32  pin;
    cyg_uint32  function;
} EthPinInfo;

/*
 * Define used to convert a textual pin, for example A1, to the values needed
 * for the array of EPIPinInfo structures.
 */
#define SET_EPI_PIN_INFO( pin ) CYGHWR_HAL_LM3S_ETH_P ## pin,         \
                                CYGHWR_HAL_LM3S_ETH_FUNC_P ## pin

/*
 * Control structure for the ethernet functinality.
 */
static lm3s9xxx_eth  lm3s9xxx_eth_info;

/*
 * These set up the driver structure and link it into eCos.
 */
ETH_DRV_SC( lm3s9xxx_eth_sc0,
            (void*)&lm3s9xxx_eth_info,
            CYGDAT_DEVS_ETH_LM3S9XXX_NAME,
            lm3s9xxx_eth_start,
            lm3s9xxx_eth_stop,
            lm3s9xxx_eth_ioctl,
            lm3s9xxx_eth_can_send,
            lm3s9xxx_eth_send,
            lm3s9xxx_eth_recv,
            lm3s9xxx_eth_deliver,
            lm3s9xxx_eth_poll,
            lm3s9xxx_eth_int_vector
    );

NETDEVTAB_ENTRY( lm3s9xxx_eth_netdev0,
                 "lm3s9xxx-" CYGDAT_DEVS_ETH_LM3S9XXX_NAME,
                 lm3s9xxx_eth_init,
                 &lm3s9xxx_eth_sc0 );

/*
 * Structure used to define the pins needed to be configured for the Ethernet
 * Interface. We only need to do this for the LEDs.
 */
const EthPinInfo eth0_pin_info[] =
{
    { SET_EPI_PIN_INFO( F3 ) },
    { 0, 0 }
};

/*
 * Forward declarations for the DSR and ISR.
 */
static cyg_uint32 lm3s9xxx_eth_isr( cyg_vector_t, cyg_addrword_t );
static void lm3s9xxx_eth_dsr( cyg_vector_t, cyg_ucount32, cyg_addrword_t );

/*
 * Enable specific interrupts, typicall TX and RX.
 */
static void lm3s9xxx_eth_interrupt_enable( struct lm3s9xxx_eth* eth,    
                                           cyg_uint32           enable_bits )
{
    ROM_EthernetIntEnable( eth->base_address, enable_bits );
}

/*
 * Disable specific interrupts, typicall TX and RX.
 */
static void lm3s9xxx_eth_interrupt_disable( struct lm3s9xxx_eth* eth,    
                                            cyg_uint32           enable_bits )
{
    ROM_EthernetIntDisable( eth->base_address, enable_bits );
}

/*
 * Called to disable the interrupt. Also sets the int_disabled flag to make
 * sure that the DSR doesn't re-enable the interrupt. Note that this should
 * not be used to disable within the ISR.
 */
static void lm3s9xxx_eth_disable_eth_interrupt( struct lm3s9xxx_eth* eth )
{
    eth->int_disabled = true;
    cyg_drv_interrupt_mask( CYGNUM_HAL_INTERRUPT_ETH0 );
}

/*
 * Called to disable the interrupt. Also clears the int_disabled flag to make
 * sure that the DSR re-enables the interrupt. Note that this should
 * not be used to re-enable within the DSR.
 */
static void lm3s9xxx_eth_enable_eth_interrupt( struct lm3s9xxx_eth* eth )
{
    eth->int_disabled = false;
    cyg_drv_interrupt_unmask( CYGNUM_HAL_INTERRUPT_ETH0 );
}

/*
 * Read a received packet from the hardware.
 */
static cyg_uint16 lm3s9xxx_eth_read_packet( struct lm3s9xxx_eth* eth )
{
    long data_len = ROM_EthernetPacketGetNonBlocking( eth->base_address,
                                                      eth->rx_data,
                                                      sizeof( eth->rx_data ));

    if( data_len < 0 )
    {
        /*
         * Treat errors as empty packets.
         */
        data_len = 0;
    }

    if( data_len != 0 )
    {
        eth->rx_data_len = (cyg_uint16)data_len;
        eth->rx_packet = true;
    }

    return( (cyg_uint16)data_len );
}

/*
 * Pass a packet for transmission to the hardware. We don't use the ROM
 * functions for this as we need to have afiner control of interrupts.
 */
static void lm3s9xxx_eth_send_packet( struct lm3s9xxx_eth*  eth,
                                      cyg_uint8*            data,
                                      cyg_uint32            data_len,
                                      unsigned long         key )
{
    cyg_uint32 data_reg;
    cyg_uint32 count = 0;

    /*
     * We need to wait until this packet has been sent before accepting
     * another for transmission.
     */
    eth->tx_can_send = false;

    /*
     * Store the key for later.
     */
    eth->tx_key      = key;
    eth->tx_complete = false;

    /*
     * Build and write WORD 0 (see format above) to the transmit FIFO.
     */
    data_reg = ( data_len - 14 );
    data_reg |= ( data[count++] << 16 );
    data_reg |= ( data[count++] << 24 );
    HAL_WRITE_UINT32(( eth->base_address + CYGHWR_HAL_LM3S_ETH_MACDATA ),
                       data_reg );

    /*
     * Write each subsequent WORD n to the transmit FIFO, except for the last
     * WORD (if the word does not contain 4 bytes).
     */
    while( count <= ( data_len - 4 ))
    {
        HAL_WRITE_UINT32(( eth->base_address + CYGHWR_HAL_LM3S_ETH_MACDATA ),
                           *(unsigned int *)&data[ count ] );
        count += 4;
    }

    /*
     * Build the last word of the remaining 1, 2, or 3 bytes, and store
     * the WORD into the transmit FIFO.
     */
    if( count != data_len )
    {
        cyg_uint32 bytes_left = ( data_len  - count );

        data_reg  = ( data[ count++ ] <<  0 );

        if( bytes_left != 1 )
        {
            data_reg |= ( data[ count++ ] <<  8 );

            if( bytes_left != 2 )
            {
                data_reg |= ( data[ count++ ] << 16 );
            }
        }

        HAL_WRITE_UINT32(( eth->base_address + CYGHWR_HAL_LM3S_ETH_MACDATA ),
                           data_reg );
    }

    /*
     * Disable interrupts to avoid any race conditions betweem activating the
     * transmitter and setting the tx_in_progress flag to true.
     */
    lm3s9xxx_eth_disable_eth_interrupt( eth );

    /*
     * Activate the transmitter
     */
    HAL_WRITE_UINT32(( eth->base_address + CYGHWR_HAL_LM3S_ETH_MACTR ),
                       CYGHWR_HAL_LM3S_ETH_MACTR_NEWTX );

    eth->tx_in_progress = true;

    lm3s9xxx_eth_enable_eth_interrupt( eth );
}

/*
 * Process an interrupt. This is called within the DSR and checks the
 * hardware status.
 */
static void lm3s9xxx_eth_process_interrupt( struct lm3s9xxx_eth* eth )
{
    cyg_uint32 int_status;

    /*
     * Read and clear the interrupt bits. Only the bits not masked off to be ignored
     */
    int_status = ROM_EthernetIntStatus( eth->base_address, true );
    ROM_EthernetIntClear( eth->base_address , int_status );

    if( int_status & CYGHWR_HAL_LM3S_ETH_INT_RX )
    {
        /*
         * Disable RX interrupts until we have processed the packet.
         */
        lm3s9xxx_eth_interrupt_disable( eth, CYGHWR_HAL_LM3S_ETH_INT_RX );

        /*
         * Read the packet.
         */
        if( lm3s9xxx_eth_read_packet( eth ) == 0 )
        {
            /*
             * Re-enable RX interrupts as we had no real packet.
             */
            lm3s9xxx_eth_interrupt_enable( eth, CYGHWR_HAL_LM3S_ETH_INT_RX );
        }
    }

    if( int_status & CYGHWR_HAL_LM3S_ETH_INT_TX )
    {
        /*
         * The TX FIFO is empty. Check to see if a transmsission
         * is outstanding.
         */
        if( eth->tx_in_progress )
        {
            /*
             * A transmission has completed.
             */
            eth->tx_complete = true;
        }
    }

}

/*
 * Function called by eCos on startup to intialise the ethernet functionality.
 */
static bool
lm3s9xxx_eth_init(struct cyg_netdevtab_entry* ndp)
{
    cyg_uint32 int_status;
    cyg_uint32 reg;

    int index;

    struct eth_drv_sc*   sc  = (struct eth_drv_sc*) ndp->device_instance;
    struct lm3s9xxx_eth* eth = (struct lm3s9xxx_eth*) sc->driver_private;

    eth->base_address = CYGHWR_HAL_LM3S_ETH0;

    eth->started        = 0;

    cyg_drv_interrupt_create( CYGNUM_HAL_INTERRUPT_ETH0,
                              CYGNUM_DEVS_ETH_LM3S9XXX_ISR_PRIORITY,
                              (CYG_ADDRWORD) sc,
                              &lm3s9xxx_eth_isr,
                              &lm3s9xxx_eth_dsr,
                              &(eth->interrupt_handle),
                              &(eth->interrupt_data) );
    cyg_drv_interrupt_attach(eth->interrupt_handle);
        
    /*
     * First enable the EPI module.
     * We need to do this explicitly because the existing eCos code can't
     * cope with register bits > 28!
     */
    HAL_READ_UINT32(( CYGHWR_HAL_LM3S_SC + CYGHWR_HAL_LM3S_SC_RCGC2 ), reg );
    reg |= ( CYGHWR_HAL_LM3S_SC_RCGC2_EMAC0 | CYGHWR_HAL_LM3S_SC_RCGC2_EPHY0 );
    HAL_WRITE_UINT32(( CYGHWR_HAL_LM3S_SC + CYGHWR_HAL_LM3S_SC_RCGC2 ), reg );

    /*
     * Pins F2 and F3 are used to control Ethernet LEDs.
     */
    CYGHWR_HAL_LM3S_PERIPH_SET( CYGHWR_HAL_LM3S_P_GPIOF, 1 );

    /*
     * Configure the LEDs for their desired use.
     */
    HAL_WRITE_UINT32(( eth->base_address + CYGHWR_HAL_LM3S_ETH_MACLED ),
                     ( CYGHWR_HAL_LM3S_ETH_MACLED_LED0_ACTIVITY ));

    /*
     * Configure the pins and functions.
     */
    index = 0;
    while( eth0_pin_info[index].pin )
    {
        CYGHWR_HAL_LM3S_GPIO_SET( eth0_pin_info[index].pin );
        CYGHWR_HAL_LM3S_GPIO_PCTL_SET( eth0_pin_info[index].function );
        index++;
    }

    /*
     * Initialize the Ethernet Controller and disable all Ethernet Controller
     * interrupt sources.
     */
    lm3s9xxx_eth_interrupt_disable( eth,
                                    ( CYGHWR_HAL_LM3S_ETH_INT_PHY   |
                                      CYGHWR_HAL_LM3S_ETH_INT_MDIO  |
                                      CYGHWR_HAL_LM3S_ETH_INT_RXER  |
                                      CYGHWR_HAL_LM3S_ETH_INT_RXOF  |
                                      CYGHWR_HAL_LM3S_ETH_INT_TX    |
                                      CYGHWR_HAL_LM3S_ETH_INT_TXER  |
                                      CYGHWR_HAL_LM3S_ETH_INT_RX ));

    int_status = ROM_EthernetIntStatus( eth->base_address, false );
    ROM_EthernetIntClear( eth->base_address, int_status );

    /*
     * Initialize the upper level driver
     */
    (sc->funs->eth_drv->init)( sc, eth->mac );

    return 1;
}

/*
 * Called by eCos to determine if the driver can send a pcaket. The driver
 * only supports one transmit at a time.
 */
static int
lm3s9xxx_eth_can_send(struct eth_drv_sc* sc)
{
    struct  lm3s9xxx_eth*    eth = (struct lm3s9xxx_eth*) sc->driver_private;

    return eth->tx_can_send;
}

/*
 * Called by eCos to transmit a packet.
 */
static void
lm3s9xxx_eth_send( struct eth_drv_sc*   sc,
                   struct eth_drv_sg*   sg_list,
                   int                  sg_len,
                   int                  total_len,
                   unsigned long        key )
{
    struct lm3s9xxx_eth* eth = (struct lm3s9xxx_eth*) sc->driver_private;
    ETH_LM3S9XXX_DIAG(( "lm3s9xxx_eth_send\n" ));

    if( eth->tx_can_send )
    {
        if( total_len > sizeof( eth->tx_data ))
        {
            ETH_LM3S9XXX_DIAG(( "Can't send %u %d\n",
                                eth->tx_can_send, total_len ));

            /*
             * Can't send this packet, just mark it as complete.
             */
            eth->tx_key      = key;
            eth->tx_complete = true;

            /*
             * We can't send another, however, until we've passed the key back.
             */
            eth->tx_can_send = false;
        }
        else if( sg_len == 1 )
        {
            /*
             * Only one element in the list, just send it.
             */
            lm3s9xxx_eth_send_packet( eth,
                                      (cyg_uint8*)sg_list->buf,
                                      sg_list->len,
                                      key );
        }
        else
        {
            /*
             * Multiple elements in the list, we need to combine them.
             */
            cyg_uint32 scatter_count;
            cyg_uint32 data_len = 0;

            for( scatter_count = 0; scatter_count < sg_len; scatter_count++ )
            {
                struct eth_drv_sg* sg_current = &sg_list[ scatter_count ];
            
                if(( data_len + sg_current->len ) > sizeof( eth->tx_data ))
                {
                    ETH_LM3S9XXX_DIAG(( "data too long!\n" ));
                    data_len = 0;
                    break;
                }

                bcopy( (void*)sg_current->buf,
                       (void*)&eth->tx_data[ data_len ],
                       sg_current->len );
                data_len += sg_current->len;
            }

            lm3s9xxx_eth_send_packet( eth, eth->tx_data, data_len, key );
        }
    }
}

/*
 * Called from eCos after we have reported that we have a received packet
 * to pass up.
 */
static void
lm3s9xxx_eth_recv( struct eth_drv_sc* sc,
                   struct eth_drv_sg* sg_list,
                   int                sg_len )
{
    lm3s9xxx_eth* eth = (lm3s9xxx_eth*) sc->driver_private;

    unsigned int data_len = eth->rx_data_len;
    unsigned int copied_count = 0;

    unsigned int scatter_count = 0;

    ETH_LM3S9XXX_DIAG(( "lm3s9xxx_eth_recv %u\n", data_len ));

    for( scatter_count = 0; scatter_count < sg_len; scatter_count++ )
    {
        struct eth_drv_sg* sg_current = &sg_list[ scatter_count ];
        unsigned int copy_len = data_len;

        if( sg_current->buf == 0 )
        {
            ETH_LM3S9XXX_DIAG(( "NULL buffer on receive\n" ));
            break;
        }

        if( copy_len > sg_current->len )
        {
            copy_len = sg_current->len;
        }
        else
        {
            sg_current->len = copy_len;
        }

        memcpy( (void*)sg_current->buf,
                (void*)&eth->rx_data[ copied_count ],
                copy_len );

        copied_count += copy_len;
        data_len -= copy_len;
    }

    /*
     * Now we can accept further incoming packets. Re-enable the RX interrupt.
     */
    lm3s9xxx_eth_interrupt_enable( eth, CYGHWR_HAL_LM3S_ETH_INT_RX );
}

/*
 * Interrupt handler for the driver. It simply disables the interrupt and
 * schedules the DSR to do the real work.
 */
static cyg_uint32
lm3s9xxx_eth_isr( cyg_vector_t vector, cyg_addrword_t data )
{
    cyg_drv_interrupt_acknowledge( vector );

    /*
     * Simply disable the interrupt and let the DSR do the work.
     */
    cyg_drv_interrupt_mask( CYGNUM_HAL_INTERRUPT_ETH0 );

    return CYG_ISR_CALL_DSR;
}

/*
 * DSR for the interrupt. It will store any received packets and determine
 * if any outstanding transmissions have completed.
 */
static void
lm3s9xxx_eth_dsr( cyg_vector_t   vector,
                  cyg_ucount32   count,
                  cyg_addrword_t data )
{
    static lm3s9xxx_eth *eth = &lm3s9xxx_eth_info;

    lm3s9xxx_eth_process_interrupt( eth );

    eth_drv_dsr( vector, count, data );

    /*
     * Re-enable the ethernet interrupt interrupt - unless it is supposed to
     * remain disabled.
     */
    if( !eth->int_disabled )
    {
        cyg_drv_interrupt_unmask( CYGNUM_HAL_INTERRUPT_ETH0 );
    }
}

/*
 * deliver() is usually called from thread context, after the DSR has woken
 * up the packet handling thread. It needs to report completed transmits so
 * that higher-level code can release the mbuf, and report all received
 * frames.
 */
static void
lm3s9xxx_eth_deliver(struct eth_drv_sc* sc)
{
    lm3s9xxx_eth* eth = (lm3s9xxx_eth*) sc->driver_private;

    ETH_LM3S9XXX_DIAG(( "lm3s9xxx_eth_deliver rx %d tx %d\n",
                        eth->rx_packet,
                        eth->tx_complete ));

    if( eth->tx_in_progress && eth->tx_complete )
    {
        eth->tx_in_progress = false;
        eth->tx_complete = false;
        (*sc->funs->eth_drv->tx_done)( sc, eth->tx_key, 1 );
        eth->tx_can_send = true;
    }

    if( eth->rx_packet )
    {
        eth->rx_packet = false;
        (*sc->funs->eth_drv->recv)( sc, eth->rx_data_len );
    }
}

/*
 * Called by eCos to poll for data. Just check for pending interrupts
 * and act as if deliver had been called.
 */
static void
lm3s9xxx_eth_poll(struct eth_drv_sc* sc)
{
    struct lm3s9xxx_eth* eth = (struct lm3s9xxx_eth*) sc->driver_private;

    ETH_LM3S9XXX_DIAG(( "lm3s9xxx_eth_poll\n" ));

    lm3s9xxx_eth_process_interrupt( eth );

    lm3s9xxx_eth_deliver(sc);
}

/*
 * int_vector() is used by RedBoot/stubs to ensure ctrl-C will get through
 * if ethernet is used for the debug channel.
 */
static int
lm3s9xxx_eth_int_vector(struct eth_drv_sc* sc)
{
    return CYGNUM_HAL_INTERRUPT_ETH0;
}

/*
 * Called by eCos to process an IOCTL. Currently we only support
 * SET_MAC_ADDRESS.
 */
static int
lm3s9xxx_eth_ioctl( struct eth_drv_sc*  sc,
                    unsigned long       key,
                    void*               data,
                    int                 data_length )
{
    int ret = 0;

    lm3s9xxx_eth* eth = (lm3s9xxx_eth*) sc->driver_private;

    ETH_LM3S9XXX_DIAG(( "lm3s9xxx_eth_ioctl: key = %u\n", key ));

    switch(key)
    {
        case ETH_DRV_SET_MAC_ADDRESS:
        {
            ETH_LM3S9XXX_DIAG(( "ETH_DRV_SET_MAC_ADDRESS\n" ));

            /*
             * Copy the MAC Address.
             */
            memcpy( eth->mac, data, sizeof( eth->mac ));

            /*
             * Program the MAC Address.
             */
            ROM_EthernetMACAddrSet( eth->base_address, eth->mac );

            break;
        }

        case ETH_DRV_GET_MEDIA_STATUS:
        {
            /*
             * Read the MII MR1 register to determin if the link is up or not.
             */
            int* media_status = (int*)data;

            cyg_uint32 mri1 = ROM_EthernetPHYRead(
                                                eth->base_address,
                                                CYGHWR_HAL_LM3S_ETH_MII_MR1 );

            *media_status = (( mri1 & CYGHWR_HAL_LM3S_ETH_MII_MR1_LINK ) != 0 );

            break;
        }

        default:
        {
            ret = 1;

            break;
        }
    }

    return ret;
}

/*
 * Called by eCos to start the Ethernet interface operating.
 */
static void
lm3s9xxx_eth_start(struct eth_drv_sc* sc, unsigned char* enaddr, int flags)
{
    lm3s9xxx_eth*    eth = (lm3s9xxx_eth*) sc->driver_private;

    ETH_LM3S9XXX_DIAG(( "lm3s9xxx_eth_start %d\n", eth->started ));

    if( eth->started )
    {
        ETH_LM3S9XXX_DIAG(( "already started\n" ));
        return;
    }

    eth->rx_data_len = 0;
    eth->tx_in_progress = false;
    eth->rx_packet = false;
    eth->int_disabled = false;

    ROM_EthernetInitExpClk( eth->base_address, ROM_SysCtlClockGet() );

    /*
     * Configure the Ethernet Controller for:
     *      Full Duplex
     *      TX CRC Auto Generation
     *      TX Padding Enabled
     */
    ROM_EthernetConfigSet( eth->base_address,
                           ( CYGHWR_HAL_ETH_CFG_TX_DPLXEN |
                             CYGHWR_HAL_ETH_CFG_TX_CRCEN  |
                             CYGHWR_HAL_ETH_CFG_TX_PADEN ));

    /*
     * Program the MAC Address.
     */
    ROM_EthernetMACAddrSet( eth->base_address, eth->mac );

    /*
     * Enable the Ethernet Controller.
     */
    ROM_EthernetEnable( eth->base_address );

    cyg_drv_interrupt_unmask( CYGNUM_HAL_INTERRUPT_ETH0 );

    lm3s9xxx_eth_interrupt_enable( eth, ( CYGHWR_HAL_LM3S_ETH_INT_RX |
                                          CYGHWR_HAL_LM3S_ETH_INT_TX ));

    /*
     * Enable all processor interrupts.
     */
    ROM_IntMasterEnable();

    eth->tx_can_send    = 1;

    eth->started = 1;
}

/*
 * Called by eCos to stop the Ethernet interface operating.
 */
static void
lm3s9xxx_eth_stop(struct eth_drv_sc* sc)
{
    ETH_LM3S9XXX_DIAG(( "lm3s9xxx_eth_stop\n" ));

    lm3s9xxx_eth*    eth = (lm3s9xxx_eth*) sc->driver_private;
    
    eth->started    = 0;

    if(! eth->tx_can_send )
    {
        (*sc->funs->eth_drv->tx_done)( sc, eth->tx_key, 1 );
        eth->tx_can_send    = 1;
    }

    cyg_drv_interrupt_mask( CYGNUM_HAL_INTERRUPT_ETH0 );

    /*
     * Make sure that the hardware is disabled.
     */
    ROM_EthernetDisable( eth->base_address );
}

