//==========================================================================
//
//        pmutex5.c
//
//        Mutex test 5 - mutex timeouts
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2005, 2006 Free Software Foundation, Inc.                  
// Copyright (C) 2005, 2006 eCosCentric Limited                             
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     nickg
// Contributors:  nickg
// Date:          2005-12-19
// Description:   Tests mutex timeout functionality.
//
//####DESCRIPTIONEND####
//==========================================================================

#include <cyg/infra/testcase.h>
#include <pkgconf/posix.h>
#include <pkgconf/system.h>
//#ifdef CYGPKG_KERNEL
//#include <pkgconf/kernel.h>
//#endif

#include <cyg/infra/diag.h>

#include <sys/types.h>
#include <pthread.h>
#include <time.h>
//# include <unistd.h>

#include <errno.h>
#include <signal.h>

// ------------------------------------------------------------------------

#define NTHREADS 1

#define STACKSIZE CYGNUM_HAL_STACK_SIZE_TYPICAL

static pthread_t thread[NTHREADS] = { 0 };

static char stack[NTHREADS][STACKSIZE];

// ------------------------------------------------------------------------
//

#if defined(CYGVAR_KERNEL_COUNTERS_CLOCK) &&                                    \
    (CYGNUM_KERNEL_SCHED_PRIORITIES > 20) &&                                    \
    defined(CYGFUN_KERNEL_API_C) &&                                             \
    !defined(CYGPKG_KERNEL_SMP_SUPPORT)


// ------------------------------------------------------------------------

pthread_mutex_t mx1;

volatile int expect;

volatile int fails = 0;

// ------------------------------------------------------------------------

static void delay_ms( int ms )
{
    struct timespec timeout;

    timeout.tv_sec = ms/1000;
    timeout.tv_nsec = (ms%1000)*1000000;

    nanosleep( &timeout, NULL );
}

static long long int ms_time( void )
{
    struct timespec t;

    clock_gettime( CLOCK_REALTIME, &t );

    return t.tv_sec*1000 + t.tv_nsec/1000000;
}

//--------------------------------------------------------------------------
// Signal handler functions

volatile int sigusr1_called;

static void sigusr1( int signo )
{
    CYG_TEST_INFO( "sigusr1() handler called" );
    CYG_TEST_CHECK( signo == SIGUSR1, "Signal not SIGUSR1");
    CYG_TEST_CHECK( pthread_equal(pthread_self(), thread[0]), "Not called in thread1");

    sigusr1_called++;
}

// ------------------------------------------------------------------------
// This thread just loops on a mutex timeout. We then prod it in
// various ways from the control thread to check that it behaves
// itself properly.

static void *thread1( void *arg )
{
    struct timespec timeout;
    sigset_t mask;
    struct sigaction sa;

    // Mask all signals except USR1.
    sigfillset( &mask );
    sigdelset( &mask, SIGUSR1 );
    pthread_sigmask( SIG_SETMASK, &mask, NULL );

    sa.sa_handler = sigusr1;
    sa.sa_mask = mask;
    sa.sa_flags = 0;

    sigaction( SIGUSR1, &sa, NULL );

    
    clock_gettime( CLOCK_REALTIME, &timeout );
    timeout.tv_sec += 2;
                
    for(;;)
    {
        int result;

        result = pthread_mutex_timedlock( &mx1, &timeout );

        if( result != expect )
        {
            diag_printf("<INFO>: %6lld: pthread_mutex_timedlock returned unexpected error %d, expected %d\n", ms_time(), result, expect);            
            CYG_TEST_FAIL("Result not as expected");
            fails++;
           
        }
        
        if( result == 0 )
        {
            diag_printf("<INFO>: %6lld: pthread_mutex_timedlock returned zero\n", ms_time());

            pthread_mutex_unlock( &mx1 );

            delay_ms(1200);
        }
        else
        {
            if( result == ETIMEDOUT )
            {
                diag_printf("<INFO>: %6lld: pthread_mutex_timedlock timeout\n", ms_time());
                clock_gettime( CLOCK_REALTIME, &timeout );
                timeout.tv_sec += 2;
            }
            else
            {
                diag_printf("<INFO>: %6lld: pthread_mutex_timedlock non-timeout: %d\n", ms_time(), result);
            }
        }
    }
}

// ------------------------------------------------------------------------

static void control_thread( void )
{
    pthread_attr_t attr;
    struct sched_param schedparam;
    pthread_mutexattr_t mattr;

    pthread_mutexattr_init( &mattr );

    pthread_mutex_init( &mx1, &mattr );
    pthread_mutex_lock( &mx1 );

    expect = ETIMEDOUT;
    
    // Start thread1

    schedparam.sched_priority = 15;
    pthread_attr_init( &attr );
    pthread_attr_setstackaddr( &attr, (void *)((char *)(&stack[0])+STACKSIZE) );        
    pthread_attr_setstacksize( &attr, STACKSIZE );
    pthread_attr_setinheritsched( &attr, PTHREAD_EXPLICIT_SCHED );
    pthread_attr_setschedpolicy( &attr, SCHED_RR );
    pthread_attr_setschedparam( &attr, &schedparam );
    
    pthread_create( &thread[0],
                    &attr,
                    thread1,
                    NULL);

    
    // Wait a bit and then signal thread1 to check that it doesn't return.

    delay_ms( 1500 );
    expect = -1;
    pthread_kill( thread[0], SIGUSR1 );

    delay_ms( 100 );
    
    CYG_TEST_CHECK( sigusr1_called == 1, "Signal not delivered" );
        
    delay_ms( 100 );

    // Next, unlock the mutex so that thread1 acquires it.
    
    expect = ENOERR;    
    pthread_mutex_unlock( &mx1 );
    delay_ms( 1000 );
    pthread_mutex_lock( &mx1 );

    
    // Let the thread run for a few more ticks
    expect = ETIMEDOUT;
    delay_ms(10000);

    if( fails > 0 )
        CYG_TEST_FAIL_FINISH("Pmutex 5");
    else
        CYG_TEST_PASS_FINISH("Pmutex 5");
}

// ------------------------------------------------------------------------

externC int
main( int argc, char **argv )
{
    CYG_TEST_INIT();
    
    control_thread();
}

#else // CYGVAR_KERNEL_COUNTERS_CLOCK &c


externC int
main( int argc, char **argv )
{
    CYG_TEST_INIT();
    CYG_TEST_INFO("PMutex5 test requires:\n"
                         "CYGFUN_KERNEL_API_C &&\n"
                         "CYGVAR_KERNEL_COUNTERS_CLOCK &&\n"
                         "!defined(CYGPKG_KERNEL_SMP_SUPPORT) &&\n"
                         "(CYGNUM_KERNEL_SCHED_PRIORITIES > 20) &&\n"
        );
    CYG_TEST_NA("PMutex5 test requirements");
}
#endif // CYGVAR_KERNEL_COUNTERS_CLOCK &c

// ------------------------------------------------------------------------
// EOF pmutex5.c
