#ifndef CYGONCE_DEVS_ETH_AT91_PHY_H
#define CYGONCE_DEVS_ETH_AT91_PHY_H
//==========================================================================
//
//      at91_phy.h
//
//      Access to PHY access for use by AT91 ethernet driver
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2007 Free Software Foundation, Inc.                        
// Copyright (C) 2007 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    jlarmour
// Contributors: 
// Date:         2007-09-08
// Purpose:
// Description:  PHY access for use by AT91 ethernet driver
//
//####DESCRIPTIONEND####
//
//========================================================================*/

#include <pkgconf/system.h>
#include <pkgconf/hal.h> // clock speed definitions
#include <pkgconf/devs_eth_arm_at91.h>
#if defined(CYGPKG_REDBOOT)
# include <pkgconf/redboot.h>
#endif

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/infra/diag.h>
#include <cyg/hal/hal_io.h>
#include <cyg/hal/hal_diag.h> // HAL_DELAY_US
#include <cyg/io/eth_phy.h>

//============================================================================
// PHY access bits and pieces
// 
// Enable/disable management port.

static void
at91_mdio_enable(void)
{
   cyg_uint32 val;
   HAL_READ_UINT32(AT91_EMAC + AT91_EMAC_NCR, val);
   val |= AT91_EMAC_NCR_MPE;    /* enable management port */
   HAL_WRITE_UINT32(AT91_EMAC + AT91_EMAC_NCR, val);
}

// ---------------------------------------------------------------------

static void 
at91_mdio_disable(void)
{
   cyg_uint32 val;
   HAL_READ_UINT32(AT91_EMAC + AT91_EMAC_NCR, val);
   val &= ~AT91_EMAC_NCR_MPE;    /* disable management port */
   HAL_WRITE_UINT32(AT91_EMAC + AT91_EMAC_NCR, val);
}

// ---------------------------------------------------------------------
// Write one of the PHY registers via the MII bus

static void
at91_write_phy(int reg_addr, int phy_addr, unsigned short data)
{
   cyg_uint32 val, cnt;

   CYG_ASSERTC(reg_addr >= 0 && reg_addr <= AT91_EMAC_MAN_REGA_MASK);
   CYG_ASSERTC(phy_addr >= 0 && phy_addr <= AT91_EMAC_MAN_PHY_MASK);

   phy_debug_printf("phy %d reg %d data %04x\n", phy_addr, reg_addr, (unsigned)data);
   
   val = (AT91_EMAC_MAN_SOF  |
         AT91_EMAC_MAN_WR   |
         AT91_EMAC_MAN_CODE |
         AT91_EMAC_MAN_PHYA(phy_addr) |
         AT91_EMAC_MAN_REGA(reg_addr) |
         AT91_EMAC_MAN_DATA(data));

   HAL_WRITE_UINT32(AT91_EMAC + AT91_EMAC_MAN, val);

   /* Wait until IDLE bit in Network Status register is cleared */
   for (cnt=0; cnt < 1000000; cnt++)
   {
      HAL_READ_UINT32((AT91_EMAC + AT91_EMAC_NSR), val);
      if (!(val & AT91_EMAC_NSR_IDLE))
         break;
   }
   CYG_ASSERTC(cnt < 1000000);
}

// ---------------------------------------------------------------------
// Read one of the PHY registers via the MII bus

static bool
at91_read_phy(int reg_addr, int phy_addr, unsigned short *data)
{
   cyg_uint32 val;

   CYG_ASSERTC(reg_addr >= 0 && reg_addr <= AT91_EMAC_MAN_REGA_MASK);
   CYG_ASSERTC(phy_addr >= 0 && phy_addr <= AT91_EMAC_MAN_PHY_MASK);

   val = (AT91_EMAC_MAN_SOF  |
         AT91_EMAC_MAN_RD   |
         AT91_EMAC_MAN_CODE |
         AT91_EMAC_MAN_PHYA(phy_addr) |
         AT91_EMAC_MAN_REGA(reg_addr));

   HAL_WRITE_UINT32(AT91_EMAC + AT91_EMAC_MAN, val);
   /* Wait until IDLE bit in Network Status register is cleared */
   do
   {
      HAL_READ_UINT32((AT91_EMAC + AT91_EMAC_NSR), val);
   }while(val & AT91_EMAC_NSR_IDLE);
   
   HAL_DELAY_US(50);

   HAL_READ_UINT32(AT91_EMAC + AT91_EMAC_MAN, val);
   *data = val & AT91_EMAC_MAN_DATA_MASK;
   phy_debug_printf("phy %d reg %d data %04x\n", phy_addr, reg_addr, (unsigned)*data);   

   return (true);
}

// ---------------------------------------------------------------------
// Enable the MDIO bit in MAC control register so that we can talk to
// the PHY. Also set the clock divider so that MDC is less than 2.5MHz.

static void 
at91_init_phy(void)
{
   cyg_uint32 cfg;
   cyg_uint32 div;

   HAL_READ_UINT32(AT91_EMAC + AT91_EMAC_NCFG, cfg);
   cfg &=~ AT91_EMAC_NCFG_CLK_MASK;

#if defined(CYGNUM_HAL_ARM_AT91_CLOCK_SPEED)
   div = (CYGNUM_HAL_ARM_AT91_CLOCK_SPEED / 2500000);
#elif defined(CYGARC_HAL_ARM_ARM9_SAM9_CLOCK)
   div = (CYGARC_HAL_ARM_ARM9_SAM9_MCK / 2500000);
#else
#error Undefined clock speed
#endif

   if (div < 8)
   {
      cfg |= AT91_EMAC_NCFG_CLK_HCLK_8;
   }
   else if (div < 16)
   {
      cfg |= AT91_EMAC_NCFG_CLK_HCLK_16;
   }
   else if (div < 32)
   {
      cfg |= AT91_EMAC_NCFG_CLK_HCLK_32;
   }
   else if (div < 64)
   {
      cfg |= AT91_EMAC_NCFG_CLK_HCLK_64;
   }
   else
   {
      CYG_FAIL("Unable to program MII clock");
   }

   HAL_WRITE_UINT32(AT91_EMAC + AT91_EMAC_NCFG, cfg);
}

ETH_PHY_REG_LEVEL_ACCESS_FUNS(at91_phy, 
                              at91_init_phy,
                              NULL,
                              at91_write_phy,
                              at91_read_phy);

#endif // ifdef CYGONCE_DEVS_ETH_AT91_PHY_H
//======================================================================
// EOF at91_phy.h
