//==========================================================================
//
//      io/serial/arm/mc9328mxl/ser_mc9328mxl.c
//
//      Generic ARM MC9328MXL macrocell serial driver
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004 Free Software Foundation, Inc.
// Copyright (C) 2004 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    nickg
// Contributors: (16x5x driver) gthomas, jlarmour, jskov
// Date:         2004-05-04
// Purpose:      MC9328MXL generic serial driver
// Description: 
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/system.h>
#include <pkgconf/io_serial.h>
#include <pkgconf/io.h>

#include <cyg/io/io.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/io/devtab.h>
#include <cyg/io/serial.h>
#include <cyg/infra/diag.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/hal/hal_io.h>

// Only compile driver if an inline file with driver details was selected.
#ifdef CYGDAT_IO_SERIAL_MC9328MXL_INL

//==========================================================================

typedef struct mc9328mxl_serial_info {
    cyg_addrword_t base;
    int            rx_int_num;
    int            tx_int_num;
#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS    
    int            rts_int_num;
    int            uartc_int_num;
    int            pferr_int_num;
#endif
    cyg_interrupt  rx_serial_interrupt;
    cyg_interrupt  tx_serial_interrupt;
    cyg_handle_t   rx_serial_interrupt_handle;
    cyg_handle_t   tx_serial_interrupt_handle;
    int rx_isr_count;
    int tx_isr_count;
    int rx_dsr_count;
    int tx_dsr_count;
#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS
#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW    
    cyg_interrupt  rts_serial_interrupt;
    cyg_handle_t   rts_serial_interrupt_handle;
    int rts_isr_count;
    int rts_dsr_count;
    int throttle_on_count;
    int throttle_off_count;
#endif
    cyg_interrupt  uartc_serial_interrupt;
    cyg_interrupt  pferr_serial_interrupt;
    cyg_handle_t   uartc_serial_interrupt_handle;
    cyg_handle_t   pferr_serial_interrupt_handle;    
    int uartc_isr_count;
    int pferr_isr_count;
    int uartc_dsr_count;
    int pferr_dsr_count;
#endif

} mc9328mxl_serial_info;

static bool mc9328mxl_serial_init(struct cyg_devtab_entry *tab);
static bool mc9328mxl_serial_putc(serial_channel *chan, unsigned char c);
static Cyg_ErrNo mc9328mxl_serial_lookup(struct cyg_devtab_entry **tab, 
                                  struct cyg_devtab_entry *sub_tab,
                                  const char *name);
static unsigned char mc9328mxl_serial_getc(serial_channel *chan);
static Cyg_ErrNo mc9328mxl_serial_set_config(serial_channel *chan, cyg_uint32 key,
                                      const void *xbuf, cyg_uint32 *len);
static void mc9328mxl_serial_start_xmit(serial_channel *chan);
static void mc9328mxl_serial_stop_xmit(serial_channel *chan);

static cyg_uint32 mc9328mxl_serial_rx_ISR(cyg_vector_t vector, cyg_addrword_t data);
static cyg_uint32 mc9328mxl_serial_tx_ISR(cyg_vector_t vector, cyg_addrword_t data);
static void       mc9328mxl_serial_rx_DSR(cyg_vector_t vector, cyg_ucount32 count,
                                cyg_addrword_t data);
static void       mc9328mxl_serial_tx_DSR(cyg_vector_t vector, cyg_ucount32 count,
                                cyg_addrword_t data);

#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS

#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW    
static cyg_uint32 mc9328mxl_serial_rts_ISR(cyg_vector_t vector, cyg_addrword_t data);
static void       mc9328mxl_serial_rts_DSR(cyg_vector_t vector, cyg_ucount32 count,
                                cyg_addrword_t data);
#endif

static cyg_uint32 mc9328mxl_serial_uartc_ISR(cyg_vector_t vector, cyg_addrword_t data);
static cyg_uint32 mc9328mxl_serial_pferr_ISR(cyg_vector_t vector, cyg_addrword_t data);
static void       mc9328mxl_serial_uartc_DSR(cyg_vector_t vector, cyg_ucount32 count,
                                cyg_addrword_t data);
static void       mc9328mxl_serial_pferr_DSR(cyg_vector_t vector, cyg_ucount32 count,
                                cyg_addrword_t data);
#endif

static SERIAL_FUNS(mc9328mxl_serial_funs, 
                   mc9328mxl_serial_putc, 
                   mc9328mxl_serial_getc,
                   mc9328mxl_serial_set_config,
                   mc9328mxl_serial_start_xmit,
                   mc9328mxl_serial_stop_xmit
    );

//==========================================================================

#include CYGDAT_IO_SERIAL_MC9328MXL_INL

#ifndef CYG_IO_SERIAL_MC9328MXL_INT_PRIORITY
# define CYG_IO_SERIAL_MC9328MXL_INT_PRIORITY 4
#endif

//==========================================================================

static unsigned long select_word_length[] = {
    0,                          // 5 bits -- unsupported - treat as 7
    0,                          // 6 bits -- unsupported - treat as 7
    0,                          // 7 bits
    MC9328MXL_UART_CR2_WS       // 8 bits
};

static unsigned long select_stop_bits[] = {
    0,
    0,                          // 1 stop bit
    0,                          // 1.5 stop bit
    MC9328MXL_UART_CR2_STPB     // 2 stop bits
};

static unsigned long select_parity[] = {
    0,                                                  // No parity
    MC9328MXL_UART_CR2_PREN,                            // Even parity
    MC9328MXL_UART_CR2_PREN|MC9328MXL_UART_CR2_PROE,    // Odd parity
    0,                                                  // Mark parity -- unsupported
    0                                                   // Space parity -- unsupported
};

static unsigned int select_baud[] = {
    0,          // Unused
    50,
    75,
    110,
    0,          // 134.5 unsupported
    150,
    200,
    300,
    600,
    1200,
    1800,
    2400,
    3600,
    4800,
    7200,
    9600,
    14400,
    19200,
    38400,
    57600,
    115200,
    230400
};

//==========================================================================
// Internal function to actually configure the hardware to desired
// baud rate, etc.

static bool
serial_config_port(serial_channel *chan, 
                   cyg_serial_info_t *new_config, bool init)
{
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    unsigned int baud = select_baud[new_config->baud];
    unsigned long cr1, cr2, cr3, cr4;

    if (baud == 0) return false;  // Invalid configuration

    HAL_READ_UINT32(base+MC9328MXL_UART_CR1, cr1);
    HAL_READ_UINT32(base+MC9328MXL_UART_CR3, cr3);
    HAL_READ_UINT32(base+MC9328MXL_UART_CR4, cr4);
    
    // Disable UART while changing hardware
    HAL_WRITE_UINT32(base+MC9328MXL_UART_CR1, 0);
    HAL_WRITE_UINT32(base+MC9328MXL_UART_CR3, 0);

    // Set clock
    cr4 |= MC9328MXL_UART_CR4_CTSTL_32 |
           MC9328MXL_UART_CR4_REF16    |
           MC9328MXL_UART_CR4_DREN;
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR4, cr4 );
    
    // Enable FIFOs
    HAL_WRITE_UINT32( base+MC9328MXL_UART_FCR,
                      (1 << MC9328MXL_UART_FCR_RXTL_SHF) |
                      MC9328MXL_UART_FCR_RFDIV_1 |
                      (2 << MC9328MXL_UART_FCR_TXTL_SHF) );
    
    cr2 = select_word_length[new_config->word_length - CYGNUM_SERIAL_WORD_LENGTH_5] | 
        select_stop_bits[new_config->stop] |
        select_parity[new_config->parity];

    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR2, cr2 );
    
    if (init) {

        // TODO: set FIFO interrupt levels
        // Currently default to 1/2 full
        
        if (chan->out_cbuf.len != 0) {
            cr1 |= MC9328MXL_UART_CR1_RRDYEN;
        }
    }

    
#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS
    cr3 |= MC9328MXL_UART_CR3_PARERREN | MC9328MXL_UART_CR3_FRAERREN;
    cr3 |= MC9328MXL_UART_CR3_DTREN;
    cr4 |= MC9328MXL_UART_CR4_BKEN | MC9328MXL_UART_CR4_OREN;
#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW
    cr1 |= MC9328MXL_UART_CR1_RTSDEN;
    cr2 |= MC9328MXL_UART_CR2_CTS;
#endif
#endif

#ifndef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW
    // Ignore the CTS line. If the hardware has left this line
    // floating then it can interfere with transmission. Note that
    // Motorola, for their own reasons, call this line RTS.
    cr2 |= MC9328MXL_UART_CR2_IRTS;
#endif
    
    HAL_WRITE_UINT32(base+MC9328MXL_UART_CR3, cr3);
    HAL_WRITE_UINT32(base+MC9328MXL_UART_CR4, cr4);

    // Enable UART
    cr1 |= MC9328MXL_UART_CR1_UARTCLKEN | MC9328MXL_UART_CR1_UARTEN;
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR1, cr1 );
    cr2 |= MC9328MXL_UART_CR2_SRST_ | MC9328MXL_UART_CR2_RXEN  | MC9328MXL_UART_CR2_TXEN;
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR2, cr2 );

    // There is a slight delay between writing the registers above and
    // the UART being available. Sometimes this results in the BIR
    // register not being written. The following loop delays a little
    // to allow the UART to get ready.
    { int i; for( i = 0; i < 100; i++ ); }
    
    // Set baud rate -- these registers only take if the UART is enabled.
    HAL_WRITE_UINT32( base+MC9328MXL_UART_BIR, baud/100 - 1 );
    HAL_WRITE_UINT32( base+MC9328MXL_UART_BRM, 10000 - 1 );

    // Clear all status flags
    HAL_WRITE_UINT32( base+MC9328MXL_UART_SR1, 0xFFFFFFFF );
    HAL_WRITE_UINT32( base+MC9328MXL_UART_SR2, 0xFFFFFFFF );
    
    if (new_config != &chan->config) {
        chan->config = *new_config;
    }
    return true;
}

//==========================================================================
// Function to initialize the device.  Called at bootstrap time.

static bool 
mc9328mxl_serial_init(struct cyg_devtab_entry *tab)
{
    serial_channel *chan = (serial_channel *)tab->priv;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;

#ifdef CYGDBG_IO_INIT
    diag_printf("mc9328mxl SERIAL init - dev: %x.%d.%d\n", 
                ser_chan->base, ser_chan->rx_int_num, ser_chan->tx_int_num);
#endif

    ser_chan->base = CYGARC_VIRTUAL_ADDRESS( ser_chan->base );
    
    // Really only required for interrupt driven devices
    (chan->callbacks->serial_init)(chan);

    if (chan->out_cbuf.len != 0) {
        cyg_drv_interrupt_create(ser_chan->rx_int_num,
                                 CYG_IO_SERIAL_MC9328MXL_INT_PRIORITY,
                                 (cyg_addrword_t)chan,
                                 mc9328mxl_serial_rx_ISR,
                                 mc9328mxl_serial_rx_DSR,
                                 &ser_chan->rx_serial_interrupt_handle,
                                 &ser_chan->rx_serial_interrupt);
        cyg_drv_interrupt_attach(ser_chan->rx_serial_interrupt_handle);
        cyg_drv_interrupt_unmask(ser_chan->rx_int_num);

        cyg_drv_interrupt_create(ser_chan->tx_int_num,
                                 CYG_IO_SERIAL_MC9328MXL_INT_PRIORITY,
                                 (cyg_addrword_t)chan,
                                 mc9328mxl_serial_tx_ISR,
                                 mc9328mxl_serial_tx_DSR,
                                 &ser_chan->tx_serial_interrupt_handle,
                                 &ser_chan->tx_serial_interrupt);
        cyg_drv_interrupt_attach(ser_chan->tx_serial_interrupt_handle);
        cyg_drv_interrupt_unmask(ser_chan->tx_int_num);

#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS

#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW            
        cyg_drv_interrupt_create(ser_chan->rts_int_num,
                                 CYG_IO_SERIAL_MC9328MXL_INT_PRIORITY,
                                 (cyg_addrword_t)chan,
                                 mc9328mxl_serial_rts_ISR,
                                 mc9328mxl_serial_rts_DSR,
                                 &ser_chan->rts_serial_interrupt_handle,
                                 &ser_chan->rts_serial_interrupt);
        cyg_drv_interrupt_attach(ser_chan->rts_serial_interrupt_handle);
        cyg_drv_interrupt_unmask(ser_chan->rts_int_num);
#endif
        
        cyg_drv_interrupt_create(ser_chan->uartc_int_num,
                                 CYG_IO_SERIAL_MC9328MXL_INT_PRIORITY,
                                 (cyg_addrword_t)chan,
                                 mc9328mxl_serial_uartc_ISR,
                                 mc9328mxl_serial_uartc_DSR,
                                 &ser_chan->uartc_serial_interrupt_handle,
                                 &ser_chan->uartc_serial_interrupt);
        cyg_drv_interrupt_attach(ser_chan->uartc_serial_interrupt_handle);
        cyg_drv_interrupt_unmask(ser_chan->uartc_int_num);
        
        cyg_drv_interrupt_create(ser_chan->pferr_int_num,
                                 CYG_IO_SERIAL_MC9328MXL_INT_PRIORITY,
                                 (cyg_addrword_t)chan,
                                 mc9328mxl_serial_pferr_ISR,
                                 mc9328mxl_serial_pferr_DSR,
                                 &ser_chan->pferr_serial_interrupt_handle,
                                 &ser_chan->pferr_serial_interrupt);
        cyg_drv_interrupt_attach(ser_chan->pferr_serial_interrupt_handle);
        cyg_drv_interrupt_unmask(ser_chan->pferr_int_num);

#endif
        
    }
    
    serial_config_port(chan, &chan->config, true);
    return true;
}

//==========================================================================
// This routine is called when the device is "looked" up (i.e. attached)

static Cyg_ErrNo 
mc9328mxl_serial_lookup(struct cyg_devtab_entry **tab, 
                 struct cyg_devtab_entry *sub_tab,
                 const char *name)
{
    serial_channel *chan = (serial_channel *)(*tab)->priv;

    // Really only required for interrupt driven devices
    (chan->callbacks->serial_init)(chan);
    return ENOERR;
}

//==========================================================================
// Send a character to the device output buffer.
// Return 'true' if character is sent to device

static bool
mc9328mxl_serial_putc(serial_channel *chan, unsigned char c)
{
    cyg_uint32 status;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;

    HAL_READ_UINT32(base+MC9328MXL_UART_TS, status);
    if( status & MC9328MXL_UART_TS_TXFULL )
        return false;

    HAL_WRITE_UINT32(base+MC9328MXL_UART_TXR, c);

    return true;
}

//==========================================================================
// Fetch a character from the device input buffer, waiting if necessary

static unsigned char 
mc9328mxl_serial_getc(serial_channel *chan)
{
    cyg_uint32 c;
    cyg_uint32 status;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;

    // Wait for char
    do {
        HAL_READ_UINT32(base+MC9328MXL_UART_TS, status);
    } while (status & MC9328MXL_UART_TS_RXEMPTY);

    HAL_READ_UINT32(base+MC9328MXL_UART_RXR, c);

    return c&0xFF;
}

//==========================================================================
// Set up the device characteristics; baud rate, etc.

static Cyg_ErrNo
mc9328mxl_serial_set_config(serial_channel *chan, cyg_uint32 key, const void *xbuf,
                     cyg_uint32 *len)
{
    switch (key) {
    case CYG_IO_SET_CONFIG_SERIAL_INFO:
      {
        cyg_serial_info_t *config = (cyg_serial_info_t *)xbuf;
        if ( *len < sizeof(cyg_serial_info_t) ) {
            return -EINVAL;
        }
        *len = sizeof(cyg_serial_info_t);
        if ( true != serial_config_port(chan, config, false) )
            return -EINVAL;
      }
      break;
#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW
    case CYG_IO_SET_CONFIG_SERIAL_HW_RX_FLOW_THROTTLE:
      {
          // Set the RTS line according to what the upper level
          // wants. Due to Motorola's weird world view, this line is
          // actually labelled CTS on this hardware.

          mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
          cyg_addrword_t base = ser_chan->base;
          cyg_uint32 *f = (cyg_uint32 *)xbuf;
          cyg_uint32 cr2;
          
          if ( *len < sizeof(*f) )
              return -EINVAL;

          HAL_READ_UINT32( base+MC9328MXL_UART_CR2, cr2 );

          if( *f )      // throttle, set CTS bit
              cr2 |= MC9328MXL_UART_CR2_CTS, ser_chan->throttle_on_count++;
          else          // un-throttle, clear CTS bit
              cr2 &= ~MC9328MXL_UART_CR2_CTS, ser_chan->throttle_off_count++;
          
          HAL_WRITE_UINT32( base+MC9328MXL_UART_CR2, cr2 );
      }
      break;
      
    case CYG_IO_SET_CONFIG_SERIAL_HW_FLOW_CONFIG:
        // Clear any unsupported flags here and then return -ENOSUPP -
        // the higher layer can then query what flags are set and
        // decide what to do. This is optimised for the most common
        // case - i.e. that authors know what their hardware is
        // capable of.

        // Clear DSR/DTR flag as it's not supported.
        if (chan->config.flags &
            (CYGNUM_SERIAL_FLOW_DSRDTR_RX|CYGNUM_SERIAL_FLOW_DSRDTR_TX))
        {
            chan->config.flags &= ~(CYGNUM_SERIAL_FLOW_DSRDTR_RX|
                                    CYGNUM_SERIAL_FLOW_DSRDTR_TX);
            return -EINVAL;
        }
      
        
      break;
#endif
    default:
        return -EINVAL;
    }
    return ENOERR;
}

//==========================================================================
// Enable the transmitter on the device

static void
mc9328mxl_serial_start_xmit(serial_channel *chan)
{
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_uint32 cr1;

    HAL_READ_UINT32( base+MC9328MXL_UART_CR1, cr1 );
    cr1 |= MC9328MXL_UART_CR1_TRDYEN;
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR1, cr1 );

    (chan->callbacks->xmt_char)(chan);    
}

//==========================================================================
// Disable the transmitter on the device

static void 
mc9328mxl_serial_stop_xmit(serial_channel *chan)
{
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_uint32 cr1;

    HAL_READ_UINT32( base+MC9328MXL_UART_CR1, cr1 );
    cr1 &= ~MC9328MXL_UART_CR1_TRDYEN;
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR1, cr1 );
}

//==========================================================================
// Serial I/O - low level interrupt handler (ISR)

static cyg_uint32 
mc9328mxl_serial_rx_ISR(cyg_vector_t vector, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    ser_chan->rx_isr_count++;
    cyg_drv_interrupt_mask(ser_chan->rx_int_num);
    cyg_drv_interrupt_acknowledge(ser_chan->rx_int_num);
    return CYG_ISR_CALL_DSR;  // Cause DSR to be run
}

static cyg_uint32 
mc9328mxl_serial_tx_ISR(cyg_vector_t vector, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    ser_chan->tx_isr_count++;
    cyg_drv_interrupt_mask(ser_chan->tx_int_num);
    cyg_drv_interrupt_acknowledge(ser_chan->tx_int_num);
    return CYG_ISR_CALL_DSR;  // Cause DSR to be run
}

#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS

#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW    
static cyg_uint32 
mc9328mxl_serial_rts_ISR(cyg_vector_t vector, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    ser_chan->rts_isr_count++;
    cyg_drv_interrupt_mask(ser_chan->rts_int_num);
    cyg_drv_interrupt_acknowledge(ser_chan->rts_int_num);
    return CYG_ISR_CALL_DSR;  // Cause DSR to be run
}
#endif

static cyg_uint32 
mc9328mxl_serial_uartc_ISR(cyg_vector_t vector, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    ser_chan->uartc_isr_count++;    
    cyg_drv_interrupt_mask(ser_chan->uartc_int_num);
    cyg_drv_interrupt_acknowledge(ser_chan->uartc_int_num);
    return CYG_ISR_CALL_DSR;  // Cause DSR to be run
}

static cyg_uint32 
mc9328mxl_serial_pferr_ISR(cyg_vector_t vector, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    ser_chan->pferr_isr_count++;
    cyg_drv_interrupt_mask(ser_chan->pferr_int_num);
    cyg_drv_interrupt_acknowledge(ser_chan->pferr_int_num);
    return CYG_ISR_CALL_DSR;  // Cause DSR to be run
}

#endif

//==========================================================================
// Serial I/O - high level interrupt handler (DSR)

static void       
mc9328mxl_serial_rx_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_int32 sr1;

    ser_chan->rx_dsr_count++;
    HAL_READ_UINT32(base+MC9328MXL_UART_SR1, sr1 );

    if( sr1 & MC9328MXL_UART_SR1_RRDY )
    {
        cyg_uint32 c;
        HAL_READ_UINT32(base+MC9328MXL_UART_RXR, c);
            
        while( c & MC9328MXL_UART_RXR_RDY )
        {
            (chan->callbacks->rcv_char)(chan, c&0xFF);
            HAL_READ_UINT32(base+MC9328MXL_UART_RXR, c);            
        }
    }

    cyg_drv_interrupt_unmask(ser_chan->rx_int_num);    
}

static void       
mc9328mxl_serial_tx_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_int32 sr1;

    ser_chan->tx_dsr_count++;
    HAL_READ_UINT32(base+MC9328MXL_UART_SR1, sr1 );
    
    if( sr1 & MC9328MXL_UART_SR1_TRDY )
            (chan->callbacks->xmt_char)(chan);

    cyg_drv_interrupt_unmask(ser_chan->tx_int_num);    
}

#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS

#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW    

static void       
mc9328mxl_serial_rts_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_int32 sr1;

    ser_chan->rts_dsr_count++;
    HAL_READ_UINT32(base+MC9328MXL_UART_SR1, sr1 );

    if ( chan->config.flags & CYGNUM_SERIAL_FLOW_RTSCTS_TX )
    {
        cyg_serial_line_status_t stat;
        HAL_WRITE_UINT32(base+MC9328MXL_UART_SR1, MC9328MXL_UART_SR1_RTSD );
        stat.which = CYGNUM_SERIAL_STATUS_FLOW;
        stat.value = (0 != (sr1 & MC9328MXL_UART_SR1_RTSS));
        (chan->callbacks->indicate_status)(chan, &stat );
    }

    cyg_drv_interrupt_unmask(ser_chan->rts_int_num);        
}

#endif

static void       
mc9328mxl_serial_uartc_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_int32 sr2;

    ser_chan->uartc_dsr_count++;    
    HAL_READ_UINT32(base+MC9328MXL_UART_SR2, sr2 );

    if( sr2 & MC9328MXL_UART_SR2_ORE )
    {
        cyg_serial_line_status_t stat;
        HAL_WRITE_UINT32( base+MC9328MXL_UART_SR2, MC9328MXL_UART_SR2_ORE );
        stat.which = CYGNUM_SERIAL_STATUS_OVERRUNERR;
        (chan->callbacks->indicate_status)(chan, &stat );
    }

    if( sr2 & MC9328MXL_UART_SR2_BRCD )
    {
        cyg_serial_line_status_t stat;
        HAL_WRITE_UINT32( base+MC9328MXL_UART_SR2, MC9328MXL_UART_SR2_BRCD );        
        stat.which = CYGNUM_SERIAL_STATUS_BREAK;
        (chan->callbacks->indicate_status)(chan, &stat );
    }

    cyg_drv_interrupt_unmask(ser_chan->uartc_int_num);        
}

static void       
mc9328mxl_serial_pferr_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    mc9328mxl_serial_info *ser_chan = (mc9328mxl_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_int32 sr1, sr2;

    ser_chan->pferr_dsr_count++;    
    HAL_READ_UINT32(base+MC9328MXL_UART_SR1, sr1 );

    if( sr1 & MC9328MXL_UART_SR1_PARITYERR )
    {
        cyg_serial_line_status_t stat;
        HAL_WRITE_UINT32(base+MC9328MXL_UART_SR1, MC9328MXL_UART_SR1_PARITYERR );
        stat.which = CYGNUM_SERIAL_STATUS_PARITYERR;
        (chan->callbacks->indicate_status)(chan, &stat );
    }

    HAL_READ_UINT32(base+MC9328MXL_UART_SR1, sr2 );
    
    if( sr1 & MC9328MXL_UART_SR1_FRAMERR )
    {
        cyg_serial_line_status_t stat;
        HAL_WRITE_UINT32(base+MC9328MXL_UART_SR1, MC9328MXL_UART_SR1_FRAMERR );
        stat.which = CYGNUM_SERIAL_STATUS_FRAMEERR;
        (chan->callbacks->indicate_status)(chan, &stat );
    }

    cyg_drv_interrupt_unmask(ser_chan->pferr_int_num);        
}


#endif
    
//==========================================================================

#endif // CYGDAT_IO_SERIAL_MC9328MXL_INL

//==========================================================================
// EOF ser_mc9328mxl.c
