//==========================================================================
//
//      devs/wallclock/arm/at91/VERSION/src/wallclock_at91.cxx
//
//      Wallclock implementation for Atmel AT91 CPUs
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2004, 2005, 2009 eCosCentric Limited                             
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2005 Free Software Foundation, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     jlarmour
// Contributors:  
// Date:          2005-08-18
// Purpose:       Wallclock driver for on-board RTC of Atmel AT91/AT91RM9200 CPUs.
// Description:   Based on eCosCentric LPC2xxx wallclock driver.
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/wallclock.h>          // Wallclock device config
#include <pkgconf/system.h>             // System package configuration

#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_intr.h>           // Interrupt macros
#include <cyg/infra/cyg_type.h>         // Common type definitions and support
#include <cyg/infra/cyg_ass.h>          // Assertions
#include <cyg/infra/cyg_trac.h>          // Assertions

#include <cyg/io/wallclock.hxx>         // The WallClock API
#include <cyg/io/wallclock/wallclock.inl> // Helpers

//#include <cyg/infra/diag.h>             // For debugging

#ifdef CYGPKG_KERNEL
#include <pkgconf/kernel.h>
#include <cyg/kernel/ktypes.h>
#include <cyg/kernel/mutex.hxx>
#include <cyg/kernel/sema.hxx>
#include <cyg/kernel/intr.hxx>
#endif

// Helper macros to get/set two digit values in BCD, masked to 7 bits
#undef TO_BCD
#undef FROM_BCD
#define TO_BCD(_x_)      (((_x_)%10) | (((_x_)/10)<<4) & 0x7f)
#define FROM_BCD(_x_)    (((((_x_)>>4)&0xf)*10) + ((_x_)&0xf))

//-----------------------------------------------------------------------------

// The AT91 (not -RM9200) HAL has slightly different names for things.
// Map them to the AT91RM9200 naming.
#ifdef CYGPKG_HAL_ARM_AT91
# include <cyg/hal/hal_io.h>
# define _RTC_CR _RTC_MR
# define  _RTC_CR_UPDTIM             _RTC_MR_UPDTIM             
# define  _RTC_CR_UPDCAL             _RTC_MR_UPDCAL             
# define  _RTC_CR_TIMEVSEL_MIN       _RTC_MR_TIMEVSEL_MIN       
# define  _RTC_CR_TIMEVSEL_HOUR      _RTC_MR_TIMEVSEL_HOUR      
# define  _RTC_CR_TIMEVSEL_MIDNIGHT  _RTC_MR_TIMEVSEL_MIDNIGHT  
# define  _RTC_CR_TIMEVSEL_NOON      _RTC_MR_TIMEVSEL_NOON      
# define  _RTC_CR_CALVSEL_WEEK       _RTC_MR_CALVSEL_WEEK       
# define  _RTC_CR_CALVSEL_MONTH      _RTC_MR_CALVSEL_MONTH      
# define  _RTC_CR_CALVSEL_YEAR       _RTC_MR_CALVSEL_YEAR       
# undef _RTC_MR
# define _RTC_MR _RTC_HMR
# define CYGNUM_HAL_INTERRUPT_RTCH CYGNUM_HAL_INTERRUPT_RTC0
#endif

//-----------------------------------------------------------------------------

// Program the clock registers
static void
set_hwclock_regs( cyg_uint32 calr, cyg_uint32 timr )
{
    cyg_uint32 cr;

    HAL_WRITE_UINT32( _RTC_CALR, calr );
    HAL_WRITE_UINT32( _RTC_TIMR, timr );
    
    // Now set UPDTIM/UPDCAL in the control register to 0 as per
    // the docs (ignoring the fact the docs also say the CR is read-only
    // and writing has no effect!)
    HAL_READ_UINT32( _RTC_CR, cr );
    cr &= ~(_RTC_CR_UPDTIM|_RTC_CR_UPDCAL);
    HAL_WRITE_UINT32( _RTC_CR, cr );

    // And clear the SEC field in the SR so we know when it's safe to do a future
    // reprogram
    HAL_WRITE_UINT32( _RTC_SCCR, _RTC_SR_SEC );
}

#ifdef CYGPKG_KERNEL

static volatile cyg_uint32 new_calr, new_timr;

// Need a mutex to prevent something retrieving (or indeed setting) values mid-update.
static Cyg_Mutex CYGBLD_ATTRIB_INIT_AFTER( CYG_INIT_KERNEL ) update_mutex;

// And a condition variable to implement a wait
static Cyg_Condition_Variable CYGBLD_ATTRIB_INIT_AFTER( CYG_INIT_KERNEL ) update_cond( update_mutex );

// The secwait_sem is used to signal threads waiting for 1 second after prior
// programming so that they can now wake up
static Cyg_Binary_Semaphore CYGBLD_ATTRIB_INIT_AFTER( CYG_INIT_KERNEL ) secwait_sem;

static volatile cyg_bool updating = false;

static volatile cyg_uint8 dsr_action;
#define DSR_ACTION_SIGNAL_COND        (1<<0)
#define DSR_ACTION_WAKE_SEM_WAITER    (1<<1)

static cyg_uint32
rtc_isr(cyg_vector vector, CYG_ADDRWORD data)
{
    cyg_uint32 sr,imr;

    CYG_REPORT_FUNCTION();
    CYG_REPORT_FUNCARG2XV(vector, data);

    CYG_ASSERT( vector == CYGNUM_HAL_INTERRUPT_RTCH, "RTC ISR called with wrong vector");

    HAL_READ_UINT32( _RTC_SR, sr );
    HAL_READ_UINT32( _RTC_IMR, imr );

    sr &= imr; // only act on bits explicitly unmasked

    if ( sr & ~(_RTC_SR_SEC|_RTC_SR_ACKUPD) ) // Anything other than SEC and ACKUPD
        CYG_FAIL("RTC interrupt for unrequested event");

    // An ISR should only run for one of the events since the update thread should
    // only do one at any one time; and due to update_mutex, there should only be
    // one thread fiddling anyway.
    if ( (sr & (_RTC_SR_SEC|_RTC_SR_ACKUPD)) == (_RTC_SR_SEC|_RTC_SR_ACKUPD)) // Both set
        CYG_FAIL("RTC interrupt for multiple events");

    if ( sr & _RTC_SR_SEC )
    {
        // Unset sec interrupt for future
        HAL_WRITE_UINT32( _RTC_IDR, _RTC_SR_SEC );
        dsr_action |= DSR_ACTION_WAKE_SEM_WAITER;
    }
    else if ( sr & _RTC_SR_ACKUPD )
    {
        // Ready to update time, so set it right now from stashed values
        // rather than waiting for a thread to be scheduled in at some unknown
        // future point
        set_hwclock_regs( new_calr, new_timr );

        // And disable this interrupt for the future
        HAL_WRITE_UINT32( _RTC_IDR, _RTC_SR_ACKUPD );

        // Regs now programmed so clock readers can now run
        dsr_action |= DSR_ACTION_SIGNAL_COND;
    }

    Cyg_Interrupt::acknowledge_interrupt(CYGNUM_HAL_INTERRUPT_RTCH);

    CYG_REPORT_RETVAL( Cyg_Interrupt::HANDLED|Cyg_Interrupt::CALL_DSR );
    return ( Cyg_Interrupt::HANDLED|Cyg_Interrupt::CALL_DSR );
} // rtc_isr()

static void
rtc_dsr(cyg_vector vector, cyg_ucount32 count, CYG_ADDRWORD data)
{
    // An ISR should never pre-empt us to change dsr_action
    // here since no thread should have wanted two ISRs to happen
    // (due to update_mutex preventing two threads)

    // Can now indicate okay to read.
    if (dsr_action & DSR_ACTION_SIGNAL_COND)
    {
        updating = false;
        update_cond.broadcast();
    }
    else if (dsr_action & DSR_ACTION_WAKE_SEM_WAITER)
    {
        // wake waiting thread
        secwait_sem.post();
    }
    dsr_action = 0;
} // rtc_dsr()

static Cyg_Interrupt CYGBLD_ATTRIB_INIT_AFTER(CYG_INIT_KERNEL) rtcint(
    CYGNUM_HAL_INTERRUPT_RTCH,
    1,
    0,
    rtc_isr,
    rtc_dsr
  );

static int arm_global_ints_are_on( void )
{
    CYG_INTERRUPT_STATE intstate;
    int ret;
    HAL_QUERY_INTERRUPTS( intstate );
    // We know what the contents are for ARM
    ret = (intstate & 0x80) == 0;  // IRQs disabled bit set
    return ret;
}
#endif // ifdef CYGPKG_KERNEL


//-----------------------------------------------------------------------------
// Functions required for the hardware-driver API.

// Returns the number of seconds elapsed since 1970-01-01 00:00:00.
cyg_uint32 
Cyg_WallClock::get_hw_seconds(void)
{
    cyg_uint32 timr, timr_prev, calr, calr_prev;
    cyg_uint32 year, month, mday, hour, minute, second;
    int timeout;

#ifdef CYGPKG_KERNEL
    // We lock a mutex to ensure that code cannot be setting the clock at the
    // same time as reading it. Reads are fast enough that clashes between
    // multiple readers aren't an issue worth wasting extra code/memory on IMHO.
    update_mutex.lock();
    while( updating )
        update_cond.wait();
#endif

    // As per datasheet, due to async update, must be read multiply to
    // ensure the value is stable.
    HAL_READ_UINT32( _RTC_TIMR, timr_prev );
    HAL_READ_UINT32( _RTC_CALR, calr_prev );
    for ( timeout=100000; timeout>0 ; timeout-- )
    {
        HAL_READ_UINT32( _RTC_TIMR, timr );
        HAL_READ_UINT32( _RTC_CALR, calr );
        if ( (timr == timr_prev) && (calr == calr_prev) )
            break;
        timr_prev = timr;
        calr_prev = calr;
    }

#ifdef CYGPKG_KERNEL
    // Can now indicate okay for another thread to update clock if they happened to
    // want to at the same time.
    update_mutex.unlock();
#endif

    CYG_ASSERT( timeout > 0, "Failed to read stable RTC within timeout" );

    year = FROM_BCD( calr >>_RTC_CALR_YEAR_SHIFT ) + 
        FROM_BCD( calr >>_RTC_CALR_CENT_SHIFT) * 100;
    month = FROM_BCD( (calr >> _RTC_CALR_MONTH_SHIFT)&0x1f );
    //    day = FROM_BCD( (calr >> _RTC_CALR_DAY_SHIFT)&0x7 );
    mday = FROM_BCD( calr >> _RTC_CALR_DATE_SHIFT );
    hour = FROM_BCD( (timr >> _RTC_TIMR_HOUR_SHIFT)&0x3f );
    minute = FROM_BCD( timr >> _RTC_TIMR_MIN_SHIFT );
    second = FROM_BCD( timr >> _RTC_TIMR_SEC_SHIFT );

#if 0
    // This will cause the test to eventually fail due to these printouts
    // causing timer interrupts to be lost...
    diag_printf("year %02d\n", year);
    diag_printf("month %02d\n", month);
    diag_printf("mday %02d\n", mday);
    diag_printf("hour %02d\n", hour);
    diag_printf("minute %02d\n", minute);
    diag_printf("second %02d\n", second);
#endif

    cyg_uint32 now = _simple_mktime(year, month, mday, hour, minute, second);
    return now;
}

static void
set_hwclock(cyg_uint32 year, cyg_uint32 month, cyg_uint32 mday,
            cyg_uint32 hour, cyg_uint32 minute, cyg_uint32 second)
{
    cyg_uint32 sr;
#ifdef CYGPKG_KERNEL
    cyg_bool ints_on = arm_global_ints_are_on();

    // We make sure other clock operations, including reads can't
    // come in while we're updating.
    update_mutex.lock();
    while( updating )
        update_cond.wait();
    updating = true;
#endif

    // A previous update may not have fully completed - you are meant to wait
    // for a second. We determine that from whether there's an uncleared SEC bit
    // in the status reg. If that applies here, then wait.

    HAL_READ_UINT32( _RTC_SR, sr );

    if ( (sr & _RTC_SR_SEC) == 0 )
    {
#ifdef CYGPKG_KERNEL
        // Are global interrupts on? We can't use an interrupt handler if not.
        if ( ints_on )
        {
            // enable sec interrupt
            HAL_WRITE_UINT32( _RTC_IER, _RTC_SR_SEC );
            // check again just to be sure (race condition)
            if ( sr & _RTC_SR_SEC )
            {
                // disable int again
                HAL_WRITE_UINT32( _RTC_IDR, _RTC_SR_SEC );
            } else {
                // Need to wait for the event
                secwait_sem.wait(); // Zzzzz
            }
        }
        else
#endif
        {
            // No kernel or no interrupts. So poll.
            do {
                HAL_READ_UINT32( _RTC_SR, sr );
            } while ( (sr & _RTC_SR_SEC) == 0 );
        }
    }

    // We let the ISR do the job of actually setting, once ACKUPD is set.
    // So instead here we just set up the new timr/calr and allow the ISR
    // to do its thing. (Or if in polled mode, poll and set directly)
    cyg_uint32 timr, calr;
    calr  = TO_BCD( year/100 ) << _RTC_CALR_CENT_SHIFT;
    calr |= TO_BCD( year%100 ) << _RTC_CALR_YEAR_SHIFT;
    calr |= TO_BCD( month ) << _RTC_CALR_MONTH_SHIFT;
    // We shouldn't care about day, but it needs to be valid
    // which means non-zero. We always set to 1.
    calr |= TO_BCD( 1 ) << _RTC_CALR_DAY_SHIFT;
    calr |= TO_BCD( mday ) << _RTC_CALR_DATE_SHIFT;
    
    timr  = TO_BCD( second ) << _RTC_TIMR_SEC_SHIFT;
    timr |= TO_BCD( minute ) << _RTC_TIMR_MIN_SHIFT;
    timr |= TO_BCD( hour ) << _RTC_TIMR_HOUR_SHIFT;

#ifdef CYGPKG_KERNEL
    // Update the globals, ready for an interrupt
    if ( ints_on )
    {
        new_timr = timr;
        new_calr = calr;

        // Enable interrupt
        HAL_WRITE_UINT32( _RTC_IER, _RTC_SR_ACKUPD );
    }
#endif
    
    // Request update
    cyg_uint32 cr;
    HAL_READ_UINT32( _RTC_CR, cr );
    cr |= (_RTC_CR_UPDTIM|_RTC_CR_UPDCAL);
    HAL_WRITE_UINT32( _RTC_CR, cr );
    
    // With kernel and ints, no need to wait for completion after
    // update - the ISR does that. However, we exit, we need to unlock
    // the mutex.
    
#ifdef CYGPKG_KERNEL
    if ( ints_on )
    {
        update_mutex.unlock();    
        return;
    }
#endif
    // Otherwise we poll
    do {
        HAL_READ_UINT32( _RTC_SR, sr );
    } while ( (sr & _RTC_SR_ACKUPD) == 0 );

    set_hwclock_regs( calr, timr );

#ifdef CYGPKG_KERNEL
    updating = false;
    update_mutex.unlock();    
#endif
}

#ifdef CYGSEM_WALLCLOCK_SET_GET_MODE
// Sets the clock. Argument is seconds elapsed since 1970-01-01 00:00:00.
void
Cyg_WallClock::set_hw_seconds( cyg_uint32 secs )
{
    cyg_uint32 year, month, mday, hour, minute, second;

    _simple_mkdate(secs, &year, &month, &mday, &hour, &minute, &second);

    set_hwclock(year, month, mday, hour, minute, second);
}
#endif

void
Cyg_WallClock::init_hw_seconds(void)
{
    // Force to known good state, just in case.

    // First let's set 24 hour mode.
    HAL_WRITE_UINT32( _RTC_MR, 0 );
    // Make sure we're not in programming mode
    HAL_WRITE_UINT32( _RTC_CR, 0 );
    // Disable alarms
    HAL_WRITE_UINT32( _RTC_TIMALR, 0 );
    HAL_WRITE_UINT32( _RTC_CALALR, 0 );
    // Ensure all ints are disabled
    HAL_WRITE_UINT32( _RTC_IDR, _RTC_SR_ACKUPD | _RTC_SR_ALARM | _RTC_SR_SEC | _RTC_SR_TIMEV | _RTC_SR_CALEV);    
    // And clear all status bits *EXCEPT* SEC, since we don't want to wait a second
    HAL_WRITE_UINT32( _RTC_SCCR, _RTC_SR_ACKUPD | _RTC_SR_ALARM | _RTC_SR_TIMEV | _RTC_SR_CALEV);    
    
    // Now, if we're part of the kernel, install the interrupt handler
#ifdef CYGPKG_KERNEL
  rtcint.attach();
  rtcint.acknowledge_interrupt(CYGNUM_HAL_INTERRUPT_RTCH);
  rtcint.unmask_interrupt(CYGNUM_HAL_INTERRUPT_RTCH);
#endif

#ifndef CYGSEM_WALLCLOCK_SET_GET_MODE
    // This is our base: 1970-01-01 00:00:00
    // Set the HW clock - if for nothing else, just to be sure it's in a
    // legal range. Any arbitrary base could be used.
    // After this the hardware clock is only read.

    set_hwclock(1970,1,1,0,0,0);
#endif    
}

//-----------------------------------------------------------------------------
// End of devs/wallclock/arm/at91/wallclock_at91.cxx
