//==========================================================================
//
//      devs/wallclock/m48t35a.cxx
//
//      Wallclock implementation for ST M48T35A Timekeeper SRAM chips
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2004 Free Software Foundation, Inc.                        
// Copyright (C) 2004 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     bartv
// Date:          2004-05-14
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/wallclock.h>
#include <pkgconf/devs_wallclock_st_m48t35a.h>

#include <cyg/hal/hal_io.h>
#include <cyg/hal/drv_api.h>
#include <cyg/infra/cyg_type.h>
#include <cyg/infra/diag.h>

#include <cyg/io/wallclock.hxx>
#include <cyg/io/wallclock/wallclock.inl>
#include <cyg/io/wallclock_m48t35a.h>

#ifndef HAL_WALLCLOCK_M48T35A_BASE
# error The platform HAL should define the base address of the M48T35A clock
#endif

// Allow the platform HAL to override the default accessor macros
#ifndef HAL_WALLCLOCK_M48T35A_STRIDE
# define HAL_WALLCLOCK_M48T35A_STRIDE 1
#endif
#ifndef HAL_WALLCLOCK_M48T35A_READ_UINT8
# define HAL_WALLCLOCK_M48T35A_READ_UINT8(_x_, _y_)     HAL_READ_UINT8((_x_), (_y_))
#endif
#ifndef HAL_WALLCLOCK_M48T35A_WRITE_UINT8
# define HAL_WALLCLOCK_M48T35A_WRITE_UINT8(_x_, _y_)     HAL_WRITE_UINT8((_x_), (_y_))
#endif

// The various registers
#define M48T35A_CONTROL                     (HAL_WALLCLOCK_M48T35A_BASE + (0x00 * HAL_WALLCLOCK_M48T35A_STRIDE))
#define M48T35A_SECONDS                     (HAL_WALLCLOCK_M48T35A_BASE + (0x01 * HAL_WALLCLOCK_M48T35A_STRIDE))
#define M48T35A_MINUTES                     (HAL_WALLCLOCK_M48T35A_BASE + (0x02 * HAL_WALLCLOCK_M48T35A_STRIDE))
#define M48T35A_HOURS                       (HAL_WALLCLOCK_M48T35A_BASE + (0x03 * HAL_WALLCLOCK_M48T35A_STRIDE))
#define M48T35A_DAY                         (HAL_WALLCLOCK_M48T35A_BASE + (0x04 * HAL_WALLCLOCK_M48T35A_STRIDE))
#define M48T35A_DATE                        (HAL_WALLCLOCK_M48T35A_BASE + (0x05 * HAL_WALLCLOCK_M48T35A_STRIDE))
#define M48T35A_MONTH                       (HAL_WALLCLOCK_M48T35A_BASE + (0x06 * HAL_WALLCLOCK_M48T35A_STRIDE))
#define M48T35A_YEAR                        (HAL_WALLCLOCK_M48T35A_BASE + (0x07 * HAL_WALLCLOCK_M48T35A_STRIDE))

#define M48T35A_CONTROL_WRITE_BIT           (0x01 << 7)
#define M48T35A_CONTROL_READ_BIT            (0x01 << 6)
#define M48T35A_CONTROL_SIGN_BIT            (0x01 << 5)
#define M48T35A_CONTROL_CALIBRATION_MASK    (0x1F << 0)
#define M48T35A_CONTROL_CALIBRATION_SHIFT   0
#define M48T35A_SECONDS_STOP_BIT            (0x01 << 6)
#define M48T35A_SECONDS_SECONDS_MASK        (0x7F << 0)
#define M48T35A_DAY_FREQUENCY_TEST_BIT      (0x01 << 6)
#define M48T35A_DAY_CENTURY_ENABLE_BIT      (0x01 << 5)
#define M48T35A_DAY_CENTURY_BIT             (0x01 << 4)

cyg_uint32
Cyg_WallClock::get_hw_seconds(void)
{
    cyg_uint32      year, month, date, hour, minutes, seconds;
    cyg_uint8       control_r, secs_r, mins_r, hours_r, day_r, date_r, month_r, year_r;
    cyg_uint32      result;
    
    // Disabling interrupts here is probably overkill, but
    // theoretically there could be a concurrent read and write
    cyg_drv_isr_lock();
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_CONTROL, control_r);
    control_r |= M48T35A_CONTROL_READ_BIT;
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_CONTROL, control_r);

    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_SECONDS, secs_r);
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_MINUTES, mins_r);
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_HOURS,   hours_r);
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_DAY,     day_r);
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_DATE,    date_r);
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_MONTH,   month_r);
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_YEAR,    year_r);
    
    control_r &= ~M48T35A_CONTROL_READ_BIT;
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_CONTROL, control_r);
    cyg_drv_isr_unlock();

    secs_r  &= M48T35A_SECONDS_SECONDS_MASK;
    seconds = (cyg_uint32) TO_DEC(secs_r);
    minutes = (cyg_uint32) TO_DEC(mins_r);
    hour    = (cyg_uint32) TO_DEC(hours_r);
    date    = (cyg_uint32) TO_DEC(date_r);
    month   = (cyg_uint32) TO_DEC(month_r);
    year    = 1900 + (cyg_uint32) TO_DEC(year_r);
#ifndef HAL_WALLCLOCK_M48T35A_NO_CENTURY_BIT    
    if (0 != (day_r & M48T35A_DAY_CENTURY_BIT)) {
        year += 100;
    }
#else
    if (year < 1970) {
        year += 100;
    }
#endif    

    result = _simple_mktime(year, month, date, hour, minutes, seconds);
    return result;
}

#ifdef CYGSEM_WALLCLOCK_SET_GET_MODE
void
Cyg_WallClock::set_hw_seconds(cyg_uint32 new_time)
{
    cyg_uint32      year, month, date, hour, minutes, seconds;
    cyg_uint8       control_r, secs_r, mins_r, hours_r, day_r, date_r, month_r, year_r;

    _simple_mkdate(new_time, &year, &month, &date, &hour, &minutes, &seconds);

#ifndef HAL_WALLCLOCK_M48T35A_NO_CENTURY_BIT    
    // A cyg_uint32 will take you from 1970 to 2106
    if (year > 2100) {
        day_r   = M48T35A_DAY_CENTURY_ENABLE_BIT;
        year   -= 2100;
    } else if (year > 2000) {
        day_r   = M48T35A_DAY_CENTURY_ENABLE_BIT | M48T35A_DAY_CENTURY_BIT;
        year   -= 2000;
    } else {
        day_r   = M48T35A_DAY_CENTURY_ENABLE_BIT;
        year   -= 1900;
    }
#else
    day_r   = 0;
    year   %= 100;
#endif
    
    // The actual day-of-the-week (0 == Sunday as per gmtime()) is not
    // filled in. It is not required get_hw_seconds() so would just
    // add code bloat for an unnecessary calculation.
    secs_r      = TO_BCD(seconds);          // Leave the stop bit clear
    mins_r      = TO_BCD(minutes);
    hours_r     = TO_BCD(hour);
    date_r      = TO_BCD(date);
    month_r     = TO_BCD(month);
    year_r      = TO_BCD(year);
        
    cyg_drv_isr_lock();
    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_CONTROL, control_r);
    control_r |= M48T35A_CONTROL_WRITE_BIT;
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_CONTROL, control_r);

    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_SECONDS, secs_r);
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_MINUTES, mins_r);
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_HOURS,   hours_r);
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_DAY,     day_r);
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_DATE,    date_r);
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_MONTH,   month_r);
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_YEAR,    year_r);
    
    control_r &= ~M48T35A_CONTROL_WRITE_BIT;
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_CONTROL, control_r);
    cyg_drv_isr_unlock();
}
#endif

// Called during system initialization by the wallclock constructor.
// First clear the READ and WRITE bits so that the clock is actually
// ticking, but leave the calibration in place. Then clear the stop bit.
// Since the WRIIE bit is clear this will not affect the seconds.
//
// If SET_GET_MODE is disabled then some other wallclock device drivers
// reset the wallclock to 1970. I don't do that here. The wallclock may
// have been initialized during manufacture, and if the application
// never updates it then disabling SET_GET_MODE makes sense (although
// it should have no real impact because of linker garbage collection).
void
Cyg_WallClock::init_hw_seconds(void)
{
    cyg_uint8   control;

    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_CONTROL, control);
    control &= ~(M48T35A_CONTROL_WRITE_BIT | M48T35A_CONTROL_READ_BIT);
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_CONTROL, control);

    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_SECONDS, 0);
}

// Additional functions specific to this device.
externC cyg_int32
cyg_wallclock_m48t35a_get_calibration(void)
{
    cyg_int32   result;
    cyg_uint8   control;

    HAL_WALLCLOCK_M48T35A_READ_UINT8(M48T35A_CONTROL, control);
    control &= ~(M48T35A_CONTROL_WRITE_BIT | M48T35A_CONTROL_READ_BIT);
    result = control & M48T35A_CONTROL_CALIBRATION_MASK;
    if (0 == (control & M48T35A_CONTROL_SIGN_BIT)) {
        result = 0 - result;
    }
    return result;
}

externC void
cyg_wallclock_m48t35a_set_calibration(cyg_int32 val)
{
    cyg_uint8   control;

    if (val < -31) {
        val = -31;
    } else if (val > 31) {
        val = 31;
    }

    if (val < 0) {
        control = (0 - val) & M48T35A_CONTROL_CALIBRATION_MASK;
    } else {
        control = M48T35A_CONTROL_SIGN_BIT | val;
    }
    HAL_WALLCLOCK_M48T35A_WRITE_UINT8(M48T35A_CONTROL, control);
}
