//==========================================================================
//
//      fatfs8.c
//
//      Test FAT filesystem
//
//==========================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
// Copyright (C) 2004, 2006, 2007 eCosCentric Limited
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           nickg
// Contributors:        nickg
// Date:                2007-03-28
// Purpose:             Test FAT system
// Description:         This test gets some performance figures from the
//                      FAT filesystem.
//                      
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>

#include CYGDAT_DEVS_DISK_CFG           // testing config defines

#include <cyg/kernel/ktypes.h>         // base kernel types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>
#include <stdlib.h>

#include <cyg/fileio/fileio.h>

#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output

#ifdef JNGPKG_USB_STACK
extern int usb_stack_init(void);
#endif

//==========================================================================

#define SHOW_RESULT( _fn, _res )                                                                \
{                                                                                               \
    diag_printf("<FAIL>: " #_fn "() returned %ld %s\n", (long)_res, _res<0?strerror(errno):"");        \
}

//==========================================================================

typedef struct
{
    cyg_int64           ticks;
    cyg_int32           halticks;
} timestamp;

typedef struct
{
    timestamp           start;
    timestamp           end;
    cyg_int64           interval;       // In HAL ticks
    cyg_int64           us_interval;    // Interval in microseconds
} timing;

static cyg_int64        ticks_overhead = 0;
static cyg_int32        us_per_haltick = 0;
static cyg_int32        halticks_per_us = 0;

static cyg_int64        rtc_resolution[] = CYGNUM_KERNEL_COUNTERS_RTC_RESOLUTION;
static cyg_int64        rtc_period = CYGNUM_KERNEL_COUNTERS_RTC_PERIOD;
static cyg_int32        ticks_per_second;

//==========================================================================

static char date_string[] = __DATE__;
static char time_string[] = __TIME__;

cyg_uint32 timehash;

//==========================================================================

static void wait_for_tick( void )
{
    cyg_tick_count_t now = cyg_current_time();

    while( cyg_current_time() == now )
        continue;
}

static void get_timestamp( timestamp *ts )
{
    ts->ticks = cyg_current_time();
    HAL_CLOCK_READ( &ts->halticks );
}

static cyg_int64 ticks_to_us( cyg_int64 ticks )
{
    cyg_int64 us;
    
    if( us_per_haltick != 0 )
        us = ticks * us_per_haltick;
    else
        us = ticks / halticks_per_us;

    return us;
}

static void calculate_interval( timing *t )
{
    t->interval = t->end.halticks - t->start.halticks;

    t->interval += (t->end.ticks - t->start.ticks) * rtc_period;

    t->interval -= ticks_overhead;
    
    t->us_interval = ticks_to_us( t->interval );
}

static void init_timing( void )
{
    timing t;

    ticks_per_second = (1000000000ULL*rtc_resolution[1])/rtc_resolution[0];
    us_per_haltick = 1000000/(rtc_period * rtc_resolution[1]);
    halticks_per_us = (rtc_period * rtc_resolution[1])/1000000; 
    
    wait_for_tick();

    get_timestamp( &t.start );
    get_timestamp( &t.end );

    calculate_interval( &t );

    ticks_overhead = t.interval;

    diag_printf("Timing overhead %lld ticks\n", ticks_overhead );
    
}

//==========================================================================

#ifndef CYGPKG_LIBC_STRING

char *strcat( char *s1, const char *s2 )
{
    char *s = s1;
    while( *s1 ) s1++;
    while( (*s1++ = *s2++) != 0);
    return s;
}

#endif

//==========================================================================

static void createfile( char *name, size_t asize, size_t block_size )
{
    size_t size = asize;
    cyg_uint8 *buf;
    cyg_uint32 *b;
    int fd;
    ssize_t wrote;
    int i;
    int err;
    timing opent, writet;
        
    diag_printf("\n--------------------------------------------------------------------\n");
    diag_printf("<INFO>: create file %s size %ld block_size %ld\n",name, size, block_size);

    buf = malloc( block_size );
    if( buf == NULL )
    {
        SHOW_RESULT( malloc, -1 );
        return;
    }
    b = (cyg_uint32 *)buf;
    
    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    i = 0;
    while( i < block_size/sizeof(cyg_uint32) )
    {
        b[i] = i;
        i++;
        b[i++] = 0;
        b[i++] = timehash;
        b[i++] = 0x12345678;
    }

    
    get_timestamp( &opent.start );        
    fd = open( name, O_WRONLY|O_CREAT );
    get_timestamp( &opent.end );        
    if( fd < 0 ) SHOW_RESULT( open, fd );

    get_timestamp( &writet.start );    
    while( size > 0 )
    {
        ssize_t len = size;
        if ( len > block_size ) len = block_size;

        b[1] = asize-size;
        b[block_size/sizeof(cyg_uint32)-3] = asize-size;
        wrote = write( fd, buf, len );
        if( wrote != len )
        {
            SHOW_RESULT( write, wrote );
            break;
        }

        size -= wrote;

        if( (size % (128*1024)) == 0 )
        {
//            diag_printf(".");
//            diag_printf("%d\n", size );

            static int x = 0;
            char *spin = "|/-\\";
            diag_printf("\r%c ",spin[(x++)&3]);

            if( (size % (1024*1024)) == 0 )
                diag_printf("%4ld", size/(1024*1024) );
        }
    }
    get_timestamp( &writet.end );        

    diag_printf("\n");
    free( buf );
    
    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );

    calculate_interval( &opent );
    calculate_interval( &writet );

    asize -= size;
    
    diag_printf("CreateFile Results: %s size %ld\n", name, asize);
    diag_printf("         Open        %8d.%03d ms\n", (int)(opent.us_interval/1000), (int)(opent.us_interval%1000) );
    diag_printf("        Write        %8d.%03d ms\n", (int)(writet.us_interval/1000), (int)(writet.us_interval%1000) );
    diag_printf("    Data Rate          %10lld KiB/s\n", (1000000LL*asize)/1024/writet.us_interval);
}

//==========================================================================

static void checkfile( char *name, size_t block_size )
{
    cyg_uint8 *buf;
    cyg_uint32 *b;
    int fd;
    ssize_t done;
    int err;
    off_t pos = 0;
    timing opent, readt;
    
    diag_printf("\n--------------------------------------------------------------------\n");
    diag_printf("<INFO>: check file %s\n",name);
    
    buf = malloc( block_size );
    if( buf == NULL )
    {
        SHOW_RESULT( malloc, -1 );
        return;
    }
    b = (cyg_uint32 *)buf;
    
    err = access( name, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    get_timestamp( &opent.start );
    fd = open( name, O_RDONLY );
    get_timestamp( &opent.end );            
    if( fd < 0 ) SHOW_RESULT( open, fd );

    get_timestamp( &readt.start );            
    for(;;)
    {
        done = read( fd, buf, block_size );
        if( done < 0 ) SHOW_RESULT( read, done );

        if( done == 0 ) break;

        // Check that this is the block we wrote
        if( b[0] != 0 )
            diag_printf("b[%d] != %08x\n", 0, 0 );

        if( b[1] != pos )
            diag_printf("b[%d] != %08x\n", 1, (unsigned)pos );

        if( b[2] != timehash )
            diag_printf("b[%d] != %08x\n", 2, timehash );

        if( b[3] != 0x12345678 )
            diag_printf("b[%d] != %08x\n", 3, 0x12345678 );

        if( b[(block_size/sizeof(cyg_uint32))-4] != (block_size/sizeof(cyg_uint32)-4) )
            diag_printf("b[%lu](%08x) != %08x\n", block_size/sizeof(cyg_uint32)-4,
                                                  b[(block_size/sizeof(cyg_uint32))-4],
                                                  (unsigned)(block_size/sizeof(cyg_uint32)-4) );

        if( b[block_size/sizeof(cyg_uint32)-3] != pos )
            diag_printf("b[%lu] != %08x\n", block_size/sizeof(cyg_uint32)-3, (unsigned)pos );

        if( b[block_size/sizeof(cyg_uint32)-2] != timehash )
            diag_printf("b[%lu] != %08x\n", block_size/sizeof(cyg_uint32)-2, timehash );

        if( b[block_size/sizeof(cyg_uint32)-1] != 0x12345678 )
            diag_printf("b[%lu] != %08x\n", block_size/sizeof(cyg_uint32)-1, 0x12345678 );
        
        pos += done;
    }
    get_timestamp( &readt.end );            

    free( buf );
    
    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );

    calculate_interval( &opent );
    calculate_interval( &readt );

    diag_printf("CheckFile Results: %s size %ld\n", name, pos);
    diag_printf("         Open        %8d.%03d ms\n", (int)(opent.us_interval/1000), (int)(opent.us_interval%1000) );
    diag_printf("         Read        %8d.%03d ms\n", (int)(readt.us_interval/1000), (int)(readt.us_interval%1000) );
    diag_printf("    Data Rate          %10lld KiB/s\n", (1000000LL*pos)/1024/readt.us_interval);
}

//==========================================================================

void checkcwd( const char *cwd )
{
    static char cwdbuf[PATH_MAX];
    char *ret;

    ret = getcwd( cwdbuf, sizeof(cwdbuf));
    if( ret == NULL ) SHOW_RESULT( getcwd, ret );    

    if( strcmp( cwdbuf, cwd ) != 0 )
    {
        diag_printf( "cwdbuf %s cwd %s\n",cwdbuf, cwd );
        CYG_TEST_FAIL( "Current directory mismatch");
    }
}

//==========================================================================
// main

void fatfs8_main( CYG_ADDRESS id )
{
    int err;
    int i;
    timing mountt, umountt;
    
#if defined(CYGSEM_FILEIO_INFO_DISK_USAGE)
    struct cyg_fs_disk_usage usage;
#endif
    
#ifdef JNGPKG_USB_STACK
    err = usb_stack_init();
    cyg_thread_delay( 1000 );
#endif
    
    CYG_TEST_INIT();

    init_timing();

    for( i = 0 ; date_string[i] != 0 ; i++ )
        timehash = (timehash<<1) ^ date_string[i];
    for( i = 0 ; time_string[i] != 0 ; i++ )
        timehash = (timehash<<1) ^ time_string[i];
    
    // --------------------------------------------------------------

    get_timestamp( &mountt.start );
    err = mount( CYGDAT_DEVS_DISK_TEST_DEVICE, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, "fatfs" );    
//    err = mount( CYGDAT_DEVS_DISK_TEST_DEVICE, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, "fatfs:sync=write" );    
//    err = mount( CYGDAT_DEVS_DISK_TEST_DEVICE, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, "fatfs:sync=close" );    
//    err = mount( CYGDAT_DEVS_DISK_TEST_DEVICE, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, "fatfs:sync=data" );    
    get_timestamp( &mountt.end );
    if( err < 0 ) {
        SHOW_RESULT( mount, err );
        CYG_TEST_FAIL_FINISH("fatfs8");
    }
    
    calculate_interval( &mountt );
    diag_printf("        Mount        %8d.%03d ms\n", (int)(mountt.us_interval/1000), (int)(mountt.us_interval%1000) );
    
    err = access( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    if( err < 0 && errno == EACCES )
    {
        diag_printf("<INFO>: create %s\n",
                    CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY);
        err = mkdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY, 0 );
        if( err < 0 ) SHOW_RESULT( mkdir, err );
    }

    err = chdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY );    
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY );
    
    // --------------------------------------------------------------
#if defined(CYGSEM_FILEIO_INFO_DISK_USAGE)
    err = cyg_fs_getinfo(CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, FS_INFO_DISK_USAGE,
                         &usage, sizeof(usage));
    if( err < 0 ) SHOW_RESULT( cyg_fs_getinfo, err );
    diag_printf("<INFO>: total size: %6lld blocks, %10lld bytes\n",
               usage.total_blocks, usage.total_blocks * usage.block_size);
    diag_printf("<INFO>: free size:  %6lld blocks, %10lld bytes\n",
               usage.free_blocks, usage.free_blocks * usage.block_size);
    diag_printf("<INFO>: block size: %6u bytes\n", usage.block_size);
#endif
    // --------------------------------------------------------------
    
#if 1
    err = unlink( "fatfs8.0" );
    createfile( "fatfs8.0", 30*1024*1024, 8*1024 );
    sync();    
#endif

#if 0
    err = unlink( "fatfs8.1" );
    createfile( "fatfs8.1", 300*1024*1024, 8*1024 );
    sync();
#endif

#if 1
    err = unlink( "fatfs8.2" );
    createfile( "fatfs8.2", 20*1024*1024, 8*1024 );
    sync();
#endif

#if 1
    err = unlink( "fatfs8.0" );    
    err = unlink( "fatfs8.1" );
    createfile( "fatfs8.4", 40*1024*1024, 8*1024 );
    sync();
#endif

    err = unlink( "fatfs8.0" );        
    err = unlink( "fatfs8.1" );        
    err = unlink( "fatfs8.2" );
    err = unlink( "fatfs8.3" );            
    err = unlink( "fatfs8.4" );        

    
    // --------------------------------------------------------------
#if defined(CYGSEM_FILEIO_INFO_DISK_USAGE)
    err = cyg_fs_getinfo(CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, FS_INFO_DISK_USAGE,
                         &usage, sizeof(usage));
    if( err < 0 ) SHOW_RESULT( cyg_fs_getinfo, err );
    diag_printf("<INFO>: total size: %6lld blocks, %10lld bytes\n",
               usage.total_blocks, usage.total_blocks * usage.block_size);
    diag_printf("<INFO>: free size:  %6lld blocks, %10lld bytes\n",
               usage.free_blocks, usage.free_blocks * usage.block_size);
    diag_printf("<INFO>: block size: %6u bytes\n", usage.block_size);
#endif
    // --------------------------------------------------------------

    diag_printf("<INFO>: cd /\n");    
    err = chdir( "/" );
    if( err == 0 )
    {
        checkcwd( "/" );
    
        diag_printf("<INFO>: umount %s\n",CYGDAT_DEVS_DISK_TEST_MOUNTPOINT);
        get_timestamp( &umountt.start );        
        err = umount( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );
        get_timestamp( &umountt.end );        
        if( err < 0 ) SHOW_RESULT( umount, err );

        calculate_interval( &umountt );
        diag_printf("      Umount        %8d.%03d ms\n", (int)(umountt.us_interval/1000), (int)(umountt.us_interval%1000) );
        
    }

    diag_printf("<INFO>: syncing\n");    
    sync();
        
    CYG_TEST_PASS_FINISH("fatfs8");
}

// -------------------------------------------------------------------------

#include <cyg/kernel/kapi.h>

static cyg_handle_t thread_handle;
static cyg_thread thread;
static char stack[CYGNUM_HAL_STACK_SIZE_TYPICAL+4*1024];
//static char stack[64*1024];

externC void
cyg_start( void )
{
    cyg_thread_create(3,                // Priority - just a number
                      fatfs8_main,      // entry
                      0,                // index
                      0,                // no name
                      &stack[0],        // Stack
                      sizeof(stack),    // Size
                      &thread_handle,   // Handle
                      &thread           // Thread data structure
        );
    cyg_thread_resume(thread_handle);

    cyg_scheduler_start();
}

// -------------------------------------------------------------------------
// EOF fatfs8.c
