//==========================================================================
//
//      jffs2-fseek1.c
//
//      Test fseek on the JFFS2 filesystem
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2004, 2006 Free Software Foundation, Inc.                  
// Copyright (C) 2004,2006 eCosCentric Limited                              
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           asl
// Contributors:        jlarmour
// Date:                2004-03-29
// Purpose:             Test fseek on a filesystem
// Description:         This tests fseek operation on the JFFS2 filesystem.
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/isoinfra.h>
#if !CYGINT_ISO_MAIN_STARTUP
# define NA_MSG "No main() startup"
#elif !defined(CYGINT_ISO_STDIO_FILETYPES)
# define NA_MSG "No stdio filetypes"
#elif !defined(CYGINT_ISO_STDIO_STREAMS)
# define NA_MSG "No stdio streams"
#elif !defined(CYGINT_ISO_STDIO_FILEOPS)
# define NA_MSG "No stdio file operation functions"
#elif !defined(CYGINT_ISO_STDIO_FILEACCESS)
# define NA_MSG "No stdio file access functions"
#elif !defined(CYGINT_ISO_STRING_MEMFUNCS)
# define NA_MSG "No string mem* functions"
#endif

#ifdef NA_MSG
#include <cyg/infra/testcase.h>
void cyg_start(void)
{
    CYG_TEST_INIT();
    CYG_TEST_NA(NA_MSG);
    CYG_TEST_FINISH("Done");
}
#else


#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include <cyg/io/io.h>
#include <cyg/io/config_keys.h>
#include <cyg/fileio/fileio.h>
#include <cyg/io/flash.h>

#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output
//==========================================================================
// Mount details

#define stringify2(_x_) #_x_
#define stringify(_x_) stringify2(_x_)

#if defined(CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1)
# define JFFS2_TEST_DEV CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1
#elif defined(CYGFUN_IO_FLASH_BLOCK_FROM_FIS)
# define JFFS2_TEST_DEV "/dev/flash/fis/jffs2test"
#else
// fall back to using a user set area in the first device (only)
# define JFFS2_TEST_DEV "/dev/flash/0/" stringify(CYGNUM_FS_JFFS2_TEST_OFFSET) "," stringify(CYGNUM_FS_JFFS2_TEST_LENGTH)
// But just in case, we check it's blank.
# define CHECK_FLASH_BLANK
#endif

//==========================================================================

#define SHOW_RESULT( _fn, _res ) \
diag_printf("FAIL: " #_fn "() returned %d %s\n", _res, _res<0?strerror(errno):"");

//==========================================================================
// prepare_device
//
// This does a quick check for existence of the underlying device.
// It also checks the region is blank if required.

void prepare_device( const char *dev )
{
    Cyg_ErrNo err;
    cyg_io_handle_t h;
    cyg_io_flash_getconfig_devsize_t devsize;
#ifdef CYGHWR_IO_FLASH_BLOCK_LOCKING
    cyg_io_flash_getconfig_unlock_t unlockconf;
#endif
    cyg_uint32 conf_size = sizeof(devsize);
    cyg_bool isblank = false;

    err = cyg_io_lookup( dev, &h );
    if (err == -ENOENT)
    {
        CYG_TEST_NA("Flash test device not found, test environment not set up by user.");
    }
    else if (err)
    {
        diag_printf("cyg_io_lookup returned: %d\n", err );
        CYG_TEST_FAIL_FINISH( "Test device lookup failed" );
    }

    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_DEVSIZE, &devsize,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config failed");
    }

    // this check is really for sanity with the CHECK_FLASH_BLANK code, but
    // we may as well always do it.
    if ((devsize.dev_size % 128) != 0)
    {
        diag_printf("device size == 0x%08x\n",devsize.dev_size);
        CYG_TEST_FAIL_FINISH("prepare_device: bizarre size returned - not multiple of 128");
    }
    
#ifdef CHECK_FLASH_BLANK
    {
        cyg_uint32 pos;
        cyg_uint8 buf[128];

        for (pos = 0; pos < devsize.dev_size; pos += sizeof(buf) )
        {
            cyg_uint32 i;
            cyg_uint32 len = sizeof(buf);
            cyg_uint32 *buf32 = (cyg_uint32 *)&buf[0];

            err = cyg_io_bread(h, &buf[0], &len, pos );
            if (err || (len < sizeof(buf)))
            {
                diag_printf("Read failure at %u, err=%d, new len=%u\n", pos, err, len );
                CYG_TEST_FAIL_FINISH("prepare_device: read failed");
            }
            for (i=0; i<(sizeof(buf)/4); i++)
            {
                if (buf32[i] != 0xFFFFFFFF)
                {
                    CYG_TEST_FAIL_FINISH("Supplied test device not blank! Not erasing.");
                }
            } // for
        } // for
        isblank = true;
    }
#endif    

#ifdef CYGHWR_IO_FLASH_BLOCK_LOCKING
    // This device might need unlocking before it can be written to.
    conf_size = sizeof(unlockconf);

    // unlock the whole "device"
    unlockconf.offset = 0;
    unlockconf.len = devsize.dev_size;
    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_UNLOCK, &unlockconf,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config unlock failed");
    }
    if (unlockconf.flasherr != CYG_FLASH_ERR_OK)
    {
        diag_printf("flash error @0x%08x:%d:%s\n", unlockconf.err_address,
                    unlockconf.flasherr,
                    cyg_flash_errmsg(unlockconf.flasherr) );
        CYG_TEST_FAIL_FINISH("prepare_device: unlock failed");
    }
#endif // ifdef CYGHWR_IO_FLASH_BLOCK_LOCKING

    if (!isblank)
    {
        // erase - this is our test device and we need it to be clean, rather
        // than containing a potentially corrupt FS.
        cyg_io_flash_getconfig_erase_t eraseconf;
        conf_size = sizeof(eraseconf);
        
        // erase the whole "device"
        eraseconf.offset = 0;
        eraseconf.len = devsize.dev_size;
        CYG_TEST_INFO("Erasing Flash test region");
        err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_ERASE, &eraseconf,
                                 &conf_size );
        if (err)
        {
            CYG_TEST_FAIL_FINISH("prepare_device: get_config for erase failed");
        }
        if (eraseconf.flasherr != CYG_FLASH_ERR_OK)
        {
            diag_printf("flash error @0x%08x:%d:%s\n", eraseconf.err_address,
                        eraseconf.flasherr,
                        cyg_flash_errmsg(eraseconf.flasherr) );
            CYG_TEST_FAIL_FINISH("prepare_device: erase failed");
        }
        CYG_TEST_INFO("Flash test region erase complete");
    }

    // flash block layer is known to need close when last instance stops using
    // handle.
    err = cyg_io_set_config( h, CYG_IO_SET_CONFIG_CLOSE, NULL, NULL );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: CLOSE via cyg_io_set_config failed");
    }
}

//==========================================================================

char buf[1024];
char buf1[1024];

//==========================================================================
// main

int main( int argc, char **argv )
{
    int err;
    FILE *stream;
    long pos;
    int i;
    
    CYG_TEST_INIT();

    prepare_device( JFFS2_TEST_DEV );

    // --------------------------------------------------------------

    CYG_TEST_INFO("mount /");    
    err = mount( JFFS2_TEST_DEV, "/", "jffs2" );

    if( err < 0 ) SHOW_RESULT( mount, err );    
    
    CYG_TEST_INFO("creating /fseek");    
    stream = fopen("/fseek","w+");
    if (!stream) {
      diag_printf("FAIL: fopen() returned NULL, %s\n", strerror(errno));
      CYG_TEST_FINISH("done");          \
    }

    /* Write a buffer full of cyclic numbers */
    for (i = 0; i < sizeof(buf); i++) {
      buf[i] = i % 256;
    }
    
    CYG_TEST_INFO("writing test pattern");    
    err=fwrite(buf,sizeof(buf), 1, stream);
    if ( err < 0 ) SHOW_RESULT( fwrite, err );
    
    /* The current position should be the same size as the buffer */
    pos = ftell(stream);
    
    if (pos < 0) SHOW_RESULT( ftell, (int)pos );
    if (pos != sizeof(buf))
      diag_printf("<FAIL>: ftell is not telling the truth.");
    
    CYG_TEST_INFO("fseek()ing to beginning and writing");    

    /* Seek back to the beginning of the file */
    err = fseek(stream, 0, SEEK_SET);
    if ( err < 0 ) SHOW_RESULT( fseek, err );

    pos = ftell(stream);
    
    if (pos < 0) SHOW_RESULT( ftell, (int)pos );
    if (pos != 0) CYG_TEST_FAIL("ftell is not telling the truth");

    /* Write 4 zeros to the beginning of the file */
    for (i = 0; i < 4; i++) {
      buf[i] = 0;
    }

    err = fwrite(buf, 4, 1, stream);
    if ( err < 0 ) SHOW_RESULT( fwrite, err );

    /* Check the pointer is at 4 */
    pos = ftell(stream);
    
    if (pos < 0) SHOW_RESULT( ftell, (int)pos );
    if (pos != 4)  CYG_TEST_FAIL("ftell is not telling the truth");

    CYG_TEST_INFO("closing file");
    
    /* Close the file, open it up again and read it back */
    err = fclose(stream);
    if (err != 0) SHOW_RESULT( fclose, err );

    CYG_TEST_INFO("open file /fseek");
    stream = fopen("/fseek", "r+");
    if (!stream) {
      diag_printf("<FAIL>: fopen() returned NULL, %s\n", strerror(errno));
    }

    err = fread(buf1,sizeof(buf1),1, stream);
    if (err != 1) SHOW_RESULT( fread, err );
    
    CYG_TEST_INFO("Comparing contents");
    if (memcmp(buf, buf1, sizeof(buf1))) {
      CYG_TEST_FAIL("File contents inconsistent");
    }
    
    CYG_TEST_INFO("closing file");

    err = fclose(stream);
    if (err != 0) SHOW_RESULT( fclose, err );

    CYG_TEST_INFO("open file /fseek");
    stream = fopen("/fseek", "r+");
    if (!stream) {
      diag_printf("<FAIL>: fopen() returned NULL, %s\n", strerror(errno));
    }

    CYG_TEST_INFO("fseek()ing past the end to create a hole");
    /* Seek 1K after the end of the file */
    err = fseek(stream, sizeof(buf), SEEK_END);
    if ( err < 0 ) SHOW_RESULT( fseek, err );

    pos = ftell(stream);
    
    if (pos < 0) SHOW_RESULT( ftell, (int)pos );
    if (pos != (2*sizeof(buf))) CYG_TEST_FAIL("ftell is not telling the truth");
    
    CYG_TEST_INFO("writing test pattern");    
    err=fwrite(buf,sizeof(buf), 1, stream);
    if ( err < 0 ) SHOW_RESULT( fwrite, err );
    
    pos = ftell(stream);
    
    if (pos < 0) SHOW_RESULT( ftell, (int)pos );
    if (pos != (3*sizeof(buf))) CYG_TEST_FAIL("ftell is not telling the truth");

    CYG_TEST_INFO("closing file");
    err = fclose(stream);
    if (err != 0) SHOW_RESULT( fclose, err );

    CYG_TEST_INFO("open file /fseek");
    stream = fopen("/fseek", "r+");
    if (!stream) {
      diag_printf("<FAIL>: fopen() returned NULL, %s\n", strerror(errno));
    }
    
    err = fread(buf1,sizeof(buf1),1, stream);
    if (err != 1) SHOW_RESULT( fread, err );
    
    CYG_TEST_INFO("Comparing contents");
    if (memcmp(buf, buf1, sizeof(buf1))) {
      CYG_TEST_FAIL("File contents inconsistent");
    }

    err = fread(buf1,sizeof(buf1),1, stream);
    if (err != 1) SHOW_RESULT( fread, err );
    
    for (i = 0; i< sizeof(buf); i++) {
      if (buf1[i] != 0)
        CYG_TEST_FAIL("Hole does not contain zeros");
    }
    
    err = fread(buf1,sizeof(buf1),1, stream);
    if (err != 1) SHOW_RESULT( fread, err );
    
    if (memcmp(buf, buf1, sizeof(buf1))) {
      CYG_TEST_FAIL("File contents inconsistent");
    }

    CYG_TEST_INFO("closing file");

    /* Close the file */
    err = fclose(stream);
    if (err != 0) SHOW_RESULT( fclose, err );
    
    CYG_TEST_INFO("umount /");    
    err = umount( "/" );
    if( err < 0 ) SHOW_RESULT( umount, err );    
    
    CYG_TEST_PASS_FINISH("fseek1");
}

#endif // ifndef NA_MSG

// EOF jffs2-fseek1.c
