//==========================================================================
//
//      jffs2_3.c
//
//      Test garbage collection on a filesystem
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2005, 2007 Free Software Foundation, Inc.                  
// Copyright (C) 2007 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           asl
// Contributors:        asl, jlarmour
// Date:                2005-01-16
// Purpose:             Test garbage collect on a filesystem
// Description:         This test creates and deletes files in order
//                      to test the garbage collection code.
//                      
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/system.h>
#include <pkgconf/io_flash.h>
#include <pkgconf/isoinfra.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include <cyg/fileio/fileio.h>
#include <cyg/io/io.h>
#include <cyg/io/config_keys.h>
#include <cyg/io/flash.h>

#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output

#include <pkgconf/fs_jffs2.h>	// Address of JFFS2

//==========================================================================

#define stringify2(_x_) #_x_
#define stringify(_x_) stringify2(_x_)

#if defined(CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1)
# define JFFS2_TEST_DEV CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1
#elif defined(CYGFUN_IO_FLASH_BLOCK_FROM_FIS)
# define JFFS2_TEST_DEV "/dev/flash/fis/jffs2test"
#else
// fall back to using a user set area in the first device (only)
# define JFFS2_TEST_DEV "/dev/flash/0/" stringify(CYGNUM_FS_JFFS2_TEST_OFFSET) "," stringify(CYGNUM_FS_JFFS2_TEST_LENGTH)
// But just in case, we check it's blank.
# define CHECK_FLASH_BLANK
#endif

//==========================================================================

#ifdef CYGPKG_CRC
# include <cyg/crc/crc.h>
# define jcksum32(_b,_s) cyg_posix_crc32((_b), (_s))
#else

// Simple version - convoluted checksum, not crc
static cyg_uint32
jcksum32( unsigned char *s, int len )
{
    cyg_uint32 ret = 0;

    for (;len; len--)
    {
        ret += (s[len-1] << ((ret+len) % 24)) ^ ret;
    }
}

#endif

//==========================================================================
// Substitute if rand is not around
#if !CYGINT_ISO_RAND
int seed;

#define srand(i)                                \
    CYG_MACRO_START                             \
    seed = (int)(i);                            \
    CYG_MACRO_END

static inline int rand( void )
{
    seed = (seed * 1103515245) + 12345;
    return seed;
}
#endif

//==========================================================================

#define ITERATIONS 2001
#define NELEM(_x_) (sizeof(_x_)/sizeof(*(_x_)))

#define SHOW_RESULT( _fn, _res ) \
diag_printf("FAIL:<" #_fn "() returned %d %s>\n", _res, \
            _res<0?strerror(errno):"");

//==========================================================================
// prepare_device
//
// This does a quick check for existence of the underlying device.
// It also checks the region is blank if required.

void prepare_device( const char *dev )
{
    Cyg_ErrNo err;
    cyg_io_handle_t h;
    cyg_io_flash_getconfig_devsize_t devsize;
#if defined(CYGHWR_IO_FLASH_BLOCK_LOCKING)
    cyg_io_flash_getconfig_unlock_t unlockconf;
#endif
    cyg_uint32 conf_size = sizeof(devsize);
    cyg_bool isblank = false;

    err = cyg_io_lookup( dev, &h );
    if (err == -ENOENT)
    {
        CYG_TEST_NA("Flash test device not found, test environment not set up by user.");
    }
    else if (err)
    {
        diag_printf("cyg_io_lookup returned: %d\n", err );
        CYG_TEST_FAIL_FINISH( "Test device lookup failed" );
    }

    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_DEVSIZE, &devsize,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config failed");
    }

    // this check is really for sanity with the CHECK_FLASH_BLANK code, but
    // we may as well always do it.
    if ((devsize.dev_size % 128) != 0)
    {
        diag_printf("device size == 0x%08x\n",(unsigned)devsize.dev_size);
        CYG_TEST_FAIL_FINISH("prepare_device: bizarre size returned - not multiple of 128");
    }
    
#ifdef CHECK_FLASH_BLANK
    {
        cyg_uint32 pos;
        cyg_uint8 buf[128];

        for (pos = 0; pos < devsize.dev_size; pos += sizeof(buf) )
        {
            cyg_uint32 i;
            cyg_uint32 len = sizeof(buf);
            cyg_uint32 *buf32 = (cyg_uint32 *)&buf[0];

            err = cyg_io_bread(h, &buf[0], &len, pos );
            if (err || (len < sizeof(buf)))
            {
                diag_printf("Read failure at %u, err=%d, new len=%u\n", pos, err, len );
                CYG_TEST_FAIL_FINISH("prepare_device: read failed");
            }
            for (i=0; i<(sizeof(buf)/4); i++)
            {
                if (buf32[i] != 0xFFFFFFFF)
                {
                    CYG_TEST_FAIL_FINISH("Supplied test device not blank! Not erasing.");
                }
            } // for
        } // for
        isblank = true;
    }
#endif    

#if defined(CYGHWR_IO_FLASH_BLOCK_LOCKING)
    // This device might need unlocking before it can be written to.
    conf_size = sizeof(unlockconf);

    // unlock the whole "device"
    unlockconf.offset = 0;
    unlockconf.len = devsize.dev_size;
    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_UNLOCK, &unlockconf,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config unlock failed");
    }
    if (unlockconf.flasherr != CYG_FLASH_ERR_OK)
    {
        diag_printf("flash error @0x%08x:%d:%s\n", unlockconf.err_address,
                    unlockconf.flasherr,
                    cyg_flash_errmsg(unlockconf.flasherr) );
        CYG_TEST_FAIL_FINISH("prepare_device: unlock failed");
    }
#endif // if defined(CYGHWR_IO_FLASH_BLOCK_LOCKING)

    if (!isblank)
    {
        // erase - this is our test device and we need it to be clean, rather
        // than containing a potentially corrupt FS.
        cyg_io_flash_getconfig_erase_t eraseconf;
        conf_size = sizeof(eraseconf);
        
        // erase the whole "device"
        eraseconf.offset = 0;
        eraseconf.len = devsize.dev_size;
        CYG_TEST_INFO("Erasing Flash test region");
        err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_ERASE, &eraseconf,
                                 &conf_size );
        if (err)
        {
            CYG_TEST_FAIL_FINISH("prepare_device: get_config for erase failed");
        }
        if (eraseconf.flasherr != CYG_FLASH_ERR_OK)
        {
            diag_printf("flash error @0x%08x:%d:%s\n", eraseconf.err_address,
                        eraseconf.flasherr,
                        cyg_flash_errmsg(eraseconf.flasherr) );
            CYG_TEST_FAIL_FINISH("prepare_device: erase failed");
        }
        CYG_TEST_INFO("Flash test region erase complete");
    }

    // flash block layer is known to need close when last instance stops using
    // handle.
    err = cyg_io_set_config( h, CYG_IO_SET_CONFIG_CLOSE, NULL, NULL );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: CLOSE via cyg_io_set_config failed");
    }
}

//==========================================================================
static void
num_to_testfile( int i, char *buf, int len)
{
    int j;

    buf[0] = 't'; buf[1] = 'e'; buf[2] = 's'; buf[3] = 't';

    // sprintf is easier but is not always around
    // the number digits will be in the other order - who cares
    for ( j=4; ; j++ )
    {
        int shift = 4*(j-4);
        int thisnum = ( i >> shift ) & 0xf;

        // check if we've gone past the start of the number
        if (!thisnum && (1 << (shift+1)) > i)
            break;

        if (j == len)
        {
            CYG_TEST_FAIL_FINISH("Exceeded file name array bounds");
        }

        if (thisnum > 9)
            buf[j] = 'a'+thisnum-10;
        else
            buf[j] = '0'+thisnum;

    }
    buf[j] = '\0';
}


//==========================================================================

// file creation, deletion and testing functions

static void create_file(int i)
{
  static cyg_int32 buffer[1020]; // static because very big for stack
  char name[16];
  cyg_uint32 j;
  int fd, err;

  num_to_testfile(i, name, sizeof(name));

  fd = creat(name, S_IRWXU);
  if (fd == -1) SHOW_RESULT( creat, fd );
  
  for (j=1; j < NELEM(buffer); j++) {
    buffer[j] = rand();
  }
  
  buffer[0] = 0;
  buffer[0] = jcksum32((unsigned char *)buffer, sizeof(buffer));
  
  err = write(fd, buffer, sizeof(buffer));
  if (err == -1) SHOW_RESULT( write, err );
  
  err = close(fd);
  if (err == -1) SHOW_RESULT( close, err );
}

static void delete_file(int i)
{
  char name[16];
  int err;

  num_to_testfile(i, name, sizeof(name));

  err = unlink(name);
  if (err == -1) SHOW_RESULT( unlink, err );
}

static void check_file(int i)
{
  char name[16];
  int err, fd;
  cyg_int32 buffer[1020];
  cyg_uint32 crc;
  
  num_to_testfile(i, name, sizeof(name));

  fd = open(name, O_RDONLY);
  if (fd == -1) SHOW_RESULT( open, fd );
  
  err = read(fd, buffer, sizeof(buffer));
  if (err == -1) SHOW_RESULT( read, err );
  
  crc = buffer[0];
  buffer[0] = 0;
  
  if (crc != jcksum32((unsigned char *)buffer, sizeof(buffer))) {
    CYG_TEST_FAIL("File corrupt");
  }

  err = close(fd);
  if (err == -1) SHOW_RESULT( read, err );
}


//==========================================================================
// main

static void
gc_test( void )
{
    int err, iteration;
    struct mallinfo minfo;

    // --------------------------------------------------------------

    prepare_device ( JFFS2_TEST_DEV );

    CYG_TEST_INFO("mount /");    
    err = mount( JFFS2_TEST_DEV, "/", "jffs2" );
    if( err < 0 ) SHOW_RESULT( mount, err );    
    
    chdir ("/");
    
    create_file(0);
    for (iteration = 0; iteration < ITERATIONS; iteration++)
    {
      if (!(iteration % 500)) {
        minfo = mallinfo();
        diag_printf("INFO:<Iteration %07d fordblks = %7d>\n", 
                    iteration, minfo.fordblks);
      }
      create_file(iteration+1);
      check_file(iteration);
      delete_file(iteration);
      check_file(iteration+1);
#ifdef CYGOPT_FS_JFFS2_GC_THREAD
      if (!(iteration % 10)) {
          /* Occasionally let GC thread have a chance to run,
           * to exercise that code a bit too */
          cyg_thread_delay(1);
      }
#endif
    }
    
    CYG_TEST_INFO("umount /");
    err = umount( "/" );
    if( err < 0 ) SHOW_RESULT( umount, err );    
    
    CYG_TEST_PASS_FINISH("jffs2_3");
}

#if CYGINT_ISO_MAIN_STARTUP
int main( int argc, char **argv )
{
    CYG_TEST_INIT();

    gc_test();
    return 0;
}

#else
void cyg_start(void)
{
    CYG_TEST_INIT();

    gc_test();
}
#endif

// EOF jffs2_3.c
