//=============================================================================
//
//      csb337_i2c.c
//
//      CSB337 I2C devices
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2005 Free Software Foundation, Inc.                        
// Copyright (C) 2005 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   bartv
// Date:        2005-02-28
//
//####DESCRIPTIONEND####
//=============================================================================

#include <pkgconf/system.h>
#ifdef CYGPKG_IO_I2C
# include <cyg/io/i2c.h>
# include <cyg/hal/hal_intr.h>

// The AT91RM9200 on the CSB337 has a dedicated I2C/TWI interface, but
// for now bit-banging should suffice - the attached device(s) are low
// bandwidth. The relevant GPIO pins are PA25 for SDA and PA26 for SCL.
//
// The AT91RM9200 GPIO support is unusually flexible, see the chapter
// "Parallel Input/Output Controller (PIO)" in the Atmel AT91RM9200
// datasheet. Bit banging could be implemented in a number of ways,
// but the most appropriate would seem to involve the built-in pull-up
// plus enabling multi-drive open drain mode. That combination matches
// a standard I2C layout. This involves the following register settings:
//
//   PER/PDR/PSR    - set PA25 and PA26 to GPIO mode, disconnecting the
//                    TWI peripheral outputs
//   MDER/MDDR/MDSR - enable for PA25 and PA26, setting the output to
//                    GND
//   SODR/CODR/ODSR - should have no effect, but set to 0 anyway
//   PUER/PUDR/PUSR - enable for PA25 and PA26 to activate the pull-up
//   OER/ODR/OSR    - toggle these to control the state. Enabling means
//                    GND, disabling means the pull-up.
//
//   IFER/IFDR/IFSR - clear, disabling the glitch filter
//   IER/IDR/IMR    - clear, disabling interrupts
//   PDSR           - read the data
//   
// NOTE: figure 149 (I/O Line Control Logic) in the datasheet does not
// seem quite right. It implies that setting MDER would connect the
// output to SODR instead of GND, and similar for the tristate buffer
// control.
//
// There is no need to worry about read/modify/write cycles and
// disabling interrupts, everything can be handled by separate
// set/clear registers.
//
// NOTE: there is no detailed board documentation, so it is not
// totally clear how the pins are hooked up to the I2C bus. For
// example, is there already an external pull-up?

#define SCL (0x01 << 26)
#define SDA (0x01 << 25)

static cyg_bool
hal_csb337_i2c_bitbang(cyg_i2c_bus* bus, cyg_i2c_bitbang_op op)
{
    cyg_bool    result  = 0;
    cyg_uint32  data;
    
    switch(op) {
      case CYG_I2C_BITBANG_INIT:
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_IDR,  SCL | SDA);     // Inputs first, disable interrupts
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_IFDR, SCL | SDA);     // and the glitch filter
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_PUER, SCL | SDA);     // Enable the pull-up
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_MDER, SCL | SDA);     // Switch to multi-drain mode
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_CODR, SCL | SDA);     // Clear output lines (should have no effect)
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_ODR,  SCL | SDA);     // Output control, set to pull-ups
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_PER,  SCL | SDA);     // Set the two lines to GPIO mode
        // We should now have the lines set to GPIO mode, currently
        // pulled-up, running in pull-up multi-drain mode, with no
        // interrupts or glitch filtering.
        break;
        
      case CYG_I2C_BITBANG_SCL_HIGH:
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_ODR, SCL);
        break;
        
      case CYG_I2C_BITBANG_SCL_LOW:
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_OER, SCL);
        break;

      case CYG_I2C_BITBANG_SCL_HIGH_CLOCKSTRETCH:
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_ODR, SCL);
        // Now loop until SCL reads high.
        do {
            HAL_READ_UINT32(AT91RM9200_PIOA+_PIO_PDSR, data);
        } while (0 == (data & SCL));
        break;
        
      case CYG_I2C_BITBANG_SCL_LOW_SDA_INPUT:
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_OER, SCL);
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_ODR, SDA);    // Just switch to pull-up
        break;
        
      case CYG_I2C_BITBANG_SDA_OUTPUT:
        // A no-op. SDA is always set to an output, albeit possibly via the pull-up
        break;
        
      case CYG_I2C_BITBANG_SDA_HIGH:
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_ODR, SDA);
        break;
        
      case CYG_I2C_BITBANG_SDA_LOW:
        HAL_WRITE_UINT32(AT91RM9200_PIOA+_PIO_OER, SDA);
        break;
        
      case CYG_I2C_BITBANG_SDA_READ:
        HAL_READ_UINT32(AT91RM9200_PIOA+_PIO_PDSR, data);
        result = (0 != (data & SDA));
        break;
    }

    CYG_UNUSED_PARAM(cyg_i2c_bus*, bus);
    return result;
}

CYG_I2C_BITBANG_BUS(hal_csb337_i2c_bus, &hal_csb337_i2c_bitbang);

// There is a single device on the main CSB337 board, a DS1307
// wallclock. If the board is hooked up to a CSB900 then that will
// provide some additional devices for keypad and touch panel.

CYG_I2C_DEVICE(cyg_i2c_wallclock_ds1307,            \
               &hal_csb337_i2c_bus,                 \
               0x68,                                \
               0x00,                                \
               CYG_I2C_DEFAULT_DELAY);

#endif // CYGPKG_IO_I2C
