//==========================================================================
//
//      mc9328mxl_misc.c
//
//      HAL misc board support code for ARM9/MC9328MXL
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2006 Free Software Foundation, Inc.
// Copyright (C) 2003, 2004, 2006 eCosCentric Limited                       
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    nickg
// Contributors: nickg
// Date:         2004-07-05
// Purpose:      HAL board support
// Description:  Implementations of HAL board interfaces
//
//####DESCRIPTIONEND####
//
//========================================================================*/

#include <pkgconf/hal.h>
#include <pkgconf/system.h>
#include CYGBLD_HAL_PLATFORM_H

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing macros
#include <cyg/infra/cyg_ass.h>          // assertion macros

#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_arch.h>           // Register state info
#include <cyg/hal/hal_diag.h>
#include <cyg/hal/hal_intr.h>           // Interrupt names
#include <cyg/hal/hal_cache.h>
#include <cyg/hal/hal_arbiter.h>         // hal_call_isr
#include <cyg/hal/mc9328mxl.h>         // Platform specifics

#include <cyg/infra/diag.h>             // diag_printf

//--------------------------------------------------------------------------

CYG_ADDRESS hal_mc9328mxl_aitc = MC9328MXL_AITC;
CYG_ADDRESS hal_mc9328mxl_timer1 = MC9328MXL_TIMER1;
CYG_ADDRESS hal_mc9328mxl_gpio = MC9328MXL_GPIOA;

// -------------------------------------------------------------------------

static cyg_uint32 period;

void hal_clock_initialize(cyg_uint32 _period)
{
    cyg_uint32 ctl;
    period = _period;
    
    hal_mc9328mxl_timer1 = CYGARC_VIRTUAL_ADDRESS( MC9328MXL_TIMER1 );

    ctl = MC9328MXL_TIMER_CONTROL_PERCLK1_DIV16   |
        MC9328MXL_TIMER_CONTROL_IRQEN     |
        MC9328MXL_TIMER_CONTROL_CAP_DIS   |
        MC9328MXL_TIMER_CONTROL_RESTART;

    HAL_WRITE_UINT32( hal_mc9328mxl_timer1+MC9328MXL_TIMER_COMPARE, period );

    HAL_WRITE_UINT32( hal_mc9328mxl_timer1+MC9328MXL_TIMER_CONTROL, ctl );
    ctl |= MC9328MXL_TIMER_CONTROL_TEN;
    HAL_WRITE_UINT32( hal_mc9328mxl_timer1+MC9328MXL_TIMER_CONTROL, ctl );    
}
    

// This routine is called during a clock interrupt.
void hal_clock_reset(cyg_uint32 vector, cyg_uint32 period)
{
    cyg_uint32 status;
    // Clear the interrupt

    HAL_READ_UINT32( hal_mc9328mxl_timer1+MC9328MXL_TIMER_STATUS, status );    
    HAL_WRITE_UINT32( hal_mc9328mxl_timer1+MC9328MXL_TIMER_STATUS, 0 );    
}

// Read the current value of the clock, returning the number of hardware
// "ticks" that have occurred (i.e. how far away the current value is from
// the start)

// Note: The "contract" for this function is that the value is the number
// of hardware clocks that have happened since the last interrupt (i.e.
// when it was reset).

void hal_clock_read(cyg_uint32 *pvalue)
{
    cyg_uint32 val;

    HAL_READ_UINT32(hal_mc9328mxl_timer1+MC9328MXL_TIMER_COUNT, val);
    *pvalue = val;
}

void 
hal_delay_us(int us)
{
    while( us > 0 )
    {
        cyg_uint32 us1 = us;
        cyg_int32 ticks;
        cyg_uint32 cval1, cval2;

        // Wait in bursts of 1s to avoid overflow problems with the
        // multiply by 1000 below.
        
        if( us1 > 1000000 )
            us1 = 1000000;

        us -= us1;
        
        // The timer ticks at 1000ns per tick. So we convert the us
        // value we were given to clock ticks and wait for that many
        // to pass.

        ticks = (us1 * 1000UL) / 1000UL; // 1MHz clock

        HAL_CLOCK_READ( &cval1 );

        // We just loop, waiting for clock ticks to happen,
        // and subtracting them from ticks when they do.
        
        while( ticks > 0 )
        {
            cyg_int32 diff;
            HAL_CLOCK_READ( &cval2 );

            diff = cval2 - cval1;

            // Cope with counter wrap-around.
            if( diff < 0 )
                diff += period;

            ticks -= diff;
            cval1 = cval2;

        }
    }
}

// ----------------------------------------------------------------------------
// Profiling support. This uses TIMER2, set up in much the same way as
// the system timer.

#ifdef CYGPKG_PROFILE_GPROF
#include <cyg/profile/profile.h>
#include <cyg/hal/drv_api.h>

CYG_ADDRESS hal_mc9328mxl_timer2 = MC9328MXL_TIMER2;

static cyg_interrupt    profile_interrupt;
static cyg_handle_t     profile_handle;

static  void
profile_dsr(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    CYG_UNUSED_PARAM(cyg_vector_t, vector);
    CYG_UNUSED_PARAM(cyg_ucount32, count);
    CYG_UNUSED_PARAM(cyg_addrword_t, data);
}

static cyg_uint32
profile_isr(cyg_vector_t vector, cyg_addrword_t data, HAL_SavedRegisters* regs)
{
    cyg_uint32  status;
    HAL_READ_UINT32( hal_mc9328mxl_timer2+MC9328MXL_TIMER_STATUS, status );
    HAL_WRITE_UINT32( hal_mc9328mxl_timer2+MC9328MXL_TIMER_STATUS, 0 );
    HAL_INTERRUPT_ACKNOWLEDGE( vector );
    __profile_hit(regs->pc);
    return CYG_ISR_HANDLED;
}

int
hal_enable_profile_timer(int resolution)
{
    // The requested resolution is in microseconds. Rather than try to
    // figure out the platform's frequency we use the existing
    // calculations for the system clock. We have
    // CYGNUM_HAL_RTC_PERIOD for a value corresponding to
    // (CYGNUM_HAL_RTC_NUMERATOR / CYGNUM_HAL_RTC_DENOMINATOR)
    // nanoseconds. This gives:
    cyg_uint32 ctl;
    cyg_uint32 profile_period = (cyg_uint32)(((long long)resolution * (long long)CYGNUM_HAL_RTC_PERIOD * (long long)CYGNUM_HAL_RTC_DENOMINATOR * 1000ll) \
        / (long long) CYGNUM_HAL_RTC_NUMERATOR);

    resolution = (int)(((long long)profile_period * (long long)CYGNUM_HAL_RTC_NUMERATOR) /
        ((long long)CYGNUM_HAL_RTC_PERIOD * (long long)CYGNUM_HAL_RTC_DENOMINATOR * 1000ll));

    cyg_drv_interrupt_create( CYGNUM_HAL_INTERRUPT_TIMER2, 1, 0, (cyg_ISR_t*) &profile_isr, &profile_dsr, &profile_handle, &profile_interrupt);
    cyg_drv_interrupt_attach(profile_handle);

    hal_mc9328mxl_timer2 = CYGARC_VIRTUAL_ADDRESS( MC9328MXL_TIMER2 );

    ctl = MC9328MXL_TIMER_CONTROL_PERCLK1_DIV16   |
        MC9328MXL_TIMER_CONTROL_IRQEN     |
        MC9328MXL_TIMER_CONTROL_CAP_DIS   |
        MC9328MXL_TIMER_CONTROL_RESTART;

    HAL_WRITE_UINT32( hal_mc9328mxl_timer2+MC9328MXL_TIMER_COMPARE, profile_period );

    HAL_WRITE_UINT32( hal_mc9328mxl_timer2+MC9328MXL_TIMER_CONTROL, ctl );
    ctl |= MC9328MXL_TIMER_CONTROL_TEN;
    HAL_WRITE_UINT32( hal_mc9328mxl_timer2+MC9328MXL_TIMER_CONTROL, ctl );    

    cyg_drv_interrupt_unmask(CYGNUM_HAL_INTERRUPT_TIMER2);
       
    return resolution;
}
#endif

// -------------------------------------------------------------------------

// This routine is called to respond to a hardware interrupt (IRQ).  It
// should interrogate the hardware and return the IRQ vector number.
unsigned int hal_IRQ_handler(void)
{
    unsigned int vec;

    HAL_READ_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_VSR, vec );
    vec >>= 16;
    if (vec >= 64)
        vec = CYGNUM_HAL_INTERRUPT_NONE;
    return vec;
}

#ifdef CYGHWR_HAL_ARM_ARM9_MC9328MXL_DECODE_GPIO

// Springboard for decoding a GPIO interrupt. The data argument is the
// base of the group of vectors mapped to this GPIO port. The
// calculation of the base address of the port relies on the vector
// groups being 32 entries apart and the port hardware registers being
// 256 bytes apart in memory.

int hal_gpio_springboard( int vector, CYG_ADDRWORD data )
{
    cyg_uint32 mask;
    cyg_uint32 ret;
    cyg_uint32 gpio = hal_mc9328mxl_gpio+((data-CYGNUM_HAL_INTERRUPT_GPIOA_BASE)*8); 
    HAL_READ_UINT32( gpio+MC9328MXL_GPIO_ISR, mask );
    HAL_LSBIT_INDEX( vector, mask );
    vector += data;

    ret = hal_call_isr( vector );

    return ret;
}

#endif

//
// Interrupt control
//

void hal_interrupt_mask(int vector)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");
    

    if( vector < CYGNUM_HAL_INTERRUPT_GPIOA_BASE )
    {
        HAL_WRITE_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_DISNUM, vector );
        return;
    }

#ifdef CYGHWR_HAL_ARM_ARM9_MC9328MXL_DECODE_GPIO

    {
        cyg_uint32 gpio = hal_mc9328mxl_gpio;
        cyg_uint32 reg;

        if( vector > CYGNUM_HAL_INTERRUPT_GPIOD_BASE )
            gpio += 0x300, vector -= CYGNUM_HAL_INTERRUPT_GPIOD_BASE;
        else if( vector > CYGNUM_HAL_INTERRUPT_GPIOC_BASE )
            gpio += 0x200, vector -= CYGNUM_HAL_INTERRUPT_GPIOC_BASE;
        else if( vector > CYGNUM_HAL_INTERRUPT_GPIOB_BASE )
            gpio += 0x100, vector -= CYGNUM_HAL_INTERRUPT_GPIOB_BASE;
        else
            vector -= CYGNUM_HAL_INTERRUPT_GPIOA_BASE;
        
        HAL_READ_UINT32( gpio+MC9328MXL_GPIO_IMR, reg );
        reg &= ~(1<<vector);
        HAL_WRITE_UINT32( gpio+MC9328MXL_GPIO_IMR, reg );
    }

#endif
}

void hal_interrupt_unmask(int vector)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

    if( vector < CYGNUM_HAL_INTERRUPT_GPIOA_BASE )
    {
        HAL_WRITE_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_ENNUM, vector );
        return;
    }

#ifdef CYGHWR_HAL_ARM_ARM9_MC9328MXL_DECODE_GPIO

    {
        cyg_uint32 gpio = hal_mc9328mxl_gpio;
        cyg_uint32 reg;
    

        if( vector > CYGNUM_HAL_INTERRUPT_GPIOD_BASE )
            gpio += 0x300, vector -= CYGNUM_HAL_INTERRUPT_GPIOD_BASE;
        else if( vector > CYGNUM_HAL_INTERRUPT_GPIOC_BASE )
            gpio += 0x200, vector -= CYGNUM_HAL_INTERRUPT_GPIOC_BASE;
        else if( vector > CYGNUM_HAL_INTERRUPT_GPIOB_BASE )
            gpio += 0x100, vector -= CYGNUM_HAL_INTERRUPT_GPIOB_BASE;
        else
            vector -= CYGNUM_HAL_INTERRUPT_GPIOA_BASE;
        
        HAL_READ_UINT32( gpio+MC9328MXL_GPIO_IMR, reg );
        reg |= (1<<vector);
        HAL_WRITE_UINT32( gpio+MC9328MXL_GPIO_IMR, reg );
    }

#endif
}

void hal_interrupt_acknowledge(int vector)
{

    if ( vector == CYGNUM_HAL_INTERRUPT_NONE )
    {
        // Spurious interrupt (no interrupt source could be found)
        return;
    }
    
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

    if( vector < CYGNUM_HAL_INTERRUPT_GPIOA_BASE )
    {
        // No ack required for these interrupts
        return;
    }

#ifdef CYGHWR_HAL_ARM_ARM9_MC9328MXL_DECODE_GPIO

    // For GPIO interrupts, we must clear it in the GPIO interrupt
    // status register.
    {
        cyg_uint32 gpio = hal_mc9328mxl_gpio;
        cyg_uint32 reg;
    

        if( vector > CYGNUM_HAL_INTERRUPT_GPIOD_BASE )
            gpio += 0x300, vector -= CYGNUM_HAL_INTERRUPT_GPIOD_BASE;
        else if( vector > CYGNUM_HAL_INTERRUPT_GPIOC_BASE )
            gpio += 0x200, vector -= CYGNUM_HAL_INTERRUPT_GPIOC_BASE;
        else if( vector > CYGNUM_HAL_INTERRUPT_GPIOB_BASE )
            gpio += 0x100, vector -= CYGNUM_HAL_INTERRUPT_GPIOB_BASE;
        else
            vector -= CYGNUM_HAL_INTERRUPT_GPIOA_BASE;
        
        reg = (1<<vector);
        HAL_WRITE_UINT32( gpio+MC9328MXL_GPIO_ISR, reg );
    }

#else

    // When we are not decoding, identify the GPIO device and clear
    // all interrupts from that device.
    {
        cyg_uint32 gpio = hal_mc9328mxl_gpio;
        cyg_uint32 reg;
    

        if( vector == CYGNUM_HAL_INTERRUPT_GPIO_PORTD )
            gpio += 0x300;
        else if( vector == CYGNUM_HAL_INTERRUPT_GPIO_PORTC )
            gpio += 0x200;
        else if( vector == CYGNUM_HAL_INTERRUPT_GPIO_PORTB )
            gpio += 0x100;
        
        HAL_READ_UINT32( gpio+MC9328MXL_GPIO_ISR, reg );
        HAL_WRITE_UINT32( gpio+MC9328MXL_GPIO_ISR, reg );
       
    }

    
#endif
    

}

void hal_interrupt_configure(int vector, int level, int up)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN, "Invalid vector");

    // No support available
}

void hal_interrupt_set_level(int vector, int level)
{
    cyg_uint32 pr;
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN, "Invalid vector");
    CYG_ASSERT(level <= 16 &&
               level >= 0, "Invalid interrupt priority level");

    if( vector < CYGNUM_HAL_INTERRUPT_GPIOA_BASE )
    {
        HAL_READ_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_PRI(vector>>3), pr );
        pr &= ~(0xf<<((vector&7)*4));
        pr |= (level<<((vector&7)*4));
        HAL_WRITE_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_PRI(vector>>3), pr );
    }
}

//
// Reset the processor/board
//
void cyg_hal_mc9328mxl_reset(void)
{
    cyg_uint32 wd = CYGARC_VIRTUAL_ADDRESS( MC9328MXL_WDOG );
    HAL_WRITE_UINT32( wd+MC9328MXL_WDOG_CR, MC9328MXL_WDOG_CR_TMD );
    HAL_WRITE_UINT32( wd+MC9328MXL_WDOG_CR, MC9328MXL_WDOG_CR_WDE|MC9328MXL_WDOG_CR_TMD );
    for(;;);
    
#if 0
    cyg_uint32 stat;
    // Reset the watchdog & force it to timeout
    HAL_WRITE_UINT32(_ST_WDMR, (_ST_WDMR_RSTEN|_ST_WDMR_EXTEN|0x1));  // Minimal count
    HAL_WRITE_UINT32(_ST_CR, _ST_CR_WDRST);        // Reset watchdog
    HAL_READ_UINT32(_ST_SR, stat);                 // Clear status
    while (true) {
        HAL_READ_UINT32(_ST_SR, stat);
        if ((stat & _ST_SR_WDOVF) != 0) {
            diag_printf("Why no RESET?\n");
        }
    }
#endif
}

//--------------------------------------------------------------------------
//
// CPU specific initialization
//

void
_mc9328mxl_hardware_init(void)
{
    hal_mc9328mxl_aitc = CYGARC_VIRTUAL_ADDRESS( MC9328MXL_AITC );
#ifndef CYGSEM_HAL_USE_ROM_MONITOR
    // Only do this if we are not running under a ROM monitor,
    // otherwise we are likely to mess up any interrupts that the
    // monitor has enabled, for example for Ctrl-C handling.
    HAL_WRITE_UINT32(hal_mc9328mxl_aitc+MC9328MXL_AITC_ENHI, 0);
    HAL_WRITE_UINT32(hal_mc9328mxl_aitc+MC9328MXL_AITC_ENLO, 0);
    HAL_WRITE_UINT32(hal_mc9328mxl_aitc+MC9328MXL_AITC_TYPEHI, 0);
    HAL_WRITE_UINT32(hal_mc9328mxl_aitc+MC9328MXL_AITC_TYPELO, 0);
#endif

#ifdef CYGHWR_HAL_ARM_ARM9_MC9328MXL_DECODE_GPIO

    hal_mc9328mxl_gpio = CYGARC_VIRTUAL_ADDRESS( MC9328MXL_GPIOA );

    // Mask all GPIO interrupts
    HAL_WRITE_UINT32( hal_mc9328mxl_gpio+0x000+MC9328MXL_GPIO_IMR, 0 );
    HAL_WRITE_UINT32( hal_mc9328mxl_gpio+0x100+MC9328MXL_GPIO_IMR, 0 );
    HAL_WRITE_UINT32( hal_mc9328mxl_gpio+0x200+MC9328MXL_GPIO_IMR, 0 );
    HAL_WRITE_UINT32( hal_mc9328mxl_gpio+0x300+MC9328MXL_GPIO_IMR, 0 );
    
    // Install springboards for GPIO interrupts
    HAL_INTERRUPT_ATTACH( CYGNUM_HAL_INTERRUPT_GPIO_PORTA, hal_gpio_springboard, CYGNUM_HAL_INTERRUPT_GPIOA_BASE, NULL );
    HAL_INTERRUPT_ATTACH( CYGNUM_HAL_INTERRUPT_GPIO_PORTB, hal_gpio_springboard, CYGNUM_HAL_INTERRUPT_GPIOB_BASE, NULL );
    HAL_INTERRUPT_ATTACH( CYGNUM_HAL_INTERRUPT_GPIO_PORTC, hal_gpio_springboard, CYGNUM_HAL_INTERRUPT_GPIOC_BASE, NULL );
    HAL_INTERRUPT_ATTACH( CYGNUM_HAL_INTERRUPT_GPIO_PORTD, hal_gpio_springboard, CYGNUM_HAL_INTERRUPT_GPIOD_BASE, NULL );

    // Unmask GPIO vectors in AITC
    HAL_WRITE_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_ENNUM, CYGNUM_HAL_INTERRUPT_GPIO_PORTA );    
    HAL_WRITE_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_ENNUM, CYGNUM_HAL_INTERRUPT_GPIO_PORTB );    
    HAL_WRITE_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_ENNUM, CYGNUM_HAL_INTERRUPT_GPIO_PORTC );    
    HAL_WRITE_UINT32( hal_mc9328mxl_aitc+MC9328MXL_AITC_ENNUM, CYGNUM_HAL_INTERRUPT_GPIO_PORTD );
    
#endif
    
    hal_clock_initialize(CYGNUM_HAL_RTC_PERIOD);
}

//--------------------------------------------------------------------------
// End mc9328mxl_misc.c
