/*==========================================================================
//
//      timer_st.c
//
//      HAL timer code using the Simple Timer
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2006 Free Software Foundation, Inc.
// Copyright (C) 2006 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    gthomas
// Contributors: gthomas, jskov, nickg, tkoeller
// Date:         2001-07-12
// Purpose:      HAL board support
// Description:  Implementations of HAL board interfaces
//
//####DESCRIPTIONEND####
//
//========================================================================*/

#include <pkgconf/hal.h>

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_ass.h>          // assertion macros

#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_arch.h>           // Register state info
#include <cyg/hal/hal_intr.h>           // necessary?

// -------------------------------------------------------------------------
// Clock support

static cyg_uint32 _period = 0;

void hal_clock_initialize(cyg_uint32 period)
{
    CYG_ADDRESS timer = AT91_ST0;
    

    CYG_ASSERT(period < 0x10000, "Invalid clock period");

    _period = period;
    
    // Turn timer on 
    HAL_WRITE_UINT32(timer+AT91_ST_ECR, AT91_ST_ECR_ST);
    
    // Reset counter
    HAL_WRITE_UINT32(timer+AT91_ST_CR, AT91_ST_CR_SWRST);

    // Set prescaler to divide CORECLK by 2**4 == 16, with the
    // default divide by 2 in the SYSCAL prescaler this gives a
    // total divide of 32, e.g. 937.5KHz from a 30MHz CORECLK.
    // Autoreload counter on end.
    HAL_WRITE_UINT32(timer+AT91_ST_PR0, AT91_ST_PR_AUTOREL | 4 );

    // Set period
    HAL_WRITE_UINT32(timer+AT91_ST_CT0, period);

    // Enable channel 0 interrupt    
    HAL_WRITE_UINT32(timer+AT91_ST_IER, AT91_ST_IR_CHEND0);

    // Enable timer
    HAL_WRITE_UINT32(timer+AT91_ST_CR, AT91_ST_CR_CHEN0 );

}

void hal_clock_reset(cyg_uint32 vector, cyg_uint32 period)
{
    CYG_ADDRESS timer = AT91_ST0;

    CYG_ASSERT(period < 0x10000, "Invalid clock period");

    // Clear interrupt
    HAL_WRITE_UINT32(timer+AT91_ST_CSR, AT91_ST_SR_CHEND0);

    _period = period;
}

void hal_clock_read(cyg_uint32 *pvalue)
{
    CYG_ADDRESS timer = AT91_ST0;
    cyg_uint32 val;

    HAL_READ_UINT32(timer+AT91_ST_CCV0, val);
    *pvalue = _period-val;
}

// -------------------------------------------------------------------------
//
// Delay for some number of micro-seconds

void 
hal_delay_us(int us)
{
    if( _period == 0 )
        hal_clock_initialize( CYGNUM_HAL_RTC_PERIOD );
    
    while( us > 0 )
    {
        cyg_uint32 us1 = us;
        cyg_int32 ticks;
        cyg_uint32 cval1, cval2;

        // Wait in bursts of 1s to avoid overflow problems with the
        // multiply by 1000 below.
        
        if( us1 > 1000000 )
            us1 = 1000000;

        us -= us1;
        
        // The timer ticks at 937.5KHz, 1066ns per tick. So we convert the us
        // value we were given to clock ticks and wait for that many
        // to pass.

        ticks = (us1 * 1000UL) / 1066UL; 

        HAL_CLOCK_READ( &cval1 );

        // We just loop, waiting for clock ticks to happen,
        // and subtracting them from ticks when they do.
        
        while( ticks > 0 )
        {
            cyg_int32 diff;
            HAL_CLOCK_READ( &cval2 );

            diff = cval2 - cval1;

            // Cope with counter wrap-around.
            if( diff < 0 )
                diff += _period;

            ticks -= diff;
            cval1 = cval2;

        }
    }
}

// -------------------------------------------------------------------------
// timer_st.c
