//==========================================================================
//
//      ixdp4xx_watchdog.cxx
//
//      Watchdog implementation for Intel IXP4xx variant CPUs
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006 Free Software Foundation, Inc.
// Copyright (C) 2004,2005,2006 eCosCentric Limited                         
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    jlarmour
// Contributors: 
// Date:         2006-03-21
// Purpose:      Watchdog class implementation
// Description:  Contains an implementation of the Watchdog class for use
//               with the Intel IXP4xx on-chip watchdog timer.
//               Based on AT91RM9200 watchdog driver.
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/kernel.h>
#include <pkgconf/infra.h>
#include <pkgconf/kernel.h>
#include <pkgconf/watchdog.h>
#include <pkgconf/hal_arm_xscale_ixp425.h>

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/infra/cyg_trac.h>
#include <cyg/hal/hal_io.h>
#include <cyg/hal/hal_diag.h>

#include <cyg/io/watchdog.hxx>

#if !defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT)
#include <cyg/hal/hal_platform_ints.h>
#include <cyg/kernel/intr.hxx>
#endif

//==========================================================================

#define WD_TICK_PERIOD_NS       (15)  // Watchdog driven by 66.66 MHz timer == every 15ns
#define WD_TIMER_CONSTANT_INIT  ((((cyg_uint64)CYGNUM_HAL_IXP4XX_WATCHDOG_DESIRED_TIMEOUT_US*1000)+WD_TICK_PERIOD_NS-1) / WD_TICK_PERIOD_NS)

//==========================================================================

#if defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT)

void
Cyg_Watchdog::init_hw(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();
  resolution = (cyg_uint64)CYGNUM_HAL_IXP4XX_WATCHDOG_DESIRED_TIMEOUT_US*1000;
  CYG_REPORT_RETURN();
}

#else /* defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT) */

//==========================================================================

static Cyg_Watchdog *wd;

//==========================================================================

static cyg_uint32
isr(cyg_vector vector, CYG_ADDRWORD data)
{
    cyg_uint32 status;

    CYG_REPORT_FUNCTION();
    CYG_REPORT_FUNCARG2XV(vector, data);

    CYG_ASSERT( vector == CYGNUM_HAL_INTERRUPT_WDOG, "IXP4xx WDOG ISR called with wrong vector");
    wd->trigger();

    /* Put in magic key and disable watchdog until it is reset - otherwise the int cannot be cleared */
    *IXP425_OST_WDOG_KEY = IXP425_OST_WDOG_KEY_VALUE;
    *IXP425_OST_WDOG_ENA = 0;

    /* Clear magic key to prevent accidental overwriting of watchdog values */
    *IXP425_OST_WDOG_KEY = 0;
    
    Cyg_Interrupt::acknowledge_interrupt(CYGNUM_HAL_INTERRUPT_WDOG);
    CYG_REPORT_RETVAL(Cyg_Interrupt::HANDLED);
    return Cyg_Interrupt::HANDLED;
}

//==========================================================================

// Grab the watchdog interrupt

static Cyg_Interrupt CYGBLD_ATTRIB_INIT_PRI(CYG_INIT_DRIVERS) wdint(
    CYGNUM_HAL_INTERRUPT_WDOG,
    1,
    0,
    isr,
    NULL
  );

//==========================================================================

void
Cyg_Watchdog::init_hw(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();

  wd = this;
  resolution = (cyg_uint64)CYGNUM_HAL_IXP4XX_WATCHDOG_DESIRED_TIMEOUT_US*1000;
  CYG_REPORT_RETURN();
}

#endif	/* defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT) */

//==========================================================================
/*
 * Reset watchdog timer. This needs to be called regularly to prevent
 * the watchdog from firing.
 */

void
Cyg_Watchdog::reset(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();

  // First put in magic key to allow access to watchdog registers
  *IXP425_OST_WDOG_KEY = IXP425_OST_WDOG_KEY_VALUE;

  // Then reset watchdog to timer value first, to avoid any risk of a
  // watchdog reset happening while we're doing the setup the first time,
  // since it is continuously running once started.
  *IXP425_OST_WDOG = WD_TIMER_CONSTANT_INIT;

  // Now enable watchdog, configured either to interrupt or reset
#ifdef CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT
  *IXP425_OST_WDOG_ENA = (IXP425_OST_WDOG_ENA_CNT|IXP425_OST_WDOG_ENA_RST);
#else
  *IXP425_OST_WDOG_ENA = (IXP425_OST_WDOG_ENA_CNT|IXP425_OST_WDOG_ENA_INT);
#endif

  /* Clear magic key to prevent accidental overwriting of watchdog values */
  *IXP425_OST_WDOG_KEY = 0;

  CYG_REPORT_RETURN();
}

//==========================================================================
/*
 * Start watchdog to generate a hardware reset
 * or interrupt when expiring.
 */

void
Cyg_Watchdog::start(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();

  // The watchdog is disabled until activated.

  // A reset does everything we need to start it
  reset();
  
  // Enable interrupt if needed
#ifndef CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT
  wdint.attach();
  wdint.acknowledge_interrupt(CYGNUM_HAL_INTERRUPT_WDOG);
  wdint.unmask_interrupt(CYGNUM_HAL_INTERRUPT_WDOG);
#endif

  CYG_REPORT_RETURN();
}

//==========================================================================
// End of ixp4xx_watchdog.cxx
