//==========================================================================
//
//      io/disk/disk.c
//
//      High level disk driver
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2003, 2004, 2005, 2006, 2007, 2008 Free Software Foundation, Inc.
// Copyright (C) 2003, 2004, 2005, 2006, 2007, 2008 eCosCentric Limited     
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     savin 
// Date:          2003-06-10
// Purpose:       Top level disk driver
// Description: 
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/io.h>
#include <pkgconf/io_disk.h>

#include <cyg/io/io.h>
#include <cyg/io/devtab.h>
#include <cyg/io/disk.h>
#include <cyg/infra/cyg_ass.h>      // assertion support
#include <cyg/infra/diag.h>         // diagnostic output
#if CYGINT_IO_DISK_ALIGN_BUFS_TO_CACHELINE
# include <cyg/hal/hal_cache.h>     // HAL_DCACHE_LINE_SIZE
#endif

// ---------------------------------------------------------------------------

#ifdef CYGDBG_IO_DISK_DEBUG
#define DEBUG 1
#endif

#ifdef DEBUG
# define D(_args_) diag_printf _args_
#else
# define D(_args_)
#endif

// ---------------------------------------------------------------------------

// Master Boot Record defines
#define MBR_SIG_ADDR  0x1FE   // signature address 
#define MBR_SIG_BYTE0 0x55    // signature first byte value
#define MBR_SIG_BYTE1 0xAA    // signature second byte value
#define MBR_PART_ADDR 0x1BE   // first partition address
#define MBR_PART_SIZE 0x10    // partition size
#define MBR_PART_NUM  4       // number of partitions

// Get cylinders, heads and sectors from data (MBR partition format)
#define READ_CHS(_data_, _c_, _h_, _s_)                     \
    do {                                                    \
        _h_ = (*((cyg_uint8 *)_data_));                     \
        _s_ = (*(((cyg_uint8 *)_data_)+1) &  0x3F);         \
        _c_ = (*(((cyg_uint8 *)_data_)+1) & ~0x3F) << 2 |   \
              (*(((cyg_uint8 *)_data_)+2));                 \
    } while (0)

// Get double word from data (MBR partition format)
#define READ_DWORD(_data_, _val_)                           \
    do {                                                    \
        _val_ = *((cyg_uint8 *)_data_)           |          \
                *(((cyg_uint8 *)_data_)+1) << 8  |          \
                *(((cyg_uint8 *)_data_)+2) << 16 |          \
                *(((cyg_uint8 *)_data_)+3) << 24;           \
    } while (0)

// Convert cylinders, heads and sectors to LBA sectors 
#define CHS_TO_LBA(_info_, _c_, _h_, _s_, _lba_) \
    (_lba_=(((_c_)*(_info_)->heads_num+(_h_))*(_info_)->sectors_num)+(_s_)-1)
    
// ---------------------------------------------------------------------------

static Cyg_ErrNo disk_bread(cyg_io_handle_t  handle, 
                            void            *buf, 
                            cyg_uint32      *len, 
                            cyg_uint32       pos);

static Cyg_ErrNo disk_bwrite(cyg_io_handle_t  handle, 
                             const void      *buf, 
                             cyg_uint32      *len, 
                             cyg_uint32       pos);

static Cyg_ErrNo disk_select(cyg_io_handle_t handle, 
                             cyg_uint32      which, 
                             CYG_ADDRWORD    info);

static Cyg_ErrNo disk_get_config(cyg_io_handle_t  handle, 
                                 cyg_uint32       key, 
                                 void            *buf, 
                                 cyg_uint32      *len);

static Cyg_ErrNo disk_set_config(cyg_io_handle_t  handle, 
                                 cyg_uint32       key, 
                                 const void      *buf, 
                                 cyg_uint32      *len);

BLOCK_DEVIO_TABLE(cyg_io_disk_devio,
                  disk_bwrite,
                  disk_bread,
                  disk_select,
                  disk_get_config,
                  disk_set_config
);

static cyg_bool disk_init(struct cyg_devtab_entry *tab);

static Cyg_ErrNo disk_connected(struct cyg_devtab_entry *tab,
                                cyg_disk_identify_t     *ident);

static Cyg_ErrNo disk_disconnected(struct disk_channel *chan);

static Cyg_ErrNo disk_lookup(struct cyg_devtab_entry **tab,
                             struct cyg_devtab_entry  *sub_tab,
                             const char               *name);

static void disk_transfer_done(struct disk_channel *chan, Cyg_ErrNo res);

DISK_CALLBACKS(cyg_io_disk_callbacks, 
               disk_init,
               disk_connected,
               disk_disconnected,
               disk_lookup,
               disk_transfer_done
); 

// ---------------------------------------------------------------------------
//
// Read partition from data
// 
// Fills in *part with the partition details. If the partition is not
// valid then part->type will be set to zero.
//
// Returns 1 if this partition table entry is consistent with this
// being a valid partition table. This means that it is either a valid
// partition, or all zeroes.
// Returns 0 if this entry is non-zero and does not describe a valid
// partition.

static int
read_partition(cyg_uint8            *data,
               cyg_disk_info_t      *info,
               cyg_disk_partition_t *part)
{
    cyg_disk_identify_t *ident = &info->ident;
    cyg_uint16 c, h, s;
    cyg_uint32 start, end, size;
    int i;
    
#ifdef DEBUG
    diag_printf("Partition data:\n");
    diag_dump_buf( data, 16 );
    diag_printf("Disk geometry: %d/%d/%d\n",info->ident.cylinders_num,
                info->ident.heads_num, info->ident.sectors_num );
#endif

    // Check for an entirely zero partition table entry. Count this as
    // a valid partition for MBR recognition purposes, but it is not a
    // valid partition otherwise.
    for( i = 0; i < 16; i++ )
        if( data[i] != 0 ) break;
    if( i == 16 )
    {
        part->type = 0;
        return 1;
    }
           
    // Retrieve basic information
    part->type  = data[4];
    part->state = data[0];
    READ_DWORD(&data[12], part->size);

    READ_DWORD(&data[8], start);        
    READ_DWORD(&data[12], size);

    // Use the LBA start and size fields if they are valid. Otherwise
    // fall back to CHS.
    
    if( start > 0 && size > 0 )
    {
        READ_DWORD(&data[8], start);    
        end = start + size - 1;

#ifdef DEBUG
        diag_printf("Using LBA partition parameters\n");
        diag_printf("      LBA start %d\n",start);
        diag_printf("      LBA size  %d\n",size);
        diag_printf("      LBA end   %d\n",end);
#endif
        
    }
    else
    {
        READ_CHS(&data[1], c, h, s);
        CHS_TO_LBA(ident, c, h, s, start);
#ifdef DEBUG
        diag_printf("Using CHS partition parameters\n");
        diag_printf("      CHS start %d/%d/%d => %d\n",c,h,s,start);
#endif
    
        READ_CHS(&data[5], c, h, s);
        CHS_TO_LBA(ident, c, h, s, end);
#ifdef DEBUG
        diag_printf("      CHS end %d/%d/%d => %d\n",c,h,s,end);
        diag_printf("      CHS size %d\n",size);
#endif

    }

    
    // Check that this looks like a reasonable partition
    if( start <= info->blocks_num  &&
        end <= info->blocks_num    &&
        end > start                &&
        (end-start) == (size-1)    &&
        (part->state == 0x00       ||
         part->state == 0x80 ))
    {
        part->size = size;
        part->start = start;
        part->end = end;
    }
    else
        part->type = 0;

#ifdef DEBUG
    diag_printf("Partition %svalid\n", part->type==0?"in":"");
#endif

    return part->type==0?0:1;

}

// ---------------------------------------------------------------------------

#define MBR_BUF_SIZE CYGINT_IO_DISK_MAX_SECTOR_SIZE

// Some drivers require buffers to be aligned to cache lines.
// FIXME: this should be controlled by a config option.
#if CYGINT_IO_DISK_ALIGN_BUFS_TO_CACHELINE && \
    defined(HAL_DCACHE_LINE_SIZE) && \
    (HAL_DCACHE_LINE_SIZE > 0)
static cyg_uint8 mbr_buf[MBR_BUF_SIZE+HAL_DCACHE_LINE_SIZE-1];

#else

static cyg_uint8 mbr_buf[MBR_BUF_SIZE];

#endif

static cyg_drv_mutex_t mbr_buf_mutex;
static cyg_bool mbr_buf_mutex_initialized = false;

// ---------------------------------------------------------------------------
//
// Read Master Boot Record (partitions)
//

static Cyg_ErrNo 
read_mbr(disk_channel *chan)
{
    cyg_disk_info_t *info = chan->info;
    disk_funs       *funs = chan->funs;
    disk_controller *ctlr = chan->controller;
    Cyg_ErrNo res = ENOERR;
    int i;

#if CYGINT_IO_DISK_ALIGN_BUFS_TO_CACHELINE && \
    defined(HAL_DCACHE_LINE_SIZE) && \
    (HAL_DCACHE_LINE_SIZE > 0)
    cyg_uint8 *buf = (cyg_uint8*)((CYG_ADDRESS)(&mbr_buf[HAL_DCACHE_LINE_SIZE-1]) & ~(HAL_DCACHE_LINE_SIZE-1));
#else
    cyg_uint8 *buf = (cyg_uint8*)&mbr_buf[0];
#endif
    
    D(("read MBR\n"));

    // Check we have space for the MBR block
    if( info->block_size > MBR_BUF_SIZE )
        return EINVAL;
    
    for (i = 0; i < info->partitions_num; i++)
        info->partitions[i].type = 0x00;    

    cyg_drv_dsr_lock();
    if( !mbr_buf_mutex_initialized )
    {
        // Initialize the buffer memory mutex
        cyg_drv_mutex_init( &mbr_buf_mutex );
        mbr_buf_mutex_initialized = true;
    }
    cyg_drv_dsr_unlock();
        
    cyg_drv_mutex_lock( &mbr_buf_mutex );
    
    cyg_drv_mutex_lock( &ctlr->lock );

    while( ctlr->busy )
        cyg_drv_cond_wait( &ctlr->queue );

    ctlr->busy = true;
    
    ctlr->result = -EWOULDBLOCK;

    for( i = 0; i < info->block_size; i++ )
        buf[i] = 0;
    
    res = (funs->read)(chan, (void *)buf, 1, 0);
    
    if( res == -EWOULDBLOCK )
    {
        // If the driver replys EWOULDBLOCK, then the transfer is
        // being handled asynchronously and when it is finished it
        // will call disk_transfer_done(). This will wake us up here
        // to continue.

        while( ctlr->result == -EWOULDBLOCK )
            cyg_drv_cond_wait( &ctlr->async );

        res = ctlr->result;
    }
        
    ctlr->busy = false;
    
    cyg_drv_mutex_unlock( &ctlr->lock );

    if (ENOERR != res)
        goto done;

#ifdef DEBUG
    diag_dump_buf_with_offset( buf, info->block_size, buf );
#endif
    
    if (MBR_SIG_BYTE0 == buf[MBR_SIG_ADDR+0] && MBR_SIG_BYTE1 == buf[MBR_SIG_ADDR+1])
    {
        int npart;
        int vpart = 0;

        D(("disk MBR found\n")); 
 
        npart = info->partitions_num < MBR_PART_NUM ? 
            info->partitions_num : MBR_PART_NUM;

        for (i = 0; i < npart; i++)
        {
            cyg_disk_partition_t *part = &info->partitions[i];
            
            vpart += read_partition(&buf[MBR_PART_ADDR+MBR_PART_SIZE*i], info, part); 
#ifdef DEBUG
            if (0x00 != part->type)
            {
                D(("\ndisk MBR partition %d:\n", i));
                D(("      type  = %02X\n", part->type));
                D(("      state = %02X\n", part->state));
                D(("      start = %d\n",   part->start));
                D(("      end   = %d\n",   part->end));
                D(("      size  = %d\n\n", part->size));
            }
#endif
        } 

        
#ifdef CYGSEM_IO_DISK_DETECT_FAT_BOOT
        /* This is a workaround for poor and inconsistent Windows behaviour.
         * Removeable media usually comes with partition tables, and Windows
         * respects that. But if it gets corrupted or the card is fully wiped,
         * then Windows has *no way* to put a partition table back on. This is
         * lame. But it results in us not knowing what type of media this will
         * be - with MBR, or without. We don't normally autodetect stuff like
         * that. So instead we allow for FAT format media in this form to be
         * detected, and use the FAT boot sector info to provide "partition"
         * information.
         *
         * Auto-detection of this is of course a total layering violation.
         *
         * We use the system ID field to look for "FAT". That should be good
         * enough. However the location of the system ID field depends on
         * whether it is FAT12/16 or FAT32, which means we have to go through
         * some machinations to determine which this would be and check it
         * matches. There are many on-line sites that describe the layout of
         * the FAT boot sector, so that or some other FAT-specific
         * documentation should be referred to in order to understand the
         * below.
         *
         * Unfortunately the above is not quite good enough. Devices
         * also exist which appear to have a partition table embedded
         * in a superficially valid FAT boot block. The only way to
         * detect this is to look for invalid or inconsistent values
         * in the BPB. So far the only example of this has been an
         * iPod where the sectors_per_fat and root_cluster fields of
         * the fake BPB happen to be zero. So this is what is tested
         * here. Further tests may need to be added as new
         * misidentifications are encountered. Ultimately, it is
         * possible for a single sector to contain both a valid BPB
         * and a valid partition table, at which point whatever we
         * choose to believe will not satisfy someone.
         */

        if( vpart != npart )
        {
            cyg_disk_info_t *info = chan->info;
            cyg_disk_partition_t *part = &info->partitions[0];
            cyg_uint32 reserved_sectors, num_fats, sectors_per_fat, root_start, root_entries;
            cyg_uint32 bytes_per_sector, clusters_start, num_sectors, num_clusters;
            cyg_uint32 sectors_per_cluster, sectors_per_head, heads_per_cylinder;
            cyg_uint32 root_cluster = 1; // fake value for FAT12/16
            cyg_uint8 fat_part_type;
            
            sectors_per_head     = (buf[0x19] << 8) | buf[0x18];
            heads_per_cylinder   = (buf[0x1b] << 8) | buf[0x1a];
            reserved_sectors     = (buf[0x0f] << 8) | buf[0x0e];
            num_fats             = (buf[0x10]);
            sectors_per_fat      = (buf[0x17] << 8) | buf[0x16];
            bytes_per_sector     = (buf[0x0c] << 8) | buf[0x0b];
            root_entries         = (buf[0x12] << 8) | buf[0x11];
            num_sectors          = (buf[0x14] << 8) | buf[0x13];
            if (0 == num_sectors) // use big number of sectors instead
                num_sectors = (buf[0x23] << 24) | (buf[0x22] << 16) | (buf[0x21] << 8) | buf[0x20];
            if (0 == sectors_per_fat)
            {
                // May be a FAT32 filesystem
                // use big number of sectors per fat instead
                sectors_per_fat = (buf[0x23] << 27) | (buf[0x26] << 16) | (buf[0x25] << 8) | buf[0x24];
                // Fetch root directory cluster number
                root_cluster = (buf[0x2F] << 27) | (buf[0x2E] << 16) | (buf[0x2D] << 8) | buf[0x2C];
            }
            sectors_per_cluster  = buf[0x0d];

            // take particular care with bytes_per_sector and sectors_per_cluster as 
            // they are used as divisors
            // We fall through to parsing as partition table otherwise
            if ( bytes_per_sector && sectors_per_cluster && sectors_per_fat && root_cluster )
            {
                root_start = reserved_sectors + (num_fats*sectors_per_fat);
                clusters_start = root_start + (((root_entries*32)+bytes_per_sector-1) / bytes_per_sector);
                num_clusters = 2 + (num_sectors - clusters_start) / sectors_per_cluster;

                /* Based on num_clusters we can determine FAT12 vs FAT16 vs FAT32. We need to verify
                 * that guess by checking the system identifier.
                 * FAT12/16 have the id at offset 0x36, FAT32's one is at 0x52
                 */

                fat_part_type = 0x00;
                if ( num_clusters < 4087 )
                {
                    if ( (buf[0x36] == 'F') && (buf[0x37] == 'A') && (buf[0x38] == 'T') &&
                         (buf[0x39] == '1') && (buf[0x3a] == '2') )
                        fat_part_type = 0x01; // FAT12
                }
                else if ( num_clusters < 65527 )
                {
                    if ( (buf[0x36] == 'F') && (buf[0x37] == 'A') && (buf[0x38] == 'T') &&
                         (buf[0x39] == '1') && (buf[0x3a] == '6') )
                        fat_part_type = 0x06; // FAT16
                }
                else if ( (buf[0x52] == 'F') && (buf[0x53] == 'A') && (buf[0x54] == 'T') &&
                          (buf[0x55] == '3') && (buf[0x56] == '2') )
                    fat_part_type = 0x0b; // FAT32

                if ( fat_part_type )
                {
                    // fill in partition info artificially
                    part->state = 0x80;             // state 0x00 - inactive, 0x80 - active
                    part->start = 0;                // first sector number
                    part->type = fat_part_type;     // Partition type code.
                    part->end = num_sectors - 1;    // last sector number
                    part->size = num_sectors;       // size in sectors
                    res = ENOERR;

                    D(("Detected FAT%s fs at block 0 instead of partition table\n",
                      (part->type == 0x01) ? "12" :
                       (part->type == 0x06) ? "16" : "32" ));
                    
                    D(( "FAT secs/hd %d, hd/cyl %d, root_start %d, clust_start %d, "
                       "num_secs %d, num_clusters %d\n",
                       sectors_per_head, heads_per_cylinder,
                       root_start, clusters_start, num_sectors, num_clusters ));
                    
                    goto done;
                }
            } // if
        }
#endif        

    }

    res = ENOERR;
    
done:
    cyg_drv_mutex_unlock( &mbr_buf_mutex );
    return res;
}

// ---------------------------------------------------------------------------

void disk_channel_dummy_event( cyg_uint32 event, cyg_uint32 devno, CYG_ADDRWORD data )
{
    D(("disk_channel_dummy_event( %s, %d, %08x)\n",
       (event==CYG_DISK_EVENT_CONNECT)?"CONNECT":
       (event==CYG_DISK_EVENT_DISCONNECT)?"DISCONNECT":"???",
       devno, data ));
}

// ---------------------------------------------------------------------------

static cyg_bool 
disk_init(struct cyg_devtab_entry *tab)
{
    disk_channel    *chan = (disk_channel *) tab->priv;
    cyg_disk_info_t *info = chan->info;
    int i;

    if (!chan->init)
    {
        disk_controller *controller = chan->controller;
        
        if( !controller->init )
        {
            cyg_drv_mutex_init( &controller->lock );
            cyg_drv_cond_init( &controller->queue, &controller->lock );
            cyg_drv_cond_init( &controller->async, &controller->lock );
            controller->busy = false;
            
            controller->init = true;
        }
        
        info->connected = false;

        chan->event = disk_channel_dummy_event;
        chan->event_data = 0;
        
        // clear partition data
        for (i = 0; i < info->partitions_num; i++)
            info->partitions[i].type = 0x00;

        chan->init = true;
    }
    return true;
}

// ---------------------------------------------------------------------------

static Cyg_ErrNo
disk_connected(struct cyg_devtab_entry *tab,
               cyg_disk_identify_t     *ident)
{
    disk_channel    *chan = (disk_channel *) tab->priv;
    cyg_disk_info_t *info = chan->info;
    Cyg_ErrNo res = ENOERR;
 
    if (!chan->init)
        return -EINVAL;

    // If the device is already connected, nothing more to do
    if( info->connected )
        return ENOERR;

    // If any of these assertions fire, it is probable that the
    // hardware driver has not been updated to match the current disk
    // API.
    CYG_ASSERT( ident->sector_size > 0, "Bad sector size" );
    CYG_ASSERT( ident->lba_sectors_num > 0, "Bad LBA sector count" );
    CYG_ASSERT( ident->phys_block_size > 0, "Bad physical block size");
    CYG_ASSERT( ident->max_transfer > 0, "Bad max transfer size");
    
    info->ident      = *ident;

    info->block_size = ident->sector_size;
    info->blocks_num = ident->lba_sectors_num;
    info->phys_block_size = ident->phys_block_size;
    info->mounts = 0;
    
    D(("disk connected\n")); 
    D(("    serial            = '%s'\n", ident->serial)); 
    D(("    firmware rev      = '%s'\n", ident->firmware_rev)); 
    D(("    model num         = '%s'\n", ident->model_num)); 
    D(("    block_size        = %d\n",   info->block_size));
    D(("    blocks_num        = %u\n",   info->blocks_num));
    D(("    phys_block_size   = %d\n",   info->phys_block_size));
    
    if (chan->mbr_support)
    {    
        // read disk master boot record
        res = read_mbr(chan);
    }

    if (ENOERR == res)
    {    
        // now declare that we are connected
        info->connected = true;
        chan->valid     = true; 
    }
    return res;
}

// ---------------------------------------------------------------------------

static Cyg_ErrNo
disk_disconnected(disk_channel *chan)
{
    cyg_disk_info_t *info = chan->info;
    int i;

    if (!chan->init)
        return -EINVAL;

    info->connected = false;
    chan->valid     = false;
     
    // clear partition data and invalidate partition devices 
    for (i = 0; i < info->partitions_num; i++)
    {
        info->partitions[i].type  = 0x00;
        chan->pdevs_chan[i].valid = false;
    }

    
    D(("disk disconnected\n")); 

    return ENOERR;    
}
    
// ---------------------------------------------------------------------------

static Cyg_ErrNo
disk_lookup(struct cyg_devtab_entry **tab,
            struct cyg_devtab_entry  *sub_tab,
            const char *name)
{
    disk_channel    *chan = (disk_channel *) (*tab)->priv;
    cyg_disk_info_t *info = chan->info;
    struct cyg_devtab_entry *new_tab;
    disk_channel            *new_chan;
    int dev_num;
    
    if (!info->connected)
        return -EINVAL;

    dev_num = 0;

    while ('\0' != *name)
    {
        if (*name < '0' || *name > '9')
            return -EINVAL;

        dev_num = 10 * dev_num + (*name - '0');
        name++;
    }
   
    if (dev_num > info->partitions_num)
        return -EINVAL;

    D(("disk lookup dev number = %d\n", dev_num)); 

    if (0 == dev_num)
    {
        // 0 is the root device number
        return ENOERR; 
    }
    if (0x00 == info->partitions[dev_num-1].type)
    {
        D(("disk NO partition for dev\n")); 
        return -EINVAL;
    }

    new_tab  = &chan->pdevs_dev[dev_num-1];
    new_chan = &chan->pdevs_chan[dev_num-1];
    
    // copy device data from parent
    *new_tab  = **tab; 
    *new_chan = *chan;

    new_tab->priv = (void *)new_chan;

    // set partition ptr
    new_chan->partition = &info->partitions[dev_num-1];

    // return device tab 
    *tab = new_tab;
        
    return ENOERR;
}

// ---------------------------------------------------------------------------

static Cyg_ErrNo 
disk_bread(cyg_io_handle_t  handle, 
           void            *buf, 
           cyg_uint32      *len,  // In blocks
           cyg_uint32       pos)  // In blocks
{
    cyg_devtab_entry_t *t    = (cyg_devtab_entry_t *) handle;
    disk_channel       *chan = (disk_channel *) t->priv;
    disk_controller    *ctlr = chan->controller;
    disk_funs          *funs = chan->funs;
    cyg_disk_info_t    *info = chan->info;
    cyg_uint32  size = *len;
    cyg_uint8  *bbuf = (cyg_uint8  *)buf;
    Cyg_ErrNo   res  = ENOERR;
    cyg_uint32  last;

    cyg_drv_mutex_lock( &ctlr->lock );

    while( ctlr->busy )
        cyg_drv_cond_wait( &ctlr->queue );

    if (info->connected && chan->valid)
    {
        ctlr->busy = true;
    
        if (NULL != chan->partition)
        {
            pos += chan->partition->start;
            last = chan->partition->end;
        }
        else
        {
            last = info->blocks_num-1;
        }
 
        D(("disk read block=%d len=%d buf=%p\n", pos, *len, buf));

        while( size > 0 )
        {
            cyg_uint32 tfr = size;
            
            if (pos > last)
            {
                res = -EIO;
                break;
            }

            if( tfr > info->ident.max_transfer )
                tfr = info->ident.max_transfer;
            
            ctlr->result = -EWOULDBLOCK;

            cyg_drv_dsr_lock();
            
            res = (funs->read)(chan, (void*)bbuf, tfr, pos);

            if( res == -EWOULDBLOCK )
            {
                // If the driver replys EWOULDBLOCK, then the transfer is
                // being handled asynchronously and when it is finished it
                // will call disk_transfer_done(). This will wake us up here
                // to continue.

                while( ctlr->result == -EWOULDBLOCK )
                    cyg_drv_cond_wait( &ctlr->async );

                res = ctlr->result;
            }

            cyg_drv_dsr_unlock();
            
            if (ENOERR != res)
                goto done;

            if (!info->connected)
            {
                res = -EINVAL;
                goto done;
            }

            bbuf        += tfr * info->block_size;
            pos         += tfr;
            size        -= tfr;
        }

    done:        
        ctlr->busy = false;
        cyg_drv_cond_signal( &ctlr->queue );
    }
    else
        res = -EINVAL;

    cyg_drv_mutex_unlock( &ctlr->lock );
#ifdef CYGPKG_KERNEL
    cyg_thread_yield();
#endif
    
    *len -= size;
    return res;
}

// ---------------------------------------------------------------------------

static Cyg_ErrNo 
disk_bwrite(cyg_io_handle_t  handle, 
            const void      *buf, 
            cyg_uint32      *len,   // In blocks
            cyg_uint32       pos)   // In blocks
{
    cyg_devtab_entry_t *t    = (cyg_devtab_entry_t *) handle;
    disk_channel       *chan = (disk_channel *) t->priv;
    disk_controller    *ctlr = chan->controller;    
    disk_funs          *funs = chan->funs;
    cyg_disk_info_t    *info = chan->info;
    cyg_uint32  size = *len;
    cyg_uint8  *bbuf = (cyg_uint8 * const) buf;
    Cyg_ErrNo   res  = ENOERR;
    cyg_uint32  last;

    cyg_drv_mutex_lock( &ctlr->lock );

    while( ctlr->busy )
        cyg_drv_cond_wait( &ctlr->queue );

    if (info->connected && chan->valid)
    {
        ctlr->busy = true;
        
        if (NULL != chan->partition)
        {
            pos += chan->partition->start;
            last = chan->partition->end;
        }
        else
        {
            last = info->blocks_num-1;
        }
    
        D(("disk write block=%d len=%d buf=%p\n", pos, *len, buf));

        while( size > 0 )
        {
            cyg_uint32 tfr = size;
        
            if (pos > last)
            {
                res = -EIO;
                goto done;
            }

            if( tfr > info->ident.max_transfer )
                tfr = info->ident.max_transfer;

            ctlr->result = -EWOULDBLOCK;

            cyg_drv_dsr_lock();
            
            res = (funs->write)(chan, (void*)bbuf, tfr, pos);

            if( res == -EWOULDBLOCK )
            {
                // If the driver replys EWOULDBLOCK, then the transfer is
                // being handled asynchronously and when it is finished it
                // will call disk_transfer_done(). This will wake us up here
                // to continue.

                while( ctlr->result == -EWOULDBLOCK )
                    cyg_drv_cond_wait( &ctlr->async );

                res = ctlr->result;
            }

            cyg_drv_dsr_unlock();
            
            if (ENOERR != res)
                goto done;
 
            if (!info->connected)
            {
                res = -EINVAL;
                goto done;
            }

            bbuf        += tfr * info->block_size;
            pos         += tfr;
            size        -= tfr;
            
        }

    done:        
        ctlr->busy = false;
        cyg_drv_cond_signal( &ctlr->queue );
    }
    else
        res = -EINVAL;

    cyg_drv_mutex_unlock( &ctlr->lock );
#ifdef CYGPKG_KERNEL    
    cyg_thread_yield();
#endif
    *len -= size;
    return res;
}

// ---------------------------------------------------------------------------

static void disk_transfer_done(struct disk_channel *chan, Cyg_ErrNo res)
{
    disk_controller    *ctlr = chan->controller;    

    ctlr->result = res;
    
    cyg_drv_cond_signal( &ctlr->async );
}

// ---------------------------------------------------------------------------

static cyg_bool
disk_select(cyg_io_handle_t handle, cyg_uint32 which, CYG_ADDRWORD info)
{
    cyg_devtab_entry_t *t     = (cyg_devtab_entry_t *) handle;
    disk_channel       *chan  = (disk_channel *) t->priv;
    cyg_disk_info_t    *cinfo = chan->info;
 
    if (!cinfo->connected || !chan->valid)
        return false;
    else
        return true;
}

// ---------------------------------------------------------------------------

static Cyg_ErrNo 
disk_get_config(cyg_io_handle_t  handle, 
                cyg_uint32       key, 
                void            *xbuf,
                cyg_uint32      *len)
{
    cyg_devtab_entry_t *t    = (cyg_devtab_entry_t *) handle;
    disk_channel       *chan = (disk_channel *) t->priv;
    disk_controller    *ctlr = chan->controller;    
    cyg_disk_info_t    *info = chan->info;
    cyg_disk_info_t    *buf  = (cyg_disk_info_t *) xbuf;
    disk_funs          *funs = chan->funs;
    Cyg_ErrNo res = ENOERR;
 
    cyg_drv_mutex_lock( &ctlr->lock );

    while( ctlr->busy )
        cyg_drv_cond_wait( &ctlr->queue );

    if (info->connected && chan->valid)
    {
        ctlr->busy = true;
    
        D(("disk get config key=%d\n", key)); 
    
        switch (key) {
        case CYG_IO_GET_CONFIG_DISK_INFO:
            if (*len < sizeof(cyg_disk_info_t)) {
                res = -EINVAL;
                break;
            }
            D(("chan->info->block_size %u\n", chan->info->block_size ));
            D(("chan->info->blocks_num %u\n", chan->info->blocks_num ));
            D(("chan->info->phys_block_size %u\n", chan->info->phys_block_size ));
            *buf = *chan->info;
            *len = sizeof(cyg_disk_info_t);
            break;       

#ifdef CYGSEM_IO_DISK_REMOVABLE_MEDIA_SUPPORT            
        case CYG_IO_GET_CONFIG_DISK_EVENT:
        {
            cyg_disk_event_t *event = (cyg_disk_event_t *)xbuf;
            if (*len < sizeof(cyg_disk_event_t)) {
                res = -EINVAL;
                break;
            }            
            event->event = chan->event;
            event->event_data = chan->event_data;
            D(("ECOS: disk set event: event %p data %08x\n", chan->event, chan->event_data ));
            *len = sizeof(cyg_disk_event_t);
            res = ENOERR;
            break;
        }
#endif
        
        default:
            // pass down to lower layers
            res = (funs->get_config)(chan, key, xbuf, len);
        }
        
        ctlr->busy = false;
        cyg_drv_cond_signal( &ctlr->queue );
    }
    else
        res = -EINVAL;

    cyg_drv_mutex_unlock( &ctlr->lock );    
    
    return res;
}

// ---------------------------------------------------------------------------

static Cyg_ErrNo 
disk_set_config(cyg_io_handle_t  handle, 
                cyg_uint32       key, 
                const void      *xbuf, 
                cyg_uint32      *len)
{
    cyg_devtab_entry_t *t    = (cyg_devtab_entry_t *) handle;
    disk_channel       *chan = (disk_channel *) t->priv;
    disk_controller    *ctlr = chan->controller;    
    cyg_disk_info_t    *info = chan->info;
    disk_funs          *funs = chan->funs;
    Cyg_ErrNo res = ENOERR;
    
    cyg_drv_mutex_lock( &ctlr->lock );

    while( ctlr->busy )
        cyg_drv_cond_wait( &ctlr->queue );

    // Event registration may be performed on unconnected devices - after all
    // the point is usually to detect them later! So we do not check for
    // validity first in that case.

    if ( (info->connected && chan->valid) || 
#ifdef CYGSEM_IO_DISK_REMOVABLE_MEDIA_SUPPORT
         (CYG_IO_SET_CONFIG_DISK_EVENT==key) ||
#endif
         0)
    {
        ctlr->busy = true;
        
        D(("disk set config key=%d\n", key)); 

        switch ( key )
        {
        case CYG_IO_SET_CONFIG_DISK_MOUNT:
            chan->mounts++;
            info->mounts++;
            D(("disk mount: chan %d disk %d\n",chan->mounts, info->mounts));
            break;
            
        case CYG_IO_SET_CONFIG_DISK_UMOUNT:
            chan->mounts--;
            info->mounts--;
            D(("disk umount: chan %d disk %d\n",chan->mounts, info->mounts));            
            break;

#ifdef CYGSEM_IO_DISK_REMOVABLE_MEDIA_SUPPORT
        case CYG_IO_SET_CONFIG_DISK_EVENT:
        {
            cyg_disk_event_t *event = (cyg_disk_event_t *)xbuf;
            if (*len < sizeof(cyg_disk_event_t)) {
                res = -EINVAL;
                break;
            }            
            chan->event = event->event;
            chan->event_data = event->event_data;
            D(("disk set event: event %p data %08x\n", chan->event, chan->event_data ));
            break;
        }
#endif
        
        default:
            break;
        }
        
        // pass down to lower layers
        res = (funs->set_config)(chan, key, xbuf, len);
        
        ctlr->busy = false;
        cyg_drv_cond_signal( &ctlr->queue );
    }
    else
        res = -EINVAL;
    
    cyg_drv_mutex_unlock( &ctlr->lock );

    return res;
    
}

// ---------------------------------------------------------------------------
// EOF disk.c
