/*========================================================================
//
//      mathfp/s_pow.c
//
//      Part of the standard mathematical function library
//
//========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004 Free Software Foundation, Inc.
// Copyright (C) 2004 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     jlarmour
// Contributors:  
// Date:          2004-02-30
// Purpose:       
// Description:   Derived from newlib libm
//                http://sources.redhat.com/newlib
//                in turn based on SunSoft fdlibm.
//                See copyright/license information below.
// Usage:         #include <math.h>
//
//####DESCRIPTIONEND####
//======================================================================*/


/* @(#)z_pow.c 1.0 98/08/13 */

/*
FUNCTION
        <<pow>>, <<powf>>---x to the power y
INDEX
        pow
INDEX
        powf


ANSI_SYNOPSIS
        #include <math.h>
        double pow(double <[x]>, double <[y]>);
        float pow(float <[x]>, float <[y]>);

TRAD_SYNOPSIS
        #include <math.h>
        double pow(<[x]>, <[y]>);
        double <[x]>, <[y]>;

        float pow(<[x]>, <[y]>);
        float <[x]>, <[y]>;

DESCRIPTION
        <<pow>> and <<powf>> calculate <[x]> raised to the exponent <[y]>.
        @tex
        (That is, $x^y$.)
        @end tex

RETURNS
        On success, <<pow>> and <<powf>> return the value calculated.

        When the argument values would produce overflow, <<pow>>
        returns <<HUGE_VAL>> and set <<errno>> to <<ERANGE>>.  If the
        argument <[x]> passed to <<pow>> or <<powf>> is a negative
        noninteger, and <[y]> is also not an integer, then <<errno>>
        is set to <<EDOM>>.  If <[x]> and <[y]> are both 0, then
        <<pow>> and <<powf>> return <<1>>.

        You can modify error handling for these functions using <<matherr>>.

PORTABILITY
        <<pow>> is ANSI C. <<powf>> is an extension.  */

#include <float.h>
#include "mathincl/fdlibm.h"
#include "zmath.h"

#ifndef _DOUBLE_IS_32BITS

double pow (double x, double y)
{
  double d, k, t, r = 1.0;
  int n, sign, exponent_is_even_int = 0;
  __uint32_t px;

  GET_HIGH_WORD (px, x);

  k = modf (y, &d);

  if (k == 0.0)
    {
      /* Exponent y is an integer. */
      if (modf (ldexp (y, -1), &t))
        {
          /* y is odd. */
          exponent_is_even_int = 0;
        }
      else
        {
          /* y is even. */
          exponent_is_even_int = 1;
        }
    }

  if (x == 0.0 && y <= 0.0)
    {
      errno = EDOM;
    }
  else if ((t = y * log (fabs (x))) >= BIGX) 
    {
      errno = ERANGE;
      if (px & 0x80000000) 
        {
          /* x is negative. */
          if (k) 
            {
              /* y is not an integer. */
              errno = EDOM;
              x = 0.0;
            }
          else if (exponent_is_even_int)
            x = z_infinity.d;
          else
            x = -z_infinity.d;
        }
      else
        {
          x = z_infinity.d;
        }
    }
  else if (t < SMALLX)
    {
      errno = ERANGE;
      x = 0.0;
    }
  else 
    {
      if ( !k && fabs(d) <= 32767 ) 
        {
          n = (int) d;

          if ((sign = (n < 0)))
            n = -n;

          while ( n > 0 ) 
            {
              if ((unsigned int) n % 2) 
                r *= x;
              x *= x;
              n = (unsigned int) n / 2;
            }

          if (sign)
            r = 1.0 / r;

          return r;
        }
      else 
        {
          if ( px & 0x80000000 ) 
            {
              /* x is negative. */
              if ( k ) 
                {
                  /* y is not an integer. */
                  errno = EDOM;
                  return 0.0;
                }
            }

          x = exp (t);

          if (!exponent_is_even_int)
            {
              if (px & 0x80000000)
                {
                  /* y is an odd integer, and x is negative,
                     so the result is negative. */
                  GET_HIGH_WORD (px, x);
                  px |= 0x80000000;
                  SET_HIGH_WORD (x, px);
                }
            }
        }
    }

  return x;
}

#endif _DOUBLE_IS_32BITS

// EOF s_pow.c
