//==========================================================================
//
//      io/serial/arm/pl011/ser_pl011.c
//
//      Generic ARM PL011 macrocell serial driver
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2008 Free Software Foundation, Inc.
// Copyright (C) 2004, 2008, 2009 eCosCentric Limited                             
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    nickg
// Contributors: (16x5x driver) gthomas, jlarmour, jskov
// Date:         2004-05-04
// Purpose:      PL011 generic serial driver
// Description: 
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/system.h>
#include <pkgconf/io_serial.h>
#include <pkgconf/io.h>

#include <cyg/io/io.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/io/devtab.h>
#include <cyg/io/serial.h>
#include <cyg/infra/diag.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/hal/hal_io.h>

#include "ser_pl011.h"

// Only compile driver if an inline file with driver details was selected.
#ifdef CYGDAT_IO_SERIAL_PL011_INL

//==========================================================================

static unsigned char select_word_length[] = {
    _UART_LCRH_WLEN_5,    // 5 bits / word (char)
    _UART_LCRH_WLEN_6,    
    _UART_LCRH_WLEN_7,    
    _UART_LCRH_WLEN_8
};

static unsigned char select_stop_bits[] = {
    0,
    0,                  // 1 stop bit
    0,                  // 1.5 stop bit
    _UART_LCRH_STP2     // 2 stop bits
};

static unsigned char select_parity[] = {
    0,                                                  // No parity
    _UART_LCRH_PEN|_UART_LCRH_EPS,                      // Even parity
    _UART_LCRH_PEN,                                     // Odd parity
    _UART_LCRH_PEN|_UART_LCRH_SPS,                      // Mark parity
    _UART_LCRH_PEN|_UART_LCRH_EPS|_UART_LCRH_SPS        // Space parity
};

static unsigned int select_baud[] = {
    0,          // Unused
    50,
    75,
    110,
    0,          // 134.5 unsupported
    150,
    200,
    300,
    600,
    1200,
    1800,
    2400,
    3600,
    4800,
    7200,
    9600,
    14400,
    19200,
    38400,
    57600,
    115200,
    230400
};

//==========================================================================

typedef struct pl011_serial_info {
    cyg_addrword_t base;        /* Base address (in virtual address space) */
    int            int_num;
    cyg_interrupt  serial_interrupt;
    cyg_handle_t   serial_interrupt_handle;
} pl011_serial_info;

static bool pl011_serial_init(struct cyg_devtab_entry *tab);
static bool pl011_serial_putc(serial_channel *chan, unsigned char c);
static Cyg_ErrNo pl011_serial_lookup(struct cyg_devtab_entry **tab, 
                                  struct cyg_devtab_entry *sub_tab,
                                  const char *name);
static unsigned char pl011_serial_getc(serial_channel *chan);
static Cyg_ErrNo pl011_serial_set_config(serial_channel *chan, cyg_uint32 key,
                                      const void *xbuf, cyg_uint32 *len);
static void pl011_serial_start_xmit(serial_channel *chan);
static void pl011_serial_stop_xmit(serial_channel *chan);

static cyg_uint32 pl011_serial_ISR(cyg_vector_t vector, cyg_addrword_t data);
static void       pl011_serial_DSR(cyg_vector_t vector, cyg_ucount32 count,
                                cyg_addrword_t data);

static SERIAL_FUNS(pl011_serial_funs, 
                   pl011_serial_putc, 
                   pl011_serial_getc,
                   pl011_serial_set_config,
                   pl011_serial_start_xmit,
                   pl011_serial_stop_xmit
    );

//==========================================================================

#include CYGDAT_IO_SERIAL_PL011_INL

#ifndef CYG_IO_SERIAL_PL011_INT_PRIORITY
# define CYG_IO_SERIAL_PL011_INT_PRIORITY 4
#endif

#ifndef _UART_CLK
# define _UART_CLK(base) 24000000
#endif

//==========================================================================
// Internal function to actually configure the hardware to desired
// baud rate, etc.

static bool
serial_config_port(serial_channel *chan, 
                   cyg_serial_info_t *new_config, bool init)
{
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    unsigned int baud = select_baud[new_config->baud];
    unsigned long lcr, ier;

    if (baud == 0) return false;  // Invalid configuration

    HAL_READ_UINT32(base+_UART_IMSC, ier);
    
    // Disable UART while changing hardware
    HAL_WRITE_UINT32(base+_UART_CR, 0);

    lcr = select_word_length[new_config->word_length - CYGNUM_SERIAL_WORD_LENGTH_5] | 
        select_stop_bits[new_config->stop] |
        select_parity[new_config->parity];

    // Set baud rate
    // IBRD = UART_CLK / (16 * BPS)
    // FBRD = ROUND((64 * MOD(UART_CLK,(16 * BPS))) / (16 * BPS))
    {
        unsigned int temp      = 16 * baud;
        unsigned int ref_clk   = _UART_CLK(base);
        unsigned int divider   = ref_clk / temp;
        unsigned int remainder = ref_clk % temp;
        unsigned int fraction;
        
        temp      = (8 * remainder) / baud;
        fraction  = (temp >> 1) + (temp & 1);

        HAL_WRITE_UINT32(base+_UART_IBRD, divider);
        HAL_WRITE_UINT32(base+_UART_FBRD, fraction);
    }

    // Baud rate doesn't change until LCRH is written
    HAL_WRITE_UINT32(base+_UART_LCRH, lcr | _UART_LCRH_FEN );
    
    if (init) {

        // TODO: set FIFO interrupt levels
        // Currently default to 1/2 full
        
        if (chan->out_cbuf.len != 0) {
            ier = _UART_IMSC_RXIM|_UART_IMSC_RTIM;
        }
    }

    
#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS
    ier |= _UART_IMSC_OEIM|_UART_IMSC_PEIM|_UART_IMSC_FEIM|_UART_IMSC_BEIM;
    ier |= _UART_IMSC_DSRMIM|_UART_IMSC_DCDMIM|_UART_IMSC_CTSMIM|_UART_IMSC_RIMIM;
#endif

    HAL_WRITE_UINT32(base+_UART_IMSC, ier);

    // Enable UART
    {
        cyg_uint32 cr = _UART_CR_UARTEN | _UART_CR_TXE | _UART_CR_RXE | _UART_CR_RTS;

        if (new_config->flags & CYGNUM_SERIAL_FLOW_RTSCTS_RX)
            cr |= _UART_CR_RTSEN;
        if (new_config->flags & CYGNUM_SERIAL_FLOW_RTSCTS_TX)
            cr |= _UART_CR_CTSEN;
        
        HAL_WRITE_UINT32(base+_UART_CR, cr);
    }
    
    if (new_config != &chan->config) {
        chan->config = *new_config;
    }
    return true;
}

//==========================================================================
// Function to initialize the device.  Called at bootstrap time.

static bool 
pl011_serial_init(struct cyg_devtab_entry *tab)
{
    serial_channel *chan = (serial_channel *)tab->priv;
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;

#ifdef CYGDBG_IO_INIT
    diag_printf("pl011 SERIAL init - dev: %x.%d\n", 
                ser_chan->base, ser_chan->int_num);
#endif

#ifdef CYGHWR_IO_SERIAL_PL011_CONFIGURE
    CYGHWR_IO_SERIAL_PL011_CONFIGURE( ser_chan );
#endif
    
    // Really only required for interrupt driven devices
    (chan->callbacks->serial_init)(chan);

    if (chan->out_cbuf.len != 0) {
        cyg_drv_interrupt_create(ser_chan->int_num,
                                 CYG_IO_SERIAL_PL011_INT_PRIORITY,
                                 (cyg_addrword_t)chan,
                                 pl011_serial_ISR,
                                 pl011_serial_DSR,
                                 &ser_chan->serial_interrupt_handle,
                                 &ser_chan->serial_interrupt);
        cyg_drv_interrupt_attach(ser_chan->serial_interrupt_handle);
        cyg_drv_interrupt_unmask(ser_chan->int_num);
    }
    serial_config_port(chan, &chan->config, true);
    return true;
}

//==========================================================================
// This routine is called when the device is "looked" up (i.e. attached)

static Cyg_ErrNo 
pl011_serial_lookup(struct cyg_devtab_entry **tab, 
                 struct cyg_devtab_entry *sub_tab,
                 const char *name)
{
    serial_channel *chan = (serial_channel *)(*tab)->priv;

    // Really only required for interrupt driven devices
    (chan->callbacks->serial_init)(chan);
    return ENOERR;
}

//==========================================================================
// Send a character to the device output buffer.
// Return 'true' if character is sent to device

static bool
pl011_serial_putc(serial_channel *chan, unsigned char c)
{
    bool result = false;
    cyg_uint32 fr;
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;

    // Disable interrupts here briefly so that a potential TXINTR does
    // not vanish immedately and confuse the interrupt controller into
    // provoking a spurious interrupt. This only happens under very
    // high load on the AHB bus.
    
    cyg_drv_isr_lock();
    
    HAL_READ_UINT32(base+_UART_FR, fr);
    if ((fr & _UART_FR_TXFF) != _UART_FR_TXFF) {
        // Transmit fifo is not full
        HAL_WRITE_UINT32(base+_UART_DR, c);
        result = true;
    }
    // No space

    cyg_drv_isr_unlock();
    
    return result;
}

//==========================================================================
// Fetch a character from the device input buffer, waiting if necessary

static unsigned char 
pl011_serial_getc(serial_channel *chan)
{
    cyg_uint32 c;
    cyg_uint32 fr;
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;

    // Wait for char
    do {
        HAL_READ_UINT32(base+_UART_FR, fr);
    } while ((fr & _UART_FR_RXFF) == _UART_FR_RXFE);

    HAL_READ_UINT32(base+_UART_DR, c);
    return c&0xFF;
}

//==========================================================================
// Set up the device characteristics; baud rate, etc.

static Cyg_ErrNo
pl011_serial_set_config(serial_channel *chan, cyg_uint32 key, const void *xbuf,
                     cyg_uint32 *len)
{
    switch (key) {
    case CYG_IO_SET_CONFIG_SERIAL_INFO:
      {
        cyg_serial_info_t *config = (cyg_serial_info_t *)xbuf;
        if ( *len < sizeof(cyg_serial_info_t) ) {
            return -EINVAL;
        }
        *len = sizeof(cyg_serial_info_t);
        if ( true != serial_config_port(chan, config, false) )
            return -EINVAL;
      }
      break;
#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW
    case CYG_IO_SET_CONFIG_SERIAL_HW_RX_FLOW_THROTTLE:
      {
          // The hardware supports automatic RTS/CTS flow control
          // with its FIFOs, but we have to cope with our own
          // buffers overflowing.
          pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
          cyg_uint32 *f = (cyg_uint32 *)xbuf;
          cyg_uint32 cr;

          if ( *len < sizeof(*f) )
              return -EINVAL;
          HAL_READ_UINT32( ser_chan->base + _UART_CR, cr );
          // If we need to throttle rx, then that means disabling RTS.
          // And the PL011 docs indicate we also need to disable the
          // hardware's own RTS hardware flow control if we do that.
          if (*f) // we should throttle
          {
              cr &= (_UART_CR_RTSEN|_UART_CR_RTS);
          } else { // we should no longer throttle
              cr |= _UART_CR_RTSEN;
          }
          HAL_WRITE_UINT32( ser_chan->base + _UART_CR, cr );
      }
      break;
      
    case CYG_IO_SET_CONFIG_SERIAL_HW_FLOW_CONFIG:
        // Clear any unsupported flags here and then return -ENOSUPP -
        // the higher layer can then query what flags are set and
        // decide what to do. This is optimised for the most common
        // case - i.e. that authors know what their hardware is
        // capable of.

        // Clear DSR/DTR flag as it's not supported.
        if (chan->config.flags &
            (CYGNUM_SERIAL_FLOW_DSRDTR_RX|CYGNUM_SERIAL_FLOW_DSRDTR_TX))
        {
            chan->config.flags &= ~(CYGNUM_SERIAL_FLOW_DSRDTR_RX|
                                    CYGNUM_SERIAL_FLOW_DSRDTR_TX);
            return -EINVAL;
        }
      
        
      break;
#endif
    default:
        return -EINVAL;
    }
    return ENOERR;
}

//==========================================================================
// Enable the transmitter on the device

static void
pl011_serial_start_xmit(serial_channel *chan)
{
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_uint32 imsc;
    
    HAL_READ_UINT32(base+_UART_IMSC, imsc);
    imsc |= _UART_IMSC_TXIM;    
    HAL_WRITE_UINT32(base+_UART_IMSC, imsc);

    (chan->callbacks->xmt_char)(chan);    
}

//==========================================================================
// Disable the transmitter on the device

static void 
pl011_serial_stop_xmit(serial_channel *chan)
{
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_uint32 imsc;
    
    HAL_READ_UINT32(base+_UART_IMSC, imsc);
    imsc &= ~_UART_IMSC_TXIM;
    HAL_WRITE_UINT32(base+_UART_IMSC, imsc);
}

//==========================================================================
// Serial I/O - low level interrupt handler (ISR)

static cyg_uint32 
pl011_serial_ISR(cyg_vector_t vector, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
    cyg_drv_interrupt_mask(ser_chan->int_num);
    cyg_drv_interrupt_acknowledge(ser_chan->int_num);
    return CYG_ISR_CALL_DSR;  // Cause DSR to be run
}

//==========================================================================
// Serial I/O - high level interrupt handler (DSR)

static void       
pl011_serial_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    pl011_serial_info *ser_chan = (pl011_serial_info *)chan->dev_priv;
    cyg_addrword_t base = ser_chan->base;
    cyg_int32 mis;

    HAL_READ_UINT32(base+_UART_MIS, mis);
    while (mis != 0) {
        cyg_int32 cur = mis & -mis;
        switch (cur) {
        case _UART_IMSC_RXIM:
        case _UART_IMSC_RTIM:
        {
            cyg_uint32 fr;
            cyg_uint32 c;
            HAL_READ_UINT32(base+_UART_FR, fr);
            while((fr & _UART_FR_RXFE) == 0) {
                HAL_READ_UINT32(base+_UART_DR, c);
                (chan->callbacks->rcv_char)(chan, c&0xFF);
                HAL_READ_UINT32(base+_UART_FR, fr);
            }
            break;
        }
        case _UART_IMSC_TXIM:
            (chan->callbacks->xmt_char)(chan);
            break;

#ifdef CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS
            
        case _UART_IMSC_OEIM:
        case _UART_IMSC_PEIM:
        case _UART_IMSC_FEIM:
        case _UART_IMSC_BEIM:
        {
            cyg_serial_line_status_t stat;
            cyg_uint32 rsr;
            HAL_READ_UINT32(base+_UART_RSR, rsr);

            // this might look expensive, but it is rarely the case that
            // more than one of these is set
            stat.value = 1;
            if ( rsr & _UART_RSR_OE ) {
                stat.which = CYGNUM_SERIAL_STATUS_OVERRUNERR;
                (chan->callbacks->indicate_status)(chan, &stat );
            }
            if ( rsr & _UART_RSR_PE ) {
                stat.which = CYGNUM_SERIAL_STATUS_PARITYERR;
                (chan->callbacks->indicate_status)(chan, &stat );
            }
            if ( rsr & _UART_RSR_FE ) {
                stat.which = CYGNUM_SERIAL_STATUS_FRAMEERR;
                (chan->callbacks->indicate_status)(chan, &stat );
            }
            if ( rsr & _UART_RSR_BE ) {
                stat.which = CYGNUM_SERIAL_STATUS_BREAK;
                (chan->callbacks->indicate_status)(chan, &stat );
            }
        }
        break;
            
#ifdef CYGOPT_IO_SERIAL_FLOW_CONTROL_HW
        case _UART_IMSC_DSRMIM:
            if ( chan->config.flags & CYGNUM_SERIAL_FLOW_DSRDTR_TX ) {
                cyg_serial_line_status_t stat;
                cyg_uint32 fr;
                HAL_READ_UINT32(base+_UART_FR, fr);
                stat.which = CYGNUM_SERIAL_STATUS_FLOW;
                stat.value = (0 != (fr & _UART_FR_DSR));
                (chan->callbacks->indicate_status)(chan, &stat );
            }
            break;
        case _UART_IMSC_CTSMIM:
            if ( chan->config.flags & CYGNUM_SERIAL_FLOW_RTSCTS_TX ) {
                cyg_serial_line_status_t stat;
                cyg_uint32 fr;
                HAL_READ_UINT32(base+_UART_FR, fr);
                stat.which = CYGNUM_SERIAL_STATUS_FLOW;
                stat.value = (0 != (fr & _UART_FR_CTS));
                (chan->callbacks->indicate_status)(chan, &stat );
            }
            break;
#endif // CYGOPT_IO_SERIAL_FLOW_CONTROL_HW
        case _UART_IMSC_DCDMIM:
        {
            cyg_serial_line_status_t stat;
            cyg_uint32 fr;
            HAL_READ_UINT32(base+_UART_FR, fr);
            stat.which = CYGNUM_SERIAL_STATUS_CARRIERDETECT;
            stat.value = (0 != (fr & _UART_FR_DCD));
            (chan->callbacks->indicate_status)(chan, &stat );
        }
        break;
        case _UART_IMSC_RIMIM:
        {
            cyg_serial_line_status_t stat;
            cyg_uint32 fr;
            HAL_READ_UINT32(base+_UART_FR, fr);
            stat.which = CYGNUM_SERIAL_STATUS_RINGINDICATOR;
            stat.value = (0 != (fr & _UART_FR_RI));
            (chan->callbacks->indicate_status)(chan, &stat );
        }
        break;
#endif // CYGOPT_IO_SERIAL_SUPPORT_LINE_STATUS
        
        default:
            // Yes, this assertion may well not be visible. *But*
            // if debugging, we may still successfully hit a breakpoint
            // on cyg_assert_fail, which _is_ useful
            CYG_FAIL("unhandled serial interrupt state");
        }

        HAL_WRITE_UINT32(base+_UART_ICR, cur);
        HAL_READ_UINT32(base+_UART_MIS, mis);
    } // while

    cyg_drv_interrupt_unmask(ser_chan->int_num);
}

//==========================================================================

#endif // CYGDAT_IO_SERIAL_PL011_INL

//==========================================================================
// EOF ser_pl011.c
