//==========================================================================
//
//      fatfs3.c
//
//      Test fileio system
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2005, 2007 Free Software Foundation, Inc.
// Copyright (C) 2004, 2005, 2007 eCosCentric Limited                       
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           nickg
// Contributors:        nickg
// Date:                2000-05-25
// Purpose:             Test fileio system
// Description:         This test uses the testfs to check out the initialization
//                      and basic operation of the fileio system
//                      
//                      
//                      
//                      
//                      
//              
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>

#include CYGDAT_DEVS_DISK_CFG           // testing config defines

#include <cyg/kernel/ktypes.h>         // base kernel types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros

#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>

#include <cyg/fileio/fileio.h>

#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output

#ifdef JNGPKG_USB_STACK
extern int usb_stack_init(void);
#endif

//==========================================================================

#define SHOW_RESULT( _fn, _res )                                                                \
{                                                                                               \
    diag_printf("<FAIL>: " #_fn "() returned %ld %s\n", (long)_res, _res<0?strerror(errno):"");        \
}

//==========================================================================

#define IOSIZE  100

#define LONGNAME1       "long_file_name_that_should_take_up_more_than_one_directory_entry_1"
#define LONGNAME2       "long_file_name_that_should_take_up_more_than_one_directory_entry_2"


//==========================================================================

#ifndef CYGPKG_LIBC_STRING

char *strcat( char *s1, const char *s2 )
{
    char *s = s1;
    while( *s1 ) s1++;
    while( (*s1++ = *s2++) != 0);
    return s;
}

#endif

//==========================================================================

static void listdir( char *name, int statp, int numexpected, int *numgot )
{
    int err;
    DIR *dirp;
    int num=0;
    
    diag_printf("<INFO>: reading directory %s\n",name);
    
//    diag_printf("<INFO>: opendir(%s)\n",name);
    dirp = opendir( name );
    if( dirp == NULL ) SHOW_RESULT( opendir, -1 );

    for(;;)
    {
        struct dirent *entry = readdir( dirp );
        
        if( entry == NULL )
            break;
        num++;
        diag_printf("<INFO>: entry %14s",entry->d_name);
        if( statp )
        {
            char fullname[PATH_MAX];
            struct stat sbuf;

            if( name[0] )
            {
                strcpy(fullname, name );
                if( !(name[0] == '/' && name[1] == 0 ) )
                    strcat(fullname, "/" );
            }
            else fullname[0] = 0;
            
            strcat(fullname, entry->d_name );
            
            err = stat( fullname, &sbuf );
            if( err < 0 )
            {
                if( errno == ENOSYS )
                    diag_printf(" <no status available>");
                else SHOW_RESULT( stat, err );
            }
            else
            {
                diag_printf(" [mode %08x ino %08x nlink %d size %ld]",
                            sbuf.st_mode,sbuf.st_ino,sbuf.st_nlink,sbuf.st_size);
            }
        }

        diag_printf("\n");
    }

//    diag_printf("<INFO>: closedir(%s)\n",name);    
    err = closedir( dirp );
    if( err < 0 ) SHOW_RESULT( stat, err );
    if (numexpected >= 0 && num != numexpected)
        CYG_TEST_FAIL("Wrong number of dir entries\n");
    if ( numgot != NULL )
        *numgot = num;
}

//==========================================================================

static void createfile( char *name, size_t size )
{
    char buf[IOSIZE];
    int fd;
    ssize_t wrote;
    int i;
    int err;

    diag_printf("<INFO>: create file %s size %ld\n",name,size);

    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );
    
    for( i = 0; i < IOSIZE; i++ ) buf[i] = i%256;
 
    fd = open( name, O_WRONLY|O_CREAT );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    while( size > 0 )
    {
        ssize_t len = size;
        if ( len > IOSIZE ) len = IOSIZE;
        
        wrote = write( fd, buf, len );
        if( wrote != len ) SHOW_RESULT( write, wrote );        

        size -= wrote;
    }

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

#if 0
static void maxfile( char *name )
{
    char buf[IOSIZE];
    int fd;
    ssize_t wrote;
    int i;
    int err;
    size_t size = 0;
    size_t prevsize = 0;
    
    diag_printf("<INFO>: create maximal file %s\n",name);
    diag_printf("<INFO>: This may take a few minutes\n");

    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );
    
    for( i = 0; i < IOSIZE; i++ ) buf[i] = i%256;
 
    fd = open( name, O_WRONLY|O_CREAT );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    do
    {
        wrote = write( fd, buf, IOSIZE );
        //if( wrote < 0 ) SHOW_RESULT( write, wrote );        

        if( wrote >= 0 )
            size += wrote;

        if( (size-prevsize) > 100000 )
        {
            diag_printf("<INFO>: size = %ld \n", size);
            prevsize = size;
        }
        
    } while( wrote == IOSIZE );

    diag_printf("<INFO>: file size == %ld\n",size);

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}
#endif

//==========================================================================

static void checkfile( char *name )
{
    char buf[IOSIZE];
    int fd;
    ssize_t done;
    int i;
    int err;
    off_t pos = 0;

    diag_printf("<INFO>: check file %s\n",name);
    
    err = access( name, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    fd = open( name, O_RDONLY );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    for(;;)
    {
        done = read( fd, buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, done );

        if( done == 0 ) break;

        for( i = 0; i < done; i++ )
            if( buf[i] != i%256 )
            {
                diag_printf("buf[%ld+%d](%02x) != %02x\n",pos,i,buf[i],i%256);
                CYG_TEST_FAIL("Data read not equal to data written\n");
                for(;;);
            }
        
        pos += done;
    }

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

static void copyfile( char *name2, char *name1 )
{

    int err;
    char buf[IOSIZE];
    int fd1, fd2;
    ssize_t done, wrote;

    diag_printf("<INFO>: copy file %s -> %s\n",name2,name1);

    err = access( name1, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    err = access( name2, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );
    
    fd1 = open( name1, O_WRONLY|O_CREAT );
    if( fd1 < 0 ) SHOW_RESULT( open, fd1 );

    fd2 = open( name2, O_RDONLY );
    if( fd2 < 0 ) SHOW_RESULT( open, fd2 );
    
    for(;;)
    {
        done = read( fd2, buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, done );

        if( done == 0 ) break;

        wrote = write( fd1, buf, done );
        if( wrote != done ) SHOW_RESULT( write, wrote );

        if( wrote != done ) break;
    }

    err = close( fd1 );
    if( err < 0 ) SHOW_RESULT( close, err );

    err = close( fd2 );
    if( err < 0 ) SHOW_RESULT( close, err );
    
}

//==========================================================================

static void comparefiles( char *name2, char *name1 )
{
    int err;
    char buf1[IOSIZE];
    char buf2[IOSIZE];
    int fd1, fd2;
    ssize_t done1, done2;
    int i;

    diag_printf("<INFO>: compare files %s == %s\n",name2,name1);

    err = access( name1, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    err = access( name1, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );
    
    fd1 = open( name1, O_RDONLY );
    if( fd1 < 0 ) SHOW_RESULT( open, fd1 );

    fd2 = open( name2, O_RDONLY );
    if( fd2 < 0 ) SHOW_RESULT( open, fd2 );
    
    for(;;)
    {
        done1 = read( fd1, buf1, IOSIZE );
        if( done1 < 0 ) SHOW_RESULT( read, done1 );

        done2 = read( fd2, buf2, IOSIZE );
        if( done2 < 0 ) SHOW_RESULT( read, done2 );

        if( done1 != done2 )
            diag_printf("Files different sizes\n");
        
        if( done1 == 0 ) break;

        for( i = 0; i < done1; i++ )
            if( buf1[i] != buf2[i] )
            {
                diag_printf("buf1[%d](%02x) != buf1[%d](%02x)\n",i,buf1[i],i,buf2[i]);
                CYG_TEST_FAIL("Data in files not equal\n");
            }
    }

    err = close( fd1 );
    if( err < 0 ) SHOW_RESULT( close, err );

    err = close( fd2 );
    if( err < 0 ) SHOW_RESULT( close, err );
    
}

//==========================================================================

void checkcwd( const char *cwd )
{
    static char cwdbuf[PATH_MAX];
    char *ret;

    ret = getcwd( cwdbuf, sizeof(cwdbuf));
    if( ret == NULL ) SHOW_RESULT( getcwd, ret );    

    if( strcmp( cwdbuf, cwd ) != 0 )
    {
        diag_printf( "cwdbuf %s cwd %s\n",cwdbuf, cwd );
        CYG_TEST_FAIL( "Current directory mismatch");
    }
}

//==========================================================================
// show memory allocated
// returns free memory size

static int show_memory(void)
{
    struct mallinfo info;

    info = mallinfo();

    diag_printf("Heap Info:\n");
    diag_printf("Arena Size: %d top: %d maxfree: %d\n",
                info.arena,info.keepcost,info.maxfree);
    diag_printf("Ordinary Blocks: %d used: %d free %d\n",
                info.ordblks, info.uordblks, info.fordblks);
    diag_printf("Small Blocks: %d used: %d free %d\n",
                info.smblks, info.usmblks, info.fsmblks);

    return info.fordblks+info.fsmblks;
}

//==========================================================================
// Change the disk

#ifdef CYGDAT_IO_DISK_ECOSYNTH_DISK0_FILENAME
externC cyg_bool synth_disk_change( int unit, char *filename, int size,
                                    int cyls, int heads, int sectors);
#endif
    
static void change_disk(void)
{
#ifdef CYGDAT_IO_DISK_ECOSYNTH_DISK0_FILENAME

    synth_disk_change( 0,
                       "/tmp/disk1.img",
                       CYGNUM_IO_DISK_ECOSYNTH_DISK0_SIZE,
                       CYGIMP_IO_DISK_ECOSYNTH_DISK0_CYLINDERS,
                       CYGIMP_IO_DISK_ECOSYNTH_DISK0_HEADS,
                       CYGIMP_IO_DISK_ECOSYNTH_DISK0_SECTORS);
#endif
}

//==========================================================================
// main

void fileio3_main( CYG_ADDRESS id )
{
    int err;
    int existingdirents=-1;
    int fd;
    int freemem, curfree;
    
    char dirname[64];
    char filename1[64];
    char filename2[64];

#define NEW_FILENAME( fn, f )                           \
{                                                       \
    fn[0]=0;                                            \
    strcat(fn, dirname);                                \
    strcat(fn, "/");                                    \
    strcat(fn,f);                                       \
}

#define NEW_FILENAME1( f ) NEW_FILENAME( filename1, f )
#define NEW_FILENAME2( f ) NEW_FILENAME( filename2, f )
    
#ifdef JNGPKG_USB_STACK
    err = usb_stack_init();
    cyg_thread_delay( 1000 );
#endif

    CYG_TEST_INIT();
    
    // --------------------------------------------------------------

    freemem = show_memory();
    
    err = mount( CYGDAT_DEVS_DISK_TEST_DEVICE, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, "fatfs" );    
    if( err < 0 ) {
        SHOW_RESULT( mount, err );
        CYG_TEST_FAIL_FINISH("fatfs3");
    }

    err = access( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    if( err < 0 && errno == EACCES )
    {
        diag_printf("<INFO>: create %s\n",
                    CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY);
        err = mkdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY, 0 );
        if( err < 0 ) SHOW_RESULT( mkdir, err );
    }

    err = chdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY );    
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY );
    
    listdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY, true, -1, &existingdirents );

    // --------------------------------------------------------------
    
    dirname[0] = 0;
    strcat(dirname, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );
    strcat(dirname, CYGDAT_DEVS_DISK_TEST_DIRECTORY );
    
    // --------------------------------------------------------------

    createfile( "foo", 20257 );
    checkfile( "foo" );
    copyfile( "foo", "fee");
    checkfile( "fee" );
    comparefiles( "foo", "fee" );
    diag_printf("<INFO>: mkdir bar\n");
    err = mkdir( "bar", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    listdir( dirname , true, existingdirents+3, NULL );

    copyfile( "fee", "bar/fum" );
    checkfile( "bar/fum" );
    comparefiles( "fee", "bar/fum" );

    diag_printf("<INFO>: cd bar\n");
    err = chdir( "bar" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    NEW_FILENAME1( "bar" );
    checkcwd( filename1 );
    
    diag_printf("<INFO>: rename ../foo bundy\n");    
    err = rename( "../foo", "bundy" );
    if( err < 0 ) SHOW_RESULT( rename, err );
    
    listdir( dirname, true, existingdirents+2, NULL );
    listdir( "." , true, 4, NULL );

    checkfile( "../bar/bundy" );
    comparefiles("../fee", "bundy" );

    // --------------------------------------------------------------
    // Check for some common error cases

    diag_printf("<INFO>: rename fum bundy -- should fail\n");    
    err = rename( "fum", "bundy" );
    if( err < 0 && errno != EEXIST ) SHOW_RESULT( rename, err );

    NEW_FILENAME1( "x/y" )
    diag_printf("<INFO>: rename fum %s -- should fail\n",filename1);    
    err = rename( "fum", filename1 );
    if( err < 0 && errno != ENOENT ) SHOW_RESULT( rename, err );
    
    diag_printf("<INFO>: unlink fee -- should fail\n");        
    err = unlink( "fee" );
    if( err < 0 && errno != ENOENT ) SHOW_RESULT( unlink, err );

    NEW_FILENAME1( "bar" );
    diag_printf("<INFO>: rmdir %s -- should fail\n",filename1);        
    err = rmdir( filename1 );
    if( err < 0 && errno != EBUSY ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: unlink %s -- should fail\n",filename1);        
    err = unlink( filename1 );
    if( err < 0 && errno != EBUSY ) SHOW_RESULT( unlink, err );

    NEW_FILENAME1( "bar/fum" );
    diag_printf("<INFO>: rmdir %s -- should fail\n",filename1);        
    err = rmdir( filename1 );
    if( err < 0 && errno != ENOENT ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: mkdir %s -- should fail\n",filename1);        
    err = mkdir( filename1, 0 );
    if( err < 0 && errno != EEXIST ) SHOW_RESULT( mkdir, err );
    
    diag_printf("<INFO>: open bundy exclusive -- should fail\n");        
    fd = open( "bundy", O_WRONLY|O_CREAT|O_EXCL );
    if( fd < 0 && errno != EEXIST ) SHOW_RESULT( open, fd );

    diag_printf("<INFO>: open %s -- should fail\n",dirname);        
    fd = open( dirname, O_WRONLY );
    if( fd < 0 && errno != EISDIR ) SHOW_RESULT( open, fd );
    
    
    listdir( dirname, true, existingdirents+2, NULL );
    listdir( "." , true, 4, NULL );
    
    // --------------------------------------------------------------    

    diag_printf("<INFO>: unlink ../fee\n");    
    err = unlink( "../fee" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink fum\n");        
    err = unlink( "fum" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink ../bar/bundy\n");        
    err = unlink( "../bar/bundy" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: cd ..\n");        
    err = chdir( ".." );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( dirname );
    
    diag_printf("<INFO>: rmdir bar\n");        
    err = rmdir( "bar" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );
    
    listdir( dirname, false, existingdirents, NULL );

    // --------------------------------------------------------------

    diag_printf("<INFO>: cd /\n");    
    err = chdir( "/" );
    if( err == 0 )
    {
        checkcwd( "/" );

        curfree = show_memory();

        diag_printf("<INFO>: umount %s\n",CYGDAT_DEVS_DISK_TEST_MOUNTPOINT);    
        err = umount( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );
        if( err < 0 ) SHOW_RESULT( umount, err );
    }

    diag_printf("<INFO>: syncing\n");    
    sync();

    curfree = show_memory();

    if( curfree < freemem )
        diag_printf("<FAIL>: Memory leak detected: curfree %d < %d\n",curfree,freemem);
    
    // --------------------------------------------------------------

    change_disk();

    // --------------------------------------------------------------

    diag_printf("<INFO>: mount %s\n", CYGDAT_DEVS_DISK_TEST_MOUNTPOINT);    
    err = mount( CYGDAT_DEVS_DISK_TEST_DEVICE, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, "fatfs" );

    if( err < 0 ) {
        SHOW_RESULT( mount, err );
        CYG_TEST_FAIL_FINISH("fatfs3");
    }

    show_memory();

    err = access( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    if( err < 0 && errno == EACCES )
    {
        diag_printf("<INFO>: create %s\n",
                    CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY);
        err = mkdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT CYGDAT_DEVS_DISK_TEST_DIRECTORY, 0 );
        if( err < 0 ) SHOW_RESULT( mkdir, err );
    }
    
    listdir( dirname , true, -1, &existingdirents);

    NEW_FILENAME1( "tinky" );
    NEW_FILENAME2( "laalaa" );
    
    createfile( filename1, 4567 );
    copyfile( filename1, filename2 );
    checkfile( filename1);
    checkfile( filename2);
    comparefiles( filename1, filename2 );

    diag_printf("<INFO>: cd %s\n",dirname);    
    err = chdir( dirname );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( dirname );
        
    diag_printf("<INFO>: mkdir noonoo\n");    
    err = mkdir( "noonoo", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    listdir( dirname , true, existingdirents+3, NULL);

    diag_printf("<INFO>: cd noonoo\n");
    err = chdir( "noonoo" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    NEW_FILENAME1( "noonoo" );
    checkcwd( filename1 );
    
    createfile( "tinky", 6789 );
    checkfile( "tinky" );

    createfile( "dipsy", 34567 );
    checkfile( "dipsy" );
    copyfile( "dipsy", "po" );
    checkfile( "po" );
    comparefiles( "dipsy", "po" );

    listdir( ".", true, 5, NULL );
    listdir( "..", true, existingdirents+3, NULL );

    // --------------------------------------------------------------

    diag_printf("<INFO>: unlink tinky\n");    
    err = unlink( "tinky" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink dipsy\n");    
    err = unlink( "dipsy" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink po\n");    
    err = unlink( "po" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: cd ..\n"); 
    err = chdir( ".." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( dirname );
    
    diag_printf("<INFO>: rmdir noonoo\n"); 
    err = rmdir( "noonoo" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    // --------------------------------------------------------------

    err = mkdir( "x", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );
    
    err = mkdir( "x/y", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );
    
    err = mkdir( "x/y/z", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    err = mkdir( "x/y/z/w", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    NEW_FILENAME1( "x/y/z/w" );
    diag_printf("<INFO>: cd %s\n",filename1);
    err = chdir( filename1 );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( filename1 );

    NEW_FILENAME1( "x/y/z" );    
    diag_printf("<INFO>: cd ..\n");
    err = chdir( ".." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( filename1 );

    NEW_FILENAME1( "x/y/z" );
    diag_printf("<INFO>: cd .\n");
    err = chdir( "." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( filename1 );

    NEW_FILENAME1( "x/y" );    
    diag_printf("<INFO>: cd ../../y\n");
    err = chdir( "../../y" );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( filename1 );

    diag_printf("<INFO>: cd ../..\n");
    err = chdir( "../.." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( dirname );

    diag_printf("<INFO>: rmdir x/y/z/w\n"); 
    err = rmdir( "x/y/z/w" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: rmdir x/y/z\n"); 
    err = rmdir( "x/y/z" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: rmdir x/y\n"); 
    err = rmdir( "x/y" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: rmdir x\n"); 
    err = rmdir( "x" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );
    
    // --------------------------------------------------------------

    checkcwd( dirname );
    
    diag_printf("<INFO>: unlink tinky\n");    
    err = unlink( "tinky" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink laalaa\n");    
    err = unlink( "laalaa" );
    if( err < 0 ) SHOW_RESULT( unlink, err );


    // --------------------------------------------------------------

    diag_printf("<INFO>: cd /\n");    
    err = chdir( "/" );
    if( err == 0 )
    {
        checkcwd( "/" );
    
        diag_printf("<INFO>: umount %s\n",CYGDAT_DEVS_DISK_TEST_MOUNTPOINT);    
        err = umount( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );
        if( err < 0 ) SHOW_RESULT( umount, err );
    }

    diag_printf("<INFO>: syncing\n");    
    sync();

    curfree = show_memory();

    if( curfree < freemem )
        diag_printf("<FAIL>: Memory leak detected: curfree %d < %d\n",curfree,freemem);
    
    // --------------------------------------------------------------

#if 0
    maxfile("file.max");

    listdir( "/", true, -1, NULL );    
        
    diag_printf("<INFO>: unlink file.max\n");    
    err = unlink( "file.max" );
    if( err < 0 ) SHOW_RESULT( unlink, err );    
#endif

    
    CYG_TEST_PASS_FINISH("fatfs3");
}

// -------------------------------------------------------------------------

#include <cyg/kernel/kapi.h>

static cyg_handle_t thread_handle;
static cyg_thread thread;
static char stack[CYGNUM_HAL_STACK_SIZE_TYPICAL+1024];

externC void
cyg_start( void )
{
    cyg_thread_create(3,                // Priority - just a number
                      fileio3_main,     // entry
                      0,                // index
                      0,                // no name
                      &stack[0],        // Stack
                      sizeof(stack),    // Size
                      &thread_handle,   // Handle
                      &thread           // Thread data structure
        );
    cyg_thread_resume(thread_handle);

    cyg_scheduler_start();
}

// -------------------------------------------------------------------------
// EOF fatfs3.c
