//==========================================================================
//
//      fatfs7.c
//
//      Test FAT UTF-8 support
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2005, 2006, 2007 Free Software Foundation, Inc.
// Copyright (C) 2004, 2005, 2006, 2007 eCosCentric Limited                 
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           nickg
// Contributors:        nickg
// Date:                2005-07-25
// Purpose:             Test fileio system
// Description:         This test uses the testfs to check out the initialization
//                      and basic operation of the fileio system.
//                      This is essentially a copy of fatfs4 with UTF8 encoded
//                      file names.
//                      
//                       
//                      
//                      
//              
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>
#include <pkgconf/fs_fat.h>

#include CYGDAT_DEVS_DISK_CFG           // testing config defines

#include <cyg/kernel/ktypes.h>         // base kernel types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>

#include <cyg/fileio/fileio.h>

#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output

#ifdef JNGPKG_USB_STACK
extern int usb_stack_init(void);
#endif

//==========================================================================

#define SHOW_RESULT( _fn, _res )                                                                \
{                                                                                               \
    diag_printf("<FAIL>: " #_fn "() returned %ld %s\n", (long)_res, _res<0?strerror(errno):""); \
}

//==========================================================================

#define IOSIZE  100

//==========================================================================

#ifndef CYGPKG_LIBC_STRING

char *strcat( char *s1, const char *s2 )
{
    char *s = s1;
    while( *s1 ) s1++;
    while( (*s1++ = *s2++) != 0);
    return s;
}

#endif

//==========================================================================

#if 1 //ndef CYGCFG_FS_FAT_LONG_FILE_NAMES
static void listdir( char *name, int statp, int numexpected, int *numgot )
{
    int err;
    DIR *dirp;
    int num=0;
    
    diag_printf("<INFO>: reading directory %s\n",name);
    
//    diag_printf("<INFO>: opendir(%s)\n",name);
    dirp = opendir( name );
    if( dirp == NULL ) SHOW_RESULT( opendir, -1 );

    for(;;)
    {
        struct dirent *entry = readdir( dirp );
        
        if( entry == NULL )
            break;
        num++;

       diag_printf("<INFO>: entry ");
        
        if( statp )
        {
            char fullname[PATH_MAX];
            struct stat sbuf;

            if( name[0] )
            {
                strcpy(fullname, name );
                if( !(name[0] == '/' && name[1] == 0 ) )
                    strcat(fullname, "/" );
            }
            else fullname[0] = 0;
            
            strcat(fullname, entry->d_name );
            
            err = stat( fullname, &sbuf );
            if( err < 0 )
            {
                if( errno == ENOSYS )
                    diag_printf(" <no status available>");
                else SHOW_RESULT( stat, err );
            }
            else
            {
                diag_printf(" [mode %08x ino %08x nlink %d size %6ld]",
                            sbuf.st_mode,sbuf.st_ino,sbuf.st_nlink,sbuf.st_size);
            }
        }

        diag_printf(" %s\n",entry->d_name);
    }

//    diag_printf("<INFO>: closedir(%s)\n",name);    
    err = closedir( dirp );
    if( err < 0 ) SHOW_RESULT( stat, err );
    if (numexpected >= 0 && num != numexpected)
        CYG_TEST_FAIL("Wrong number of dir entries");
    if ( numgot != NULL )
        *numgot = num;
}

//==========================================================================

static void createfile( char *name, size_t size )
{
    char buf[IOSIZE];
    int fd;
    ssize_t wrote;
    int i;
    int err;

    diag_printf("<INFO>: create file %s size %ld\n",name,size);

    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );
    
    for( i = 0; i < IOSIZE; i++ ) buf[i] = i%256;
 
    fd = open( name, O_WRONLY|O_CREAT );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    while( size > 0 )
    {
        ssize_t len = size;
        if ( len > IOSIZE ) len = IOSIZE;
        
        wrote = write( fd, buf, len );
        if( wrote != len ) SHOW_RESULT( write, wrote );        

        size -= wrote;
    }

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

#if 0
static void maxfile( char *name )
{
    char buf[IOSIZE];
    int fd;
    ssize_t wrote;
    int i;
    int err;
    size_t size = 0;
    size_t prevsize = 0;
    
    diag_printf("<INFO>: create maximal file %s\n",name);
    diag_printf("<INFO>: This may take a few minutes\n");

    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );
    
    for( i = 0; i < IOSIZE; i++ ) buf[i] = i%256;
 
    fd = open( name, O_WRONLY|O_CREAT );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    do
    {
        wrote = write( fd, buf, IOSIZE );
        //if( wrote < 0 ) SHOW_RESULT( write, wrote );        

        if( wrote >= 0 )
            size += wrote;

        if( (size-prevsize) > 100000 )
        {
            diag_printf("<INFO>: size = %ld \n", size);
            prevsize = size;
        }
        
    } while( wrote == IOSIZE );

    diag_printf("<INFO>: file size == %ld\n",size);

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}
#endif

//==========================================================================

static void checkfile( char *name )
{
    char buf[IOSIZE];
    int fd;
    ssize_t done;
    int i;
    int err;
    off_t pos = 0;

    diag_printf("<INFO>: check file %s\n",name);
    
    err = access( name, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    fd = open( name, O_RDONLY );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    for(;;)
    {
        done = read( fd, buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, done );

        if( done == 0 ) break;

        for( i = 0; i < done; i++ )
            if( buf[i] != i%256 )
            {
                diag_printf("buf[%ld+%d](%02x) != %02x\n",pos,i,buf[i],i%256);
                CYG_TEST_FAIL("Data read not equal to data written\n");
                for(;;);
            }
        
        pos += done;
    }

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

static void copyfile( char *name2, char *name1 )
{

    int err;
    char buf[IOSIZE];
    int fd1, fd2;
    ssize_t done, wrote;

    diag_printf("<INFO>: copy file %s -> %s\n",name2,name1);

    err = access( name1, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    err = access( name2, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );
    
    fd1 = open( name1, O_WRONLY|O_CREAT );
    if( fd1 < 0 ) SHOW_RESULT( open, fd1 );

    fd2 = open( name2, O_RDONLY );
    if( fd2 < 0 ) SHOW_RESULT( open, fd2 );
    
    for(;;)
    {
        done = read( fd2, buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, done );

        if( done == 0 ) break;

        wrote = write( fd1, buf, done );
        if( wrote != done ) SHOW_RESULT( write, wrote );

        if( wrote != done ) break;
    }

    err = close( fd1 );
    if( err < 0 ) SHOW_RESULT( close, err );

    err = close( fd2 );
    if( err < 0 ) SHOW_RESULT( close, err );
    
}

//==========================================================================

static void comparefiles( char *name2, char *name1 )
{
    int err;
    char buf1[IOSIZE];
    char buf2[IOSIZE];
    int fd1, fd2;
    ssize_t done1, done2;
    int i;

    diag_printf("<INFO>: compare files %s == %s\n",name2,name1);

    err = access( name1, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    err = access( name1, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );
    
    fd1 = open( name1, O_RDONLY );
    if( fd1 < 0 ) SHOW_RESULT( open, fd1 );

    fd2 = open( name2, O_RDONLY );
    if( fd2 < 0 ) SHOW_RESULT( open, fd2 );
    
    for(;;)
    {
        done1 = read( fd1, buf1, IOSIZE );
        if( done1 < 0 ) SHOW_RESULT( read, done1 );

        done2 = read( fd2, buf2, IOSIZE );
        if( done2 < 0 ) SHOW_RESULT( read, done2 );

        if( done1 != done2 )
            diag_printf("Files different sizes\n");
        
        if( done1 == 0 ) break;

        for( i = 0; i < done1; i++ )
            if( buf1[i] != buf2[i] )
            {
                diag_printf("buf1[%d](%02x) != buf1[%d](%02x)\n",i,buf1[i],i,buf2[i]);
                CYG_TEST_FAIL("Data in files not equal\n");
            }
    }

    err = close( fd1 );
    if( err < 0 ) SHOW_RESULT( close, err );

    err = close( fd2 );
    if( err < 0 ) SHOW_RESULT( close, err );
    
}

#endif

//==========================================================================

void checkcwd( const char *cwd )
{
    static char cwdbuf[PATH_MAX];
    char *ret;

    ret = getcwd( cwdbuf, sizeof(cwdbuf));
    if( ret == NULL ) SHOW_RESULT( getcwd, ret );    

    if( strcmp( cwdbuf, cwd ) != 0 )
    {
        diag_printf( "cwdbuf %s cwd %s\n",cwdbuf, cwd );
        CYG_TEST_FAIL( "Current directory mismatch");
    }
}

//==========================================================================
// Common code page translation structure.

#ifdef CYGCFG_FS_FAT_LONG_FILE_NAMES

typedef unsigned short mbcs_t;
typedef unsigned short unicode_t;

struct cp2unicode_row
{
    int             row_index;
    int             row_base;
    int             row_size;
    unicode_t       *codes;
};
struct cp2unicode        
{
    int     col_base;
    int     col_count;
    struct cp2unicode_row **row;
};

#define __DONT_DEFINE_STRUCTS__

#include "cyrillic.c"
#include "korean.c"

//-----------------------------------------------------------------------------
// Common code page decoding function.

//#define dbprintf diag_printf
#define dbprintf(__fmt, ... )

unicode_t decode( struct cp2unicode *cp2uc, mbcs_t mc )
{
    int col = mc>>8;
    int i;

    for( i = 0; i < cp2uc->col_count; i++ )
    {
        struct cp2unicode_row *r = cp2uc->row[i];
        int row = mc&0xff;

        if( r->row_index != col )
            continue;

        row -= r->row_base;

        if( row < 0 || row > r->row_size )
            break;

        return r->codes[row];
    }

    return 0xFFFF;
}

# if (CYG_BYTEORDER == CYG_MSBFIRST)
   // need to swap endian-ness on big-endian targets
#  define SWAP_UNICODE(__ucode) ((((__ucode)&0xFF) << 8) | (((__ucode) >> 8)&0xFF))
# elif (CYG_BYTEORDER == CYG_LSBFIRST)
#  define SWAP_UNICODE(__ucode) (__ucode)
# else
#  error SWAP_UNICODE not defined
# endif

//-----------------------------------------------------------------------------
// Cyrillic translation routines. The Cyrillic code page only contains
// single byte characters.

static int mbcs_to_utf16le_cyrillic( CYG_ADDRWORD data, const cyg_uint8 *mbcs, int size, cyg_uint16 *utf16le)
{
    struct cp2unicode *cp2uc = (struct cp2unicode *)data;
    int result = 0;
    mbcs_t mb;
    unicode_t uc;
    const cyg_uint8 *p = mbcs;

    dbprintf("<INFO>: mbcs_to_utf16le_cyrillic()\n");
    {
        int i;
        dbprintf("<INFO>:    mbcs[%d]: ", size );
        for( i = 0; i < size; i++ )
            dbprintf("  %02x ", mbcs[i] );
        dbprintf("\n");
    }
    
    while( p-mbcs < size )
    {
        mb = *p++;

        uc = decode( cp2uc, mb );

        if( uc == 0xFFFF )
            break;

        utf16le[result++] = SWAP_UNICODE(uc);
    }

    {
        int i;
        dbprintf("<INFO>: utf16le[%d]: ", result );
        for( i = 0; i < result; i++ )
            dbprintf("%04x ", utf16le[i] );
        dbprintf("\n");
    }
    
    return result;
}

static int utf16le_to_mbcs_cyrillic( CYG_ADDRWORD data, const cyg_uint16 *utf16le, int size, cyg_uint8 *mbcs)
{
    dbprintf("utf16le_to_mbcs_cyrillic\n");
    data = data;
    utf16le = utf16le;
    size = size;
    mbcs = mbcs;
    return 0;
}


//-----------------------------------------------------------------------------
// Korean translation routines. Korean The korean codepage contains
// 2byte characters.

static int mbcs_to_utf16le_korean( CYG_ADDRWORD data, const cyg_uint8 *mbcs, int size, cyg_uint16 *utf16le)
{
    struct cp2unicode *cp2uc = (struct cp2unicode *)data;
    int result = 0;
    mbcs_t mb;
    unicode_t uc;
    const cyg_uint8 *p = mbcs;

    dbprintf("<INFO>: mbcs_to_utf16le_korean()\n");
    {
        int i;
        dbprintf("<INFO>:    mbcs[%d]: ", size );
        for( i = 0; i < size; i++ )
            dbprintf("  %02x ", mbcs[i] );
        dbprintf("\n");
    }
    
    while( p-mbcs < size )
    {
        mb = *p++;

        dbprintf("mb %x\n", mb);
        if( mb >= 0x80 )
        {
            mb <<= 8;
            mb |= *p++;
            dbprintf("mb %x\n", mb);            
        }

        uc = decode( cp2uc, mb );

        if( uc == 0xFFFF )
            break;

        utf16le[result++] = SWAP_UNICODE(uc);
    }

    {
        int i;
        dbprintf("<INFO>: utf16le[%d]: ", result );
        for( i = 0; i < result; i++ )
            dbprintf("%04x ", utf16le[i] );
        dbprintf("\n");
    }
    
    return result;
}

static int utf16le_to_mbcs_korean( CYG_ADDRWORD data, const cyg_uint16 *utf16le, int size, cyg_uint8 *mbcs)
{
    dbprintf("utf16le_to_mbcs_korean\n");
    data = data;
    utf16le = utf16le;
    size = size;
    mbcs = mbcs;
    return 0;
}

#endif // CYGCFG_FS_FAT_LONG_FILE_NAMES

//==========================================================================

void dir_list (char * dirpath, int depth) {
	DIR * dir;
	struct dirent * entry;
	struct stat sbuf;
	int result, i;
	
	/* open the directory */
	dir = opendir (dirpath);
	
	/* loop over all directory entries */
	result = chdir (dirpath);
	while ((entry = readdir (dir))) {
	  	if (stat (entry->d_name, &sbuf))
			break;
		diag_printf ("<INFO>: ");
		for (i = 0; i < depth; i++)
			diag_printf ("        ");
		if (S_ISDIR (sbuf.st_mode)) { /* if a directory */
			diag_printf ("%s/\n", entry->d_name);
			if ('.' != entry->d_name [0])
				dir_list (entry->d_name, depth + 1); /* recurse */
		} else /* a regular file */
			diag_printf ("%s [%ld bytes]\n", entry->d_name, sbuf.st_size);
	}
	result = chdir ("..");
	
	/* close the directory */
	result = closedir (dir);
	return;		
}

//-----------------------------------------------------------------------------

void dir_clean (char * dirpath )
{
    DIR * dir;
    struct dirent * entry0, entry;
    struct stat sbuf;
    int result;

    /* open the directory */
    dir = opendir (dirpath);

    if( dir == NULL )
        return;
	
    /* loop over all directory entries */
    result = chdir (dirpath);
    if( result < 0 ) SHOW_RESULT( chdir, result );
    
    while ((entry0 = readdir (dir)))
    {
        entry = *entry0;
        result = stat (entry.d_name, &sbuf);
        if( result < 0 ) SHOW_RESULT( stat, result );
        
        if ( result != 0 )
            break;
        if (S_ISDIR (sbuf.st_mode))
        {
            /* if a directory */
            if ('.' != entry.d_name [0])
            {
                dir_clean (entry.d_name); /* recurse */
                result = rmdir( entry.d_name );
                if( result < 0 ) SHOW_RESULT( rmdir, result );
            }
        }
        else
        {
            /* a regular file */
            result = unlink(entry.d_name );
            if( result < 0 ) SHOW_RESULT( unlink, result );
        }
    }
    result = chdir ("..");
    if( result < 0 ) SHOW_RESULT( chdir, result );
	
    /* close the directory */
    result = closedir (dir);
    if( result < 0 ) SHOW_RESULT( closedir, result );
    return;		
}

//==========================================================================
// Test control information

struct test_info
{
    char        *dir;
    char        *file1;
    char        *file2;
    char        *file3;
    char        *file4;
    char        *file5;    
    char        *dir2;

    struct cyg_fs_mbcs_translate translate;
};

//==========================================================================
//
// This program behaves differently depending on whether long filename
// (LFN) support is enabled in the filesystem.
//
// When LFN is disabled it tests that standard 8.3 DOS names are 8 bit
// clean when they contain non-ASCII characters. At the end of its
// run, this program leaves the files it has created in place.
//
// When LFN is enabled it looks for the files that a non-LFN run left
// and checks that the code page to UNICODE character code translation
// routines are functioning correctly.



#ifndef CYGCFG_FS_FAT_LONG_FILE_NAMES

//==========================================================================
//
// The filenames in the following info structures are encoded
// according to Microsoft code page 1251 for cyrillic and code page
// 949 for korean.

struct test_info cyrillic_test_info =
{
    .dir        = "cyrillic",
    .file1      = "\xc0\xc1\xc2\xc3\xc4\xc5",
    .file2      = "\xc6\xc7\xc8\xc9\xca\xcb",
    .file3      = "A\xe0\xe1\xe2\xe3_12.\x9c\x9d\x9e",
    .file4      = "\xf8\xf9\xfa\xfb\xfc",
    .dir2       = "\xc8\xc9\xcamNo.dir",
};

struct test_info korean_test_info =
{
    .dir        = "korean",
    .file1      = "\xa4\x41\xa4\x42\xa4\x43\xa4\x44",
    .file2      = "\xb0\x61\xb0\x62\xb2\x63",
    .file3      = "A\xc1\x50\xc1\x51_12.\xc1\x52z",
    .file4      = "\xd8\xa1\xd8\xa2",
    .dir2       = "\xf2\xc0\xf2\xc1mNo.dir",
};

void fatfs7_test( struct test_info *info )
{
    int err;
    int existingdirents=-1;
    char file3[40];
    char file4[20];

    diag_printf("\n<INFO>: Running non-LFN tests in %s\n", info->dir);
    
    err = access( info->dir, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    if( err < 0 && errno == EACCES )
    {
        diag_printf("<INFO>: create %s\n", info->dir );
        err = mkdir( info->dir, 0 );
        if( err < 0 ) SHOW_RESULT( mkdir, err );
    }

    dir_clean( info->dir );
//    dir_list( info->dir, 0 );
    
    err = chdir( info->dir );    
    if( err < 0 ) SHOW_RESULT( chdir, err );

    listdir( ".", true, -1, &existingdirents );

    createfile ( info->file1, 31223 );
    checkfile( info->file1 );
    copyfile( info->file1, info->file2 );
    comparefiles( info->file1, info->file2 );
    
    diag_printf("<INFO>: mkdir %s\n", info->dir2);
    err = mkdir( info->dir2, 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    strcpy( file3, info->dir2 );
    strcat( file3, "/" );
    strcat( file3, info->file3 );
    
    diag_printf("<INFO>: rename %s -> %s\n", info->file2, file3);    
    err = rename( info->file2, file3 );
    if( err < 0 ) SHOW_RESULT( rename, err );

    err = chdir( info->dir2 );    
    if( err < 0 ) SHOW_RESULT( chdir, err );

    strcpy(file4, info->file4); strcat(file4, ".0");
    createfile( file4, 12345 );    
    strcpy(file4, info->file4); strcat(file4, ".1");
    createfile( file4, 12345 );
    strcpy(file4, info->file4); strcat(file4, ".2");
    createfile( file4, 12345 );
    strcpy(file4, info->file4); strcat(file4, ".3");
    createfile( file4, 12345 );
    strcpy(file4, info->file4); strcat(file4, ".4");
    createfile( file4, 12345 );
    strcpy(file4, info->file4); strcat(file4, ".5");
    createfile( file4, 12345 );
    strcpy(file4, info->file4); strcat(file4, ".6");
    createfile( file4, 12345 );
    
    chdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );

    dir_list( info->dir, 0 );
}

#else

//==========================================================================
//
// The filenames in the following structures are encoded in UTF-8 and
// are the same names as the code page encoding given above.
//
// WARNING: Only edit this file with an editor that is UTF-8 aware (I
// use EMACS on Linux, but Notepad on Windows also works). Saving this
// file from an unaware editor could cause these strings to be
// corrupted and make this test fail. If the strings below do not
// appear as clean Cyrillic characters then exit now without saving
// and look for a better editor.

struct test_info cyrillic_test_info =
{
    .dir        = "cyrillic",
    .file1      = "АБВГДЕ",
    .file3      = "Aабвг_12.њќћ",
    .file4      = "шщъыь",
    .file5      = "Спутник",
    .dir2       = "ИЙКmNo.dir",
    
    .translate.mbcs_to_utf16le = mbcs_to_utf16le_cyrillic,
    .translate.utf16le_to_mbcs = utf16le_to_mbcs_cyrillic,
    .translate.data = (CYG_ADDRWORD)&cyrillic_table
};

struct test_info korean_test_info =
{
    .dir        = "korean",
    .file1      = "\xEC\xA7\x9E\xEC\xA7\x9F\xEC\xA7\xA1\xEC\xA7\xA3",
    .file2      = "\xEC\xBA\xBB\xEC\xBA\xBC\xEC\xBA\xBD",
    .file3      = "A\xED\x95\xBB\xED\x95\xBD_12.\xED\x95\xBEz",
    .file4      = "\xE7\xAB\x8B\xE7\xAC\xA0",
    .file5      = "\xE9\x85\x8B\xE9\x86\x9C\xE9\x8C\x90\xE9\x8C\x98\xE9\x8E\x9A",
    .dir2       = "\xE9\x81\xB2\xE7\x9B\xB4mNo.dir",
    
    .translate.mbcs_to_utf16le = mbcs_to_utf16le_korean,
    .translate.utf16le_to_mbcs = utf16le_to_mbcs_korean,
    .translate.data = (CYG_ADDRWORD)&korean_table
};


void fatfs7_test( struct test_info *info )
{
    int err;
    int existingdirents = -1;
    char file3[40];
    char file4[20];
    int i;
    
    diag_printf("\n<INFO>: Running LFN tests in %s\n", info->dir);

    // Install translation routines.
    err = cyg_fs_setinfo( ".",
                          FS_INFO_MBCS_TRANSLATE,
                          &info->translate,
                          sizeof(info->translate));
    if( err < 0 ) SHOW_RESULT( cyg_fs_setinfo, err );
    
//    dir_list( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, 0 );

    err = access( info->dir, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    if( err < 0 && errno == EACCES )
    {
        diag_printf("<INFO>: %s not present, tests not run\n", info->dir );
        return;
    }
    
    err = chdir( info->dir );    
    if( err < 0 ) SHOW_RESULT( chdir, err );

#if 1
    listdir( ".", true, -1, &existingdirents );
    
    checkfile( info->file1 );
    strcpy( file3, info->dir2 );
    strcat( file3, "/" );
    strcat( file3, info->file3 );
    checkfile( file3 );
    comparefiles( info->file1, file3 );

    copyfile( file3, info->file5 );
    comparefiles( file3, info->file5 );
    
    err = chdir( info->dir2 );    
    if( err < 0 ) SHOW_RESULT( chdir, err );
#endif
    
#if 1
    for( i = 0; i < 10; i++ )
    {
        char ext[4];
        int fd;

        strcpy(file4, info->file4 );
        ext[0] = '.';
        ext[1] = '0' + i;
        ext[2] = '\0';
        strcat( file4, ext);
        
        err = access( file4, F_OK );
        if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

        if( err < 0 && errno == EACCES )
            continue;

        checkfile( file4 );
        createfile( info->file5, 3456 );
        checkfile( info->file5 );

        diag_printf("<INFO>: rename %s -> %s -- should fail\n", info->file5, file4);    
        err = rename( info->file5, file4 );
        if( err == 0 ) SHOW_RESULT( rename, err );

        diag_printf("<INFO>: create %s -- should fail\n", file4);    
        fd = open( file4, O_WRONLY|O_CREAT|O_EXCL );
        if( fd >= 0 ) SHOW_RESULT( open, fd );

        diag_printf("<INFO>: rename %s -> %s -- should fail\n", file4, file4);    
        err = rename( file4, file4 );
        if( err == 0 ) SHOW_RESULT( rename, err );

        diag_printf("<INFO>: unlink %s\n", info->file5 );
        err = unlink( info->file5 );
        if( err < 0 ) SHOW_RESULT( unlink, err );
        
        diag_printf("<INFO>: rename %s -> %s\n", file4, info->file5);    
        err = rename( file4, info->file5 );
        if( err < 0 ) SHOW_RESULT( rename, err );

        checkfile( info->file5 );
        
        diag_printf("<INFO>: unlink %s\n", info->file5 );
        err = unlink( info->file5 );
        if( err < 0 ) SHOW_RESULT( unlink, err );
        
        break;
    }

#endif
    chdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );

    dir_list( info->dir, 0 );
    
}

#endif

//==========================================================================
// main

char *mountpoint = CYGDAT_DEVS_DISK_TEST_MOUNTPOINT;

void fatfs7_main( CYG_ADDRESS id )
{
    int err;
//    int existingdirents=-1;
//    cyg_bool mp2 = false;
//    int fd;
    
//    char dirname[256];
//    char filename1[256];
//    char filename2[256];

        
#ifdef JNGPKG_USB_STACK
    err = usb_stack_init();
    cyg_thread_delay( 1000 );
#endif
    
    CYG_TEST_INIT();
    
    // --------------------------------------------------------------

    err = mount( CYGDAT_DEVS_DISK_TEST_DEVICE, CYGDAT_DEVS_DISK_TEST_MOUNTPOINT, "fatfs" );    
    if( err < 0 ) {
        SHOW_RESULT( mount, err );
        CYG_TEST_FAIL_FINISH("fatfs7");
    }

    chdir( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    
    // --------------------------------------------------------------

#ifdef CYGCFG_FS_FAT_LONG_FILE_NAMES


#endif
    
    // --------------------------------------------------------------
    
    fatfs7_test( &cyrillic_test_info );
    fatfs7_test( &korean_test_info );
    
    // --------------------------------------------------------------

    diag_printf("<INFO>: cd /\n");    
    err = chdir( "/" );
    if( err == 0 )
    {
        checkcwd( "/" );
    
        diag_printf("<INFO>: umount %s\n",CYGDAT_DEVS_DISK_TEST_MOUNTPOINT);    
        err = umount( CYGDAT_DEVS_DISK_TEST_MOUNTPOINT );
        if( err < 0 ) SHOW_RESULT( umount, err );
    }

    diag_printf("<INFO>: syncing\n");    
    sync();
    
    CYG_TEST_PASS_FINISH("fatfs7");
}

// -------------------------------------------------------------------------

#include <cyg/kernel/kapi.h>

static cyg_handle_t thread_handle;
static cyg_thread thread;
static char stack[CYGNUM_HAL_STACK_SIZE_TYPICAL+1024*8];

externC void
cyg_start( void )
{
    cyg_thread_create(3,                // Priority - just a number
                      fatfs7_main,      // entry
                      0,                // index
                      0,                // no name
                      &stack[0],        // Stack
                      sizeof(stack),    // Size
                      &thread_handle,   // Handle
                      &thread           // Thread data structure
        );
    cyg_thread_resume(thread_handle);

    cyg_scheduler_start();
}

// -------------------------------------------------------------------------
// EOF fatfs7.c
