/*=============================================================================
//
//      timers.c
//
//      Test for STM32 Timers
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2008 Free Software Foundation, Inc.                        
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    nickg
// Date:         2008-09-11
//              
//####DESCRIPTIONEND####
//
//===========================================================================*/

#include <pkgconf/system.h>
#include <pkgconf/hal.h>

#if defined(CYGPKG_KERNEL)
#include <pkgconf/kernel.h>
#endif

#include <cyg/infra/testcase.h>

//=============================================================================
// Check all required packages and components are present

#if !defined(CYGPKG_KERNEL) || !defined(CYGPKG_KERNEL_API)

#define NA_MSG  "Configuration insufficient"

#endif

//=============================================================================
// If everything is present, compile the full test.

#ifndef NA_MSG

#include <cyg/hal/hal_arch.h>
#include <cyg/hal/hal_io.h>
#include <cyg/hal/hal_if.h>

#include <cyg/kernel/kapi.h>
#include <cyg/infra/diag.h>
#include <string.h>

//=============================================================================
// Configuration options
//
// These options control various parameters of the test to allow
// different system aspects to be investigated.
//
// LOOPS        The test prints results every 5 seconds. This controls how
//              many such loops it will perform.
//
// LOITER       When set the ISR and DSR will loop for a while before returning.
//              This increases the chance of another interrupt preempting it.
//              Of course this will have a disasterous effect on the latency of
//              any lower priority interrupts.
//
// DSRS         Cause each timer's DSR to be called one in every 10 interrupts.
//              This allows interrupting of DSRS to be tested.
//
// MHZ          Frequency of timers in MHz. Increasing this increases latency
//              measurement accuracy, but care is needed not to exceed the timer
//              input frequency, or allow the timer interval to exceed 65535.
//
// SYSTICK      If set the kernel SysTick timer is allowed to run. Disabling this
//              will remove some jitter from the latency measurements.
//
// DELAY        Use cyg_thread_delay() for timing the main loop. Otherwise the main
//               thread will loop for the necessary number of interrupts from timer 1
//              to complete. The latter mechanism will also be used if SYSTICK is
//              disabled. Disabling this will remove some jitter from the latency
//              measurements.
//
// LATENCY      Measure ISR latency. This is reported as a maximum and average over
//              each 5s cycle, plus a distibution histogram over the entire run.
//              Generally, the figures for timer 1 will show the raw hardware latency,
//              plus any system effects (mainly from the SysTick timer). Figures for
//              other timers will show accumulated latency waiting for higher priority
//              ISRs to complete.
//
// LATENCY_HIST Number of entries in latency histogram.
//
// LATENCY_BASE Timer tick count at which to start histogram. Some number of low latency
//              figures will never be seen. Instead these can be omitted to provide
//              space for more higher values. The default of 7 gives some leeway for
//              hardware variations.

#define LOOPS           24      // == 2 minutes

#define LOITER          1       // Loiter in ISR & DSR

#define DSRS            1       // Use and report DSRs

//#define MHZ              5      // Timer frequency in MHz
#define MHZ             10      // Timer frequency in MHz
//#define MHZ             16      // Timer frequency in MHz
//#define MHZ             32
//#define MHZ             48

#define SYSTICK         1       // Allow SYSTICK to run

#define DELAY           1       // Use thread delay for main loop


#define LATENCY         1       // Report interrupt latency figures

#define LATENCY_HIST    25      // Latency histogram size



#if MHZ<10
#define LATENCY_BASE    0       // Latency histogram base value
#elif MHZ < 12
#define LATENCY_BASE    7       // Latency histogram base value
#elif MHZ < 20
#define LATENCY_BASE    15      // Latency histogram base value
#else
#define LATENCY_BASE    65      // Latency histogram base value
#endif


//=============================================================================

#define STACK_SIZE 2000

static int test_stack[(STACK_SIZE/sizeof(int))];
static cyg_thread test_thread;
static cyg_handle_t main_thread;

//=============================================================================

struct timer
{
    cyg_uint32          timer;
    cyg_uint32          base;
    cyg_uint32          vector;
    cyg_uint32          clkena;
    cyg_uint32          priority;
    cyg_uint32          interval;

    cyg_uint32          prescaler;
    cyg_uint32          frequency;      // Real timer frequency
    cyg_uint32          ns_per_clock;   // ns per timer count
    
    cyg_uint32          ticks;

    cyg_uint32          preempt[10];

    cyg_uint32          preempt_dsr[10];
    cyg_uint32          dsr_count[10];

#if LATENCY
    cyg_uint32          latency_hist[LATENCY_HIST];
    cyg_int32           latency_max;
    cyg_uint32          latency_total;
    cyg_uint32          latency_samples;
#endif
    
    cyg_interrupt       interrupt_object;
    cyg_handle_t        interrupt_handle;
};

struct timer timers[] =
{
#if 0
    { 1, CYGHWR_HAL_STM32_TIM1, CYGNUM_HAL_INTERRUPT_TIM1_UP, CYGHWR_HAL_STM32_TIM1_CLOCK, 0x20,    1000 },    
#elif 1
    { 1, CYGHWR_HAL_STM32_TIM1, CYGNUM_HAL_INTERRUPT_TIM1_UP, CYGHWR_HAL_STM32_TIM1_CLOCK, 0x20,     127 },
    { 2, CYGHWR_HAL_STM32_TIM2, CYGNUM_HAL_INTERRUPT_TIM2,    CYGHWR_HAL_STM32_TIM2_CLOCK, 0x30,     355 },
    { 3, CYGHWR_HAL_STM32_TIM3, CYGNUM_HAL_INTERRUPT_TIM3,    CYGHWR_HAL_STM32_TIM3_CLOCK, 0x40,     731 },
    { 4, CYGHWR_HAL_STM32_TIM4, CYGNUM_HAL_INTERRUPT_TIM4,    CYGHWR_HAL_STM32_TIM4_CLOCK, 0x50,     999 },
    { 5, CYGHWR_HAL_STM32_TIM5, CYGNUM_HAL_INTERRUPT_TIM5,    CYGHWR_HAL_STM32_TIM5_CLOCK, 0x60,    1453 },
    { 6, CYGHWR_HAL_STM32_TIM6, CYGNUM_HAL_INTERRUPT_TIM6,    CYGHWR_HAL_STM32_TIM6_CLOCK, 0x70,    1931 },
    { 7, CYGHWR_HAL_STM32_TIM7, CYGNUM_HAL_INTERRUPT_TIM7,    CYGHWR_HAL_STM32_TIM7_CLOCK, 0x80,    2011 },
#ifdef CYGNUM_HAL_INTERRUPT_TIM8_UP
    { 8, CYGHWR_HAL_STM32_TIM8, CYGNUM_HAL_INTERRUPT_TIM8_UP, CYGHWR_HAL_STM32_TIM8_CLOCK, 0x90,    2345 },
#endif
#endif
    { 0, 0, 0, 0 }
};

//=============================================================================

volatile cyg_uint32 ticks = 0;
volatile cyg_uint32 nesting = 0;
volatile cyg_uint32 max_nesting = 0;
volatile cyg_uint32 max_nesting_seen = 0;
volatile cyg_uint32 current = 0;
volatile cyg_uint32 in_dsr = 0;
volatile cyg_uint32 started = 0;

cyg_uint32 ntimers = 0;

#if !(DELAY && SYSTICK)

volatile cyg_uint32 tim1_ticks;

#endif

//=============================================================================

__externC cyg_uint32 hal_stm32_pclk1;
__externC cyg_uint32 hal_stm32_pclk2;

void init_timer( struct timer *t )
{
    cyg_uint32 base = t->base;
    cyg_uint32 interval = t->interval;
    cyg_uint32 prescaler;
    cyg_uint32 source_clock = hal_stm32_pclk1;
    ntimers++;
    
#if LATENCY
        t->latency_max = -1;
#endif
    
    if( base == CYGHWR_HAL_STM32_TIM1 || base == CYGHWR_HAL_STM32_TIM8 )
    {
        source_clock = hal_stm32_pclk2;
        if( CYGHWR_HAL_CORTEXM_STM32_CLOCK_PCLK2_DIV != 1 )
            source_clock *= 2;
    }
    else
    {
        if( CYGHWR_HAL_CORTEXM_STM32_CLOCK_PCLK1_DIV != 1 )
            source_clock *= 2;
    }

    t->prescaler = prescaler = source_clock / (1000000*MHZ);
    t->frequency = source_clock/prescaler;
    t->ns_per_clock = 1000000000/t->frequency;

    if( (interval*MHZ) > 0xFFFF )
    {
        t->interval = interval = 1000;
    }
    
    HAL_WRITE_UINT32(base+CYGHWR_HAL_STM32_TIM_PSC, prescaler-1 );

    HAL_WRITE_UINT32(base+CYGHWR_HAL_STM32_TIM_CR2, 0 );

    HAL_WRITE_UINT32(base+CYGHWR_HAL_STM32_TIM_DIER, CYGHWR_HAL_STM32_TIM_DIER_UIE );

    HAL_WRITE_UINT32(base+CYGHWR_HAL_STM32_TIM_ARR, interval*MHZ );
    
    HAL_WRITE_UINT32(base+CYGHWR_HAL_STM32_TIM_CR1, CYGHWR_HAL_STM32_TIM_CR1_CEN);
}

//=============================================================================

cyg_uint32 timer_isr( cyg_uint32 vector, CYG_ADDRWORD data )
{
    struct timer *t = (struct timer *)data;
    cyg_uint32 preempt = current;
    CYG_ADDRWORD base = t->base;
    cyg_uint32 cnt;
    cyg_uint32 ret = 1;

    HAL_READ_UINT32( base+CYGHWR_HAL_STM32_TIM_CNT, cnt );

    if( !started )
    {
        HAL_WRITE_UINT32(t->base+CYGHWR_HAL_STM32_TIM_SR, 0 );
        return ret;
    }
        
    
    current = t->timer;
    t->ticks++;
    ticks++;
    t->preempt[preempt]++;
    nesting++;

#if LATENCY
    // skip first reading after printout, since it will have been
    // delayed considerably.
    if( t->latency_max == -1 ) 
        t->latency_max = 0;
    else
    {
        t->latency_total += cnt;
        t->latency_samples++;
        if( cnt > t->latency_max )
            t->latency_max = cnt;
        if( cnt >= LATENCY_HIST+LATENCY_BASE ) cnt = LATENCY_HIST-1+LATENCY_BASE;
        if( cnt >= LATENCY_BASE )
            t->latency_hist[cnt-LATENCY_BASE]++;
    }
#endif
    
    // Count only first ISR to preempt a DSR
    if( preempt == 0 )
        t->preempt_dsr[in_dsr]++;
    
    HAL_WRITE_UINT32(t->base+CYGHWR_HAL_STM32_TIM_SR, 0 );
    
    if( nesting > max_nesting )
        max_nesting = nesting;

#if LOITER && defined(__OPTIMIZE__)
    // Loiter here for a proportion of the timer interval to give
    // other timers the chance to preempt us.
    do
    {
        HAL_READ_UINT32( base+CYGHWR_HAL_STM32_TIM_CNT, cnt );
    } while( cnt < t->interval/10 );
#endif
#if !(DELAY && SYSTICK)
    if( t->timer == 1 )
        tim1_ticks += t->interval;
#endif

    nesting--;
    current = preempt;

#if DSRS
    if( (t->ticks % 10) == 0 )
        ret |= 2;
#endif
    return ret;
}

//=============================================================================

void timer_dsr( cyg_uint32 vector, cyg_uint32 count, CYG_ADDRWORD data )
{
    struct timer *t = (struct timer *)data;
    
    in_dsr = t->timer;

    if( count >= 8 )
        count = 8;

    t->dsr_count[count]++;

#if LOITER && defined(__OPTIMIZE__)
    // Loiter for a while
    int i;
    for( i = 0; i < t->interval/10; i++)
        continue;
#endif

    in_dsr = 0;
}

//=============================================================================

void print4( cyg_uint32 val )
{
    if( val <= 9999 )
        diag_printf(" %4d", val );
    else if( val <= 999999 )
        diag_printf(" %3dk", val/1000 );
    else
        diag_printf(" %3dM", val/10000000 );
}


//=============================================================================

void
timers_test(cyg_addrword_t data)
{
    int loops = LOOPS;
    int i;
    CYG_INTERRUPT_STATE istate;
    
    CYG_TEST_INIT();
    
    CYG_TEST_INFO("Start Timers test");

    diag_printf("Options:\n");
    diag_printf("\tLOOPS        %d\n", LOOPS );
    diag_printf("\tLOITER       %d\n", LOITER );
    diag_printf("\tDSRS         %d\n", DSRS );
    diag_printf("\tMHZ          %d\n", MHZ );
    diag_printf("\tSYSTICK      %d\n", SYSTICK );
    diag_printf("\tDELAY        %d\n", DELAY );
    diag_printf("\tLATENCY      %d\n", LATENCY );
    diag_printf("\tLATENCY_HIST %d\n", LATENCY_HIST );
    diag_printf("\tLATENCY_BASE %d\n", LATENCY_BASE );


    diag_printf("PCLK1 %10dHz\n", hal_stm32_pclk1 );
    diag_printf("PCLK2 %10dHz\n", hal_stm32_pclk2 );
    
    diag_printf("T  Interval Frequency    Tick Prescaler Vector Priority\n");
    //           X:  XXXXXus XXXXXXXXX XXXXXns     XXXXX    XXX      XXX
    for( i = 0; timers[i].timer != 0; i++ )
    {
        struct timer *t = &timers[i];

        CYGHWR_HAL_STM32_CLOCK_ENABLE( t->clkena );

        init_timer( t );

        cyg_interrupt_create( t->vector,
                              t->priority,
                              (cyg_addrword_t)t,
                              timer_isr,
                              timer_dsr,
                              &t->interrupt_handle,
                              &t->interrupt_object
            );

        cyg_interrupt_attach( t->interrupt_handle );
        cyg_interrupt_unmask( t->vector );

        diag_printf("%1d:  %5dus %9d %5dns     %5d    %3d      %3d\n",
                    t->timer, t->interval, t->frequency, t->ns_per_clock, t->prescaler, t->vector, t->priority);
//    diag_printf("%08x %d %d %d\n", base, interval, prescaler, interval*MHZ );
    
        
    }

#if !SYSTICK
    // Clear enable bit of SysTick timer
    HAL_WRITE_UINT32(CYGARC_REG_SYSTICK_BASE+CYGARC_REG_SYSTICK_CSR, 0 );
#endif
    
    started = 1;
    
    while( loops-- )
    {
        int j;

        // 5 second delay
#if DELAY && SYSTICK
        cyg_thread_delay( 5*100 );
#else
        tim1_ticks = 0;
        while( tim1_ticks <= 5000000 )
            HAL_IDLE_THREAD_ACTION( 0 );
//            continue;
#endif

        // Disable interrupts while we print details, otherwise it
        // comes out very slowly.
        HAL_DISABLE_INTERRUPTS( istate );
        
        if( max_nesting > max_nesting_seen )
            max_nesting_seen = max_nesting;
        
        diag_printf("\nISRs max_nesting %d max_nesting_seen %d\n", max_nesting, max_nesting_seen );
        max_nesting = 0;

        diag_printf("T  Ticks");

        for( j = 0; j < ntimers+1; j++ )
            print4( j );
        diag_printf("\n");
            
        for( i = 0; timers[i].timer != 0; i++ )
        {
            struct timer *t = &timers[i];

            diag_printf("%1d: ", t->timer);
            print4( t->ticks );

            for( j = 0; j < ntimers+1; j++ )
                print4( t->preempt[j] );
            diag_printf("\n");

        }

#if DSRS
        diag_printf("DSRs\n");

        diag_printf("T:           ");

        for( j = 0; j < ntimers+1; j++ )
            print4( j );
        diag_printf("\n");
        
        for( i = 0; timers[i].timer != 0; i++ )
        {
            struct timer *t = &timers[i];

            diag_printf("%1d:  preempt: ", t->timer);
            
            for( j = 0; j < ntimers+1; j++ )
                print4( t->preempt_dsr[j] );
            diag_printf("\n");

            diag_printf("      count: ");
            for( j = 0; j < ntimers+1; j++ )
                print4( t->dsr_count[j] );
            diag_printf("\n");
        }
#endif

#if LATENCY
        diag_printf("ISR Latency\n");
        diag_printf(" T:  Max  Ave  Histogram (ns)...\n");
        diag_printf("      ns   ns");

        for( j = 0; j < LATENCY_HIST; j++ )
            print4( (j+LATENCY_BASE)*timers[0].ns_per_clock );
        diag_printf("+\n");
        
        for( i = 0; timers[i].timer != 0; i++ )
        {
            struct timer *t = &timers[i];
            cyg_uint32 ave = t->latency_total/t->latency_samples;
            
            diag_printf("%2d:", t->timer );
            print4( t->latency_max*t->ns_per_clock );
            print4( ave*t->ns_per_clock);
//            print4( t->latency_max );
//            print4( ave );
            
            t->latency_max = -1;
            t->latency_total = 0;
            t->latency_samples = 0;
            
            for( j = 0; j < LATENCY_HIST; j++ )
                print4( t->latency_hist[j] );
            diag_printf("\n");
        }
#endif
        
        HAL_RESTORE_INTERRUPTS( istate );        
    }

    CYG_TEST_PASS_FINISH("Timers test");
}

//=============================================================================

void cyg_user_start(void)
{
    cyg_thread_create(0,                // Priority
                      timers_test,
                      0,               
                      "timers test",    // Name
                      test_stack,       // Stack
                      STACK_SIZE,       // Size
                      &main_thread,     // Handle
                      &test_thread      // Thread data structure
        );
    cyg_thread_resume( main_thread);
}

//=============================================================================
// Print a message if we cannot run

#else // NA_MSG

void cyg_user_start(void)
{
    CYG_TEST_NA(NA_MSG);
}

#endif // NA_MSG

//=============================================================================
/* EOF timers.c */
