//==========================================================================
//
//      disktest.c
//
//      Disk driver test
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2006, 2007 Free Software Foundation, Inc.
// Copyright (C) 2004, 2006, 2007 eCosCentric Limited                       
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           nickg
// Contributors:        nickg
// Date:                2005-11-25
// Purpose:             Test disk driver system
// Description:         This program tests the functionality of the disk
//                      driver infrastructure and of any physical disk driver.
//                      It does this mainly by attempting to determine the
//                      performance characteristics of the disk.
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>

#include CYGDAT_DEVS_DISK_CFG           // testing config defines

#include <cyg/kernel/ktypes.h>         // base kernel types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros


#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output

#include <cyg/io/io.h>
#include <cyg/io/disk.h>

#include <cyg/kernel/kapi.h>

#include <stdlib.h>

//==========================================================================

typedef struct
{
    cyg_int64           ticks;
    cyg_int32           halticks;
} timestamp;

typedef struct
{
    timestamp           start;
    timestamp           end;
    cyg_int64           interval;       // In HAL ticks
    cyg_int64           us_interval;    // Interval in microseconds
} timing;

#define TIMINGS         2000

static timing           timings[TIMINGS];

static cyg_int64        ticks_overhead = 0;
static cyg_int32        us_per_haltick = 0;
static cyg_int32        halticks_per_us = 0;

static cyg_int64        rtc_resolution[] = CYGNUM_KERNEL_COUNTERS_RTC_RESOLUTION;
static cyg_int64        rtc_period = CYGNUM_KERNEL_COUNTERS_RTC_PERIOD;

//==========================================================================

#ifdef CYGDAT_DEVS_DISK_TEST_DEVICE
# define TEST_DEV CYGDAT_DEVS_DISK_TEST_DEVICE
#else
# define TEST_DEV "/dev/hd0/0"
#endif

//#define BLOCKSIZE       (8*1024*1024)
//#define BLOCKSIZE       (1*1024*1024)
#define BLOCKSIZE       (512*1024)
//#define BLOCKSIZE       (512)

#define BLOCKEXTRA      (1024)
#if CYGINT_IO_DISK_ALIGN_BUFS_TO_CACHELINE
# include <cyg/hal/hal_cache.h>
# define BLOCKALIGN     (HAL_DCACHE_LINE_SIZE)
#else
# define BLOCKALIGN     (1)
#endif

cyg_uint8               *block;

cyg_uint8               sector_buf[512] CYGBLD_ATTRIB_ALIGN(BLOCKALIGN);

//==========================================================================

//#define MAPSIZE 1000
//#define MAPSIZE 300
#define MAPSIZE 100
//#define MAPSIZE 10

cyg_uint32              sectormap[MAPSIZE];

cyg_io_handle_t         device;

cyg_uint32              sector_size;
cyg_uint32              disk_size;
cyg_uint32              phys_block_size;

//==========================================================================

static char date_string[] = __DATE__;
static char time_string[] = __TIME__;

cyg_uint32 timehash;

//==========================================================================

static void wait_for_tick( void )
{
    cyg_tick_count_t now = cyg_current_time();

    while( cyg_current_time() == now )
        continue;
}

static void get_timestamp( timestamp *ts )
{
    ts->ticks = cyg_current_time();
    HAL_CLOCK_READ( &ts->halticks );
}

static cyg_int64 ticks_to_us( cyg_int64 ticks )
{
    cyg_int64 us;
    
    if( us_per_haltick != 0 )
        us = ticks * us_per_haltick;
    else
        us = ticks / halticks_per_us;

    return us;
}

static void calculate_interval( timing *t )
{
    t->interval = t->end.halticks - t->start.halticks;

    t->interval += (t->end.ticks - t->start.ticks) * rtc_period;

    t->interval -= ticks_overhead;
    
    t->us_interval = ticks_to_us( t->interval );
}

//==========================================================================

static void read_sector( cyg_uint32 sector )
{
    int         result;
    cyg_uint32  size    = 1;

//    diag_printf("read_sector %d\n", sector );
    
    result = cyg_io_bread( device, sector_buf, &size, sector  );

    if( result != ENOERR || size != 1 )
        diag_printf("Sector read failed %d size %d\n", result, size );
}

//==========================================================================

static void write_sector( cyg_uint32 sector )
{
    int         result;
    cyg_uint32  size    = 1;

//    diag_printf("write_sector %d\n", sector );
    
    result = cyg_io_bwrite( device, sector_buf, &size, sector  );

    if( result != ENOERR || size != 1 )
        diag_printf("Sector Write failed %d size %d\n", result, size );
}

//==========================================================================

static void read_block( cyg_uint32 sector, int size )
{
    int         result;

    size /= sector_size;

//    diag_printf("read_block %d\n", sector );    
    
    result = cyg_io_bread( device, block, &size, sector  );

    if( result != ENOERR )
        diag_printf("Block read failed %d size %d\n", result, size );
}

//==========================================================================

static void write_block( cyg_uint32 sector, int size )
{
    int         result;

    size /= sector_size;
    
//    diag_printf("write_block %d\n", sector );
    
    result = cyg_io_bwrite( device, block, &size, sector  );

    if( result != ENOERR )
        diag_printf("Block Write failed %d size %d\n", result, size );
}

//==========================================================================

static void init_block( cyg_uint32 sector, int size )
{
    int i = 0;
    cyg_uint32 *b = (cyg_uint32 *)block;

    while( i < size/sizeof(cyg_uint32) )
    {
        b[i] = i;
        i++;
        b[i++] = sector;
        b[i++] = timehash;
        b[i++] = 0x12345678;
    }

    memset( block + size, 0xcc, BLOCKEXTRA );
}

//==========================================================================

static void check_block( cyg_uint32 sector, int size )
{
    int i = 0;
    cyg_uint32 *b = (cyg_uint32 *)block;

    while( i < size/sizeof(cyg_uint32) )
    {
        if( b[i] != i )
        {
            diag_printf("Block check failure: %d[%d] : %08x != %08x\n", sector, i, b[i] , i);
            break;
        }
        i++;
        if( b[i] != sector )
        {
            diag_printf("Block check failure: %d[%d] : %08x != %08x\n", sector, i, b[i] , sector);
            break;
        }
        i++;
        if( b[i] != timehash )
        {
            diag_printf("Block check failure: %d[%d] : %08x != %08x\n", sector, i, b[i] , timehash);
            break;
        }
        i++;
        if( b[i] != 0x12345678 )
        {
            diag_printf("Block check failure: %d[%d] : %08x != %08x\n", sector, i, b[i] , 0x12345678);
            break;
        }            
        i++;
    }

    for( i = 0 ; i < BLOCKEXTRA/sizeof(cyg_uint32) ; i++ )
    {
        if( b[size/sizeof(cyg_uint32) + i] != 0xcccccccc )
            diag_printf("Block check failure: %d[%d] : %08x != %08x\n", sector, i, b[i] , 0xcccccccc);
    }
    
}

//==========================================================================

static void clear_cache( int index )
{
//    int i;
#if 0
    diag_printf("clear cache\n");
    for( i = 0; i < (8*1024*1024); i += BLOCKSIZE, index++ )
    {
        read_block( sectormap[index], BLOCKSIZE );
    }
    diag_printf("clear cache done\n");
#endif
}

//==========================================================================

static void rate_test( void (*blockfn)( cyg_uint32 sector, int size ), char *name, int size )
{
    int i;
    cyg_int64 max = 0;
    cyg_int64 min = 0x7FFFFFFFFFFFFFFFLL;
    cyg_int64 total = 0;
    cyg_int64 ave;
    
    clear_cache( MAPSIZE/2 );

    diag_printf("Start %s rate test\n", name);
    for( i = 0; i < MAPSIZE; i++ )
    {

        if( (i % (MAPSIZE/10) ) == 0 )
            diag_printf("%4d: %10d\n", i, sectormap[i] );

        if( blockfn == write_block )
            init_block( sectormap[i], size );
        else
            memset( block, 0xcc, BLOCKSIZE + BLOCKEXTRA );
        
        // Fetch a single sector from the target location to force a
        // seek.

        read_sector( sectormap[i] );
        write_sector( sectormap[i] );

        wait_for_tick();

        get_timestamp( &timings[i].start );

        blockfn( sectormap[i], size );
        
        get_timestamp( &timings[i].end );

        if( blockfn == read_block )
            check_block( sectormap[i], size );
    }

    // Now analyze timings
    
    for( i = 0; i < MAPSIZE; i++ )
    {
        timing *t = &timings[i];

        calculate_interval( t );
        
        if(t->us_interval > max )
            max = t->us_interval;

        if( t->us_interval < min )
            min = t->us_interval;

        total += t->us_interval;

#if 0
        diag_printf("%4d: %10d: %8lld+%8u %8lld+%8u %10llu %10lluus\n",
                    i, sectormap[i],
                    t->start.ticks, t->start.halticks,
                    t->end.ticks, t->end.halticks,
                    t->interval, t->us_interval );
#endif
    }

    ave = total/MAPSIZE;

    diag_printf("Times for %d byte %s:\n", size, name );
    diag_printf("          Max           %2d.%03d ms\n", (int)(max/1000), (int)(max%1000) );
    diag_printf("          Min           %2d.%03d ms\n", (int)(min/1000), (int)(min%1000) );
    diag_printf("          Ave           %2d.%03d ms\n", (int)(ave/1000), (int)(ave%1000) );

    diag_printf("Data rates:\n");
    diag_printf("          Max        %10d KiB/s\n", (1000000LL*size)/1024/min );
    diag_printf("          Min        %10d KiB/s\n", (1000000LL*size)/1024/max );
    diag_printf("          Ave        %10d KiB/s\n", (1000000LL*size)/1024/ave );
}

//==========================================================================

static void seek_test( void )
{
    int i;
    cyg_int64 max = 0;
    cyg_int64 min = 0x7FFFFFFFFFFFFFFFLL;
    cyg_int64 total = 0;
    cyg_int64 ave;
    int test = 0;

    clear_cache( MAPSIZE/2 );

#if 1
//    for( j = 0 ; j < 10; j++ )
    {
        read_sector( 0 );
    
        // Seek to each map position sequentially
        for( i = 0; i < MAPSIZE; i++ )
        {
//            CYG_INTERRUPT_STATE old;
        
            timing *t = &timings[test++];
        
            wait_for_tick();

//            HAL_DISABLE_INTERRUPTS( old );
            get_timestamp( &t->start );

            read_sector( sectormap[i] );
        
            get_timestamp( &t->end );
//            HAL_RESTORE_INTERRUPTS( old );
        }
    }
#endif
    
#if 1
    // Seek from 0 to each position in map
    for( i = 0; i < MAPSIZE; i++ )
    {
        timing *t = &timings[test++];
        
        read_sector( 0 );

        wait_for_tick();

        get_timestamp( &t->start );

        read_sector( sectormap[i] );
        
        get_timestamp( &t->end );
    }
#endif

#if 1
    // And now from end of disk to each position
    for( i = 0; i < MAPSIZE; i++ )
    {
        timing *t = &timings[test++];
        
        read_sector( disk_size - 1 );

        wait_for_tick();

        get_timestamp( &t->start );

        read_sector( sectormap[i] );
        
        get_timestamp( &t->end );
    }
#endif

#if 1
    // And now from middle of disk to each position
    for( i = 0; i < MAPSIZE; i++ )
    {
        timing *t = &timings[test++];
        
        read_sector( disk_size/2 );

        wait_for_tick();

        get_timestamp( &t->start );

        read_sector( sectormap[i] );
        
        get_timestamp( &t->end );
    }
#endif
    
    // Now analyze timings
    
    for( i = 0; i < test; i++ )
    {
        timing *t = &timings[i];

        calculate_interval( t );
        
        if( max < t->us_interval )
            max = t->us_interval;

        if( min > t->us_interval )
            min = t->us_interval;

        total += t->us_interval;

#if 0
        diag_printf("%4d: %8lld+%8u %8lld+%8u %10llu %10lluus\n", i, 
                    t->start.ticks, t->start.halticks,
                    t->end.ticks, t->end.halticks,
                    t->interval, t->us_interval );
#endif
    }

    ave = total/test;

    diag_printf("Times for seek test:\n");
    diag_printf("          Max           %2d.%03d ms\n", (int)(max/1000), (int)(max%1000) );
    diag_printf("          Min           %2d.%03d ms\n", (int)(min/1000), (int)(min%1000) );
    diag_printf("          Ave           %2d.%03d ms\n", (int)(ave/1000), (int)(ave%1000) );
    
}

//==========================================================================
// main

void disktest_main( CYG_ADDRESS id )
{
    int                 result;
    cyg_disk_info_t     info;
    cyg_uint32          len = 1;
    char                buf = 1;
    int                 i;
    int                 size;

    CYG_TEST_INIT();
    
    result = cyg_io_lookup( TEST_DEV, &device );
    CYG_ASSERT( result == ENOERR, "Unexpected error" );

    result = cyg_io_set_config(device, CYG_IO_SET_CONFIG_DISK_MOUNT, &buf, &len);
    CYG_ASSERT( result == ENOERR, "Unexpected error" );

    len = sizeof(cyg_disk_info_t);
    result = cyg_io_get_config( device, CYG_IO_GET_CONFIG_DISK_INFO, &info, &len );
    CYG_ASSERT( result == ENOERR, "Unexpected error" );
    
    sector_size        = info.block_size;
    disk_size          = info.blocks_num;
    phys_block_size    = info.phys_block_size;

#if 1
    for( i = 0; i < MAPSIZE-1; i++ )
    {
        sectormap[i] = (disk_size/MAPSIZE)*i;
    }
#else
    for( i = 0; i < MAPSIZE-1; i++ )
    {
        sectormap[i] = i*1000;
    }
#endif
    // Put last block at very end of disk.
    sectormap[MAPSIZE-1] = disk_size - BLOCKSIZE/sector_size;

#if 1
    for( i = 0 ; date_string[i] != 0 ; i++ )
        timehash = (timehash<<1) ^ date_string[i];
    for( i = 0 ; time_string[i] != 0 ; i++ )
        timehash = (timehash<<1) ^ time_string[i];
#else
    timehash = 0x01a09456;
#endif
    
    block = malloc( BLOCKSIZE + BLOCKEXTRA + BLOCKALIGN - 1);
    CYG_ASSERT( block != NULL, "block malloc failure" );
    block = (cyg_uint8*)(((CYG_ADDRESS)block + BLOCKALIGN - 1) & ~(BLOCKALIGN - 1));

//    us_per_haltick = (100000000000ULL * rtc_resolution[1])/(rtc_period *rtc_resolution[0]);
    us_per_haltick = 1000000/(rtc_period * rtc_resolution[1]);
    
//    halticks_per_us = (rtc_period * rtc_resolution[0])/(100000000000ULL * rtc_resolution[1]);
    halticks_per_us = (rtc_period * rtc_resolution[1])/1000000; 

    {
        timing *t = &timings[0];
        
        wait_for_tick();

        get_timestamp( &t->start );
        get_timestamp( &t->end );

        calculate_interval( t );

        ticks_overhead = t->interval;

        diag_printf("Timing overhead %d ticks\n", ticks_overhead );

#if 1
        read_sector( 1000 );
        read_sector( 1000 );

        wait_for_tick();
        get_timestamp( &t->start );

        read_sector( 1000 );
        
        get_timestamp( &t->end );
        calculate_interval( t );        

        diag_printf("Sector transfer overhead %dus\n", t->us_interval );
#endif
        
        wait_for_tick();
        get_timestamp( &t->start );

        wait_for_tick();
        
        get_timestamp( &t->end );
        calculate_interval( t );

        diag_printf("Period %lld, Resolution %lld/%lld\n", rtc_period, rtc_resolution[0], rtc_resolution[1] );
        diag_printf("us_per_haltick %d halticks_per_us %d\n", us_per_haltick, halticks_per_us );
        diag_printf("Test interval: %8lld+%8u %8lld+%8u %10llu ticks %10lluus\n",
                    t->start.ticks, t->start.halticks,
                    t->end.ticks, t->end.halticks,
                    t->interval, t->us_interval );
        
    }

    seek_test();

#if 0
    init_block( 0, BLOCKSIZE );

//    memset( block, 0xcc, BLOCKSIZE + BLOCKEXTRA );
    
//    write_block( 0, 32*1024 );
    write_block( 0, BLOCKSIZE );

//    read_block( 0, 17*1024 );
    read_block( 0, BLOCKSIZE );

    check_block( 0, BLOCKSIZE );
    
    diag_dump_buf_with_offset( block, 512, block );
#endif

#if 1
    size = BLOCKSIZE;

    while( size >= (32*1024) )
    {
        rate_test( write_block, "write", size );

        rate_test( read_block, "read", size );

        size >>= 1;
    }
#endif
    
    CYG_TEST_FINISH( "disktest" );
}

// -------------------------------------------------------------------------


static cyg_handle_t thread_handle;
static cyg_thread thread;
static char stack[CYGNUM_HAL_STACK_SIZE_TYPICAL+1024*8];

externC void
cyg_start( void )
{
    cyg_thread_create(3,                // Priority - just a number
                      disktest_main,    // entry
                      0,                // index
                      0,                // no name
                      &stack[0],        // Stack
                      sizeof(stack),    // Size
                      &thread_handle,   // Handle
                      &thread           // Thread data structure
        );
    cyg_thread_resume(thread_handle);

    cyg_scheduler_start();
}


// -------------------------------------------------------------------------
// EOF disktest.c
