//==========================================================================
//
//      src/lwip/eth_drv_std.c
//
//      Hardware independent ethernet driver for lwIP, for standard eth drivers
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2006, 2007 Free Software Foundation, Inc.
// Copyright (C) 2006, 2007 eCosCentric Limited                             
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    Jani Monoses <jani@iv.ro>, jlarmour
// Contributors: 
// Date:         2002-04-05
// Purpose:      Hardware independent ethernet driver
// Description:  Based on the standalone driver for RedBoot.
//               This accompanies eth_drv.c and is for use with standard
//               stack-independent ethernet drivers, not lwip-specific ones.
//               
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/system.h>
#include <pkgconf/io_eth_drivers.h>
#include <pkgconf/net_lwip.h>

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/hal/hal_arch.h>
#include <cyg/hal/hal_diag.h>
#include <cyg/infra/diag.h>
//#include <cyg/hal/drv_api.h>
#include <cyg/hal/hal_if.h>
#include <cyg/io/eth/eth_drv.h>
#include <cyg/io/eth/netdev.h>
//#include <string.h>

#include <cyg/hal/hal_tables.h>
//#include <cyg/kernel/kapi.h>

#include "lwip/opt.h"
#include "lwip/ip.h"
#include "lwip/mem.h"
#include "lwip/pbuf.h"
#include "lwip/sys.h"
#include "lwip/ip_addr.h"
#include "lwip/netif.h"
#include "lwip/tcpip.h"
#include "netif/etharp.h"


// ------------------------------------------------------------------------
// Interfaces exported to eth drivers. These functions are implemented in this file.
// eth_drv_funs is a (poorly named, grr) global used to provide the func ptrs to eth drivers.

static void eth_drv_init(struct eth_drv_sc *sc, unsigned char *enaddr);
static void eth_drv_recv(struct eth_drv_sc *sc, int total_len);
static void eth_drv_tx_done(struct eth_drv_sc *sc, CYG_ADDRWORD key, int status);

struct eth_drv_funs eth_drv_funs = { eth_drv_init, eth_drv_recv, eth_drv_tx_done };

// ------------------------------------------------------------------------
// Global variables

__externC char cyg_io_eth_net_debug; // Are we debugging this layer?

// NETDEVTAB HAL table is filled in by hardware drivers.
// We need to define HAL network device table boundaries 
CYG_HAL_TABLE_BEGIN(__NETDEVTAB__, netdev);
CYG_HAL_TABLE_END(__NETDEVTAB_END__, netdev);

// ------------------------------------------------------------------------

// This function is called back from hardware drivers when they are initted
// to register a network interface with the system.
static void eth_drv_init(struct eth_drv_sc *sc, unsigned char *enaddr)
{
    struct netif *netif = &sc->sc_arpcom.ac_if;
    err_t err;
    char do_dhcp;
    
    err = cyg_lwip_eth_drv_init_netif((void *)sc, enaddr, netif, &do_dhcp);

    if (ERR_OK == err)
    {
        // Perform any further hardware initialization to get driver going.
        // enaddr == 0 -> hardware init was incomplete (no ESA)
        if ( NULL != enaddr )
            (sc->funs->start) (sc, (unsigned char *) &netif->hwaddr, 0);

#if defined(CYGSEM_HAL_VIRTUAL_VECTOR_DIAG)
        // Set up interfaces so debug environment can share this device
        {
            void *dbg = CYGACC_CALL_IF_DBG_DATA();
            if (!dbg) {
                CYGACC_CALL_IF_DBG_DATA_SET((void *)sc);
            }
        }
#endif

        //
        // we call this after the driver was started successfully
        //
        if (do_dhcp)
            cyg_lwip_dhcp_init(netif);
    }
}



// ------------------------------------------------------------------------

__externC void
cyg_lwip_eth_ecos_init(void)
{
    cyg_netdevtab_entry_t *t;

    // Ensure safe treatment of any console output during init.
    CYG_ETH_DRV_START_CONSOLE();

    for ( t = &__NETDEVTAB__[0]; t != &__NETDEVTAB_END__; t++ )
    {
        if ( t->init(t) )
        {
            t->status = CYG_NETDEVTAB_STATUS_AVAIL;
        } else {
            // What to do if device init fails?
            t->status = 0;		// Device not [currently] available
        }
    }

    CYG_ETH_DRV_END_CONSOLE();
}

// ------------------------------------------------------------------------

//DSR called from the low level driver.Signals the input_thread
void
eth_drv_dsr(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
  struct eth_drv_sc *sc = (struct eth_drv_sc *) data;
#ifdef CYGDBG_USE_ASSERTS
    // then check that this really is a "sc"
    {
        cyg_netdevtab_entry_t *t;
        for (t = &__NETDEVTAB__[0]; t != &__NETDEVTAB_END__; t++)
            if ( ((struct eth_drv_sc *)t->device_instance) == sc )
                break; // found it
        CYG_ASSERT( t != &__NETDEVTAB_END__, "eth_drv_dsr: Failed to find sc in NETDEVTAB" );
    }
#endif // Checking code
  sc->state |= ETH_DRV_NEEDS_DELIVERY;
  cyg_lwip_eth_dsr();
}


/* This routine is called back from the tcpip thread context when the flag
 * bit set for eth tx or rx packets (by cyg_lwip_dsr()) indicates it should.
 */
__externC void
cyg_lwip_eth_run_deliveries(void)
{
    cyg_netdevtab_entry_t *t;

    for (t = &__NETDEVTAB__[0]; t != &__NETDEVTAB_END__; t++) {
        struct eth_drv_sc *sc = (struct eth_drv_sc *)t->device_instance;
        if (sc->state & ETH_DRV_NEEDS_DELIVERY) {
            sc->state &= ~ETH_DRV_NEEDS_DELIVERY;
#if defined(CYGDBG_HAL_DEBUG_GDB_CTRLC_SUPPORT)
            cyg_bool was_ctrlc_int = HAL_CTRLC_CHECK((*sc->funs->int_vector)(sc), (int)sc);
            if (!was_ctrlc_int) // Fall through and run normal code
#endif
                (sc->funs->deliver) (sc);
        }
    }
} // cyg_lwip_eth_run_deliveries()

//
// Control whether any special locking needs to take place if we intend to
// cooperate with a ROM monitor (e.g. RedBoot) using this hardware.  
//
#if defined(CYGSEM_HAL_USE_ROM_MONITOR) && \
    defined(CYGSEM_HAL_VIRTUAL_VECTOR_DIAG) && \
   !defined(CYGSEM_HAL_VIRTUAL_VECTOR_CLAIM_COMMS)

// Indicate that special locking precautions are warranted.
#define _LOCK_WITH_ROM_MONITOR

// This defines the [well known] channel that RedBoot will use when it is
// using the network hardware for the debug channel.
#define RedBoot_TCP_CHANNEL CYGNUM_HAL_VIRTUAL_VECTOR_COMM_CHANNELS

#endif

//
// cyg_lwip_eth_low_level_output():
//
// This is the function called by higher layers which should do the actual
// transmission of the packet. The packet is contained in the pbuf that is
// passed to the function. This pbuf might be chained. It calls into the
// hardware driver to do the transmission.
//

err_t
cyg_lwip_eth_low_level_output(struct netif *netif, struct pbuf *p)
{
    struct eth_drv_sg sg_list[MAX_ETH_DRV_SG];
    struct eth_drv_sc *sc = netif->state;
    int sg_len = 0;
    struct pbuf *q;

#ifdef _LOCK_WITH_ROM_MONITOR
    bool need_lock = false;
    int debug_chan;
#endif

    // Can we send?
    if ((sc->funs->can_send) (sc) <= 0)
    {
#ifdef CYGIMP_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT
        // FIXME: this is a workaround to put a limit on the wait. I don't really
        // want to, but there is some interaction with RedBoot when there's a
        // ctrl-c, at which point RedBoot "takes over" and we can be left with
        // packets in the driver. It may be an AT91RM9200 eth driver specific
        // issue, but maybe not. For now, this isn't a bad change.

        cyg_ucount32 loops = 1000;
#endif

        // If we can't send, we need to call the delivery function to make
        // sure that the reason we can't send isn't just because we are
        // either higher priority than the normal thread doing delivery,
        // and thus preventing it delivering; or that we _are_ the
        // delivery thread, in which case we would never see it complete
        // even if it had, because clearly this thread has been "doing
        // stuff" if it has stuff to send.
        cyg_lwip_eth_run_deliveries();

#ifdef CYGIMP_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT

        // FIXME: think about introducing some sort of queuing? It's hard
        // because you can't use pbuf's own queuing abilities as the pbuf
        // may be added on the stack to e.g. the unacked packet queue.
        // so we'd need to wrap the pbuf in another object and queue that.
        // Not very lightweight!
        while (((sc->funs->can_send) (sc) <= 0) && loops--)
        {
            // As above, call the delivery function each time. We may
            // be the delivery thread anyway, so if we didn't, looping here
            // and not calling it would otherwise cause everything to freeze.
            cyg_lwip_eth_run_deliveries();

# if (CYGNUM_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT_DELAY > 0)
            // Give others a chance to run
#  ifdef CYGACC_CALL_IF_DELAY_US
            CYGACC_CALL_IF_DELAY_US(CYGNUM_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT_DELAY);
#  else
            HAL_DELAY_US(CYGNUM_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT_DELAY);
#  endif
# endif
        }
        if (!loops) // timed out
            return ERR_IF;
#else // !CYGIMP_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT
        /* Otherwise if we still can't send, just drop the packet completely. 
         * We claim success because there isn't necessarily really an error as such.
         * We may just be generating data too fast.
         */
        if ((sc->funs->can_send) (sc) <= 0)
            return ERR_OK;
#endif
    }

    // There is the potential for a thread-safety issue here since we may be
    // rescheduled, or interrupted by RedBoot between the can_send test and the
    // actual send, at which point you may not be able to send any more.
    // As long as this is only called from one thread, then it should be ok.

    for (q = p; q != NULL; q = q->next)
    {
        if ( sg_len == MAX_ETH_DRV_SG )
        {
#ifdef CYGDBG_IO_ETH_DRIVERS_DEBUG
            if (cyg_io_eth_net_debug)
            {
                CYG_ETH_DRV_START_CONSOLE();
                diag_printf("eth_drv_send: too many SGs to send, dropping pkt\n");
                CYG_ETH_DRV_END_CONSOLE();
            }
#endif
            return ERR_MEM; // drop it on the floor
        }
#ifdef CYGDBG_IO_ETH_DRIVERS_DEBUG
            if ( cyg_io_eth_net_debug > 1 ) {
                CYG_ETH_DRV_START_CONSOLE();
                diag_printf("xmit %d bytes at %p sg[%d]\n", q->len, q->payload, sg_len);
                if ( cyg_io_eth_net_debug > 2 )
                    diag_dump_buf(q->payload, q->len);
                CYG_ETH_DRV_END_CONSOLE();
            }
#endif

        sg_list[sg_len].buf = (CYG_ADDRESS) q->payload;
        sg_list[sg_len++].len = q->len;
    }

    // incr ref count of pbuf, so that caller's free will not really free up
    // memory. At least not until packet is definitely sent.
    pbuf_ref( p );

#ifdef _LOCK_WITH_ROM_MONITOR
    debug_chan = CYGACC_CALL_IF_SET_DEBUG_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);
    if ( debug_chan == RedBoot_TCP_CHANNEL )
    {
        need_lock = true;
        cyg_drv_dsr_lock();
    }
#endif // _LOCK_WITH_ROM_MONITOR

    (sc->funs->send) (sc, sg_list, sg_len, p->tot_len,
                      (CYG_ADDRWORD) p);

#ifdef _LOCK_WITH_ROM_MONITOR
    // Unlock the driver & hardware.  It can once again be safely shared.
    if (need_lock)
    {
        cyg_drv_dsr_unlock();
    }
#endif // _LOCK_WITH_ROM_MONITOR

    return ERR_OK;
}

//
// This function is called from the hardware driver when an output operation
// has completed - i.e. the packet has been sent.
//
static void
eth_drv_tx_done(struct eth_drv_sc *sc, CYG_ADDRWORD key, int status)
{
  struct pbuf *p = (struct pbuf *)key;
  //  struct netif *netif = &sc->sc_arpcom.ac_if;

  CYGARC_HAL_SAVE_GP();

  // We increased pbuf ref count in send, now we can free the pbuf. If the
  // sender hasn't called pbuf_free() themselves, this will only decr the ref
  // count. Otherwise it will (probably) free the pbufs.
  (void)pbuf_free(p);

  CYGARC_HAL_RESTORE_GP();
}


#define MAX_ETH_MSG 1540
//
// This function is called from a hardware driver to indicate that an input
// packet has arrived.  The routine will set up appropriate network resources
// to hold the data and call back into the driver to retrieve the data.
//
static void
eth_drv_recv(struct eth_drv_sc *sc, int total_len)
{
  struct eth_drv_sg sg_list[MAX_ETH_DRV_SG];

  struct pbuf *p, *q;

  int sg_len = 0;
  CYGARC_HAL_SAVE_GP();

  if ((total_len > MAX_ETH_MSG) || (total_len < 0)) {
    total_len = MAX_ETH_MSG;
  }

#ifdef CYGDBG_LWIP_DEBUG
  // As a general principle, if we know that the underlying lwip code may
  // have some debug output, then there's a chance it could be output as
  // a result of some function called here. Therefore force the console
  // away from the network i/f if needed, for safety.
  CYG_ETH_DRV_START_CONSOLE();
#endif

  p = pbuf_alloc(PBUF_RAW, total_len, PBUF_POOL);

  if (p == NULL)
  {
#ifdef CYGDBG_IO_ETH_DRIVERS_DEBUG
      if (cyg_io_eth_net_debug)
      {
#ifndef CYGDBG_LWIP_DEBUG // if CYGDBG_LWIP_DEBUG set, it's already done
          CYG_ETH_DRV_START_CONSOLE();
#endif
          diag_printf("eth_drv_recv: pbuf_alloc returned NULL\n");
#ifndef CYGDBG_LWIP_DEBUG
          CYG_ETH_DRV_END_CONSOLE();
#endif
      }
#endif
      /* Can't just call return due to block fiddling by
       * CYG_ETH_DRV_START_CONSOLE.
       */
      goto end;
  }

  for (q = p; q != NULL; q = q->next) {
    sg_list[sg_len].buf = (CYG_ADDRESS) q->payload;
    sg_list[sg_len++].len = q->len;
  }
  (sc->funs->recv) (sc, sg_list, sg_len);

#ifdef CYGDBG_IO_ETH_DRIVERS_DEBUG
  if ( cyg_io_eth_net_debug > 1 ) {
      int i;

      CYG_ETH_DRV_START_CONSOLE();
      for (i = 0;  i < sg_len;  i++) {
          if (sg_list[i].buf) {
              diag_printf("rx %d bytes at %x sg[%d]\n", sg_list[i].len, sg_list[i].buf, i);
              if ( cyg_io_eth_net_debug > 2 )
                  diag_dump_buf((void *)sg_list[i].buf, sg_list[i].len);
          }
      }
      CYG_ETH_DRV_END_CONSOLE();
  }
#endif


  /* FIXME: this RX pbuf queuing should probably be an option,
   * as it's only here (instead of directly calling ecosif_input) because
   * of problems when interacting with RedBoot GDB debugging over TCP/IP.
   */

  /* Add to queue of rx pbufs */
  if (!sc->sc_arpcom.rx_pbufs)
  {
      sc->sc_arpcom.rx_pbufs = p;
  }
  else
  {
      /* Starting from queue tail, go to the end of the queue.
       * This can happen when the pbufs are allocated in a chain.
       * Even so, only having to start from the start of the last
       * packet still helps.
       */
      CYG_ASSERT( NULL != sc->sc_arpcom.rx_pbufs_queue_tail,
                  "rx_pbufs set, but not rx_pbufs_queue_tail" );
      for (q=sc->sc_arpcom.rx_pbufs_queue_tail; q->next != NULL; q = q->next)
          CYG_EMPTY_STATEMENT;
      /* q is now last portion of last packet on queue */
      q->next = p;
  }
  sc->sc_arpcom.rx_pbufs_queue_tail = p;

  cyg_lwip_eth_indicate_rx_pkt();

 end:
#ifdef CYGDBG_LWIP_DEBUG
  CYG_ETH_DRV_END_CONSOLE();
#endif

  CYGARC_HAL_RESTORE_GP();
}

/* This routine is called back from the tcpip thread context when the flag
 * bit set for rx eth packets (by cyg_lwip_dsr()) indicates it should.
 */
__externC void
cyg_lwip_eth_process_rx(void)
{
    cyg_netdevtab_entry_t *t;
    struct eth_drv_sc *sc;
    struct netif *netif;
    struct pbuf *p, *nextp;
    u16_t p_remlen;
    struct arpcom *ac;

    for (t = &__NETDEVTAB__[0]; t != &__NETDEVTAB_END__; t++)
    {
        sc = (struct eth_drv_sc *)t->device_instance;
        ac = &sc->sc_arpcom;
        netif = &ac->ac_if;

        while ( ac->rx_pbufs )
        {
            p = ac->rx_pbufs;
            p_remlen = p->tot_len;

            /* Dequeue rx pbufs from queue by iterating through
             * and adjusting calculated remaining length of this pbuf
             * until we reach the final portion of it. Then we can
             * both assign the new rx_pbufs head, as well as set the
             * last portion of 'p' to have a next of NULL.
             */
            for ( nextp = p; nextp->len != p_remlen ; nextp = nextp->next )
            {
                p_remlen -= nextp->len;
            }
            ac->rx_pbufs = nextp->next;
            nextp->next = NULL;

            /* Tail may need to be cleared if that was the only pbuf */
            if (NULL == ac->rx_pbufs)
            {
                ac->rx_pbufs_queue_tail = NULL;
            }

            /* Pass up */
            cyg_lwip_eth_drv_ecosif_input( netif, p );
        } /* while */
    } /* for */
} /* cyg_lwip_eth_process_rx() */


/* EOF eth_drv_std.c */
