//========================================================================
//
//      httpd2.c
//
//      Revised raw API lwIP httpd server for testing
//
//========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2006, 2007, 2008 Free Software Foundation, Inc.            
// Copyright (C) 2006, 2007, 2008 eCosCentric Limited                       
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================

// Derived in part from httpd.c with this licence:

/*
 * Copyright (c) 2001, Swedish Institute of Computer Science.
 * All rights reserved. 
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met: 
 * 1. Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright 
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the distribution. 
 * 3. Neither the name of the Institute nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software 
 *    without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE 
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE. 
 *
 * This file is part of the lwIP TCP/IP stack.
 * 
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

#include <pkgconf/net_lwip.h>

#include "lwip/debug.h"
#include "lwip/stats.h"
#include "lwip/tcp.h"
#include <cyg/kernel/kapi.h>
#include <cyg/hal/hal_arch.h> // CYGNUM_HAL_STACK_SIZE_TYPICAL

#define MAX_URL_LENGTH 10
#define HTTPD_CONN_TIMEOUT_SECS  15

#define HTTP_POLLS_PER_TCP_SLOW_TIMER 4
#define HTTPD_CONN_TIMEOUT_IN_POLLS  ((HTTPD_CONN_TIMEOUT_SECS*1000)/(TCP_SLOW_INTERVAL*HTTP_POLLS_PER_TCP_SLOW_TIMER))

// States for state machine in http_recv()
#define STATE_RD_METHOD     1
#define STATE_RD_URL        2
#define STATE_RD_PROT       3
#define STATE_RD_SUPPL_HDR  4
#define STATE_DONE          5
          
struct http_state {
  const struct webpage *page;
  const char *pos;
  u16_t left;
  u8_t retries;

  /* saved state for state machine in http_recv */
  u8_t flags;
  char *saved_str;
  char read_url[MAX_URL_LENGTH];
};

#define FLAG_IN_HEADER  (1<<0)
#define FLAG_HTTP1X     (1<<1)
#define FSM_STATE(hs) ((hs->flags&0xf0) >> 4)
#define SET_FSM_STATE(hs, state) do { hs->flags &= 0x0f; hs->flags |= state << 4; } while (0)

struct webpage {
    const char *location;
    const char *header;
    const char *html;
};

static const char std_html_header[] = "HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n";

static const struct webpage pages[] = {
    {
        "/", std_html_header,
        "<html><head><title>eCos lwIP httpd test page</title></head><body bgcolor=\"white\">"
        "<h1>eCos lwIP httpd test page</h1>"
        "<hr noshade size=3>"
        "<p>If you can see this, then the webserver works!</p>"
        "<p>Server build date: " __DATE__ " " __TIME__ "</p>"
        "<ul><li><a href=\"http://www.ecoscentric.com\">eCosCentric website</a>"
        "<li><a href=\"http://www.ecoscentric.com/ecos/ecospro.shtml\">eCosPro</a>"
        "<li><a href=\"http://www.ecoscentric.com/ecospro/doc/html/ecospro-ref/lwip.html\">eCosPro lwIP documentation</a>"
        "</ul><hr noshade size=3>"
        "<p><font size=-1><i>&copy; eCosCentric Limited<br>Server: eCosPro+lwIP</i></font></p>"
        "</body></html>\r\n"
    }
};

static const struct webpage error404page = {
    "", "HTTP/1.0 404 Not Found\r\nContent-Type: text/html\r\n\r\n",
    "<html><head><title>404 Not Found</title></head><body><h1>Not found</h1>"
    "<p>The requested URL was not found on the server.</p>"
    "<hr noshade size=3><p><font size=-1><i>Server: eCosPro+lwIP</i></font></p>"
    "</body></html>\r\n"
};

static const char getmethod[] = "GET ";
static const char httpprot[] = "HTTP/";
static const char crnl2[] = "\r\n\r\n";

/*-----------------------------------------------------------------------------------*/
static void
conn_err(void *arg, err_t err)
{
  struct http_state *hs;

  hs = arg;
  mem_free(hs);
}

/*-----------------------------------------------------------------------------------*/
static void
close_conn(struct tcp_pcb *pcb, struct http_state *hs)
{
  tcp_recv(pcb, NULL);
  tcp_sent(pcb, NULL);
  tcp_arg(pcb, NULL);
  mem_free(hs);
  tcp_close(pcb);
}

/*-----------------------------------------------------------------------------------*/
static void
send_data(struct tcp_pcb *pcb, struct http_state *hs)
{
  err_t err;
  u16_t len, sndbuf;

  do {
      /* We cannot send more data than space available in the send
       * buffer.
       */
      sndbuf = tcp_sndbuf(pcb);
      if(sndbuf < hs->left) {
          len = sndbuf;
      } else {
          len = hs->left;
      }

      if (!len)
          return;

      do {
          err = tcp_write(pcb, hs->pos, len, 0);
          if(err == ERR_MEM) {
              len /= 2;
          }
      } while((err == ERR_MEM) && (len > 1));

      if(err == ERR_OK) {
          hs->pos += len;
          hs->left -= len;

          if (0 == hs->left) {
              if (hs->flags & FLAG_IN_HEADER) {
                  const char *s;

                  hs->flags &= ~FLAG_IN_HEADER;
                  s = hs->pos = hs->page->html;
                  while (*s++ != '\0')
                      hs->left++;
              }
          }
      }
  } while ((err == ERR_OK) && hs->left);
}

/*-----------------------------------------------------------------------------------*/
static err_t
http_poll(void *arg, struct tcp_pcb *pcb)
{
  struct http_state *hs;

  hs = arg;
  
  if(hs == NULL) {
    tcp_abort(pcb);
    return ERR_ABRT;
  } else {
    ++hs->retries;
    if(hs->retries == HTTPD_CONN_TIMEOUT_IN_POLLS) {
      tcp_abort(pcb);
      return ERR_ABRT;
    }
    send_data(pcb, hs);
  }

  return ERR_OK;
}

/*-----------------------------------------------------------------------------------*/
static err_t
http_sent(void *arg, struct tcp_pcb *pcb, u16_t len)
{
  struct http_state *hs;

  hs = arg;

  hs->retries = 0;
  
  if(hs->left > 0) {    
    send_data(pcb, hs);
  } else {
    close_conn(pcb, hs);
  }

  return ERR_OK;
}

/*-----------------------------------------------------------------------------------*/

static const char *
next_pbuf_char(struct pbuf **p_ptr, u16_t *index_state)
{
    struct pbuf *p = *p_ptr;
    const char *pdata;

    if (*index_state >= p->len)
    {
        p = *p_ptr = p->next;
        if (!p)
            return NULL;
        *index_state=0;
    }
    pdata = p->payload;
    pdata += *index_state;
    (*index_state)++;
    return pdata;
}

static err_t
http_recv(void *arg, struct tcp_pcb *pcb, struct pbuf *p, err_t err)
{
    const char *c;
  struct http_state *hs = arg;
  struct pbuf *p_state;
  u16_t p_index_state;
  u16_t i;
  u8_t url_match;
  u8_t wp_idx;

  if (ERR_OK != err)
  {
      if (p)
          pbuf_free(p);
      return err;
  }

  if (p != NULL)
  {
      p_index_state = 0;
      p_state = p;

      /* Inform TCP that we have taken the data. */
      tcp_recved(pcb, p->tot_len);
    
      if (hs->page == NULL)
      {
          /* It is a reasonable assumption that the request arrives
           * in a single pbuf chain. Otherwise we need to save state
           * between runs, which is probably overkill for this app.
           * Action on bad requests is just to close, rather than anything fancier.
           */

          do {
              c = next_pbuf_char(&p_state, &p_index_state);
              if (!c)
              {
                  /* Need more data */
                  pbuf_free(p);
                  goto ret;
              }
              switch (FSM_STATE(hs))
              {
              case STATE_RD_METHOD:
                  if (*c != *hs->saved_str++)
                      goto free_close_ret;
                  if (*hs->saved_str == '\0')
                  {
                      SET_FSM_STATE(hs, STATE_RD_URL);
                      hs->saved_str = hs->read_url;
                  }
                  break;
              case STATE_RD_URL:
                  if ((*c == '\r') || (*c == '\n'))
                  {
                      /* Old (pre-HTTP v1) request */
                      *hs->saved_str = '\0';
                      SET_FSM_STATE(hs, STATE_DONE);
                  }
                  else if (*c == ' ')
                  {
                      /* End of URL part */
                      *hs->saved_str = '\0';
                      SET_FSM_STATE(hs, STATE_RD_PROT);
                      hs->saved_str = (char*)httpprot;
                  }
                  else {
                      *hs->saved_str++ = *c;
                      if ((hs->saved_str-hs->read_url) == MAX_URL_LENGTH)
                          goto free_close_ret;
                  }
                  break;
              case STATE_RD_PROT:
                  if (hs->flags & FLAG_HTTP1X)
                  {
                      /* Already determined this is 1.x, so keep reading to EOL.
                       * We don't care what the specific version is (and ignore any
                       * further syntax errors before EOL, but no big deal)*/
                      if (*c == '\n')
                      {
                          SET_FSM_STATE(hs, STATE_RD_SUPPL_HDR);
                          // skip straight to second crnl in sequence, because
                          // we've now already seen one.
                          hs->saved_str = (char*)&crnl2[2];
                      }
                  } else if (*c != *hs->saved_str++)
                      goto free_close_ret;
                  else if (*hs->saved_str == '\0') 
                  {
                      /* Reached end of HTTP/ and matched all the way*/
                      hs->flags |= FLAG_HTTP1X;
                  }
                  break;
              case STATE_RD_SUPPL_HDR:
                  /* Consume until we see double CRNL */
                  if ( *c == *hs->saved_str )
                  {
                      hs->saved_str++;
                      if (*hs->saved_str == '\0')
                      {
                          SET_FSM_STATE(hs, STATE_DONE);
                      }
                  } else
                      hs->saved_str = (char*)crnl2; /* Reset to start of sequence */
                  break;
              default: /* Shouldn't happen */
                  goto free_close_ret;
              }
          } while (FSM_STATE(hs) != STATE_DONE);                          

          /* Find URL */
          for (wp_idx = 0; wp_idx < (sizeof(pages)/sizeof(pages[0])); wp_idx++)
          {
              url_match = 0;
              for (i=0; !url_match; i++)
              {
                  if ( pages[wp_idx].location[i] != hs->read_url[i] )
                      break;
                  if ( hs->read_url[i] == '\0' ) /* must have matched all the way */
                      url_match = 1;
              }
              if (url_match)
                  break;
          }
          if (url_match)
              hs->page = &pages[wp_idx];
          else
              hs->page = &error404page;
          if (hs->flags & FLAG_HTTP1X)
          {
              hs->pos = hs->page->header;
              hs->flags = FLAG_IN_HEADER;
          } else {
              hs->pos = hs->page->html;
          }

          c = hs->pos;
          while (*c++ != '\0')
              hs->left++;

          /* Finished with rx pbuf */
          pbuf_free(p);

          /* Tell TCP that we wish be to informed of data that has been
             successfully sent by a call to the http_sent() function. */
          tcp_sent(pcb, http_sent);

          /* We can now act on the URL */
          send_data(pcb, hs);

          goto ret;
      } else {
        goto free_close_ret;
      }
  }
  
free_close_ret:
  if (p)
      pbuf_free(p);
  close_conn(pcb, hs);
ret:
  return ERR_OK;
}

/*-----------------------------------------------------------------------------------*/
static err_t
http_accept(void *arg, struct tcp_pcb *pcb, err_t err)
{
  struct http_state *hs;

  tcp_setprio(pcb, TCP_PRIO_MIN);
  
  /* Allocate memory for the structure that holds the state of the
     connection. */
  hs = mem_malloc(sizeof(struct http_state));

  if(hs == NULL) {
    return ERR_MEM;
  }
  
  /* Initialize the structure. */
  hs->page = NULL;
  hs->pos = NULL;
  hs->left = 0;
  hs->retries = 0;
  hs->flags = 0;
  SET_FSM_STATE(hs, STATE_RD_METHOD);
  hs->saved_str = (char *)getmethod;

  
  /* Tell TCP that this is the structure we wish to be passed for our
     callbacks. */
  tcp_arg(pcb, hs);

  /* Tell TCP that we wish to be informed of incoming data by a call
     to the http_recv() function. */
  tcp_recv(pcb, http_recv);

  tcp_err(pcb, conn_err);
  
  tcp_poll(pcb, http_poll, HTTP_POLLS_PER_TCP_SLOW_TIMER);

#if 0
  {
      static cyg_handle_t thr;
      static cyg_uint16 thr_id;
      static cyg_thread_info thr_info;

      thr = 0;
      while (cyg_thread_get_next(&thr, &thr_id))
      {
          if (!cyg_thread_get_info(thr, thr_id, &thr_info))
              break;
          diag_printf("%s: %d\n", thr_info.name, (int)thr_info.stack_used);
      }
  }
#endif
#if 0
  stats_display();
#endif

  return ERR_OK;
}

/*-----------------------------------------------------------------------------------*/
void
httpd_init(cyg_addrword_t arg)
{
  struct tcp_pcb *pcb;
  err_t err;

  cyg_lwip_init();
  pcb = tcp_new();
  err = tcp_bind(pcb, IP_ADDR_ANY, 80);

  pcb = tcp_listen(pcb);

  tcp_accept(pcb, http_accept);
}

#if 0
// This value was used successfully on AT91SAM7X during a size optimisation exercise.
#define STACK_SIZE 384
#else
#define STACK_SIZE CYGNUM_HAL_STACK_SIZE_TYPICAL
#endif
static char stack[STACK_SIZE] CYGBLD_ATTRIB_ALIGN_MAX;
static cyg_thread thread_data;

void
cyg_start(void)
{
    cyg_handle_t thread_handle;

    // Create a main thread, so we can run the scheduler and have time 'pass'
    cyg_thread_create(7,                // Priority - just a number
                      httpd_init,       // entry
                      0,                // entry parameter
                      "httpd",          // Name
                      &stack[0],        // Stack
                      STACK_SIZE,       // Size
                      &thread_handle,   // Handle
                      &thread_data      // Thread data structure
            );
    cyg_thread_resume(thread_handle);  // Start it
    cyg_scheduler_start();
}

