/*
* The contents of this file are subject to the Mozilla Public
* License Version 1.1 (the "License"); you may not use this file
* except in compliance with the License. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
* 
* Software distributed under the License is distributed on an "AS
* IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
* implied. See the License for the specific language governing
* rights and limitations under the License.
* 
* The Original Code is MPEG4IP.
* 
* The Initial Developer of the Original Code is Cisco Systems Inc.
* Portions created by Cisco Systems Inc. are
* Copyright (C) Cisco Systems Inc. 2000, 2001.  All Rights Reserved.
* 
* Contributor:  --mlt-- 
*
*/

#include "mp4creator.h"
#include "mp4.h"
#include "mp4util.h"

uint32_t import_chapter_file(MP4Chapters_t** chap_arr, FILE* inFile, double framerate);

MP4TrackId QTChapterCreator(MP4FileHandle mp4File, FILE* inFile)
{
	double			fps = 0;
	MP4Chapters_t*	chap_arr[1000];
	uint32_t		numchaps = 0;
	uint32_t		i,j;

	// try to find a video track to retrieve framerate
	MP4TrackId vidTrackId = MP4FindTrackId(mp4File, MP4_INVALID_TRACK_ID, MP4_VIDEO_TRACK_TYPE); 
	if(MP4_IS_VALID_TRACK_ID(vidTrackId))
		fps = MP4GetTrackVideoFrameRate(mp4File, vidTrackId);


	// first, attempt to read the chapter file
	memset(chap_arr, 0, sizeof(MP4Chapters_t*) * 1000);
	numchaps = import_chapter_file(&chap_arr[0], inFile, fps);

	if(numchaps == 0) {
		fprintf(stderr,
			"Error reading chapter file data - check syntax\n");
		return MP4_INVALID_TRACK_ID;
	}

	// get the audio track that will reference the chapter track 
	MP4TrackId refTrackId = MP4FindTrackId(mp4File, MP4_INVALID_TRACK_ID, MP4_AUDIO_TRACK_TYPE); 
	if (! MP4_IS_VALID_TRACK_ID(refTrackId)) { 
		fprintf(stderr, "MP4 must already include audio track in order to add QuickTime chapters");
		return MP4_INVALID_TRACK_ID;
	}	 

	// get information about the audio track 
	MP4Duration refTrackDuration = MP4GetTrackDuration(mp4File, refTrackId); 
	uint32_t refTrackTimeScale = MP4GetTrackTimeScale(mp4File, refTrackId); 

	// determine our chapter track total duration
	MP4Duration chapterTrackDuration = MP4ConvertTime(refTrackDuration, refTrackTimeScale, MP4_MILLISECONDS_TIME_SCALE);

	// validate the chapters imported from file
	MP4Duration  durtest = 0;
	for(i = 0; i < numchaps; i++) {
		if(chap_arr[i]->duration > 0 && !(chap_arr[i]->duration > durtest)) {
			fprintf(stderr, "Chapter points must be ordered sequentially in chapter file\n");
			for(j = 0; j < numchaps; j++)
				MP4Free(chap_arr[j]);

			return MP4_INVALID_TRACK_ID;
		}

		durtest = chap_arr[i]->duration;

		if(chap_arr[i]->duration > chapterTrackDuration) {
			fprintf(stderr, "Invalid chapter point in chapter file\n");
			for(j = 0; j < numchaps; j++)
				MP4Free(chap_arr[j]);

			return MP4_INVALID_TRACK_ID;
		}
	}


	// delete previous chapter markers 
	MP4DeleteChapters(mp4File); 

	// create the chapter track ... 
	MP4TrackId chapterTrack = MP4AddChapterTextTrack(mp4File, refTrackId, MP4_MILLISECONDS_TIME_SCALE); 


	// create chapters 
	MP4Duration chapterDuration; 
	uint32_t  chapCount = 1;
	for (i = 0; i < numchaps; i++) { 
		// if the first chapter does not start at 00:00:00, we auto-create
		if(chapCount == 1 && chap_arr[i]->duration > 0) {
			MP4AddQTChapter(mp4File, chapterTrack, chap_arr[i]->duration, chapCount);
			chapCount++;
		}

		// determine chapter duration
		if(i < numchaps - 1)
			chapterDuration = chap_arr[i + 1]->duration - chap_arr[i]->duration;
		else  // otherwise, this is the last chapter
			chapterDuration = chapterTrackDuration - chap_arr[i]->duration;

		// chapterDuration is expected as count of samples related to the timescale of chapterTrack 
		MP4AddQTChapter(mp4File, chapterTrack, chapterDuration, chapCount, chap_arr[i]->title); 
		chapCount++;
	} 

	// free memory
	for(i = 0; i < numchaps; i++)
		MP4Free(chap_arr[i]);

	return chapterTrack; 

}


/*
From the GPAC documentation:

The following syntaxes are supported in the chapter text file, with one chapter entry per line:

* ZoomPlayer chapter files : AddChapter(nb_frames,chapter name), 
AddChapterBySeconds(nb_sec,chapter name) and 
AddChapterByTime(h,m,s,chapter name). One chapter entry per line.
* Time codes : h:m:s name, h:m:s:ms name and h:m:s.ms name. One chapter entry per line.
* SMPTE codes : h:m:s;nb_f/fps name and h:m:s;nb_f name with 
nb_f the number of frames and fps the framerate. One chapter entry per line.
* Common syntax : CHAPTERX=h:m:s[:ms or .ms] on one line 
and CHAPTERXNAME=name on the other - the order is not 
important but chapter lines MUST be declared sequencially
(same X value expected for 2 consecutive lines).


--mlt--
This code was taken from the GPAC/MP4Box project and modified for use here
original method: gf_media_import_chapters
file: src\media_tools\isom_tools.c

*/
uint32_t import_chapter_file(MP4Chapters_t** chap_arr, FILE* inFile, double framerate)
{
	uint32_t count = 0;
	uint32_t state;
	uint32_t cur_chap;
	uint64_t ts;
	uint32_t h, m, s, ms, fr, fps;
	char line[1024];
	char szTitle[1024];

	cur_chap = 0;
	ts = 0;
	state = 0;
	while (fgets(line, 1024, inFile) != NULL) {
		char *title = NULL;
		uint32_t off = 0;
		char *sL;
		while (1) {
			uint32_t len = strlen(line);
			if (!len) break;
			switch (line[len-1]) {
			case '\n': case '\t': case '\r': case ' ':
				line[len-1] = 0;
				continue;
			}
			break;
		}

		while (line[off]==' ') off++;
		if (!strlen(line+off)) continue;
		sL = line+off;

		szTitle[0] = 0;
		//ZoomPlayer chapters
		if (!strnicmp(sL, "AddChapter(", 11)) {
			uint32_t nb_fr;
			sscanf(sL, "AddChapter(%d,%s)", &nb_fr, szTitle);
			ts = nb_fr;
			ts *= 1000;
			if (framerate != 0)
				ts = (uint64_t) (((int64_t) ts ) / framerate);
			else ts /= 25;
			sL = strchr(sL, ','); strcpy(szTitle, sL+1); sL = strrchr(szTitle, ')'); if (sL) sL[0] = 0;
		} else if (!strnicmp(sL, "AddChapterBySecond(", 19)) {
			uint32_t nb_s;
			sscanf(sL, "AddChapterBySecond(%d,%s)", &nb_s, szTitle);
			ts = nb_s;
			ts *= 1000;
			sL = strchr(sL, ','); strcpy(szTitle, sL+1); sL = strrchr(szTitle, ')'); if (sL) sL[0] = 0;
		} else if (!strnicmp(sL, "AddChapterByTime(", 17)) {
			uint32_t h, m, s;
			sscanf(sL, "AddChapterByTime(%d,%d,%d,%s)", &h, &m, &s, szTitle);
			ts = 3600*h + 60*m + s;
			ts *= 1000;
			sL = strchr(sL, ','); 
			if (sL) sL = strchr(sL+1, ','); 
			if (sL) sL = strchr(sL+1, ','); 
			strcpy(szTitle, sL+1); sL = strrchr(szTitle, ')'); if (sL) sL[0] = 0;
		}
		//regular or SMPTE time codes
		else if ((strlen(sL)>=8) && (sL[2]==':') && (sL[5]==':')) {
			title = NULL;
			if (strlen(sL)==8) {
				sscanf(sL, "%02d:%02d:%02d", &h, &m, &s);
				ts = (h*3600 + m*60+s)*1000;
			} 
			else {
				char szTS[20], *tok;
				strncpy(szTS, sL, 18);
				tok = strrchr(szTS, ' ');
				if (tok) {
					title = strchr(sL, ' ') + 1;
					while (title[0]==' ') title++;
					if (strlen(title)) strcpy(szTitle, title);
					tok[0] = 0;
				}
				ts = 0;
				h = m = s = ms = 0;

				if (sscanf(szTS, "%d:%d:%d;%d/%d", &h, &m, &s, &fr, &fps)==5) {
					ts = (h*3600 + m*60+s)*1000 + 1000*fr/fps;
				} else if (sscanf(szTS, "%d:%d:%d;%d", &h, &m, &s, &fr)==4) {
					ts = (h*3600 + m*60+s);
					if (framerate != 0)
						ts = (((framerate * ((int64_t)ts) + fr) * 1000 ) / framerate);
					else
						ts = ((ts*25 + fr) * 1000 ) / 25;
				} else if (sscanf(szTS, "%d:%d:%d.%d", &h, &m, &s, &ms) == 4) {
					ts = (h*3600 + m*60+s)*1000+ms;
				} else if (sscanf(szTS, "%d:%d:%d.%d", &h, &m, &s, &ms) == 4) {
					ts = (h*3600 + m*60+s)*1000+ms;
				} else if (sscanf(szTS, "%d:%d:%d:%d", &h, &m, &s, &ms) == 4) {
					ts = (h*3600 + m*60+s)*1000+ms;
				} else if (sscanf(szTS, "%d:%d:%d", &h, &m, &s) == 3) {
					ts = (h*3600 + m*60+s) * 1000;
				}
			}
		} 
		//CHAPTERX= and CHAPTERXNAME=
		else if (!strnicmp(sL, "CHAPTER", 7)) {
			uint32_t idx;
			char szTemp[20], *str;
			strncpy(szTemp, sL, 19);
			str = strrchr(szTemp, '=');
			if (!str) continue;
			str[0] = 0;
			strlwr(szTemp);
			idx = cur_chap;
			str = strchr(sL, '=');
			str++;
			if (strstr(szTemp, "name")) {
				sscanf(szTemp, "chapter%dname", &idx);
				strcpy(szTitle, str);
				if (idx!=cur_chap) {
					cur_chap=idx;
					state = 0;
				}
				state++;
			} else {
				sscanf(szTemp, "chapter%d", &idx);
				if (idx!=cur_chap) {
					cur_chap=idx;
					state = 0;
				}
				state++;

				ts = 0;
				h = m = s = ms = 0;
				if (sscanf(str, "%d:%d:%d.%d", &h, &m, &s, &ms) == 4) {
					ts = (h*3600 + m*60+s)*1000+ms;
				} else if (sscanf(str, "%d:%d:%d:%d", &h, &m, &s, &ms) == 4) {
					ts = (h*3600 + m*60+s)*1000+ms;
				} else if (sscanf(str, "%d:%d:%d", &h, &m, &s) == 3) {
					ts = (h*3600 + m*60+s) * 1000;
				}
			}
			if (state==2) {
				chap_arr[count] = (MP4Chapters_t*)MP4Malloc(sizeof(MP4Chapters_t));
				chap_arr[count]->duration = ts;
				strcpy(chap_arr[count]->title, szTitle);
				count++;

				if(count > 999)
					return count;

				state = 0;
			}
			continue;
		}
		else continue;

		chap_arr[count] = (MP4Chapters_t*)MP4Malloc(sizeof(MP4Chapters_t));
		chap_arr[count]->duration = ts;
		if(strlen(szTitle))
			strcpy(chap_arr[count]->title, szTitle);
		count++;

		if(count > 999)
			return count;

	}

	return count;
}
