/******************************************************************************

  Copyright (C), 2001-2011, Hisilicon Tech. Co., Ltd.

 ******************************************************************************
  File Name     : hinand.c
  Version       : 1.0
  Author        : Hisilicon multimedia software group
  Created       : 2009/02/15
  Last Modified :
  Description   : nand flash controller driver
  Function List :
  History       :
  1.Date        : 2009/02/15
    Author      : Zhan Weitao / Zhang Kuanhuai
    Modification: Created file
  2.Date        : 2009/03/16
    Author      : Zhang Kuanhuai
    Modification: Support 4K page size

******************************************************************************/

#include <common.h>

#include <nand.h>

#include <asm/io.h>
#include <asm/errno.h>
#include <malloc.h>
#include "hinand.h"

#undef debug
#define debug(fmt, args...)
//#define debug(fmt, args...) printk(fmt, ##args)

void printreg(struct hinand_data *data)
{
	unsigned int i, reg;

	for(i = 0; i < 0x30; i+=4)
	{
        reg = hinand_readl(data, i);
		printk("reg %x is %x ", i, reg);
	}
    printk("\n");
}

static void hinand_ecc_encipher(struct hinand_data *data)
{
	hinand_writel(data, HINAND_NFC_CON_ECC_ENABLE, HINAND_NFC_CON);
	hinand_writel(data, BIT_HINAND_NFC_ECC_TEST_ENC_ONLY, HINAND_NFC_ECC_TEST);
	while(!(hinand_readl(data, HINAND_NFC_ECC_TEST) & BIT_HINAND_NFC_ECC_TEST_ENC_ONLY));
	hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
}

static void hinand_ecc_decipher(struct hinand_data *data)
{
    hinand_writel(data, HINAND_NFC_CON_ECC_ENABLE, HINAND_NFC_CON);
    hinand_writel(data, BIT_HINAND_NFC_ECC_TEST_DEC_ONLY, HINAND_NFC_ECC_TEST);
    while(!(hinand_readl(data, HINAND_NFC_ECC_TEST) & BIT_HINAND_NFC_ECC_TEST_ENC_ONLY));
    hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
}

static void hinand_init(struct nand_chip *chip)
{
 	struct hinand_data *data = chip->priv;

    hinand_writel(data,
		BIT_HINAND_NFC_CON_READY_BUSY_SEL  | 
		BIT_HINAND_NFC_CON_BUS_WIDTH       |  		             
		BIT_HINAND_NFC_CON_OP_MODE         |
		BIT_HINAND_NFC_CON_PAGESIZE,
		HINAND_NFC_CON);

}

/* for 2k page, we use the buffer of controller directly */
static void hinand_read_2k(struct hinand_data *data)
{
    hinand_writel(data, HINAND_BUFFER_SIZE, HINAND_NFC_DATA_NUM);
    
	hinand_writel(data, data->hinand_address_buf[0] & 0xffff0000, HINAND_NFC_ADDRL);
	if(data->hinand_address_cycle > HINAND_ADDRESS_CYCLE_MASK)
	{
		hinand_writel(data, data->hinand_address_buf[1], HINAND_NFC_ADDRH);
	}

	hinand_writel(data, NAND_CMD_READSTART << 8 | NAND_CMD_READ0, HINAND_NFC_CMD);
	hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
	hinand_writel(data, 
			BIT_HINAND_NFC_OP_CMD2_EN |
			BIT_HINAND_NFC_OP_ADDR_EN |
			BIT_HINAND_NFC_OP_CMD1_EN | 
			BIT_HINAND_NFC_OP_READ_DATA_EN  |
			BIT_HINAND_NFC_OP_WAIT_READY_EN |
			(data->hinand_address_cycle << BIT_HINAND_NFC_OP_ADDRESS_CYCLE_BIT),
			HINAND_NFC_OP);
    while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);

    hinand_ecc_decipher(data);
    
	data->hinand_address_cycle = 0x0;
}

static void hinand_program_2k(struct hinand_data *data)
{    
    if(unlikely(!(data->column_addr)))
    {
        memset(data->chip->IO_ADDR_W, 0xff, data->column_addr);
    }

    hinand_writel(data, HINAND_BUFFER_SIZE, HINAND_NFC_DATA_NUM);

	hinand_writel(data, data->hinand_address_buf[0] & 0xffff0000, HINAND_NFC_ADDRL);
	if(data->hinand_address_cycle > HINAND_ADDRESS_CYCLE_MASK)
	{
		hinand_writel(data, data->hinand_address_buf[1], HINAND_NFC_ADDRH);
	}

    hinand_ecc_encipher(data);

	hinand_writel(data, NAND_CMD_PAGEPROG << 8 | HINAND_STATUS_CMD | NAND_CMD_SEQIN, HINAND_NFC_CMD);
	hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
	hinand_writel(data,  
			BIT_HINAND_NFC_OP_READ_STATUS_EN | 
			BIT_HINAND_NFC_OP_WAIT_READY_EN  | 
			BIT_HINAND_NFC_OP_CMD1_EN        | 
			BIT_HINAND_NFC_OP_CMD2_EN        |
			BIT_HINAND_NFC_OP_ADDR_EN        |
			BIT_HINAND_NFC_OP_WRITE_DATA_EN  |
			(data->hinand_address_cycle << BIT_HINAND_NFC_OP_ADDRESS_CYCLE_BIT),
			HINAND_NFC_OP);

    while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);
    
	data->hinand_address_cycle = 0x0;
}


/* 
    read function for 4k pagesize.
    the controller's buffer is 2k bytes. we have to use a 4k buffer in driver.
*/
static void hinand_read_4k(struct hinand_data *data)
{
	hinand_writel(data, data->hinand_address_buf[0] & 0xffff0000, HINAND_NFC_ADDRL);
	if(data->hinand_address_cycle > HINAND_ADDRESS_CYCLE_MASK)
	{
		hinand_writel(data, data->hinand_address_buf[1], HINAND_NFC_ADDRH);
	}

	hinand_writel(data, NAND_CMD_READSTART << 8 | NAND_CMD_READ0, HINAND_NFC_CMD);

	hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
    hinand_writel(data, (CONFIG_HINAND_PAGE_SIZE+CONFIG_HINAND_OOB_SIZE)/2, HINAND_NFC_DATA_NUM);
	hinand_writel(data, 
			BIT_HINAND_NFC_OP_CMD2_EN |
			BIT_HINAND_NFC_OP_ADDR_EN |
			BIT_HINAND_NFC_OP_CMD1_EN | 
			BIT_HINAND_NFC_OP_READ_DATA_EN  |
			BIT_HINAND_NFC_OP_WAIT_READY_EN |
			(data->hinand_address_cycle << BIT_HINAND_NFC_OP_ADDRESS_CYCLE_BIT),
			HINAND_NFC_OP);
	
	while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);

    hinand_ecc_decipher(data);

	memcpy(data->buffer, (unsigned char *)(data->chip->IO_ADDR_R), CONFIG_HINAND_PAGE_SIZE/2);
	memcpy(data->buffer+CONFIG_HINAND_PAGE_SIZE, 
        (unsigned char *)(data->chip->IO_ADDR_R+CONFIG_HINAND_PAGE_SIZE/2), CONFIG_HINAND_OOB_SIZE/2);

	hinand_writel(data, 
			BIT_HINAND_NFC_OP_READ_DATA_EN,
			HINAND_NFC_OP);

	while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);

    hinand_ecc_decipher(data);

	memcpy(data->buffer + CONFIG_HINAND_PAGE_SIZE/2, 
        (unsigned char *)(data->chip->IO_ADDR_R), CONFIG_HINAND_PAGE_SIZE/2);
    
	memcpy(data->buffer + CONFIG_HINAND_PAGE_SIZE+CONFIG_HINAND_OOB_SIZE/2, 
        (unsigned char *)(data->chip->IO_ADDR_R + CONFIG_HINAND_PAGE_SIZE/2), CONFIG_HINAND_OOB_SIZE/2);

	data->hinand_address_cycle = 0x0;
}

/* program function for 4k pagesize */
static void hinand_program_4k(struct hinand_data *data)
{
    if(unlikely(!(data->column_addr)))
    {
        memset(data->buffer, 0xff, data->column_addr);
    }

  	hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
    hinand_writel(data, (CONFIG_HINAND_PAGE_SIZE+CONFIG_HINAND_OOB_SIZE)/2, HINAND_NFC_DATA_NUM);
    memcpy((unsigned char *)(data->chip->IO_ADDR_W), 
        data->buffer, CONFIG_HINAND_PAGE_SIZE / 2);
    memcpy((unsigned char *)(data->chip->IO_ADDR_W) + CONFIG_HINAND_PAGE_SIZE / 2, 
        data->buffer + CONFIG_HINAND_PAGE_SIZE, CONFIG_HINAND_OOB_SIZE / 2);

    hinand_ecc_encipher(data);
    hinand_writel(data, data->hinand_address_buf[0] & 0xffff0000, HINAND_NFC_ADDRL);
    if(data->hinand_address_cycle > HINAND_ADDRESS_CYCLE_MASK)
    {
    	hinand_writel(data, data->hinand_address_buf[1], HINAND_NFC_ADDRH);
    }

    hinand_writel(data, NAND_CMD_PAGEPROG << 8 | HINAND_STATUS_CMD | NAND_CMD_SEQIN, HINAND_NFC_CMD);
    hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
    hinand_writel(data,  
    		BIT_HINAND_NFC_OP_CMD1_EN        | 
    		BIT_HINAND_NFC_OP_ADDR_EN        |
    		BIT_HINAND_NFC_OP_WRITE_DATA_EN  |
    		(data->hinand_address_cycle << BIT_HINAND_NFC_OP_ADDRESS_CYCLE_BIT),
    		HINAND_NFC_OP);

    while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);

    memcpy((unsigned char *)(data->chip->IO_ADDR_W),
        data->buffer + CONFIG_HINAND_PAGE_SIZE/2, CONFIG_HINAND_PAGE_SIZE/2);
    memcpy((unsigned char *)(data->chip->IO_ADDR_W) + CONFIG_HINAND_PAGE_SIZE/2, 
        data->buffer + CONFIG_HINAND_PAGE_SIZE + CONFIG_HINAND_OOB_SIZE/2, CONFIG_HINAND_OOB_SIZE/2);

    hinand_ecc_encipher(data);

    hinand_writel(data, 0x1d, HINAND_NFC_OP);

    while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);

    data->hinand_address_cycle = 0x0;
}

/*
 * hardware specific funtion for controlling .
 */
static void hinand_cmd_ctrl(struct mtd_info *mtd, int dat,
		unsigned int ctrl)
{
	unsigned int hinand_address = 0, hinand_address_offset;
	struct nand_chip *chip = mtd->priv;
	struct hinand_data *data = chip->priv;

    debug("%s ctrl 0x%x command 0x%x page offset %d\n", __FUNCTION__, ctrl, dat, data->page_offset);
    
	if( ctrl & NAND_ALE )
	{
		if(ctrl & NAND_CTRL_CHANGE)
		{
			data->hinand_address_cycle = 0x0;
			data->hinand_address_buf[0] = 0x0;
			data->hinand_address_buf[1] = 0x0;
		}
		hinand_address_offset =  data->hinand_address_cycle << 3;

		if(data->hinand_address_cycle >= HINAND_ADDRESS_CYCLE_MASK)
		{
			hinand_address_offset = (data->hinand_address_cycle - HINAND_ADDRESS_CYCLE_MASK) << 3;
			hinand_address = 1;
		}

		data->hinand_address_buf[hinand_address] |= ((dat & 0xff) << hinand_address_offset);

		data->hinand_address_cycle ++;
	}

	if( ( ctrl & NAND_CLE ) && ( ctrl & NAND_CTRL_CHANGE ) )
	{
		data->command = dat & 0xff;
		switch(data->command) {
			case NAND_CMD_PAGEPROG:
                if(CONFIG_HINAND_PAGE_SIZE == 2048)
                {
                    hinand_program_2k(data);
                }
                else if(CONFIG_HINAND_PAGE_SIZE == 4096)
                {
                    hinand_program_4k(data);
                }
				break;

			case NAND_CMD_READSTART: 
                if(CONFIG_HINAND_PAGE_SIZE == 2048)
                {
                    hinand_read_2k(data);
                }
                else if(CONFIG_HINAND_PAGE_SIZE == 4096)
                {
                    hinand_read_4k(data);
                }
				break;

			case NAND_CMD_ERASE2:
        		hinand_writel(data, data->hinand_address_buf[0], HINAND_NFC_ADDRL);
        		if(data->hinand_address_cycle > HINAND_ADDRESS_CYCLE_MASK)
        		{
        			hinand_writel(data, data->hinand_address_buf[1], HINAND_NFC_ADDRH);
        		}

				hinand_writel(data, (data->command << 8) | HINAND_STATUS_CMD | NAND_CMD_ERASE1, HINAND_NFC_CMD);
				hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
				hinand_writel(data,
						BIT_HINAND_NFC_OP_READ_STATUS_EN | 
						BIT_HINAND_NFC_OP_WAIT_READY_EN  | 
						BIT_HINAND_NFC_OP_CMD2_EN        |
						BIT_HINAND_NFC_OP_CMD1_EN        |
						BIT_HINAND_NFC_OP_ADDR_EN        |
						(data->hinand_address_cycle << BIT_HINAND_NFC_OP_ADDRESS_CYCLE_BIT),
						HINAND_NFC_OP);
        		while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);
				break;

			case NAND_CMD_READID:
				hinand_writel(data, data->command, HINAND_NFC_CMD);
				hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
				hinand_writel(data,  
						BIT_HINAND_NFC_OP_CMD1_EN       | 
                        BIT_HINAND_NFC_OP_ADDR_EN       |
                        BIT_HINAND_NFC_OP_READ_DATA_EN  |
						BIT_HINAND_NFC_OP_WAIT_READY_EN | 
						(1 << BIT_HINAND_NFC_OP_ADDRESS_CYCLE_BIT),
						HINAND_NFC_OP);
				data->hinand_address_cycle = 0x0;

        		while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);
                if(CONFIG_HINAND_PAGE_SIZE == 4096)
                {
                    memcpy(data->buffer, (unsigned char *)(chip->IO_ADDR_R), 0x10);
                }
				break;
	
			case NAND_CMD_STATUS:
				hinand_writel(data, HINAND_STATUS_CMD, HINAND_NFC_CMD);
				hinand_writel(data, HINAND_NFC_CON_ECC_DISABLE, HINAND_NFC_CON);
				hinand_writel(data,
						BIT_HINAND_NFC_OP_READ_STATUS_EN | 
						BIT_HINAND_NFC_OP_WAIT_READY_EN ,
						HINAND_NFC_OP);
        		while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);
				break;

			case NAND_CMD_SEQIN:
			case NAND_CMD_ERASE1:
			case NAND_CMD_READ0:
				break;
				
			default :
				break;
		}
	}

	if( (dat == NAND_CMD_NONE) && data->hinand_address_cycle)
	{
		if(data->command == NAND_CMD_SEQIN || data->command == NAND_CMD_READ0)
		{
			data->page_offset = 0x0;
            data->column_addr = data->hinand_address_buf[0] & 0xffff;
		}

        debug("command 0x%x, address is 0x%x, 0x%x, cycle %d\n", data->command, 
            data->hinand_address_buf[0], data->hinand_address_buf[1], 
            data->hinand_address_cycle);

	}

}
/*
 * hinand_dev_ready()
 *
 * returns 0 if the nand is busy, 1 if it is ready
 */
static int hinand_dev_ready(struct mtd_info *mtd)
{
	return 0x1;
}

static void hinand_select_chip(struct mtd_info *mtd, int chip)
{
	struct nand_chip *_chip = mtd->priv;
	struct hinand_data *data = _chip->priv;

	if (chip > 4)
	{
		BUG();
	}

	if (chip < 0)
	{
		return;
	}

	hinand_set_cs(data, chip);
	while((hinand_readl( data, HINAND_NFC_STATUS ) & 0x1) == 0x0);
}

static uint8_t hinand_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	unsigned int reg;
	struct hinand_data *data = chip->priv;
	if(data->command == NAND_CMD_STATUS)
	{
		reg = ( ( hinand_readl(data, HINAND_NFC_STATUS) >> 0x5 ) & 0xff );
		return reg;
	}

    if(CONFIG_HINAND_PAGE_SIZE == 2048)
    {
    	reg = readb(chip->IO_ADDR_R + data->column_addr + data->page_offset);
    }
    else if(CONFIG_HINAND_PAGE_SIZE == 4096)
    {
    	reg = readb(data->buffer + data->column_addr + data->page_offset);
    }

    debug("%s 0x%x 0x%x 0x%x\n", __FUNCTION__, data->column_addr, data->page_offset, reg);
    
    data->page_offset ++;
    
	return reg;
}

static u16 hinand_read_word(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct hinand_data *data = chip->priv;
	u16    read_data = 0;

    if(CONFIG_HINAND_PAGE_SIZE == 2048)
    {
    	read_data = readw(chip->IO_ADDR_R + data->column_addr + data->page_offset);
    }
    else if(CONFIG_HINAND_PAGE_SIZE == 4096)
    {
    	read_data = readw(data->buffer + data->column_addr + data->page_offset);
    }

	data->page_offset += 2;

    debug("%s column %d page offset %d, data 0x%x\n", __FUNCTION__, 
        data->column_addr, data->page_offset, read_data);

	return read_data;
}

static void hinand_read_buf(struct mtd_info *mtd, uint8_t *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct hinand_data *data = chip->priv;

    debug("%s column %d page offset %d, len %d\n", __FUNCTION__, 
        data->column_addr, data->page_offset, len);

    if(CONFIG_HINAND_PAGE_SIZE == 2048)
    {
    	memcpy(buf, (unsigned char *)(chip->IO_ADDR_R  + data->column_addr + data->page_offset), len);
        
    }
    else if(CONFIG_HINAND_PAGE_SIZE == 4096)
    {
    	memcpy(buf, data->buffer + data->column_addr + data->page_offset, len);
    }

	data->page_offset += len;
}

static void hinand_write_buf(struct mtd_info *mtd, const uint8_t *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct hinand_data *data = chip->priv;

    debug("%s column %d page offset %d, len %d\n", __FUNCTION__, 
        data->column_addr, data->page_offset, len);

    if(CONFIG_HINAND_PAGE_SIZE == 2048)
    {
    	memcpy((unsigned char *)(chip->IO_ADDR_W + data->column_addr + data->page_offset), buf, len);
    }
    else if(CONFIG_HINAND_PAGE_SIZE == 4096)
    {
    	memcpy(data->buffer + data->column_addr + data->page_offset, buf, len);
    }

	data->page_offset += len;
}

int board_nand_init(struct nand_chip *nand)
{
	struct hinand_data *data;
	/* Allocate memory for the device structure (and zero it) */
	data = kmalloc(sizeof(struct hinand_data), GFP_KERNEL);
	if (!data) {
		printf("failed to allocate device structure.\n");
		return -ENOMEM;
	}
    
	data->iobase = HI_NAND_BASE_VIRT_REG_ADDR;
	data->chip = nand;
	nand->priv = data;
	nand->cmd_ctrl = hinand_cmd_ctrl;
	nand->dev_ready = hinand_dev_ready;
	nand->select_chip = hinand_select_chip;
	nand->chip_delay = CHIP_DELAY;
	nand->options |= NAND_NO_AUTOINCR | NAND_BBT_SCANNED;
	nand->read_byte = hinand_read_byte;
	nand->read_word = hinand_read_word;
	nand->write_buf = hinand_write_buf;
	nand->read_buf = hinand_read_buf;

    if(CONFIG_HINAND_PAGE_SIZE == 2048)
    {
    	nand->ecc.layout = &hinand_oob_64_1bit;
    }
    else if(CONFIG_HINAND_PAGE_SIZE == 4096)
    {
//    	nand->ecc.layout = &hinand_oob_128_4bit;
    }
    
	nand->ecc.mode = NAND_ECC_NONE;
	data->hinand_address_cycle = 0;
	data->hinand_address_buf[0] = 0;
	data->hinand_address_buf[1] = 0;

    memset((char*)HI_NAND_BASE_VIRT_MEM_ADDR, 0xff, HINAND_BUFFER_SIZE);

    if(CONFIG_HINAND_PAGE_SIZE == 4096)
    {
        data->buffer = kmalloc(HINAND_BUFFER_SIZE*2, GFP_KERNEL);
        memset(data->buffer, 0xff, HINAND_BUFFER_SIZE*2);
    }

	hinand_init(nand);

	return 0;
}


