/**
 * gpio.c
 *
 * History:
 *    2005/09/03 - [Charles Chiou] created file
 *    2008/05/09 - [Charles Chiou] added macros specifying default DIR, MASK
 *    2009/10/15 - [Allen Wang] added macros for GPIO4.
 *
 * Copyright (C) 2004-2009, Ambarella, Inc.
 *
 * All rights reserved. No Part of this file may be reproduced, stored
 * in a retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Ambarella, Inc.
 */

#include <config.h>
#include <asm/types.h>
#include <asm/arch/ambhw/chip.h>
#include <asm/arch/ambhw.h>
#include <asm/arch/ambhw/gpio.h>
  
struct amb_gpio_reg
{
    u32 data;
    u32 dir;
    u32 is;
    u32 ibe;
    u32 iev;
    u32 ie;
    u32 afsel;
    u32 ris;
    u32 mis;
    u32 ic;
    u32 mask;
    u32 enable;
};

struct amb_gpio_init_data
{    
    u32 afsel;
    u32 data;
    u32 dir;
    u32 mask;
};
 
static void *gpio_bank_a5s[] = {    
    (void *)(GPIO0_BASE),      
    (void *)(GPIO1_BASE), 
#if (GPIO_INSTANCES >= 3) 
    (void *)(GPIO2_BASE),
#endif
}; 

static struct amb_gpio_init_data init_data[] = {
    {DEFAULT_GPIO0_AFSEL,DEFAULT_GPIO0_DIR, DEFAULT_GPIO0_MASK,DEFAULT_GPIO0_DATA},
    {DEFAULT_GPIO1_AFSEL,DEFAULT_GPIO1_DIR, DEFAULT_GPIO1_MASK,DEFAULT_GPIO1_DATA},
    {DEFAULT_GPIO2_AFSEL,DEFAULT_GPIO2_DIR, DEFAULT_GPIO2_MASK,DEFAULT_GPIO2_DATA},
};

int amb_gpio_init(void)
{
    int i;
    struct amb_gpio_reg *reg;
    for (i = 0; i< GPIO_INSTANCES; i++){
        reg = gpio_bank_a5s[i];
        reg->afsel = init_data[i].afsel;
        reg->dir = init_data[i].dir;
        reg->mask = init_data[i].mask;
        reg->data = init_data[i].data;
        reg->enable = 0xffffffff;
    }
    return 0;
}

int amb_gpio_config_hw(int gpio)
{
    struct amb_gpio_reg *reg;
    
    if (gpio > GPIO_MAX_LINES)
        return -1;

    reg = gpio_bank_a5s[gpio>>5];
    gpio &= 0x1f;

    reg->afsel |= (1<<gpio);
    reg->afsel &= ~(1<<gpio);
    return 0;
}

int amb_gpio_set_value(int gpio, int value)
{    
    struct amb_gpio_reg *reg;
    
    if (gpio > GPIO_MAX_LINES)
        return -1;

    reg = gpio_bank_a5s[gpio>>5];
    gpio &= 0x1f;
    value = value?1:0;
    reg->data = (reg->data &(~(1<<gpio))) | value<<gpio;
    return 0;
}

int amb_gpio_set(int gpio, int value)
{
    return amb_gpio_set_value(gpio, 1);
}

int amb_gpio_clear(int gpio, int value)
{
    return amb_gpio_set_value(gpio, 0);
}

int amb_gpio_get_value(int gpio)
{    
    struct amb_gpio_reg *reg;
    
    if (gpio > GPIO_MAX_LINES)
        return -1;

    reg = gpio_bank_a5s[gpio>>5];
    gpio &= 0x1f;
    return (reg->data&(1<<gpio))?1:0;
} 

int amb_gpio_direction(int gpio, int is_input)
{
    struct amb_gpio_reg *reg;
    
    if (gpio > GPIO_MAX_LINES)
        return -1;

    reg = gpio_bank_a5s[gpio>>5];
    gpio &= 0x1f;
    reg->dir = (reg->dir & ~(1<<gpio)) | (!is_input)<<gpio; 
    reg->afsel &= ~(1<<gpio);
    reg->mask = reg->mask | (1<<gpio);
    return 0;
}

int amb_gpio_direction_input(int gpio)
{
    return amb_gpio_direction(gpio, 1);
}

int amb_gpio_direction_output(int gpio, int value)
{
    amb_gpio_direction(gpio, 0);
    return amb_gpio_set_value(gpio,value);
}

