#include <common.h>  
#include <asm/io.h>
#include <asm/setup.h>
#include <nand.h>
#include <i2c.h>
#include <exports.h>
#include <ambimage.h>
#include <configs/ambarella_a5s_ipc.h>

#include <config.h>
#include <asm/arch/ambhw/chip.h>
#include <asm/arch/basedef.h>
#include <asm/arch/ambhw/eth.h>
#include <asm/arch/ambhw/intvec.h>
#include <asm/arch/ambhw/uart.h>
#include <asm/arch/ambhw/wdog.h>
#include <asm/arch/hal/hal.h>
#include <asm/arch/bldfunc.h>

#define UPDATE_INIT       (0xFF)
#define UPDATE_START   (0x55)
#define UPDATE_END       (0xAA)
#define UPDATE_VERIFY   (0x5A)
#define MAX_TRY_COUNT 2620

extern  int a5s_nand_read_pages(u32 block, u32 page, u32 pages, \
                const u8 *buf, u32 enable_ecc);
extern int a5s_nand_read_byte(u8 *dst, u32 from, int len);
extern  void flush_all_cache_region(void *addr, unsigned int size);
extern  int get_partinfo(struct boot_mtd_partition *mtd_parts, u32 offset);

static int load_img_by_name(char *name);
static inline loff_t get_good_block(int type);

#define nand_skip_badblock(nand, offset) \
if (nand_block_isbad (nand, offset & ~(nand->erasesize - 1))) \
{   \
    printf ("Skip bad block 0x%llx\n",offset);   \
    offset  += nand->erasesize; \
    continue;   \
}   \

struct partinfo_crc_info
{
    size_t len;
    unsigned int crc32;
};

enum BOLOCK_TYPE
{
    BLOCK_UPDATE_FLAGS = 0,
    BLOCK_ERR_COUNT,
    BLOCK_CRC_DATA,
    BLOCK_NUM,
};

enum FLAG_PAGES_TYPE
{
    PAGES_UPDATE_START,
    PAGES_UPDATE_END,
    PAGES_UPDATE_VERIFY, 
};


static loff_t good_block_table[CONFIG_UPDATE_FLAG_SIZE/(128*1024)];

/******************************************************************************* 
 *     :  __disable_mmu
 *       :  رmmu
 *       :  
 *       :   
 *  ֵ   :  
 *******************************************************************************/
inline static void __disable_mmu(void)
{
    unsigned long reg = 0;

    asm ("mrc p15, 0, %0, c1, c0, 0": :"r" (reg));/*Read control register*/
    asm ("bic %0, %0, #(1<<0)": :"r" (reg));      /* Turn off bit 0 MMU*/
    //asm ("bic %0, %0, #(1<<13)": :"r" (reg));     /* Turn off bit 13 HV*/
    asm ("mcr p15, 0, %0, c1, c0, 0": :"r" (reg));/* Write control register*/
}

/******************************************************************************* 
 *     :  load_run_boot
 *       :  زuboot
 *       :  
 *       :   
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
int load_run_boot(void)
{
    volatile unsigned char *boot_from = (unsigned char *)BOOT_FROM;
    ulong (*entry)(void) = (ulong (*)(void))CONFIG_SYS_LOAD_ADDR;

    (*boot_from) ++;
    printf("%01x..",(*boot_from)&0x0f);

    if (((*boot_from)&0x0f) == 1 )
    {
        (*boot_from) ++;
        printf("%01x..",(*boot_from)&0x0f);
    }

    if (((*boot_from) &0x0f) == 2)//run uboot
    {
        if(!load_img_by_name("uboot"))
        {
            _clean_flush_all_cache();
            entry();
        }
        goto err;
    }
    printf("boot_from:%s\n",((*boot_from)>>4)==2 ?"normal":"backup" );
    return 0;

err:
    printf("all u-boot was destory,used default\n");
    return -1;
}

/******************************************************************************* 
 *     :  load_img_by_name
 *       :  ͨFLASHѰӦӳ񣬲صӦַ 
 *       :  -name:Ҫصӳ
 *       :  
 *  ֵ   :  -0: ɹ
 *              -1: ʧ
 *******************************************************************************/
static int load_img_by_name(char *name)
{
    int i;
    int j;
    int ret;
    unsigned int crc;
    unsigned int offset;
    struct amb_image aimg;
    unsigned char *boot_from = (unsigned char *)BOOT_FROM; 
    unsigned char *data  = (unsigned char *)(CONFIG_SYS_LOAD_ADDR + UBOOT_SZ);
    
    for (i = 2; i; i--)
    {
        debug("i:%d\n",i);
        ret = a5s_nand_read_byte((u8 *)(&aimg),i*UBOOT_SZ,sizeof(aimg));
        if (ret != sizeof(aimg))
        {
            printf("read boot image failed ret:%d\n",ret);
            return -1;
        }
        crc = aimg.header_crc32;
        aimg.header_crc32 = 0;
        aimg.header_crc32 = crc32(0,(const char *)(&aimg),  sizeof(aimg) );
        if (aimg.image_magic != UBOOT_MAGIC  ||  aimg.header_crc32 != crc){
            debug("magic:%08x,crc32:%08x\n",aimg.image_magic,crc);
            continue;
        }

        ret = a5s_nand_read_byte(data, i*UBOOT_SZ + sizeof(aimg),aimg.data_size);
        if (ret != aimg.data_size)
        {
            printf("read boot image failed ret:%d\n",ret);
            return -1;
        }
        crc = crc32(0,data, aimg.data_size);

        if (crc != aimg.data_crc32)
        {
            debug("crc32:%08x,crc32 calc:%08x\n",aimg.data_crc32,crc);
            continue;
        }

        for (j = 0,offset = 0;j < aimg.node_count ;j++)
        {
            if (!strcmp(aimg.node[j].name,name))
            {
                memcpy((void *)aimg.node[j].entry,data + offset, aimg.node[j].size);
                (*boot_from) |= i <<4;
                return 0;
            }
            debug("name:%s\n",aimg.node[j].name);
            offset += aimg.node[j].size;
        }
    }

    return -1;
}

/******************************************************************************* 
 *     :  scan_good_block
 *       :  ˽гԱͷŸö
 *       :  -nand:nandָ
 *       :  -table:Чַָ
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
static int scan_good_block(nand_info_t *nand, loff_t *table)
{
    int i; 
    loff_t offset = CONFIG_UPDATE_FLAG_OFFSET;
    loff_t end = CONFIG_UPDATE_FLAG_OFFSET + CONFIG_UPDATE_FLAG_SIZE;
    for (i = 0; (offset <  end) ; )
    {
        nand_skip_badblock(nand, offset);
        table[i] = offset;
        debug("%llx ",offset);
        offset += nand->erasesize;
        i++;
    }
    debug("\n");
    return i;
} 

/******************************************************************************* 
 *     :  get_good_block
 *       :  ȡӦ־Ĵ洢ַ
 *       :  -type:־
 *       :   
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
static inline loff_t get_good_block(int type)
{
    return good_block_table[type];
}

/******************************************************************************* 
 *     :  get_err_count
 *       :  ˽гԱͷŸö
 *       :  -this:ͷŵĶָ
 *       :   
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
inline  int get_err_count(nand_info_t *nand,unsigned int *value)
{ 
    int ret = -1;
    size_t size;
    loff_t offset;
    char *buf;
    
    buf = (unsigned char *)malloc(nand->writesize);
    if (!buf)
    {
        return -1;
    }

    offset = get_good_block(BLOCK_ERR_COUNT);
    if (offset < 0)
    {
        goto out;
    }
    debug("offset:%llx\n",offset);
    size = nand->writesize;
    ret = nand_read(nand,offset,&size,buf) ;
    if (ret< 0)
    {
        goto out;
    } 
    ret = 0;
    *value = (unsigned)buf[0];
    
out:
    free(buf);
    return ret;
}

/******************************************************************************* 
 *     :  set_err_count
 *       :  ô
 *       :  -nand:nandָ
 *              -count:ֵ
 *       :   
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
inline static int set_err_count(nand_info_t *nand, int count)
{
    int ret;
    loff_t offset;
    size_t size;
    unsigned char *buf;

    buf = (unsigned char *)malloc(nand->writesize);
    if (!buf)
    {
        return -1;
    }
    offset = get_good_block(BLOCK_ERR_COUNT);
    ret = nand_erase(nand,offset,nand->erasesize);
    if (ret < 0)
    {
        free(buf);
        return ret;
    }
    
    memset(buf,0xff,nand->writesize);
    buf[0] = (unsigned char)count;
    size = nand->writesize;
    ret = nand_write(nand,offset,&size,buf); 
    if (ret <0 || (size != nand->writesize))
    {
        printf("set error count failed\n");
    }
    free(buf);
    return ret;
}

/******************************************************************************* 
 *     :  add_err_count
 *       :  Ӵ
 *       :  -nand:nandָ
 *       :   
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
inline int add_err_count(nand_info_t *nand)
{
    unsigned int err_count;
    int ret = get_err_count(nand,&err_count);
    
    if (ret < 0)
    {
        return -1;
    }
    
    err_count ++;
    ret = set_err_count(nand, err_count);
    if (ret < 0)
    {
        return ret;
    }

    ret = get_err_count(nand,&err_count);
    if (ret < 0)
    {
        return -1;
    }
    
    return 0;
}

/******************************************************************************* 
 *     :  clear_update_flag
 *       :  ״̬Ϊʹ״̬
 *       :  -nand:nandָ
 *       :   
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
inline static int clear_update_flag(nand_info_t *nand)
{
    int ret;
    
    ret = nand_erase(nand,get_good_block(BLOCK_UPDATE_FLAGS),nand->erasesize);
    if (ret < 0)
    {
        return ret;
    }
    return set_err_count(nand, 1);//ĬΪ1ӦóҪ 
}

/******************************************************************************* 
 *     :  load_run_boot
 *       :  鱸ݷǷ
 *       :  -nand:nandָ
 *              -info:CRCϢָ
 *              -mtd_parts:ָ
 *       :   
 *  ֵ   :  -0: ɹ
 *              -1: ʧ 
 *******************************************************************************/
static int check_partition(nand_info_t *nand, struct partinfo_crc_info *info,
                            struct boot_mtd_partition *mtd_parts)
{
    int i;
    int ret;
    size_t tmp;
    loff_t end;
    loff_t offset_rd;
    unsigned int crc;
    unsigned char *buf;

    buf = (unsigned char *)malloc(nand->erasesize);
    
    if (!buf)
    {
        printf("can't malloc mem\n");
        return -1;
    } 

    for (i = 0; '\0'!= mtd_parts[i].name[0]; i++)
    {
        offset_rd = mtd_parts[i].offset - CONFIG_NORMAL_OFFSET + CONFIG_BAKUP_OFFSET;
        if ((offset_rd + mtd_parts[i].size >= (CONFIG_BAKUP_OFFSET 
            +  CONFIG_BAKUP_SIZE))  || offset_rd < CONFIG_BAKUP_OFFSET)
        {
            continue;
        }

        printf("%s...", mtd_parts[i].name);
        if ( 0xFFFFFFFF == (unsigned int) info[i].len )
        {
            continue;
        }
          
        crc = 0; 
        debug("info_len:%08x,size:%llx\n",info[i].len,mtd_parts[i].size);
        end = offset_rd + mtd_parts[i].size;
        for ( ; info[i].len && (offset_rd < end); )
        {
            nand_skip_badblock(nand, offset_rd);
            debug("read:%llx\n",offset_rd);
            tmp = nand->erasesize;
            ret = nand_read(nand,offset_rd,&tmp,buf);
            if (ret || (tmp != nand->erasesize))
            {
                printf("[%s] nand read error\n",__func__);
                goto err;
            }
            crc = crc32(crc,buf,nand->erasesize);   
            offset_rd += nand->erasesize;
            info[i].len -= nand->erasesize;
        }
        
        if (crc != info[i].crc32)
        {
            printf("\nwaring %s is destory! crc:%x,info crc:%x.\n",
                    mtd_parts[i].name,crc, info[i].crc32);
            goto err;
        }  
    }

    free(buf);
    return 0;
err:
    free(buf);
    return -1;
}

/******************************************************************************* 
 *     :  set_update_flag
 *       :  ˽гԱͷŸö
 *       :  -page_no:־ڵҳ
 *              -flag:־ֵ
 *       :  
 *  ֵ   :  -0: ɹ
 *              -1: ʧ
 *******************************************************************************/
static int set_update_flag(int page_no,char flag)
{   
    int ret;
    size_t size;
    char *buf;
    nand_info_t *nand;
    loff_t offset = CONFIG_UPDATE_FLAG_OFFSET;
     
    if(nand_curr_device <0)
    {
        return -1;
    }

    nand = &nand_info[nand_curr_device];
    if (NULL == nand)
    {
        return -1;
    }
    
    offset = get_good_block(BLOCK_UPDATE_FLAGS);
    if (offset < 0)
    {
        return -1;
    }

    buf = malloc(nand->writesize);
    if (!buf)
    {
        return -1;
    }

    memset(buf,0xff,nand->writesize);
    buf[0] = flag;
    size = nand->writesize;
    offset += (page_no * nand->writesize);
    ret = nand_write(nand,offset,&size,buf);//ֱдÿѾ
    free(buf);
    return ret;
}

/******************************************************************************* 
 *     :  update_start
 *       :  ÿʼ־
 *       :  
 *       :  
 *  ֵ   :  -0: ɹ
 *              -1: ʧ
 *******************************************************************************/
int update_start(void)
{
    
    nand_info_t *nand;

    nand = &nand_info[nand_curr_device];
    if(nand_curr_device <0 || !nand)
    {
        return -1;
    }

    puts ("Erasing update flag partition.\n");

    nand_erase(nand,get_good_block(BLOCK_UPDATE_FLAGS),nand->erasesize);

    return set_update_flag(PAGES_UPDATE_START,UPDATE_START);
}

/******************************************************************************* 
 *     :  update_end
 *       :  ־
 *       :  
 *       :  
 *  ֵ   :  -0: ɹ
 *              -1: ʧ
 *******************************************************************************/
int update_end(void)
{
    int ret;
    ret = set_update_flag(PAGES_UPDATE_END,UPDATE_END);
    if (ret)
    {
        return -1;
    }
    return set_update_flag(PAGES_UPDATE_VERIFY,UPDATE_VERIFY);
}

/******************************************************************************* 
 *     :  sys_sync
 *       :  ʵֱ/ָ
 *       :  
 *       :  
 *  ֵ   :  -0: ɹ
 *              -1: ʧ
 *******************************************************************************/
int sys_sync(void)
{
    int i;
    int ret;
    size_t size;
    char *env;
    nand_info_t *nand;
    unsigned char update_state[] = {UPDATE_START, UPDATE_END,UPDATE_VERIFY}; 
    unsigned int err_count;
    unsigned char state;
    unsigned char *page_buffer ;

    if(nand_curr_device <0)
    {
        return -1;
    }

    nand = &nand_info[nand_curr_device];
    if (!nand)
    {
        return -1;
    }
    
    memset(good_block_table,0xff,sizeof(good_block_table));//ʹΪ͵-1

    if (scan_good_block(nand, good_block_table) < BLOCK_NUM)
    {
#if 1
        while(1)
        {
            printf("waring! to mach bad block\n"); 
            if (tstc() && getc() == '*')
            {
                break;
            }
            udelay(500*1000);
        } 
#else
        printf("waring! too much bad blocks\n"); 
#endif
        return -1;
    }
    
    env = getenv("sysbackup");
    if (env)
    {
        if (0 == simple_strtoul(env, NULL,10))
        {
            return 0;
        }
    }
    
    page_buffer = malloc(nand->writesize * BLOCK_NUM);
    if(NULL == page_buffer)
    {
        return -1;
    }

    ret = get_err_count(nand, &err_count);
    if (ret < 0)
    {
        printf("get err_count\n");
        err_count = 1;/**/
    }
    
    if (err_count >MAX_ERR_COUNT)//һϵ 
    {
        set_err_count(nand,1);//ֵΪ1ֹ UPDATE_VERIFY ״ֱ̬ӽб
        printf("first power?\n");//»Ҫж״̬豸ܲϽлָ
        err_count = 1;
    }
    
    size = nand->writesize*BLOCK_NUM;
    ret = nand_read_skip_bad(nand, CONFIG_UPDATE_FLAG_OFFSET, &size, page_buffer);
    if (ret || (size != (nand->writesize*BLOCK_NUM)))
    {
        printf("read flags error\n");
        goto out;
    }

    i = 0;
    state = UPDATE_INIT;
    for (;(i< sizeof(update_state)) && (update_state[i] == page_buffer[i*nand->writesize]); i++)
    {
            state = update_state[i];
    }

    if ((i < sizeof(update_state)) && (page_buffer[i*nand->writesize] !=  UPDATE_INIT))
    {
        clear_update_flag(nand);
        printf("invalid state:%02x\n",page_buffer[i*nand->writesize]);
        goto out;
    }
    
    printf("state:%02x,err_count:%02x\n",state,err_count);
out:
    free(page_buffer);
    return 0;
}
