/*
 * Command for accessing SPI flash.
 *
 * Copyright (C) 2008 Atmel Corporation
 * Licensed under the GPL-2 or later.
 */

#include <common.h>
#include <spi_flash.h>

#include <asm/io.h>

#include <image.h>
#include <hush.h>

#ifdef UBOOT_UPDATE
#include <updater.h>
#endif
#ifndef CONFIG_SF_DEFAULT_SPEED
# define CONFIG_SF_DEFAULT_SPEED	1000000
#endif
#ifndef CONFIG_SF_DEFAULT_MODE
# define CONFIG_SF_DEFAULT_MODE		SPI_MODE_3
#endif

static struct spi_flash *flash = 0;
// --> liu_hongcong
#ifdef CONFIG_DVR_CMD
//static int spi_probed = 0;
static int __do_spi_probe(unsigned int cs);
#endif


extern volatile unsigned int CPU_SPEC_VAL;


// <-- liu_hongcong
static int do_spi_flash_probe(int argc, char *argv[])
{
	unsigned int bus = 0;
	unsigned int cs;
	unsigned int speed = CONFIG_SF_DEFAULT_SPEED;
	unsigned int mode = CONFIG_SF_DEFAULT_MODE;
	char *endp;
	struct spi_flash *new;

	if (argc < 2)
		goto usage;

	cs = simple_strtoul(argv[1], &endp, 0);
	if (*argv[1] == 0 || (*endp != 0 && *endp != ':'))
		goto usage;
	if (*endp == ':') {
		if (endp[1] == 0)
			goto usage;

		bus = cs;
		cs = simple_strtoul(endp + 1, &endp, 0);
		if (*endp != 0)
			goto usage;
	}

	if (argc >= 3) {
		speed = simple_strtoul(argv[2], &endp, 0);
		if (*argv[2] == 0 || *endp != 0)
			goto usage;
	}
	if (argc >= 4) {
		mode = simple_strtoul(argv[3], &endp, 16);
		if (*argv[3] == 0 || *endp != 0)
			goto usage;
	}

	new = spi_flash_probe(bus, cs, speed, mode);
	if (!new) {
		printf("Failed to initialize SPI flash at %u:%u\n", bus, cs);
		return 1;
	}

	if (flash)
		spi_flash_free(flash);
	flash = new;

	printf("%u KiB %s at %u:%u is now current device\n",
			flash->size >> 10, flash->name, bus, cs);
// --> lhc
#ifdef CONFIG_DVR_CMD
	//spi_probed = 1;
#endif
// <-- lhc
	return 0;

usage:
	puts("Usage: sf probe [bus:]cs [hz] [mode]\n");
	return 1;
}

static int do_spi_flash_read_write(int argc, char *argv[])
{
	unsigned long addr;
	unsigned long offset;
	unsigned long len;
	void *buf;
	char *endp;
	int ret;

	if (argc < 4)
		goto usage;

	addr = simple_strtoul(argv[1], &endp, 16);
	if (*argv[1] == 0 || *endp != 0)
		goto usage;
	offset = simple_strtoul(argv[2], &endp, 16);
	if (*argv[2] == 0 || *endp != 0)
		goto usage;
	len = simple_strtoul(argv[3], &endp, 16);
	if (*argv[3] == 0 || *endp != 0)
		goto usage;

	buf = map_physmem(addr, len, MAP_WRBACK);
	if (!buf) {
		puts("Failed to map physical memory\n");
		return 1;
	}

	if (strcmp(argv[0], "read") == 0)
		ret = spi_flash_read(flash, offset, len, buf);
	else
		ret = spi_flash_write(flash, offset, len, buf);

	unmap_physmem(buf, len);

	if (ret) {
		printf("SPI flash %s failed\n", argv[0]);
		return 1;
	}

	return 0;

usage:
	printf("Usage: sf %s addr offset len\n", argv[0]);
	return 1;
}

static int do_spi_flash_erase(int argc, char *argv[])
{
	unsigned long offset;
	unsigned long len;
	char *endp;
	int ret;

	if (argc < 3)
		goto usage;

	offset = simple_strtoul(argv[1], &endp, 16);
	if (*argv[1] == 0 || *endp != 0)
		goto usage;
	len = simple_strtoul(argv[2], &endp, 16);
	if (*argv[2] == 0 || *endp != 0)
		goto usage;

	ret = spi_flash_erase(flash, offset, len);
	if (ret) {
		printf("SPI flash %s failed\n", argv[0]);
		return 1;
	}

	return 0;

usage:
	puts("Usage: sf erase offset len\n");
	return 1;
}

static int do_spi_flash_protect(int argc, char *argv[])
{
	unsigned long len;
	char *endp;
	int ret;

	if (argc < 2)
		goto usage;

	len = simple_strtoul(argv[1], &endp, 16);
	if (*argv[1] == 0 || *endp != 0)
		goto usage;

	ret = spi_flash_protect(flash,len);
	if (ret) {
		printf("SPI flash %s failed\n", argv[0]);
		return 1;
	}

	return 0;

usage:
	puts("Usage: sf protect len\n");
	return 1;
}


static int do_spi_flash(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	const char *cmd;

	/* need at least two arguments */
	if (argc < 2)
		goto usage;

	cmd = argv[1];

	if (strcmp(cmd, "probe") == 0)
		return do_spi_flash_probe(argc - 1, argv + 1);

// --> lhc
#ifdef CONFIG_DVR_CMD
	if (!flash) {
		__do_spi_probe(0);
	}
#endif
	// <-- lhc

	/* The remaining commands require a selected device */
	if (!flash) {
		puts("No SPI flash selected. Please run `sf probe'\n");
		return 1;
	}

	if (strcmp(cmd, "read") == 0 || strcmp(cmd, "write") == 0)
		return do_spi_flash_read_write(argc - 1, argv + 1);
	if (strcmp(cmd, "erase") == 0)
		return do_spi_flash_erase(argc - 1, argv + 1);
	if (strcmp(cmd, "protect") == 0)
		return do_spi_flash_protect(argc -1,argv + 1);

usage:
	cmd_usage(cmdtp);
	return 1;
}

U_BOOT_CMD(
	sf,	6,	1,	do_spi_flash,
	"SPI flash sub-system",
	"probe [bus:]cs [hz] [mode]	- init flash device on given SPI bus\n"
	"				  and chip select\n"
	"sf read addr offset len 	- read `len' bytes starting at\n"
	"				  `offset' to memory at `addr'\n"
	"sf write addr offset len	- write `len' bytes from memory\n"
	"				  at `addr' to flash at `offset'\n"
	"sf erase offset len		- erase `len' bytes from `offset'\n"
	"sf protect len		        - protect `len' bytes from 0 to `offset'"
);

// --> liu_hongcong update 20110804
#ifdef CONFIG_DVR_CMD
static void sendtofrontboard(int cmd)
{
	//FIXME: not implement
	return ;
}

# define ERR_OK				0
# define ERR_TIMOUT			1
# define ERR_NOT_ERASED			2
# define ERR_PROTECTED			4
# define ERR_INVAL			8
# define ERR_ALIGN			16
# define ERR_UNKNOWN_FLASH_VENDOR	32
# define ERR_UNKNOWN_FLASH_TYPE		64
# define ERR_PROG_ERROR			128
# define ERR_OVERFLOW_PART			256

#define DATA_ALIGN_SZ 	0x10000
#define FLASH_BUF_SZ  	0x10000
#define VIRT_SECT_SIZE  FLASH_BUF_SZ
#define UBOOT_FLASH_REGION 	0xc0000
#define   	FPGA_FLASH_SIZE        0x0800000 

#define ALIGN(size, order) ( ((size-1) >> order) +1) << order)

#define SWAP32(x)        ((((x) & 0x000000ff) << 24) | (((x) & 0x0000ff00) << 8)|\
                          (((x) & 0x00ff0000) >>  8) | (((x) & 0xff000000) >> 24) )


#define CPU_nCONFIG_N		(0*32 + 2)  /*new board is reverse  husj@add 2012-2-7*/
#define CPU_nSTATUS_N		(0*32 + 3)

#define CPU_nCONFIG		(0*32 + 3) 
#define CPU_nSTATUS		(0*32 + 2)

#define CPU_CONF_DONE	(0*32 + 5)
#define FPGA_CTRLn		(3*32 + 4)

int check_gpio_status(int gpio,int status)
{
	int retry;
	int input_data;
	for (retry = 1000;retry;retry--){
		input_data = omap_get_gpio_datain(0*32 + 5);
		if ( input_data== status)
			return 0;
		udelay(1000);
	}
	printf("time out\n");
	return -1;
}



int fpga_upgade_start(void)
{
#if 1

	int retry = 10000;
	int nCONFIG, nSTATUS;
#if 0
	volatile unsigned int value;
	unsigned int offset[] = {0x20, 0x24, 0x2c, 0x34c};
	int i;

	for(i = 0; i < sizeof(offset)/sizeof(offset[0]); i++){
		value = __raw_readl(PINCNTLX_ADDR + offset[i]);
		printf("value = 0x%x\n", value);
		if((value & 0xff) != 0x80){
			value &= (~0xff);
			value |= 0x80;
			__raw_writel(value, PINCNTLX_ADDR + offset[i]);
		}
	}
#endif

	if(!CPU_SPEC_VAL){
		nCONFIG = CPU_nCONFIG;
		nSTATUS = CPU_nSTATUS;
	}else{
		nCONFIG = CPU_nCONFIG_N;
		nSTATUS = CPU_nSTATUS_N;
	}

	omap_set_gpio_dataout(nCONFIG,1);
	omap_set_gpio_dataout(FPGA_CTRLn,1);
	
	omap_set_gpio_direction(FPGA_CTRLn,0); //output

	omap_set_gpio_direction(nSTATUS,1); //input

	omap_set_gpio_direction(nCONFIG,0);//output
	omap_set_gpio_direction(CPU_CONF_DONE,1);  //input

	
	#if 0
	padctrl_base[9] = (padctrl_base[9] & (~0xff)) | 0x80;
	padctrl_base[10] = (padctrl_base[10] & (~0xff)) | 0x80;
	padctrl_base[12] = (padctrl_base[12] & (~0xff)) | 0x80;
	padctrl_base[212] = (padctrl_base[212] & (~0xff)) | 0x80;

	padctrl_base[85] = (padctrl_base[85] & (~0xff)) | 0x01;//spi1_clk
	padctrl_base[86] = (padctrl_base[86] & (~0xff)) | 0x01;//spi1_scs0
	padctrl_base[7] = (padctrl_base[7] & (~0xff)) | 0x02; //spi1_scs1
	padctrl_base[72] = (padctrl_base[72] & (~0xff)) | 0x10;//spi1_scs3
	#endif
	
	//udelay(5);
	omap_set_gpio_dataout(nCONFIG,0);
	udelay(5);
	omap_set_gpio_dataout(nCONFIG, 1);
	
	if (check_gpio_status(nSTATUS,0))
		return -1;
	return 0;

#else
	static int val = 0;
	unsigned int reg;

	if(val == 0){
		val = 1;
	}else{
		val = 0;
	}
	printf("val = %d   ", val);
	omap_set_gpio_dataout(CPU_nCONFIG, val);
	omap_set_gpio_dataout(FPGA_CTRLn, val);
	omap_set_gpio_dataout(CPU_nSTATUS, val);
	omap_set_gpio_dataout(CPU_CONF_DONE, val);

	reg = __raw_readl(PINCNTLX_ADDR + 0xc0);
	printf("pincntl = 0x%x\n", reg);
	if((reg & 0xff) != 0x80){
			reg &= (~0xff );
			reg |= 0x80;
			__raw_writel(reg, PINCNTLX_ADDR + 0xc0);
	}	
	omap_set_gpio_dataout(20, val);
	
#endif
}

int fpga_start(void)
{
	int nCONFIG, nSTATUS;
	if(!CPU_SPEC_VAL){
		nCONFIG = CPU_nCONFIG;
		nSTATUS = CPU_nSTATUS;
	}else{
		nCONFIG = CPU_nCONFIG_N;
		nSTATUS = CPU_nSTATUS_N;
	}


	omap_set_gpio_dataout(FPGA_CTRLn,0);//SPI_CTRLn
	omap_set_gpio_dataout(nCONFIG,1);//CPU_nCONFIG
	
	omap_set_gpio_direction(FPGA_CTRLn,0);
	omap_set_gpio_direction(nCONFIG,0);//output

	if (check_gpio_status(CPU_CONF_DONE,1))
		return -1;
	return 0;
}

void flash_perror (int err)
{
	switch (err) {
	case ERR_OK:
		break;
	case ERR_TIMOUT:
		puts ("Timeout writing to Flash\n");
		break;
	case ERR_NOT_ERASED:
		puts ("Flash not Erased\n");
		break;
	case ERR_PROTECTED:
		puts ("Can't write to protected Flash sectors\n");
		break;
	case ERR_INVAL:
		puts ("Outside available Flash\n");
		break;
	case ERR_ALIGN:
		puts ("Start and/or end address not on sector boundary\n");
		break;
	case ERR_UNKNOWN_FLASH_VENDOR:
		puts ("Unknown Vendor of Flash\n");
		break;
	case ERR_UNKNOWN_FLASH_TYPE:
		puts ("Unknown Type of Flash\n");
		break;
	case ERR_PROG_ERROR:
		puts ("General Flash Programming Error\n");
		break;
	case ERR_OVERFLOW_PART:
		puts("Overflow Flash Part,Check packshop\n");
		break;
	default:
		printf ("%s[%d] FIXME: rc=%d\n", __FILE__, __LINE__, err);
		break;
	}
}

static int __do_spi_probe(unsigned int  cs)
{
	unsigned int bus = 0;
	unsigned int speed = CONFIG_SF_DEFAULT_SPEED;
	unsigned int mode = CONFIG_SF_DEFAULT_MODE;
	struct spi_flash *new;

	if (flash && flash->spi->cs == cs)
		return 0;

	if (cs == 2){//fpga
		cs = 3;
		bus = 1;
		fpga_upgade_start();
	}
	new = spi_flash_probe(bus, cs, speed, mode);
	if (!new) {
		printf("Failed to initialize SPI flash at %u:%u\n", bus, cs);
		return 1;
	}

	if (flash)
		spi_flash_free(flash);
	flash = new;

	printf("SPI probe: %u KiB %s at %u:%u is now current device\n",
			flash->size >> 10, flash->name, bus, cs);

	return 0;
}


int dvr_spi_flash_read(void *buf, int offset, int len)
{
	int ret = -1;

	__do_spi_probe((offset &0x03000000)>>24);

	ret = spi_flash_read(flash, offset&0xffffff, len, buf);
	if (ret) {
		printf("%s(): error!\n", __func__);
		return ret;
	}

	return 0;
}

int dvr_spi_flash_write(void *buf, int offset, int len)
{
	int ret = -1;
	
	__do_spi_probe((offset &0x03000000)>>24);
	
	ret = spi_flash_erase(flash, offset&0xffffff, len);
	if (ret) {
		printf("%s(): erase error!\n", __func__);
		return ret;
	}
	
	ret = spi_flash_write(flash, offset&0xffffff, len, buf);
	if (ret) {
		printf("%s(): write error!\n", __func__);
		return ret;
	}

	return 0;
}
extern uchar updater_mode;
extern short updater_progress;

static int __do_write_flash(ulong img_addr, ulong flash_addr)
{
	image_header_t header;
	void *data_addr = NULL;
	int data_len = 0;
	uint data_end;
	static void *read_buf = NULL;
	void *src_buf = NULL;
	int offset = 0;
	int i, find_different=0;
	long int start_sect, data_sects;
	int ret = 0;
	
	memcpy(&header, (void *)img_addr, sizeof(image_header_t));
	data_addr = (void *)img_addr + sizeof(image_header_t);
	data_len = SWAP32((&header)->ih_size);
	data_end = SWAP32((&header)->ih_ep);

	if ((flash_addr + data_len) > data_end){
		flash_perror (ERR_OVERFLOW_PART);
		return -1;
	}
	
	if (flash_addr+data_len > (CONFIG_SYS_FLASH_BASE + CONFIG_SPI_FLASH_SIZE) || \
		(flash_addr < CONFIG_SPI_FLASH_SIZE  && flash_addr + data_len>CONFIG_SPI_FLASH_SIZE) || \
		((flash_addr &0xffffff) + data_len) > 16*1024*1024) {
		printf("data lenght error.\n");
		flash_perror (ERR_INVAL);
		return 1;
	}
	start_sect = (flash_addr - CONFIG_SYS_FLASH_BASE) / VIRT_SECT_SIZE;
	data_sects = ((data_len - 1) / VIRT_SECT_SIZE) + 1;
	

	/* if image is ok, now we write to flash */
	if (__do_spi_probe(flash_addr>>24)) return 1;

	read_buf = malloc(VIRT_SECT_SIZE);
	if (!read_buf){
		printf("malloc failed\n");
		return 1;
	}

	/* write to flash one sector by one sector */
	printf("\rwrite : 0%%");
	for(i=start_sect; i<(data_sects+start_sect); i++) {
		src_buf = data_addr + (i - start_sect) * VIRT_SECT_SIZE;
		offset = (i * VIRT_SECT_SIZE)&0x1ffffff; //useless!!!
		//printf("src_buf:0x%08x offset:0x%08x,cs:%d\n", src_buf, offset,flash->spi->cs);
		
		/*added for fpga flash--hujg@2012.3.14*/
		if ((flash_addr>>24) == 2 && offset == 0){
			/*erase first sector*/
			ret = spi_flash_erase(flash, offset, VIRT_SECT_SIZE);
			if (ret) 
				goto err_done;
			//printf("\terase first sector");
			continue;
		}
#if 1		
		if (find_different < 3) {
			ret = spi_flash_read(flash, offset, VIRT_SECT_SIZE, read_buf);
			if (ret) 
				goto err_done;
			if (memcmp(read_buf, src_buf, VIRT_SECT_SIZE) == 0) {
				goto SECT_DONE;
			}
			find_different++;
		}
#endif
		ret = spi_flash_erase(flash, offset, VIRT_SECT_SIZE);
		if (ret) 
			goto err_done;
		ret = spi_flash_write(flash, offset, VIRT_SECT_SIZE, src_buf);
		if (ret) 
			goto err_done;
SECT_DONE:
		UpdaterSendProgress(0xffff,(i - start_sect) * 100 / data_sects);
		printf("\rwrite : %ld%%", (i - start_sect) * 100 / data_sects);
	}

	/*added for fpga flash--hujg@2012.3.14*/
	if ((flash_addr>>24) == 2 && offset == (FPGA_FLASH_SIZE - VIRT_SECT_SIZE)){
		offset = 0;
		src_buf = data_addr;
		/*write first sector*/
		ret = spi_flash_write(flash, offset, VIRT_SECT_SIZE, src_buf);
		if (ret) 
			goto err_done;
		//printf("\twrite first sector");
	}
	
	//updater_progress = 100;
	UpdaterSendProgress(0xffff,100);
	printf("\rwrite : %ld%%\ndone\n", (i - start_sect) * 100 / data_sects);
	
#if 1
	free(read_buf);
	read_buf = NULL;
#endif

	/* set enviroment to default */
	if (start_sect < (UBOOT_FLASH_REGION/VIRT_SECT_SIZE)) {
		//setenv("restore", "1");  // set restore flag
		//return saveenv();
	}

	if ((flash_addr>>24) == 2){
		fpga_start();
		spi_flash_free(flash);
		flash = NULL;
	}
	return 0;

err_done:
	printf("%s: error!\n",__func__);
	free(read_buf);
	read_buf = NULL;
	return 1;
}

static int __do_check_image(ulong addr)
{
	void *hdr = (void *)addr;

	printf ("\n## Checking Image at %08lx ...\n", addr);

	switch (genimg_get_format (hdr)) {
	case IMAGE_FORMAT_LEGACY:
		puts ("   Legacy image found\n");
		if (!image_check_magic (hdr)) {
			puts ("   Bad Magic Number\n");
			return 1;
		}

		if (!image_check_hcrc (hdr)) {
			puts ("   Bad Header Checksum\n");
			return 1;
		}

		image_print_contents (hdr);

		puts ("   Verifying Checksum ... ");
		if (!image_check_dcrc (hdr)) {
			puts ("   Bad Data CRC\n");
			return 1;
		}
		puts ("OK\n");
		return 0;
#if defined(CONFIG_FIT)
	case IMAGE_FORMAT_FIT:
		puts ("   FIT image found\n");

		if (!fit_check_format (hdr)) {
			puts ("Bad FIT image format!\n");
			return 1;
		}

		fit_print_contents (hdr);

		if (!fit_all_image_check_hashes (hdr)) {
			puts ("Bad hash in FIT image!\n");
			return 1;
		}

		return 0;
#endif
	default:
		puts ("Unknown image format!\n");
		break;
	}

	return 1;

}


static int __do_check_addr(unsigned long addr)
{
	int rc = ERR_OK;

	if ((addr < CONFIG_SYS_FLASH_BASE) 
		|| (addr >= (CONFIG_SYS_FLASH_BASE+CONFIG_SPI_FLASH_SIZE)))
	{
		printf("Dest address:0x%lx invalid!\n", addr);
		rc = ERR_INVAL;
		goto err_out;
	}

	/* "run da"=CONFIG_SYS_FLASH_BASE, so here is right */
	if ((addr > CONFIG_SYS_FLASH_BASE) &&
		(addr < (CONFIG_SYS_FLASH_BASE + 256*1024)))
	{
		rc = ERR_PROTECTED;
		goto err_out;
	}

	/* align 128KB, for the old N5 */
	if (addr % DATA_ALIGN_SZ)
	{
		rc = ERR_ALIGN;
		goto err_out;
	}

	return rc;

err_out:
	flash_perror (rc);
	return rc;
}

int __do_flwrite (cmd_tbl_t * cmdtp, ulong img_addr, int argc, char *argv[])
{
	ulong flash_baseaddr; //Ҫдλ
	
	/* Flash writing address */
	if (argc == 1) {
		image_header_t hdr; //imgļheader
		/* copy image hader */
		memcpy (&hdr, (char *)img_addr, sizeof (image_header_t));
		flash_baseaddr = SWAP32((&hdr)->ih_load);
	} else {/* argc > 1 */
		flash_baseaddr = simple_strtoul (argv[1], NULL, 16);
	}
	
	/* check flash address */
	if (__do_check_addr(flash_baseaddr)) {
		return 1;
	}

	/* check image data */
	if (__do_check_image(img_addr)) {
		printf ("   Bad Image Info\n");
		return 1;
	}

	/* write data to flash */
	printf("Programing start at: 0x%08lx\n", flash_baseaddr);
	__do_write_flash(img_addr, flash_baseaddr);
	
	return 0;
} /* end __do_flwrite */

/* ļжȡһ */
static char *get_line(char **str)
{
    char *p, *end, *tmp;
    char *pStr;

    /* Ƿָ */
    if ((str == NULL) || (*str == NULL))
    {
        return NULL;
    }

    /* ַ */
    if (**str == 0)
    {
        return NULL;
    }
    
    p = pStr = *str;
    do
    {
        if ((*pStr == '\n') 
		     || ((*pStr == '\r') && (*(pStr + 1) == '\n')))
        {
            *pStr = 0;
            pStr++;
            break;
        }

        pStr++;
    } while (*pStr);
    *str = pStr;

    return p;
}

/* ִýű */
static int __do_script(ulong img_addr)
{
    char *img_data, *new_line, *buf;
    ulong len, img_size;

    /* 龵ļ */
    if (__do_check_image(img_addr)) 
    {
        printf ("   Bad Image Info\n");
        return 1;
    }

    /* ȡϢ */
    img_data = image_get_data ((image_header_t *)img_addr);   
    img_size = image_get_data_size ((image_header_t *)img_addr);

    buf  = malloc(img_size + 4);
    if (!buf)
    {
        printf ("Fail to malloc buffer for CmdScript!\n");
        return 1;
    }
    memcpy(buf, img_data, img_size);
    buf[img_size] = 0;
    img_data = buf;

    printf ("exce update config script start!\n");
    
    /* ִýű */
    new_line = get_line(&img_data);
    while (new_line)
    {
        if (new_line[0] == '#')
        {
            new_line = get_line(&img_data);
            continue;
        }
        
        if (strstr(new_line, SCRIPT_FILE_END))
        {
            break;
        }

        len = strlen(new_line);
        if ((len < MIN_SCRIPT_LEN) || (len >= MAX_SCRIPT_LEN))
        {
            new_line = get_line(&img_data);
            continue;
        }

        printf ("%s\n", new_line);
        # ifndef CONFIG_SYS_HUSH_PARSER
        run_command (new_line, 0);
        # else
        parse_string_outer(new_line, FLAG_PARSE_SEMICOLON |
                    FLAG_EXIT_FROM_LOOP);
        # endif

        new_line = get_line(&img_data);
    }

    free(buf);
    
    printf ("exce update config script complete!\n");
    return 0;
}

int do_flwrite (cmd_tbl_t * cmdtp, int flag, int argc, char *argv[])
{
	header_t *phdr = NULL;
	int total_len, head_len, img_len;
	ulong data;
	int ret;

	phdr = (header_t*)load_addr;

	/* special update file, ex: romfs-x.cramfs.img */
	ret = strncmp((char *)phdr->id, PACK_ID, strlen(PACK_ID));
	if (ret != 0)
	{
		ret = __do_flwrite(cmdtp, load_addr, argc, argv);
		if (ret) printf("flwrite error 1!\n");
		goto out;
	}

	total_len = phdr->total_len;
	head_len = phdr->head_len;
	img_len = 0;

	/* deal every image in update.img */
	for (data = load_addr + head_len;
	     data < load_addr + total_len;
	     data += img_len)
	{
		img_len = SWAP32(((image_header_t *)data)->ih_size) + sizeof(image_header_t);
		
		/* check Ctrl-C */
		ctrlc();
		if ((had_ctrlc())) {
			
			goto out;
		}

        if (strncmp(SCRIPT_IMG_NAME, image_get_name((image_header_t *)data), 
            sizeof(SCRIPT_IMG_NAME)) == 0)
        {
            if (__do_script(data))
            {
    			printf("flwrite fail to run config script\n");
    			ret = 1;
    			goto out;
            }
        } 
        else
		{
            if (__do_flwrite(cmdtp, data, argc, argv))
            {
    			printf("flwrite error 2!\n");
    			ret = 1;
    			goto out;
            }
		}
	}

out:

#ifdef UBOOT_UPDATE
	if(ret){
	        //ʧ
	        UpdaterSendProgress(UP_FAILD, 0);
	}else{
	        //ɹ
	        UpdaterSendProgress(UP_COMPLETE, UP_COMPLETE);
	 }
#endif
	return ret;
}


U_BOOT_CMD(
	flwrite,	2,	1,	do_flwrite,
	"flwrite - write data into FLASH memory\n",
	"DestAddr\n"
	"    - write data into DestAddr(0xA0000000~0xA4000000)\n"
);
#endif
// <-- liu_hongcong update

