#include <common.h>
#include <i2c.h>



typedef enum i2c_chip_e
{
	MUX_I2C = 0,
	ALARM_I2C = 1,
}i2c_chip_e;

#define GPIO_DIR_IN		1
#define GPIO_DIR_OUT	0
#define GPIO_OUT_LOW       0
#define GPIO_OUT_HIGH      1
#define HW_REG(reg) 			*((volatile unsigned int *)(reg))
#define DELAY(us)       		udelay(us * 2)
 
#define NAME_MAX_LEN	32
#define MAX_PIN 96

extern int amb_gpio_direction_input(int gpio);
extern int amb_gpio_direction_output(int gpio, int value);
extern int amb_gpio_get_value(int gpio);
extern int amb_gpio_set_value(int gpio, int value);
extern int amb_gpio_set(int gpio);

//static unsigned int I2cBaseAddress; //current port address
static unsigned int SCL = 0; 			// current scl pin
static unsigned int SDA = 1; 			// current sda pin

typedef struct Sti2cDevice_s
{
	unsigned char dev_addr;
	unsigned char offset;
	unsigned char value;
	unsigned char reserve;
	unsigned int  cmd;
} Sti2cDevice_t;

struct i2c_resource
{
	unsigned int port; 		// port number
	unsigned long port_addr; // port address
	unsigned int scl_pin;
	unsigned long scl_addr;
	unsigned int sda_pin;
	unsigned long sda_addr;
	char name[NAME_MAX_LEN];
};

/* >>>>>>>>>>>>>>>>>>>>>>>>>> i2c HAL <<<<<<<<<<<<<<<<<<<<<<<<<<< */
static void i2c_set_direct(u8 whichline, u8 direct, u8 val)
{
	if(direct == GPIO_DIR_IN){
		amb_gpio_direction_input(whichline);
	}else{ 
		amb_gpio_direction_output(whichline, val);
	}
}

static void i2c_clr(unsigned char whichline)
{
	i2c_set_direct(whichline, GPIO_DIR_OUT, GPIO_OUT_LOW); 
}

static void  i2c_set(unsigned char whichline)
{
	i2c_set_direct(whichline, GPIO_DIR_OUT, GPIO_OUT_HIGH);
}

static unsigned char i2c_data_read(void)
{ 
	return (amb_gpio_get_value(SDA));

}
/* >>>>>>>>>>>>>>>>>>>>>>>>>> i2c HAL <<<<<<<<<<<<<<<<<<<<<<<<<<< */


/* >>>>>>>>>>>>>>>>>>>>>>>>>> i2c timing <<<<<<<<<<<<<<<<<<<<<<<<<<< */
static void i2c_start_bit(void)
{
	i2c_set(SDA);
	i2c_set(SCL);
	DELAY(5);
	i2c_clr(SDA);
	DELAY(3);
	i2c_clr(SCL);
}

static void i2c_stop_bit(void)
{
	i2c_clr(SCL);
	DELAY(2);
	i2c_clr(SDA);
	DELAY(1);
	i2c_set(SCL);
	DELAY(1);
	i2c_set(SDA);
	DELAY(1);
}

static void i2c_send_byte(unsigned char c)
{
	int i;
	//local_irq_disable();
	for (i=0; i<8; i++)
	{
		i2c_clr(SCL);

		if (c & (1<<(7-i)))
			i2c_set(SDA);
		else
			i2c_clr(SDA);

		DELAY(1);

		i2c_set(SCL);
		DELAY(1);
		i2c_clr(SCL);
	}
	//local_irq_enable();
}

static unsigned char i2c_receive_byte(void)
{
	int j=0;
	int i;
	//local_irq_disable();
	i2c_set_direct(SDA, GPIO_DIR_IN, 0);
	for (i=0; i < 8; i++)
	{
		i2c_clr(SCL);
		DELAY(2);
		i2c_set(SCL);
		DELAY(1);
		//i2c_clr(SCL);
		if (i2c_data_read())
		{
			j+=(1<<(7-i));
		}
		DELAY(1);
	}
	//local_irq_enable();
	DELAY(1);

	return j;
}

static int i2c_receive_ack(void)
{
	int nack;
	int timeout = 255;
	i2c_clr(SCL);

	i2c_set_direct(SDA, GPIO_DIR_IN, 0);

	DELAY(4);
	i2c_set(SCL);
	DELAY(1);

	nack = i2c_data_read(); //read ack

	while(nack && timeout)
	{
		timeout--;
		DELAY(1);
		nack = i2c_data_read();
	}

	DELAY(1);
	i2c_clr(SCL); /* liu_hongcong */

	if (timeout == 0)
	{
	    i2c_stop_bit();
	        //i2c_set(SCL); /* liu_hongcong */
	        //i2c_set(SDA); /* liu_hongcong */
		return -1;
	}

	DELAY(1);

	if (nack == 0)
		return 1;

	return 0;
}

/*
 * sends an acknowledge over I2C rountine.
 */
//static
void i2c_send_ack(void)
{
	i2c_clr(SDA);
	i2c_clr(SCL);
	DELAY(1);
	i2c_set(SCL);
	DELAY(1);
	i2c_clr(SCL);
	DELAY(1);
}


static void i2c_send_noack(void)
{
	DELAY(1);
	i2c_clr(SCL);
	DELAY(1);
	i2c_set(SDA);
	DELAY(1);
	i2c_set(SCL);
	DELAY(1);
	i2c_clr(SCL);
	DELAY(1);
	i2c_clr(SDA);
	DELAY(1);
}
/* >>>>>>>>>>>>>>>>>>>>>>>>>> i2c timing <<<<<<<<<<<<<<<<<<<<<<<<<<< */

/* >>>>>>>>>>>>>>>>>>>>>>>>>> inner API <<<<<<<<<<<<<<<<<<<<<<<<<<< */
static int __i2c_write(unsigned char devaddress,
		unsigned char address, unsigned char data)
{

	i2c_start_bit();
	DELAY(2);

	i2c_send_byte((unsigned char)(devaddress));

	if (i2c_receive_ack() < 0) {
		return -1;
	}

	i2c_send_byte(address);
	if (i2c_receive_ack() < 0) {
		return -1;
	}

	i2c_send_byte(data);
	if (i2c_receive_ack() < 0) {
		return -1;
	}

	i2c_stop_bit();

	return 0;
}

static int __i2c_read(unsigned char devaddress, unsigned char address)
{
	int rxdata;

	i2c_start_bit();
	DELAY(2);

	i2c_send_byte((unsigned char)(devaddress));
	if (i2c_receive_ack() < 0) {
		return -1;
	}

	i2c_send_byte(address);
	if (i2c_receive_ack() < 0)
	{
		return -1;
	}
	{
#if 0 //liu_hongcong
		/* optional stop bit */
		i2c_clr(SDA);
		i2c_stop_bit();
#endif

		/* if no stop bit, set SDA */
		i2c_set(SDA);
		DELAY(5);

		i2c_start_bit();
		DELAY(2);
	}

	i2c_send_byte(devaddress | 0x01);

	if (i2c_receive_ack() < 0) {
            return -1;
	}
	rxdata = i2c_receive_byte();
	i2c_send_noack();
	i2c_stop_bit();

	return rxdata;
}
/* >>>>>>>>>>>>>>>>>>>>>>>>>> inner API <<<<<<<<<<<<<<<<<<<<<<<<<<< */

/* >>>>>>>>>>>>>>>>>>>>>>>>>> export API <<<<<<<<<<<<<<<<<<<<<<<<<<< */
int i2c_read(uchar addr, uint off, int alen, uchar *data, int len)
//int i2c_read(u8 addr, u8 off, u8 *data, u32 len, u32 cmd)
{
    int i; 
    int ret = 0;
    int val;

    //i2c_init(0,0);
    if (!data){
        return -1;
    }

    /* init global i2c pin */
    //array_no = cmd;
    //SCL = GET_PIO_INDEX(i2c_info[array_no].port, i2c_info[array_no].scl_pin);
    //SDA = GET_PIO_INDEX(i2c_info[array_no].port, i2c_info[array_no].sda_pin);

    /* read data from i2c bus */
    addr = addr<<1;
    for (i = 0; i < len; i++) {
        if ((val = __i2c_read(addr, off)) < 0) {
        ret = -1;  
        goto out;
    }
        *(data + i) = (u8)(val & 0xff);
        off++;
    }

    out:
    return ret;
}

int i2c_write(uchar addr, uint off, int alen, uchar *data, int len)
//int i2c_write(u8 addr, u8 off, u8 *data, u32 len, u32 cmd)
{
	int i;
	int ret = 0;
	u8  val;

	if (!data) return -1;

	/* init global i2c pin */
	//array_no = cmd;
	//SCL = GET_PIO_INDEX(i2c_info[array_no].port, i2c_info[array_no].scl_pin);
	//SDA = GET_PIO_INDEX(i2c_info[array_no].port, i2c_info[array_no].sda_pin);

	/* write data to i2c bus */
	addr = addr<<1;
	for (i = 0; i < len; i++) {
		val = *(data+i);
		if ((ret = __i2c_write(addr, off, val)) < 0) {
			ret = -1;
			goto out;
		}
		off++;
	}

out:
	return ret;
}

int i2c_set_bus_num (unsigned int bus)
{ 
	return 0;
}

unsigned int i2c_get_bus_num (void)
{
	return 0;
}


void i2c_init (int speed, int slaveadd)
{
    amb_gpio_direction_output(SDA, 1);
    amb_gpio_direction_output(SCL, 1);
    
    amb_gpio_set(SDA);
    amb_gpio_set(SCL);
    return;
}
 
int i2c_probe(uchar chip)
{
    char buf[1];
    i2c_init(0,0);
    if (i2c_read(chip,0x0,1,buf,1) == 0){
       return 0;
    } 
    return -1;
}
