/**
 * system/src/bld/usb_drv.c
 *
 * History:
 *    2005/09/07 - [Arthur Yang] created file
 *
 *    Notes: The data pointer for USB DMA TX/RX should be aligned to 8-bytes
 *
 * Copyright (C) 2004-2005, Ambarella, Inc.
 *
 * All rights reserved. No Part of this file may be reproduced, stored
 * in a retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Ambarella, Inc.
 */

/* ========================================================================== */
/*                              ͷļ                                      */
/* ========================================================================== */
#include <common.h>
#include <command.h>
#include <net.h>
#include <miiphy.h>
#include <malloc.h>
#include <net.h>
#include <netdev.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <config.h>
#include <asm/arch/ambhw/chip.h>
#include <asm/arch/basedef.h>
#include <asm/arch/ambhw/eth.h>
#include <asm/arch/netdev/tftp.h>
#include <asm/arch/bldnet.h>
#include <asm/arch/amba_usb.h>
#include <asm/arch/hwusbreg.h>
#include <asm/arch/hwusbcfg.h>
#include "bld_usb_descriptors.c"
#include <asm/arch/ambhw/ambhw.h>
#include <asm/arch/fio/firmfl.h>
#include <asm/arch/fio/firmfl_api.h>
#include <asm/arch/mem_map.h>
#include <image.h>



/* ========================================================================== */
/*                           Ͷ                                   */
/* ========================================================================== */

#define _VERSION_	"Version 0.1"
#define QUEUE_DEPTH		10
#define DEFAULT_PLL_MODE	7
#define DESCRIPTOR_BASE(x) 	((u32)descriptor_buf[x])
#define BUFFER_BASE(x)		((u32)data_buf[x])
#define RETRY_COUNT		2
#define FAILED			1
#define SUCCESSED		0
#define TASK_TIMEOUT		10000	/* 3 second */
#define TEST_NUM		100
#define FLAG_LAST_TRANS		0x01
#define FLAG_ADDRESS		0x02
#define FLAG_JUMP_EXEC		0x04
#define FLAG_COMMAND		0x08
#define FLAG_FORCE_FINISH	0x10
#define PRINT_SIZE          262144
#define PACKET_SIZE         512
#define PRINT_LINE_SIZE     60

#define msleep(ms) (udelay(ms*1000))

/* ========================================================================== */
/*                          ݽṹ                                    */
/* ========================================================================== */

/*usbշbufṹ*/
typedef struct _buf_info 
{
	u32 addr;
	u32 length;
	u32 crc_value;
} buf_info_t;

/*usbݽṹ*/
typedef struct _transfer_info 
{
	volatile u32	flag;
	volatile u32	task_finished;
	u32		jump_address;
	u32		total_length;
	buf_info_t 	tx;
	buf_info_t 	rx;
} transfer_info_t;


/* ========================================================================== */
/*                          ȫֱ                                    */
/* ========================================================================== */

static struct USBDEV usbd;
static struct USB_EP_UDC_REG *udc_reg[EP_NUM];
transfer_info_t transfer;
static u32 gRxState = COMMAND_PHASE;
static int changed_to_ext_pll = 0;
static int enable_usb = 0;	 

extern u8 *uds_bld_descriptors[];
extern u8 uds_bld_configuration1_hs[];
extern u8 uds_bld_configuration1_fs[];

u32 setup_desc[64] __attribute__ ((aligned(32), section(".bss.noinit")));
u32 descriptor_buf[8][64] __attribute__ ((aligned(32), section(".bss.noinit")));
u8 data_buf[8][1024] __attribute__ ((aligned(32), section(".bss.noinit")));
DEV_RSP dev_response __attribute__ ((aligned(32), section(".bss.noinit")));

volatile COMMAND_QUEUE	command[QUEUE_DEPTH]
	__attribute__ ((section(".bss.noinit")));;
volatile WAIT_QUEUE	response[QUEUE_DEPTH]
	__attribute__ ((section(".bss.noinit")));;
volatile QUEUE_INDEX	qindex={0};
static int gUsbStartSend = 0;
static int gUsbDelay = 0;
static long  gLoadAddr = 0;


/* ========================================================================== */ 
/*                                                                  */ 
/* ========================================================================== */

static void USB_disconnect(void);


/* ========================================================================== */
/*                                                                  */
/* ========================================================================== */ 


/******************************************************************************* 
*  	: USB_startRx
*    	: Ϊ˿ڽ׼ʹDMAͶ˿ж
*    	: ep_num: Ҫݶ˿
*         : buf_ptr: ڽݵַ
*    	:  ޡ
*  ֵ	:  
*******************************************************************************/
static void USB_startRx(u32 ep_num, u32 *buf_ptr)
{
    volatile struct USB_DATA_DESC 	*rx = NULL;
    volatile struct USB_DEV_REG 	*dev_reg = NULL;
    volatile struct USB_EP_OUT_REG 	*out_reg = NULL;
    volatile u32 value=0;

    vic_disable(USB_INT_VEC);  

    out_reg = (struct USB_EP_OUT_REG *)usbd.ep[ep_num];
    rx = out_reg->descriptor_p;

    rx = (struct USB_DATA_DESC *)usbd.ep_descriptor[ep_num];
    rx->data_ptr = buf_ptr;
    rx->status = USB_DMA_BUF_HOST_RDY | USB_DMA_LAST;

    vic_enable(USB_INT_VEC);  

    dev_reg = usbd.dev_reg;
    value = dev_reg->ep_intrmask;

    if(ep_num == CTRL_OUT)
    {
        value &= ~(USB_DEV_MSK_EP0_OUT);
    }
    else 
    {
        value &= ~(USB_DEV_MSK_EP1_OUT);
    }
    
    dev_reg->ep_intrmask = value;

    dev_reg->control |= USB_DEV_RCV_DMA_EN;

    for(value=0;value < 0x100;value++);

    out_reg->control |= USB_EP_CLR_NAK;
}


/******************************************************************************* 
*  	: USB_initUdc
*    	: öжϺ󣬳ʼusb device controller
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_initUdc(void)
{
    volatile u32 value;

    value = USB_UDC_EP0_NUM 
        | USB_UDC_OUT 
        | USB_UDC_CTRL 
        | USB_UDC_CFG_NUM
        | USB_UDC_INTF_NUM 
        | USB_UDC_ALT_SET 
        | USB_UDC_CTRLOUT_MAX_PKT_SZ;
    udc_reg[CTRL_OUT]->value = value;

    value = USB_UDC_EP0_NUM 
        | USB_UDC_IN 
        | USB_UDC_CTRL
        | USB_UDC_CFG_NUM
        | USB_UDC_INTF_NUM 
        | USB_UDC_ALT_SET
        | USB_UDC_CTRLIN_MAX_PKT_SZ;
    udc_reg[CTRL_IN]->value = value;

    value = USB_UDC_EP1_NUM
        | USB_UDC_OUT 
        | USB_UDC_BULK 
        | USB_UDC_CFG_NUM
        | USB_UDC_INTF_NUM 
        | USB_UDC_ALT_SET
        | USB_UDC_BULKOUT_MAX_PKT_SZ_HI;
    udc_reg[BULK_OUT]->value = value;

    value = USB_UDC_EP1_NUM 
        | USB_UDC_IN
        | USB_UDC_BULK 
        | USB_UDC_CFG_NUM 
        | USB_UDC_INTF_NUM 
        | USB_UDC_ALT_SET
        | USB_UDC_BULKIN_MAX_PKT_SZ_HI;
    udc_reg[BULK_IN]->value = value;

    usbd.dev_reg->config |= USB_DEV_CSR_PRG_EN;

}


/******************************************************************************* 
*  	: USB_deviceInterrupt
*    	: usb豸жϣҪöٺж
*    	: int_value:豸жϱ־λڲѯж
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_deviceInterrupt(u32 int_value)
{
    volatile struct USB_EP_OUT_REG 	*out_reg = NULL;
    volatile struct USB_EP_IN_REG  	*in_reg = NULL;
    volatile struct USB_DEV_REG	*dev_reg = NULL;
    u32 				value = 0;
    u32				serviced_int = 0;

    dev_reg = usbd.dev_reg;

    /* case 1. enumeration complete */
    if(int_value & USB_DEV_ENUM_CMPL)
    {
        USB_initUdc();

        serviced_int = USB_DEV_ENUM_CMPL;
        value = dev_reg->status & USB_DEV_ENUM_SPD;

        usbd.speed = _SPEED_(value);

        if(usbd.speed == USB_HS) 
        {  
            /* high speed */
            /* set BULK_IN size register */
            in_reg = (struct USB_EP_IN_REG *)usbd.ep[BULK_IN];
            value = udc_reg[BULK_IN]->value;
            value &= (~USB_UDC_MAX_PKT_SZ);
            value |= USB_UDC_BULKIN_MAX_PKT_SZ_HI;
            udc_reg[BULK_IN]->value = value;
            in_reg->max_pkt_size = USB_EP_BULKIN_MAX_PKT_SZ_HI;

            /* set BULK_OUT size register */
            out_reg = (struct USB_EP_OUT_REG *)usbd.ep[BULK_OUT];
            value = udc_reg[BULK_OUT]->value;
            value &= (~USB_UDC_MAX_PKT_SZ);
            value |= USB_UDC_BULKIN_MAX_PKT_SZ_HI;
            udc_reg[BULK_OUT]->value = value;
            out_reg->max_pkt_size = USB_EP_BULKOUT_MAX_PKT_SZ_HI;

        }
        else if (usbd.speed == USB_FS) 
        { 
            /* full speed */
            /* set BULK_IN size register */
            in_reg = (struct USB_EP_IN_REG *)usbd.ep[BULK_IN];
            value = udc_reg[BULK_IN]->value;
            value &= (~USB_UDC_MAX_PKT_SZ);
            value |= USB_UDC_BULKIN_MAX_PKT_SZ_FU;
            udc_reg[BULK_IN]->value = value;
            in_reg->max_pkt_size = USB_EP_BULKIN_MAX_PKT_SZ_FU;

            /* set BULK_IN size register */
            out_reg = (struct USB_EP_OUT_REG *)usbd.ep[BULK_OUT];
            value = udc_reg[BULK_OUT]->value;
            value &= (~USB_UDC_MAX_PKT_SZ);
            value |= USB_UDC_BULKIN_MAX_PKT_SZ_FU;
            udc_reg[BULK_OUT]->value = value;
            out_reg->max_pkt_size = USB_EP_BULKOUT_MAX_PKT_SZ_FU;

        } 
        else
        {
            printf("Error: USB device do not support other speed.\n");
        }
    } /* ENUM COMPLETE */

    /* case 2. Get set_config or set_interface request from host */
    if (int_value & (USB_DEV_SET_CFG | USB_DEV_SET_INTF)) 
    {

        if(int_value & USB_DEV_SET_CFG)
        {
            serviced_int |= USB_DEV_SET_CFG;
        }
        
        if(int_value & USB_DEV_SET_INTF)
        {
            serviced_int |= USB_DEV_SET_INTF;
        }

        /* unlock endpoint stall status */
        in_reg = (struct USB_EP_IN_REG *)usbd.ep[CTRL_IN];
        in_reg->control &= (~USB_EP_STALL);

        /* told UDC the configuration is done, and to ack HOST */
        dev_reg->control |= USB_DEV_CSR_DONE ;

        /* The device is full configured, data IN/OUT can be started */
        USB_startRx(BULK_OUT, (u32 *)usbd.ep_buffer[BULK_OUT]);

    }

    /* acknowledge all serviced interrupt */
    if(serviced_int != int_value) 
    {
        /* some interrupts are not serviced, why ? */
        printf("serviced_int= %x",serviced_int);
        dev_reg->intr = serviced_int;
    }
}

/******************************************************************************* 
*     : USB_getDescriptor
*       : hostĻȡ
*       : 
*       : ޡ
*  ֵ   : 
*******************************************************************************/
static void USB_getDescriptor(void)
{
    USB_DEVICE_REQUEST *req;
    u16	type;
    u16 index;
    u32	tx_size = 0;
    u8	*descriptor_p = NULL;

    req = (USB_DEVICE_REQUEST *)&(usbd.setup[0]);
    type = (req->wValue & 0xff00) >> 8;
    index = (req->wValue & 0xff);

    switch(type) 
    {
        case USB_DEVICE_DESCRIPTOR:
        {
            descriptor_p = (u8 *)uds_bld_descriptors[DEVICE_DESC];
            tx_size = sizeof(USB_DEVICE_DESC);
            break;
        }
        case USB_CONFIGURATION_DESCRIPTOR:
        {
            if(usbd.speed == USB_HS) 
            {
                descriptor_p = (u8 *)uds_bld_descriptors[CONFIG_HS];
                if(req->wLength == 0xff)
                {
                    tx_size = sizeof(uds_bld_configuration1_hs);
                }
                else
                {
                    tx_size = req->wLength;
                }
            }
            else 
            {
                descriptor_p = (u8 *)uds_bld_descriptors[CONFIG_FS];
                if(req->wLength == 0xff)
                {
                    tx_size = sizeof(uds_bld_configuration1_fs);
                }
                else
                {
                    tx_size = req->wLength;
                }
            }
            break;
        }
        case USB_STRING_DESCRIPTOR:
        {
            if(index == 0)
            {
                descriptor_p = (u8 *)uds_bld_descriptors[STRING_0];
            }
            else if(index == 1)
            {
                descriptor_p = (u8 *)uds_bld_descriptors[STRING_1];
            }
            else if(index == 2)
            {
                descriptor_p = (u8 *)uds_bld_descriptors[BLD_STRING2];
            }
            else if(index == 3)
            {
                descriptor_p = (u8 *)uds_bld_descriptors[STRING_3];
            }
            else
            {
                printf("Unknown string index.\n");
            }

            if (descriptor_p){
                tx_size = descriptor_p[0];
            }
            break;
        }
        case USB_DEVICE_QUALIFIER:
        {
            descriptor_p = (u8 *)uds_bld_descriptors[DEVICE_QUA];
            tx_size = sizeof(USB_DEVICE_QUALIFIER_DESC);
            break;
        }
        case USB_OTHER_SPEED_CONFIGURATION:
        {
            printf("get other speed configuration.\n");
            break;
        }
        
        default:

            break;
    }

    qindex.cmd_end ++;
    qindex.cmd_end %= QUEUE_DEPTH; /* wrap around */

    command[qindex.cmd_end].TX_RX = TX_DIR;
    command[qindex.cmd_end].pendding = 1;
    command[qindex.cmd_end].pipe = CTRL_IN;
    command[qindex.cmd_end].need_size = tx_size;
    command[qindex.cmd_end].done_size = 0;
    command[qindex.cmd_end].pBuf = descriptor_p;
    command[qindex.cmd_end].lock = 0;
}


 /******************************************************************************* 
*  	: USB_initEndpoint
*    	: ö˿setupݽṹ
*    	: ep_num: ˿ں
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_initSetupDesc(int ep_num)
{
    struct USB_EP_OUT_REG *out_reg = NULL;
    struct USB_SETUP_PKT *setup = NULL;

    out_reg = (struct USB_EP_OUT_REG *)usbd.ep[ep_num];

    out_reg->setup_buffer_p = (void *)setup_desc;

    setup = (struct USB_SETUP_PKT *)(out_reg->setup_buffer_p);

    setup->status 	= USB_DMA_BUF_HOST_RDY;
    setup->reserved = 0xffffffff;
    setup->data0	= 0xffffffff;
    setup->data1	= 0xffffffff;
}

 
/******************************************************************************* 
*  	: USB_initDataDesc
*    	: ö˿ݰṹ
*    	: ep_num:usb˿ں
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_initDataDesc(u32 ep_num)
{
    struct USB_DATA_DESC *desc_p = NULL;
    struct USB_EP_IN_REG *in_reg = NULL;
    struct USB_EP_OUT_REG *out_reg = NULL;


    desc_p = (struct USB_DATA_DESC *)usbd.ep_descriptor[ep_num];

    desc_p->status = (USB_DMA_BUF_DMA_DONE | USB_DMA_LAST);
    desc_p->reserved = 0xffffffff;
    desc_p->data_ptr = (u32 *)(usbd.ep_buffer[ep_num]);
    desc_p->next_desc_ptr = (u32 *)desc_p;

    if((ep_num == BULK_IN) || (ep_num == CTRL_IN)) 
    {
        in_reg = (struct USB_EP_IN_REG *)usbd.ep[ep_num];
        in_reg->descriptor_p = desc_p;
    } 
    else 
    {
        out_reg = (struct USB_EP_OUT_REG *)usbd.ep[ep_num];
        out_reg->descriptor_p = desc_p;
    }
}


/******************************************************************************* 
*  	: USB_decodeRequest
*    	: usb hostͨcontrol out ˿ڷ
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_decodeRequest(void)
{
    u8 bmRequestType;
    u8 bRequest;
    u8 request;
    USB_DEVICE_REQUEST *req;

    req = (USB_DEVICE_REQUEST *)&(usbd.setup[0]);

    bmRequestType = req->bmRequestType;
    bRequest = req->bRequest;

    request = bmRequestType & USB_DEV_REQ_TYPE_TYPE;

    switch(request) 
    {
        case USB_DEV_REQ_TYPE_STANDARD: 	/* Standard request */
        {
            switch(bRequest) 
            {
                case USB_GET_DESCRIPTOR:   	/* get descriptor    */
                {
                    USB_getDescriptor();
                    break;
                }
                default:
                {
                    printf("unknown request=%x ",bRequest);
                    break;
                }
            } /* bRequest */

            break;
        }
        case USB_CLASS:
        {
            /* not implmented */
            break;
        }
        case USB_VENDOR:
        {
            /* not implmented */
            break;
        }
        
        default:
        {
            break;
        }
    }/* switch(request) */

    /* reinitialize setup packet */
    USB_initSetupDesc(CTRL_OUT);
}


/******************************************************************************* 
*  	: USB_controlOutHandler
*    	: control out ˿ڷ͹ݰ,жDMAǷɹ
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_controlOutHandler(void)
{
    volatile struct USB_EP_OUT_REG *out_reg;
    volatile struct USB_DATA_DESC *desc;
    volatile u32	dma_status;
    volatile u32	ep_status;
    volatile u32	rcv_size;

    out_reg = (struct USB_EP_OUT_REG *)usbd.ep[CTRL_OUT];
    ep_status = out_reg->status;

    desc = out_reg->descriptor_p;

    /* check if the descriptor has fresh data */
    dma_status = desc->status;

    if((dma_status & USB_DMA_BUF_STS) == USB_DMA_BUF_DMA_DONE) 
    {

        if((dma_status & USB_DMA_RXTX_STS) == USB_DMA_RXTX_SUCC) 
        {
            /* process error case */
            if(ep_status & USB_EP_BUF_NOT_AVAIL)
            {
                out_reg->status = USB_EP_BUF_NOT_AVAIL;
            }
            
            if(ep_status & USB_EP_HOST_ERR)
            {
                out_reg->status = USB_EP_HOST_ERR;
            }
            rcv_size = desc->status & USB_DMA_RXTX_BYTES;
        } 
        else
        { 
            printf("Error: fail to receive data.\n");
        }
    }
}


/******************************************************************************* 
*  	: USB_endpointInInterrupt
*    	: ˿ж
*    	: int_value:˿жϱ־λڲѯ˿
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_endpointOutInterrupt(u32 int_value)
{
    volatile struct USB_DEV_REG 	*dev_reg = NULL;
    volatile struct USB_EP_OUT_REG 	*out_reg = NULL;
    volatile struct USB_SETUP_PKT 	*setup = NULL;
    u32 int_status = 0;
    u32 ep_status = 0;
    u32 rx_length = 0;

    dev_reg = usbd.dev_reg;

    if(int_value & USB_DEV_EP1_OUT) 
    { 

        /*** BULK_OUT ***/
        dev_reg->ep_intr = USB_DEV_EP1_OUT;

        out_reg = usbd.ep[BULK_OUT];

        int_status = out_reg->descriptor_p->status;

        /* received data */
        if((int_status & USB_DMA_BUF_STS) == USB_DMA_BUF_DMA_DONE) 
        {

            rx_length = (out_reg->descriptor_p->status &
            USB_DMA_RXTX_BYTES);

            if (rx_length == 0)
            {
                printf("rx_length is 0\n");
            }
            
            /* disable OUT endpoint */
            dev_reg->ep_intrmask |= USB_DEV_EP1_OUT;

            /* set NAK until next rx is ready */
            out_reg->control |= USB_EP_SET_NAK;

            /* queue reuest for background processing */
            qindex.cmd_end ++;
            qindex.cmd_end %= QUEUE_DEPTH; /* wrap around */

            command[qindex.cmd_end].TX_RX = RX_DIR;
            command[qindex.cmd_end].pendding = 1;
            command[qindex.cmd_end].pipe = BULK_OUT;
            command[qindex.cmd_end].need_size = rx_length;
            command[qindex.cmd_end].done_size = rx_length;
            command[qindex.cmd_end].pBuf =
                (u8 *)out_reg->descriptor_p->data_ptr;
            command[qindex.cmd_end].lock = 0;

        }

        ep_status = out_reg->status;

        /* check if any endpoint error status */
        /* case 1. buffer is not available */
        if(ep_status & USB_EP_BUF_NOT_AVAIL)
        {
            out_reg->status = USB_EP_BUF_NOT_AVAIL;
        }
        
        /* case 2. host error */
        if(ep_status & USB_EP_HOST_ERR) 
        {
            out_reg->status = USB_EP_HOST_ERR;
        }

        /* Receive Clear Stall interrupt */
        if (ep_status & USB_EP_RCV_CLR_STALL) 
        {
            out_reg->status = USB_EP_RCV_CLR_STALL;
        }

        /* acknowledge endpoint status bit */
        if(out_reg->status & USB_EP_OUT_PKT) 
        {
            out_reg->status = USB_EP_OUT_PKT;
        }

    } 
	else 
    { 				

        /*** CTRL_OUT ***/
        dev_reg->ep_intr = USB_DEV_EP0_OUT;

        out_reg = usbd.ep[CTRL_OUT];
        /* check the status bits for what kind of packets in */
        int_status = out_reg->status;

        if((int_status & USB_EP_OUT_PKT_MSK) == USB_EP_OUT_PKT) 
        {
            /*data packet*/
            out_reg->status = USB_EP_OUT_PKT;

            rx_length = (out_reg->descriptor_p->status &
            USB_DMA_RXTX_BYTES);

            if(rx_length == 0) 
            {
                dev_reg->control |= USB_DEV_RCV_DMA_EN;
                USB_initSetupDesc(CTRL_OUT);

            } /* rx_length==0 */

            USB_controlOutHandler();
            return;

        } 
        else if ((int_status & USB_EP_OUT_PKT_MSK) 
         == USB_EP_SETUP_PKT) 
        {

            /* SETUP packet, always 8 bytes */
            /* ack setup packet status */
            out_reg->status = USB_EP_SETUP_PKT;

            /* read out setup packet */
            setup = (struct USB_SETUP_PKT *)
            out_reg->setup_buffer_p;
            usbd.setup[0] = setup->data0;
            usbd.setup[1] = setup->data1;
            USB_decodeRequest();
        }
        else 
        { 
            printf("Error: no this kind of data.\n");
        }
    }
	return;
}


/******************************************************************************* 
*  	: USB_bulkInHandler
*    	: bulk inжϺص
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_bulkInHandler(void)
{
	/* not implmented */
}


/******************************************************************************* 
*  	: USB_endpointInInterrupt
*    	: ˿ж
*    	: int_value:˿жϱ־λڲѯ˿
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_endpointInInterrupt(u32 int_value)
{
    volatile struct USB_DEV_REG 	*dev_reg = NULL;
    volatile struct USB_EP_IN_REG 	*in_reg = NULL;
    struct USB_DATA_DESC *desc = NULL;
    volatile u32 int_status = 0;
    volatile u32 value = 0;

    dev_reg = usbd.dev_reg;

    if(int_value & USB_DEV_EP1_IN) 
    { 

        /* BULK_IN */
        dev_reg->ep_intr = USB_DEV_EP1_IN;

        in_reg = (struct USB_EP_IN_REG *)usbd.ep[BULK_IN];
        int_status = in_reg->status;

        if(int_status & USB_EP_TRN_DMA_CMPL) 
        {

            /* if a previous TX DMA complete interrupt happens
                disable interrupt before next TX DMA is prepared */
            dev_reg->ep_intrmask |= USB_DEV_EP1_IN;

            /* acknowledge the status bit */
            in_reg->status = USB_EP_TRN_DMA_CMPL;

            /* FIXME: this is a delay for PHY to finish sending and
                prevent from stopping transmitting */
            for(value=0;value<0x8ff;value++);

            /* if this is a issued send data command
                then update the information in the command structure	*/
            if(command[qindex.cmd_start].lock) 
            {

                /* update tx data size */
                desc = (struct USB_DATA_DESC *)in_reg->descriptor_p;

                command[qindex.cmd_start].done_size +=
                    (desc->status & USB_DMA_RXTX_BYTES);
                command[qindex.cmd_start].pBuf +=
                    (desc->status & USB_DMA_RXTX_BYTES);

                /* if there is pending response, clear it */
                if(qindex.rsp_index)
                {
                    qindex.rsp_index --;
                }
                if(qindex.rsp_index)
                {
                    printf("failed in ISR\n");
                }
            }

        }
        
        if (int_status & USB_EP_IN_PKT) 
        {

            /* if a IN token is received */
            USB_bulkInHandler();

            /* NAK the request before next IN packet is ready */
            in_reg->control |= USB_EP_SET_NAK;
            /* acknowledge the status bit */
            in_reg->status = USB_EP_IN_PKT;
        }
        
        if (int_status & USB_EP_BUF_NOT_AVAIL) 
        {
            in_reg->status = USB_EP_BUF_NOT_AVAIL;
            printf("BNA\n");
        }
        
        if (int_status & USB_EP_HOST_ERR) 
        {
            in_reg->status = USB_EP_HOST_ERR;
            printf("HE\n");
        }
        
        if (int_status & USB_EP_RCV_CLR_STALL) 
        {
            in_reg->status = USB_EP_RCV_CLR_STALL;
        }

    } 
    else 
    {  			

        /* CTRL_IN */
        in_reg = (struct USB_EP_IN_REG *)usbd.ep[CTRL_IN];
        int_status = in_reg->status;

        /* ack the device interrupt */
        /* disable interrupt before next TX DMA is prepared */
        if(int_status & USB_EP_TRN_DMA_CMPL) 
        {
            /* disable EP0 IN interrupt */
            dev_reg->ep_intrmask |= USB_DEV_EP0_IN;

            /* acknowledge the DMA complete status */
            while(in_reg->status & USB_EP_TRN_DMA_CMPL)
            {
                in_reg->status = USB_EP_TRN_DMA_CMPL;
            }
            /* update tx data size */
            desc = (struct USB_DATA_DESC *)in_reg->descriptor_p;

            command[qindex.cmd_start].done_size +=
                (desc->status & USB_DMA_RXTX_BYTES);

            /* if there is pending response, clear it */
            if(qindex.rsp_index)
            {
                qindex.rsp_index --;
            }
            
            if(qindex.rsp_index)
            {
                printf("response index failed\n");
            }
        } 
        else if (int_status & USB_EP_IN_PKT) 
        {
            /* acknowledge the status bit */
            in_reg->status = USB_EP_IN_PKT;
        }
        else if (int_status & USB_EP_BUF_NOT_AVAIL)
        {
            in_reg->status = USB_EP_BUF_NOT_AVAIL;
        } 
        else if (int_status & USB_EP_HOST_ERR) 
        {
            in_reg->status = USB_EP_HOST_ERR;
        } 
        else
        {  
            /* other interrupt reasons */
            in_reg->status = int_status;
        }
        dev_reg->ep_intr = USB_DEV_EP0_IN;
    }
}


/******************************************************************************* 
*  	: USB_interrupt
*    	: usb豸Ͷ˿ж
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_interrupt(void)
{
    volatile u32	value = 0;

    /* this could be ethernet interrupt on A2, just return */
    if(!enable_usb)
    {
        return;
    }

    /* disable USB interrupt in VIC */
    vic_disable(USB_INT_VEC);

    /* 1. check if device interrupt */
    value = usbd.dev_reg->intr;
    if(value) 
    {
        /* ack the interrupt */
        usbd.dev_reg->intr = value;

        USB_deviceInterrupt(value);
    }

    /* 2. check if endpoint interrupt */
    value = usbd.dev_reg->ep_intr;
    if(value) 
    {
        if(value & USB_DEV_EP_OUT) 
        {
            _clean_flush_d_cache();
            USB_endpointOutInterrupt(value);
        }
        else if(value & USB_DEV_EP_IN) 
        {
            _clean_flush_d_cache();
            USB_endpointInInterrupt(value);
        } 
        else 
        { 
            /* something wrong in endpoint interrupt */
            printf("Error: no this kind of interrupt.\n");
        }
    }

    /* re-enable USB interrupt */
    vic_enable(USB_INT_VEC);

}


/******************************************************************************* 
*  	: USB_initEndpoint
*    	: ø˿
*    	: ep_num: ˿ں
*         : speed: ˿ٶ
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_initEndpoint(int ep_num, int speed)
{
    volatile struct USB_EP_IN_REG *in_reg = NULL;
    volatile struct USB_EP_OUT_REG *out_reg = NULL;

    if(ep_num == CTRL_OUT) 	
    {
        /*** initialize CTROL_OUT ***/
        out_reg = (struct USB_EP_OUT_REG *)usbd.ep[CTRL_OUT];
        out_reg->control = USB_EP_TYPE_CTRL;
        out_reg->max_pkt_size = USB_EP_CTRLOUT_MAX_PKT_SZ;
        udc_reg[CTRL_OUT] = 
            (struct USB_EP_UDC_REG *)USB_UDC_REG(CTRL_OUT);
        USB_initSetupDesc(CTRL_OUT);
        USB_initDataDesc(CTRL_OUT);

    } 
    else if(ep_num == CTRL_IN) 
    {	
    
        /*** initialize CTROL_IN ***/
        in_reg = (struct USB_EP_IN_REG *)usbd.ep[CTRL_IN];
        in_reg->control = USB_EP_TYPE_CTRL;
        in_reg->buf_size = USB_TXFIFO_DEPTH_CTRLIN;
        in_reg->max_pkt_size = USB_EP_CTRLIN_MAX_PKT_SZ;
        udc_reg[CTRL_IN] =
            (struct USB_EP_UDC_REG *)USB_UDC_REG(CTRL_IN);
        USB_initDataDesc(CTRL_IN);
    } 
    else if(ep_num == BULK_OUT) 
    {	

        /*** initialize BULK_OUT ***/
        out_reg = (struct USB_EP_OUT_REG *)usbd.ep[BULK_OUT];
        out_reg->control = USB_EP_TYPE_BULK | USB_EP_SET_NAK;
        if(speed == USB_HS)
        {
            out_reg->max_pkt_size = USB_EP_BULKOUT_MAX_PKT_SZ_HI;
        }
        else	/* either FS or LS */
        {
            out_reg->max_pkt_size = USB_EP_BULKOUT_MAX_PKT_SZ_FU;
        }

        udc_reg[BULK_OUT] =
            (struct USB_EP_UDC_REG *)USB_UDC_REG(BULK_OUT);
            
        USB_initDataDesc(BULK_OUT);

    } 
    else 
    {			/*** initialize BULK_IN ***/
        in_reg = (struct USB_EP_IN_REG *)usbd.ep[BULK_IN];
        in_reg->control = USB_EP_TYPE_BULK | USB_EP_SET_NAK;
        in_reg->buf_size = USB_TXFIFO_DEPTH_BULKIN;
        if(speed == USB_HS)
        {
            in_reg->max_pkt_size = USB_EP_BULKIN_MAX_PKT_SZ_HI;
        }
        else
        {
            in_reg->max_pkt_size = USB_EP_BULKIN_MAX_PKT_SZ_FU;
        }

        udc_reg[BULK_IN] =
            (struct USB_EP_UDC_REG *)USB_UDC_REG(BULK_IN);
        USB_initDataDesc(BULK_IN);
    }
}


/******************************************************************************* 
*  	: USB_configEndpoint
*    	: usb˿ڷݳ
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_configEndpoint(void)
{
    /* config control in endpoint */
    usbd.ep_info[CTRL_IN].id.physical = CTRL_IN;
    usbd.ep_info[CTRL_IN].id.logical = 0 | 0x80;
    usbd.ep_info[CTRL_IN].transfer_type = 0; 
    usbd.ep_info[CTRL_IN].maxpkt_size = MAX_CTRL_PKT_SIZE64;
    usbd.ep_info[CTRL_IN].fifo_size = MAX_CTRL_PKT_SIZE64;

    /* config control out endpoint */
    usbd.ep_info[CTRL_OUT].id.physical = CTRL_OUT;
    usbd.ep_info[CTRL_OUT].id.logical = 0;
    usbd.ep_info[CTRL_OUT].transfer_type = 0; 
    usbd.ep_info[CTRL_OUT].maxpkt_size = MAX_CTRL_PKT_SIZE64;
    usbd.ep_info[CTRL_OUT].fifo_size = MAX_CTRL_PKT_SIZE64;

    /* config bulk in endpoint */
    usbd.ep_info[BULK_IN].id.physical = BULK_IN;
    usbd.ep_info[BULK_IN].id.logical =	BULK_IN | 0x80;
    usbd.ep_info[BULK_IN].transfer_type = 0; 
    usbd.ep_info[BULK_IN].maxpkt_size = MAX_BULK_PKT_SIZE512;
    usbd.ep_info[BULK_IN].fifo_size = 0;

    /* config bulk out endpoint */
    usbd.ep_info[BULK_OUT].id.physical = BULK_OUT;
    usbd.ep_info[BULK_OUT].id.logical = BULK_OUT;
    usbd.ep_info[BULK_OUT].transfer_type = 0; 
    usbd.ep_info[BULK_OUT].maxpkt_size = MAX_BULK_PKT_SIZE512;
    usbd.ep_info[BULK_OUT].fifo_size = 0;
}


/******************************************************************************* 
*  	: USB_initHardware
*    	: usbӲĴʼ
*    	: 
*    	: ޡ
*  ֵ	: 0:ʾɹ
*******************************************************************************/
static u32 USB_initHardware(void)
{
    volatile u32 value = 0;

    rct_usb_reset();

    /* initialize each endpoint */
    USB_initEndpoint(CTRL_IN,USB_HS);
    USB_initEndpoint(CTRL_OUT,USB_HS);
    USB_initEndpoint(BULK_IN,USB_HS);
    USB_initEndpoint(BULK_OUT,USB_HS);

    /* device config register */
#if	defined (__FPGA__)
    value = USB_DEV_SPD_HI 
        | USB_DEV_SELF_POWER 
        | USB_DEV_PHY_16BIT 
        | USB_DEV_UTMI_DIR_BI 
        | USB_DEV_HALT_ACK
        | USB_DEV_SET_DESC_STALL;
#else
    value = USB_DEV_SPD_HI 
        | USB_DEV_SELF_POWER 
        | USB_DEV_PHY_8BIT 
        | USB_DEV_UTMI_DIR_BI 
        | USB_DEV_HALT_ACK 
        | USB_DEV_SET_DESC_STALL;
#endif

    usbd.dev_reg->config = value;

    /* device control register */
    value = USB_DEV_RCV_DMA_EN 
        | USB_DEV_TRN_DMA_EN 
        | USB_DEV_DESC_UPD_PYL 
        | USB_DEV_LITTLE_ENDN 
        | USB_DEV_DMA_MD 
        | USB_DEV_BURST_EN 
        | USB_DEV_BURST_LEN;

    usbd.dev_reg->control = value;

    /* device interrupt mask register */
    /* ack previous interrupt, but necessary ? */
    usbd.dev_reg->intr = 0x0000007f;
    usbd.dev_reg->ep_intr = 0xffffffff;
    
    /* from a clean status */
    usbd.dev_reg->intrmask = 0x0000007f;
    usbd.dev_reg->ep_intrmask = 0xffffffff;

    value = ~(USB_DEV_MSK_SET_CFG 
        | USB_DEV_MSK_SET_INTF 
        | USB_DEV_MSK_SPD_ENUM_CMPL);
    usbd.dev_reg->intrmask = value;

    value = ~(USB_DEV_MSK_EP0_IN 
        | USB_DEV_MSK_EP0_OUT 
        | USB_DEV_MSK_EP1_IN 
        | USB_DEV_MSK_EP1_OUT);
    usbd.dev_reg->ep_intrmask = value;

    return 0;
}


/******************************************************************************* 
*  	: USB_initDataStruct
*    	: usbݽṹʼ
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_initDataStruct(void)
{
    usbd.base 	 = USBDC_BASE;
    usbd.dev_reg 	 = (struct USB_DEV_REG *)USB_DEV_BASE;
    usbd.intr_num 	 = USB_IRQ;

    /*
    note: the start index of EP IN and EP OUT register
    should be 0 since both base address are different. */
    usbd.ep[CTRL_IN] = (void *)USB_EP_IN_CTRL_REG(CTRL_IN);
    usbd.ep[BULK_IN] = (void *)USB_EP_IN_CTRL_REG(BULK_IN);
    usbd.ep[CTRL_OUT] = (void *)USB_EP_OUT_CTRL_REG(CTRL_IN);
    usbd.ep[BULK_OUT] = (void *)USB_EP_OUT_CTRL_REG(BULK_IN);

    usbd.tx_fifo_size = TX_FIFO_SIZE;
    usbd.rx_fifo_size = RX_FIFO_SIZE;
    usbd.control_pkt_size = MAX_CTRL_PKT_SIZE64;
    usbd.bulk_pkt_size = MAX_BULK_PKT_SIZE512;

    /*descriptor will be located at */
    usbd.ep_descriptor[CTRL_IN] = DESCRIPTOR_BASE(CTRL_IN);
    usbd.ep_descriptor[CTRL_OUT]= DESCRIPTOR_BASE(CTRL_OUT);
    usbd.ep_descriptor[BULK_IN] = DESCRIPTOR_BASE(BULK_IN);
    usbd.ep_descriptor[BULK_OUT]= DESCRIPTOR_BASE(BULK_OUT);

    /* data buffer will be located at*/
    usbd.ep_buffer[CTRL_IN]	= BUFFER_BASE(CTRL_IN);
    usbd.ep_buffer[CTRL_OUT]= BUFFER_BASE(CTRL_OUT);
    usbd.ep_buffer[BULK_IN]	= BUFFER_BASE(BULK_IN);
    usbd.ep_buffer[BULK_OUT]= BUFFER_BASE(BULK_OUT);

    usbd.ctrl_tx_cnt = 0;
    usbd.ctrl_rx_cnt = 0;
    usbd.bulk_tx_cnt = 0;
    usbd.bulk_rx_cnt = 0;
    usbd.error_cnt	 = 0;
    usbd.link	 = USB_DISCONNECT;
    usbd.speed	 = USB_HS;	// default High-Speed
    usbd.mode	 = MODE_DMA;	// default DMA mode
    usbd.address	 = 0;

    /* init command/response queue */
    memset ((u8 *)command, 0,sizeof(COMMAND_QUEUE) * QUEUE_DEPTH);
    memset ((u8 *)response, 0,sizeof(WAIT_QUEUE) * QUEUE_DEPTH);
    memset ((u8 *)&qindex, 0,sizeof(QUEUE_INDEX));
    memset((u8 *)&transfer,0, sizeof(transfer_info_t));
    memset((u8 *)command,0,sizeof(COMMAND_QUEUE)*QUEUE_DEPTH);
	
}


/******************************************************************************* 
*     : USB_delay
*       : usbʱ
*       : ms:ʱʱ,ԺΪλ
*       : ޡ
*  ֵ   : 
*******************************************************************************/
static void USB_delay(int ms)
{
    udelay(ms*1000);
}

/******************************************************************************* 
*     : USB_checkConnected
*       : ȡĴжusbǷѾӵhost
*       : 
*       : ޡ
*  ֵ   : 1ʾ
*           : 0ʾû
*******************************************************************************/
static int USB_checkConnected(void)
{
	/* Setup VBUS interrupt */
	vic_set_type(USBVBUS_INT_VEC, VIRQ_LEVEL_HIGH);

	/* Read VBUS status */
    if (readl(VIC_RAW_STA_REG) & 1)
    {
        return 1;
    }
    else
    {
        return 0;
    }
}


/******************************************************************************* 
*  	: USB_SoftDisc
*    	:  ͨʽϿusb
*    	: 
*    	: 
*  ֵ	:  
*******************************************************************************/
static void USB_SoftDisc(void)
{
    /* soft disconnect defaultly */
    if (USB_checkConnected()) 
    {
        /* note don't change the order */
        *(volatile u32 *)(UDC_CTRL_REG) = UDC_CTRL_SD;	// Soft disconnect
        *(volatile u32 *)(UDC_CFG_REG) = UDC_CFG_RWKP;	// Enable remote wakeup
    }
}


/******************************************************************************* 
*     : USB_pllInit
*       : usbʱ
*       : 
*       : ޡ
*  ֵ   : 
*******************************************************************************/
static void USB_pllInit(void)
{
#if	defined (__FPGA__)
    /* do nothing */
#else
    rct_set_usb_debounce();

    USB_delay(1);

#if	(CHIP_REV == A1)

    if(!changed_to_ext_pll)
    {
        rct_set_usb_int_clk();
    }
    else
    {
        rct_set_usb_ext_clk();
    }

#else
    if(!changed_to_ext_pll)
    {
        rct_enable_usb();
    }
    else
    {
        rct_set_usb_ext_clk();
    }
#endif
    USB_delay(1);

    USB_SoftDisc(); /* check if USB cable is on now */
#endif
}


/******************************************************************************* 
*  	: USB_init
*    	: usbݽṹĴ˿õȳʼ
*    	: 
*    	: ޡ
*  ֵ	: 
*******************************************************************************/
int USB_init (void)
{
    int connect = 0;

    /* initialize software data structure */
    USB_initDataStruct();

    /* initialize hardware */
    USB_initHardware();

    /* configure endpoint register and data structure */
    USB_configEndpoint();

    /* setup VIC interrupt and install ISR */
    vic_set_type(USB_INT_VEC, VIRQ_LEVEL_HIGH);

    /* enable USB interrupt */
    vic_enable(USB_INT_VEC);

    /* re-enable USB connection by SW */
    connect = USB_checkConnected();
    if(connect)
    {
        usbd.dev_reg->control |= USB_DEV_REMOTE_WAKEUP; // resume
    }

    return connect;
}


/******************************************************************************* 
*  	: USB_startTx
*    	: ׼ʼ
*    	: ep_num: Ҫݶ˿
*         : buf_ptr: ڴŷݵַ
*         : pkt_size:Ҫݰ
*    	:  ޡ
*  ֵ	: 0:ʾûҪ;1:ʾҪ
*******************************************************************************/
static u32 USB_startTx(u32 ep_num, u32 *buf_ptr, u32 *pkt_size)
{
    volatile struct USB_DATA_DESC *tx = NULL;
    volatile struct USB_DEV_REG *dev_reg = NULL;
    volatile struct USB_EP_IN_REG *in_reg = NULL;
    volatile u32 value = 0;
    volatile u32 send_size = 0;

    in_reg = (struct USB_EP_IN_REG *)usbd.ep[ep_num];
    dev_reg = (struct USB_DEV_REG *)usbd.dev_reg;

    /* sending NULL packet, *buf_ptr should be NULL	*/
    if(*pkt_size == 0) 
    {
        buf_ptr = (u32 *)usbd.ep_buffer[ep_num];
        send_size = 0;
    } 
    else 
    {
        send_size = (*pkt_size > in_reg->max_pkt_size)?
        in_reg->max_pkt_size:*pkt_size;
    }
    
    tx = (struct USB_DATA_DESC *)usbd.ep_descriptor[ep_num];
    tx->data_ptr = buf_ptr; /* assign new packet buffer pointer */
    tx->status = USB_DMA_BUF_HOST_RDY | USB_DMA_LAST | send_size;

    vic_disable(USB_INT_VEC);

    _clean_flush_d_cache();

    in_reg->control |= (USB_EP_CLR_NAK);

    /*
    * Wait IN token before issue DMA polling
    * NOTE: Please add your own timeout here.
    * Or it will wait forever ....
    */
    while(!(in_reg->status & USB_EP_IN_PKT));

    in_reg->status &= USB_EP_IN_PKT;

    value = dev_reg->ep_intrmask;
    if(ep_num == CTRL_IN)
    {
        value &= ~(USB_DEV_MSK_EP0_IN);
    }
    else /* if(ep_num == BULK_IN) */
    {
        value &= ~(USB_DEV_MSK_EP1_IN);
    }

    dev_reg->ep_intrmask = value;

    /* issue polling bit */
    in_reg->control |= (USB_EP_POLL_DEMAND);

    for(value=0;value<0xffff;value++);

    /* update data buffer and size in current command element*/
    buf_ptr += send_size;
    *pkt_size -= send_size;

    vic_enable(USB_INT_VEC);

    if((send_size == 0) || (send_size < in_reg->max_pkt_size))
    {
        /* the last packet */
        if(*pkt_size)
        {
            printf("remaind size is not 0.\n");
        }
        
        return 0;
    } 
    else 
    {
        /* still packet to send */
        return 1;
    }
}


/******************************************************************************* 
*  	: USB_sendResponse
*    	: װظhostݰ
*    	: rsp: ڱʾ֮ǰ״̬
*         : r0: ظ
*         : r1: ظ
*    	:  ޡ
*  ֵ	:  
*******************************************************************************/
static void USB_sendResponse (u32 rsp, u32 r0, u32 r1)
{
	u32 rsp_length = sizeof(DEV_RSP);

	dev_response.signature = STATUS_TOKEN;
	dev_response.response = rsp;
	dev_response.parameter0 = r0;
	dev_response.parameter1 = r1;

	USB_startTx(BULK_IN, (u32 *)&dev_response, &rsp_length);
}


/******************************************************************************* 
*  	: USB_processRdyRecvCmd
*    	: hostյUSB_CMD_RDY_TO_RCV
*    	: host_cmd: hostյcmdṹ
*    	:  ޡ
*  ֵ	:  
*******************************************************************************/
static void USB_processRdyRecvCmd (HOST_CMD *host_cmd)
{
    u32 r0 = 0;

    if (transfer.flag & FLAG_JUMP_EXEC) 
    {
        transfer.jump_address = host_cmd->parameter2;
        if (transfer.jump_address < DRAM_START_ADDR 
            || transfer.jump_address > DRAM_START_ADDR + DRAM_SIZE) 
        {
            printf("Invalid jump address: 0x%x", transfer.jump_address);
            printf("Revert to default address: 0x%x", MEMFWPROG_RAM_START);
            transfer.jump_address = MEMFWPROG_RAM_START;
        }
    }
    if (transfer.flag & FLAG_COMMAND) 
    {
        switch (host_cmd->parameter1) 
        {
            case USB_CMD_SET_MEMORY_READ:
            {
                r0 = readl(host_cmd->parameter2);
                break;
            }
            case USB_CMD_SET_MEMORY_WRITE:
            {
                writel (host_cmd->parameter3, host_cmd->parameter2);
                break;
            }
            case USB_CMD_SET_FW_PTB:
            {
                printf("Not support this command.\n");
                break;
            }
        }
    }
    
    if (transfer.flag & FLAG_FORCE_FINISH)
    {
        transfer.task_finished = 1;
    }
    
    /* reponse */
    USB_sendResponse(USB_RSP_SUCCESS, r0, 0);
}


/******************************************************************************* 
*  	: USB_recvData
*    	: bulk out˿ڽ
*    	: host_cmd: hostյcmdṹ
*    	:  ޡ
*  ֵ	: ؽݵַ
*******************************************************************************/
static u32 * USB_recvData (HOST_CMD *host_cmd)
{
	transfer.rx.crc_value = host_cmd->parameter0;
	
	gRxState = RCV_DATA_PHASE;

	return (u32 *)transfer.rx.addr;
}


/******************************************************************************* 
*  	: USB_processSendCmd
*    	: USB_CMD_RDY_TO_SND
*    	: host_cmd: hostյcmdṹ
*    	:  ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_processSendCmd (HOST_CMD *host_cmd)
{
    buf_info_t *tx = &transfer.tx;
    u32 rsp;

    /* Init response */
    rsp = USB_RSP_SUCCESS;
    tx->addr = 0;
    tx->length = 0;

    /* check application finished */
    transfer.flag = host_cmd->parameter0;

    switch (host_cmd->parameter1) 
    {
        case USB_CMD_GET_MEMORY:
        {
            /* Save transfer information */
            tx->addr = host_cmd->parameter2;
            tx->length = host_cmd->parameter3;
            break;
        }
        case USB_CMD_GET_FW_BST:
        case USB_CMD_GET_FW_BLD:
        case USB_CMD_GET_FW_PRI:
        case USB_CMD_GET_FW_RMD:
        case USB_CMD_GET_FW_ROM:
        case USB_CMD_GET_FW_DSP:
        case USB_CMD_GET_FW_PTB:

            break;

        default:
        {
            printf("Unknown Get CMD=%x\n", host_cmd->parameter1);
        }
    }

    /* Calculate CRC */
    if (tx->length)
    {
        tx->crc_value = crc32(0,(void *)tx->addr, tx->length);
    }
    
    /* reponse with length and CRC */
    USB_sendResponse(rsp, tx->length, tx->crc_value);

    _clean_flush_d_cache();
}


 /******************************************************************************* 
*  	: USB_processSendDataCmd
*    	: USB_CMD_SND_DATA
*    	: host_cmd: hostյcmdṹ
*    	:  ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_processSendDataCmd (HOST_CMD *host_cmd)
{
	printf ("Start to send data ");
	printf("%d",transfer.tx.length);
	printf(" bytes...\r\n");

	/* Add a new command to command queue */
	qindex.cmd_end ++;
	qindex.cmd_end %= QUEUE_DEPTH;

	command[qindex.cmd_end].TX_RX = TX_DIR;
	command[qindex.cmd_end].pendding = 1;
	command[qindex.cmd_end].pipe = BULK_IN;
	command[qindex.cmd_end].need_size = transfer.tx.length;
	command[qindex.cmd_end].done_size = 0;
	command[qindex.cmd_end].pBuf = (u8 *)transfer.tx.addr;
	command[qindex.cmd_end].lock = 1;
}


/******************************************************************************* 
*  	: USB_processInquiryCmd
*    	: USB_CMD_INQUIRY_STATUS,ڲѯ
*    	: host_cmd: hostյcmdṹ
*    	:  ޡ
*  ֵ	: 
*******************************************************************************/
static void USB_processInquiryCmd (HOST_CMD *host_cmd)
{
    if (host_cmd->parameter0 == USB_BLD_INQUIRY_CHIP)
    {
        USB_sendResponse(USB_RSP_SUCCESS, CHIP_REV, 0);
    } 
    else
    {
        /* reponse */
        USB_sendResponse(USB_RSP_IN_BLD, 0, 0);
    }
}


/******************************************************************************* 
*     : USB_processBulkOut
*       : bulk out˿¼,ݴ׶
*       : 豸൱ǰcmd
*       :  ޡ
*  ֵ   :  
*******************************************************************************/
static void USB_processBulkOut(COMMAND_QUEUE *cur_cmd)
{
    HOST_CMD *host_cmd = (HOST_CMD *)cur_cmd->pBuf;
    struct USB_EP_OUT_REG *out_reg  = NULL;
    volatile u32 *next_rx_ptr       = NULL;
    static u32 count                = 0;
    static u32 totol_size           = 0;
    header_t *pHdr                  = NULL;
    char * s                        = NULL;
    long pUsbDelay                  = 0;
    ulong headData                  = 0;
    int imgType                     = 0;
    int headLen                     = 0;
    int ret;
    
    s = getenv ("bootdelay");

    out_reg = (struct USB_EP_OUT_REG *)usbd.ep[BULK_OUT];

    /* Set next_rx_ptr to EP buffer for receive next command */
    next_rx_ptr = (u32 *)usbd.ep_buffer[BULK_OUT];

    switch (gRxState) 
    {
        case COMMAND_PHASE:
        {
            gUsbStartSend = 1;
            /* Check host command */
            if (host_cmd->signature != COMMAND_TOKEN) 
            {
                printf("Wrong signature:%x\n", host_cmd->signature);
                break;
            }

            /* Process command */
            switch (host_cmd->command) 
            {
                case USB_CMD_RDY_TO_RCV:
                {
                    USB_processRdyRecvCmd (host_cmd);
                    break;
                }
                
                case USB_CMD_RCV_DATA:
                {
                    next_rx_ptr = USB_recvData(host_cmd);
                    break;
                }
                
                case USB_CMD_RDY_TO_SND:
                {
                    USB_processSendCmd (host_cmd);
                    break;
                }
                
                case USB_CMD_SND_DATA:
                {
                    USB_processSendDataCmd (host_cmd);
                    break;
                }
                
                case USB_CMD_INQUIRY_STATUS:
                {
                    USB_processInquiryCmd (host_cmd);
                    break;
                }
                
                default:
                {
                    printf ("Unknown command:%x\n", host_cmd->command);
                }
            }

            break;
        }
        
        case SND_DATA_PHASE:
        {
            break;
        }
        
        case RCV_DATA_PHASE:
        {
            /* done_size = need_size = received packet size */
            if (cur_cmd->done_size < out_reg->max_pkt_size) 
            {
                u32 cal_crc_value;

                /* Short packet received */
                transfer.rx.length = (u32)cur_cmd->pBuf +
                cur_cmd->done_size -
                transfer.rx.addr;
                transfer.total_length += transfer.rx.length;

                /* Calculate CRC */
                _clean_flush_d_cache();
                cal_crc_value = crc32(0,(void *) transfer.rx.addr,
                transfer.rx.length);
                
                /* send response according to CRC check */
                if(cal_crc_value != transfer.rx.crc_value)
                {
                    USB_sendResponse(USB_RSP_FAILED, 0, 0);
                }
                else
                {
                    USB_sendResponse (USB_RSP_SUCCESS, 0, 0);
                }

                transfer.task_finished= 1;
                gRxState = COMMAND_PHASE;
                if(totol_size < PRINT_SIZE)
                {
                    printf("\n#\n");
                }
                
                printf("\r\n Success transferred:%d bytes.\n",
                       transfer.total_length);
                       
                pHdr        =  (header_t*)gLoadAddr;
                headLen     =  pHdr->head_len;
                headData    =  gLoadAddr + headLen;
                
                ret = strncmp((char *)pHdr->id, PACK_ID, strlen(PACK_ID));
                if (0 == ret)
                {
                    imgType = genimg_get_format((void *) headData);
                    if((imgType == IMAGE_FORMAT_LEGACY)
                       || (imgType == IMAGE_FORMAT_FIT))
                    {
                        run_command("flwrite",0);
                    }
                }
                
                imgType = genimg_get_format((void *) gLoadAddr);
                if((imgType == IMAGE_FORMAT_LEGACY)
                    || (imgType == IMAGE_FORMAT_FIT))
                {
                    run_command("flwrite",0);
                }

                gUsbStartSend = 0;
                pUsbDelay = s ? (int)simple_strtol(s, NULL, 10) : CONFIG_BOOTDELAY;
                gUsbDelay = get_timer(0)+ pUsbDelay*CONFIG_SYS_HZ;
                count = 0;
            } 
            else 
            {
                totol_size += PACKET_SIZE;
                if(totol_size % PRINT_SIZE == 0)
                {
                    count++;
                    printf("#");
                    if(count % PRINT_LINE_SIZE == 0)
                    {
                        printf("\r\n");
                        count = 0;
                    }
                }
                /* advanced rx buffer pointer */
                /* done_size is "byte", but address is "word" */
                next_rx_ptr = (u32 *)cur_cmd->pBuf;
                next_rx_ptr += (cur_cmd->done_size >> 2);
            }
            
            break;
        }
        
        default:
        {
            printf("Unknown gRxState:%x\n", gRxState);
        }
    } /* switch */

    cur_cmd->pendding = 0;
    
    USB_startRx(BULK_OUT, (u32 *)next_rx_ptr);
}


/******************************************************************************* 
*  	: USB_processBulkIn
*    	: bulk in˿¼
*    	: 豸൱ǰcmd
*    	:  ޡ
*  ֵ	:  
*******************************************************************************/
static void USB_processBulkIn(COMMAND_QUEUE *cur_cmd)
{
    /* being here only as in SND_DATA state */
    if(cur_cmd->need_size == 0) 
    {

        /* finish sending, then just ACK */
        transfer.total_length += transfer.tx.length;
        qindex.rsp_index ++;

        USB_sendResponse (USB_RSP_SUCCESS, 0, 0);

        printf("Send completed!\n");

        cur_cmd->pendding = 0;
        transfer.task_finished = 1;

    } 
    else 
    {	
        /* prepare wait response data */
        /* wait for TX DMA finish */
        qindex.rsp_index ++;
        USB_startTx(BULK_IN, (u32*)cur_cmd->pBuf, &(cur_cmd->need_size));
    }
}

/******************************************************************************* 
*  	: USB_processCtrlIn
*    	: control in˿¼
*    	: 豸൱ǰcmd
*    	:  ޡ
*  ֵ	:  
*******************************************************************************/
static void USB_processCtrlIn (COMMAND_QUEUE *cur_cmd)
{
    struct USB_EP_IN_REG *in_reg = (struct USB_EP_IN_REG *)         \
                    usbd.ep[CTRL_IN];

    if(cur_cmd->need_size < in_reg->max_pkt_size)
    {
        /* the last packet */
        cur_cmd->pendding = 0;
    }

    /* prepare wait response data */
    /* wait for TX DMA finish */
    if(qindex.rsp_index) 
    {
        printf("wrong rsp_index :%x\n", qindex.rsp_index);
        /* reset response index if this is an error one */
        qindex.rsp_index = 0;
    }

    qindex.rsp_index ++;
    USB_startTx(CTRL_IN, (u32 *)cur_cmd->pBuf, &(cur_cmd->need_size));

    /* prepare the status stage */
    USB_startRx(CTRL_OUT, (u32 *)usbd.ep_buffer[CTRL_OUT]);
}


/******************************************************************************* 
*     : USB_isrProcess
*       : usb豸Ͷ˿ж
*       : 
*       : ޡ
*  ֵ   :  
*******************************************************************************/
static void USB_isrProcess(void)
{
    volatile struct USB_DEV_REG	*dev_reg = NULL;
    volatile u32 intr = 0;
    volatile u32 ep_intr = 0;

    dev_reg = usbd.dev_reg;

    intr = dev_reg->intr;
    ep_intr = dev_reg->ep_intr;

    intr &= (USB_DEV_MSK_SET_CFG 
        |  USB_DEV_MSK_SET_INTF 
        |  USB_DEV_MSK_SPD_ENUM_CMPL);

    ep_intr &= (USB_DEV_MSK_EP0_IN 
        | USB_DEV_MSK_EP0_OUT
        | USB_DEV_MSK_EP1_IN
        | USB_DEV_MSK_EP1_OUT);

    if(intr || ep_intr) 
    {
        USB_interrupt();
    }

}


/******************************************************************************* 
*     : USB_task
*       : usb¼
*       : 
*       : ޡ
*  ֵ   : SUCCESSED:ɹ
*           : FAILED:ʧ
*******************************************************************************/
int USB_task(void)
{
    COMMAND_QUEUE *cur_cmd = NULL;
    struct USB_EP_OUT_REG *out_reg = NULL;
    struct USB_DEV_REG *dev_reg = usbd.dev_reg;
    char *s = NULL;
    long pUsbDelay = 0;

    s = getenv ("bootdelay");
    pUsbDelay = s ? (int)simple_strtol(s, NULL, 10) : CONFIG_BOOTDELAY;
    pUsbDelay *= 2;
    gUsbDelay = get_timer(0)+ pUsbDelay*CONFIG_SYS_HZ;
    
    cur_cmd = NULL;
    transfer.task_finished = 0;
    printf("*****please chose the file sending by usb*****.\n");
    while(1) 
    {
        USB_isrProcess();

        if(ctrlc())
        {
            printf("\n ...cancel by user...\n");
            gUsbStartSend = 0;
            gRxState = COMMAND_PHASE;
            USB_disconnect();

            return SUCCESSED;
        }

        if(gUsbStartSend == 0)
        {
            if(ctrlc())
            {
                USB_disconnect();
                return SUCCESSED;
            }

            if(get_timer(0) > gUsbDelay)
            {
                USB_disconnect();
                return SUCCESSED;
            }
        }
        
        /* issuing RX DMA enable all the time if in RCV_DATA */
        if(gRxState == RCV_DATA_PHASE)
        {
            vic_disable(USB_INT_VEC);
            out_reg = usbd.ep[BULK_OUT];
            _clean_flush_d_cache();
            
            /* if descriptor is ready to receive */
            if(!(out_reg->descriptor_p->status & USB_DMA_BUF_STS))
            {
                dev_reg->control |= USB_DEV_RCV_DMA_EN;
            }
            vic_enable(USB_INT_VEC);
        }

        /* the rsp_index is set caused by previous command,
            and will be cleared in ISR */
        if(qindex.rsp_index) 
        {
            /* a timeout is needed ? */
            continue;
        }

        /* If no current command or current command has finished,
            * try to dequeue one from command queue. */
        if ((!cur_cmd) || (cur_cmd->pendding == 0)) 
        {
            if (qindex.cmd_start == qindex.cmd_end)
            {
                /* Command queue is empty */
                cur_cmd = NULL;
                continue;
            } 
            else 
            {
                /* get a new command from command queue */
                qindex.cmd_start ++;
                qindex.cmd_start %= QUEUE_DEPTH;
                cur_cmd = (COMMAND_QUEUE *)&command[qindex.cmd_start];
            }
        }

        /* process the pendding or new command */
        if(cur_cmd->TX_RX == RX_DIR) 
        {		

            /* RX_DIR */
            if (cur_cmd->pipe == BULK_OUT)
            {
                USB_processBulkOut(cur_cmd);
            }
        }
        else if (cur_cmd->TX_RX == TX_DIR) 
        {	

            /* TX_DIR */
            if (cur_cmd->pipe == BULK_IN)
            {
                USB_processBulkIn(cur_cmd);
            }
            else if (cur_cmd->pipe == CTRL_IN)
            {
                USB_processCtrlIn(cur_cmd);
            }
        }

 
    } /* finish_download */

    return SUCCESSED;

}

/******************************************************************************* 
*     : USB_doReboot
*       :  usb࣬ϵͳ
*       : 
*       : 
*  ֵ   :  
*******************************************************************************/
static void USB_doReboot(void)
{
    disable_interrupts();
    _clean_flush_all_cache();
    _disable_icache();
    _disable_dcache();

    rct_reset_chip();

	while (1);
}


/******************************************************************************* 
*  	: USB_downloadFile
*    	:  usbļaddrַָ
*    	: addr: Ŀַ
*    	: 
*  ֵ	:  
*******************************************************************************/
void USB_downloadFile (u32 addr)
{
    unsigned int retry_counter;

    printf("Target address: 0x%x \n",addr);

    /* run the background task */
    transfer.rx.addr = addr;
    retry_counter = 0;
    while(USB_task() == FAILED) 
    {
        retry_counter ++;
        if(retry_counter == RETRY_COUNT) 
        {
            printf("Retry failed\r\n");
            USB_doReboot();
        }
    }
}


/******************************************************************************* 
*  	: USB_downloadKernel
*    	:  ںصַָ
*    	: is_fw_prog
*    	: 
*  ֵ	:  
*******************************************************************************/
static void USB_downloadKernel(int is_fw_prog)
{

    printf("============================= \r\n");
    printf("USB download %s \n",_VERSION_);
    printf("Built @ %s  %s \n",__DATE__,__TIME__);
    printf("============================= \r\n");


    if (is_fw_prog) 
    {
        printf("Download 1 firmware programming file\r\n");
        USB_downloadFile (MEMFWPROG_RAM_START);
        printf("firmware program is loaded\r\n");
    } 
    else 
    {
        printf("Download 4 kernel files\r\n");
        USB_downloadFile (MEMFWPROG_RAM_START);
        printf("Prkernel is loaded\r\n");

        /* Flow broken if PC application ask to finish */
        if (transfer.flag & FLAG_LAST_TRANS)
        {
            return;
        }
        
        USB_downloadFile(DRAM_START_ADDR + 0x03e00000);
        printf("code is loaded\r\n");

        USB_downloadFile(DRAM_START_ADDR + 0x03f00000);
        printf("memd is loaded\r\n");

        USB_downloadFile(DRAM_START_ADDR + 0x03eb9c00);
        printf("default_bin is loaded\r\n");
    }
}


/******************************************************************************* 
*  	: USB_testDownload
*    	:  USBʱ
*    	:   
*    	: 
*  ֵ	:  
*******************************************************************************/
static void USB_testDownload (void)
{
    u32 addr = MEMFWPROG_RAM_START;
    u32 mem_start = DRAM_START_ADDR;
    u32 mem_end = DRAM_END_ADDR;
    u32 mem_jump = 0x500000;	
    u32 max_file_size = 0x400000;	
    int i;

    printf("USB download test:\r\n");
    for (i = 0; i < TEST_NUM; i++) 
    {
        USB_downloadFile(addr);

        /* Change and wrapper memory address */
        addr += mem_jump;
        if ((addr + max_file_size) > mem_end) 
        {
            addr += mem_jump;
            addr -= (mem_end - mem_start);
        }
        printf("test #%x", i);
    }
    printf("test times=%x", i);
}


/******************************************************************************* 
*  	: USB_test
*    	: usbԹģʽӿ
*    	: flagжǷΪusbģʽ 
*    	:  ޡ
*  ֵ	:  
*******************************************************************************/
static void USB_test (int flag)
{
    if (flag & USB_FLAG_TEST_DOWNLOAD)
    {
        USB_testDownload ();
    }
}


/******************************************************************************* 
*  	: usb_download
*    	: usbģ
*    	: addr: Ŀַ
*         : exec: غúǷaddrУ0ʾУ1ʾ
*         : flag: ر־
*    	:  ޡ
*  ֵ	:  
*******************************************************************************/
u32 USB_download(void *addr, int exec, int flag)
{
    void (*jump)(void) = addr;
    u32 connected_at_boot;

    enable_usb = 1;
    connected_at_boot = exec;

    /* USB initialization */
    USB_init ();

    /* Handling transfer request */
    if (flag & USB_FLAG_TEST_MASK) 
    {

        USB_test (flag);

    } else if ((flag & USB_FLAG_KERNEL) 
        || (flag & USB_FLAG_FW_PROG))
    {

        USB_downloadKernel (flag & USB_FLAG_FW_PROG);

        /* disable TX/RX DMA */
        usbd.dev_reg->control &= 0xfffffff3;

        /* mask device/endpoint interrupt */
        usbd.dev_reg->intrmask = 0xffffffff;
        usbd.dev_reg->ep_intrmask = 0xffffffff;

        /* set soft-disconnect
            * set_softdisc() got problem because it set the VIC again    */
        usbd.dev_reg->control = UDC_CTRL_SD;	// Soft disconnect
        usbd.dev_reg->config = UDC_CFG_RWKP;	// Enable remote wakeup

    }
    else if (flag & USB_FLAG_MEMORY)
    {
        transfer.rx.addr = (u32) addr;
        gLoadAddr = (long)addr;
        USB_task();

    } 
    else 
    {
        while ((transfer.flag & FLAG_LAST_TRANS) == 0)
        {
            USB_task ();
        }
        
        if (transfer.flag & FLAG_JUMP_EXEC) 
        {
            exec = 1;
            jump = (void *) transfer.jump_address;
        }

        /* disable TX/RX DMA */
        usbd.dev_reg->control &= 0xfffffff3;

        /* mask device/endpoint interrupt */
        usbd.dev_reg->intrmask = 0xffffffff;
        usbd.dev_reg->ep_intrmask = 0xffffffff;

        /* set soft-disconnect
        * set_softdisc() got problem because it set the VIC again
        */
        usbd.dev_reg->control = UDC_CTRL_SD;	// Soft disconnect
        usbd.dev_reg->config = UDC_CFG_RWKP;	// Enable remote wakeup
    }

    /* Tranfer finished */
    if(!connected_at_boot) 
    {
        usbd.dev_reg->config = 0;
        usbd.dev_reg->control = 0;
        usbd.dev_reg->intr = 0x7f;
        usbd.dev_reg->ep_intr = 0xffffffff;
        usbd.dev_reg->intrmask = 0x7f;
        usbd.dev_reg->ep_intrmask = 0xffffffff;
    }
    return transfer.total_length;
}

 
/******************************************************************************* 
*  	: USB_enableReg
*    	:  ʹusbĴд
*    	: 
*    	: 
*  ֵ	:  
*******************************************************************************/
static void USB_enableReg(void)
{
	USB_pllInit();
}

/******************************************************************************* 
*     : USB_disconnect
*       : Ͽusb豸ósuspend״̬
*       : 
*       : 
*  ֵ   : 
*******************************************************************************/
static void USB_disconnect(void)
{
    USB_enableReg();
    USB_SoftDisc();
    rct_suspend_usb();
}


/******************************************************************************* 
*  	: USB_bootLogo
*    	:  usblogo
*    	: 
*    	: 
*  ֵ	:  
*******************************************************************************/
static void USB_bootLogo(void)
{
    printf("core freq: %d \n", get_core_bus_freq_hz());

#if (RCT_DRAM_CLK_SRC_CORE == 1)
    printf("ddr freq: %d \n", get_ddr_freq_hz());

#endif

#if (RCT_IDSP_CLK_SRC_CORE == 1)
    printf("idsp freq: %d \n",get_idsp_freq_hz());
#endif

}

 /******************************************************************************* 
*  	: usb_boot
*    	:  usb
*    	: usbdl_mode: downloadģʽ
*    	: 
*  ֵ	:  
*******************************************************************************/
void usb_boot(u8 usbdl_mode)
{
    int flag = 0;

    USB_bootLogo();

    USB_enableReg();
    vic_disable(TIMER1_INT_VEC);
    switch (usbdl_mode)
    {
        case 0x80:	
        {
            flag = USB_FLAG_FW_PROG;
            break;
        }
        
        case USB_DL_NORMAL:
        {
            flag = USB_FLAG_KERNEL;
            break;
        }
        
        case USB_DL_DIRECT_USB:
        {
            flag = 0;
            break;
        }
        
        default:
        {
            printf("Error: no this download mode.\n");
            break;
        }
    }

    printf("Entered USB download mode ( %d ) ...\n",usbdl_mode);

    USB_download((void *) MEMFWPROG_RAM_START, 1, flag);
}



